/************************************************************************
*              decrypt.cpp                                              *
* This class adds some simple block decryption/encryption with file     *
* patching to Borg. By storing decryptors in blocks it is possible to   *
* reconstruct the file when it is saved to database and reloaded, even  *
* if some patches were applied to the file and some were not.           *
* Added in Borg 2.19                                                    *
* NB Any future general file patching will need to be included in this  *
* class and saved in a similar way. This opens up the way to decrypting *
* patching and reencrypting within Borg :)                              *
* Current decryptors are fairly simple, but when used in combination    *
* they are very powerful. The Xadd is a bit obscure, but could be       *
* simply modified as needed, and recompiled for some powerful routines. *
************************************************************************/
#include <windows.h>
#include <stdio.h>

#include "schedule.h"
#include "decrypt.h"
#include "dasm.h"
#include "range.h"
#include "data.h"
#include "disio.h"
#include "disasm.h"
#include "fileload.h"
#include "menuids.rh"

/************************************************************************
* global variables                                                      *
* these are used within the main decryption dialog as well as the       *
* decryption class                                                      *
************************************************************************/
dectype lastdec=decxor;
ditemtype lastditem=decbyte;
char lastvalue[20],lastseg[20],lastoffset[20];
BOOL patchexe=FALSE;
BOOL loading_db=FALSE;

/************************************************************************
* forward declarations                                                  *
************************************************************************/
int dec_comp_func(listitem i,listitem j);
void dec_del_func(listitem i);
BOOL FAR PASCAL decbox(HWND hdwnd,UINT message,WPARAM wParam,LPARAM lParam);

/************************************************************************
* constructor function                                                  *
* - sets compare and deletion function for the decryptors and resets a  *
*   few global variables                                                *
************************************************************************/
decrypt::decrypt()
{ setcomparefunc(dec_comp_func);
  setdelfunc(dec_del_func);
  nextitemnum=1;
  lastvalue[0]=0;
  lastseg[0]=0;
  lastoffset[0]=0;
}

/************************************************************************
* destructor function                                                   *
* - currently null                                                      *
************************************************************************/
decrypt::~decrypt()
{
}

/************************************************************************
* compare function for decryptor                                        *
* - these are simply stored in uid order, the uid being increased each  *
*   time a new one is applied                                           *
************************************************************************/
int dec_comp_func(listitem i,listitem j)
{ if(((declist *)i)->uid==((declist *)j)->uid)
	 return 0;
  if(((declist *)i)->uid>((declist *)j)->uid)
	 return 1;
  return -1;
}

/************************************************************************
* delete function for decryptor                                         *
* - simply deletes the decryptor                                        *
************************************************************************/
void dec_del_func(listitem i)
{ delete (declist *)i;
}

/************************************************************************
* add_decrypted                                                         *
* - just adds another item to the decrypt list                          *
* - the decrypt list is simply a list of blocks and how they were       *
*   changed and whether the exe was patched                             *
* - the list is just to enable reconstruction of the state of the file  *
*   on saving and loading databases with decryptors which may or may    *
*   not have been saved to the exe file                                 *
************************************************************************/
dword decrypt::add_decrypted(lptr dstart,lptr dend,dectype t,ditemtype ditem,dword val,lptr adr,BOOL patchedexe)
{ declist *ndec;
  ndec=new struct declist;
  ndec->dec_start=dstart;
  ndec->dec_end=dend;
  ndec->typ=t;
  ndec->dlength=ditem;
  ndec->value=val;
  ndec->addr=adr;
  ndec->patch=patchedexe;
  ndec->uid=nextitemnum;
  nextitemnum++;
  addto((listitem)ndec);
  return ndec->uid;
}

/************************************************************************
* process_dec                                                           *
* - this processes a decryptor given the uid, actually applying it to   *
*   the file in memory. If a block contains any disassembly then this   *
*   is also deleted.                                                    *
************************************************************************/
void decrypt::process_dec(dword dec_id)
{ declist fnd,*patch;
  dsegitem *pseg,*aseg;
  lptr cpos;
  int plen,ctr;
  dword doitval,lval,tval;
  fnd.uid=dec_id;
  patch=(declist *)find((listitem)&fnd);
  if(patch==NULL) return;
  if(patch->uid!=dec_id) return;
  pseg=dta.findseg(patch->dec_start);
  if(pseg==NULL) return;
  ctr=0;
  switch(patch->dlength)
  { case decbyte:
      plen=1;
      break;
    case decword:
      plen=2;
      break;
    case decdword:
      plen=4;
      break;
    case decarray:
      plen=1;
      aseg=dta.findseg(patch->addr);
      if(aseg==NULL) return;
      ctr=patch->addr-aseg->addr;
      break;
    default:
      plen=1;
      break;
  }
  cpos=patch->dec_start;
  lval=patch->value;
  while(cpos<=patch->dec_end)
  { // check within seg, and move to the next seg if we arent
    while(cpos>pseg->addr+(pseg->size-plen))
    { dta.nextseg(&cpos);
      if(!cpos.segm) break;
      if(cpos>patch->dec_end) break;
      pseg=dta.findseg(cpos);
    }
    if(!cpos.segm) break;
    if(cpos>patch->dec_end) break;
    switch(plen)
    { case 1:
        doitval=((byte *)(pseg->data+(cpos-pseg->addr)))[0];
        break;
      case 2:
        doitval=((word *)(pseg->data+(cpos-pseg->addr)))[0];
        break;
      case 4:
        doitval=((dword *)(pseg->data+(cpos-pseg->addr)))[0];
        break;
    }
    if(patch->dlength==decarray)
    { if(ctr+plen>aseg->size) break;
      switch(plen)
      { case 1:
          patch->value=((byte *)(aseg->data+ctr))[0];
          break;
        case 2:
          patch->value=((word *)(aseg->data+ctr))[0];
          break;
        case 4:
          patch->value=((dword *)(aseg->data+ctr))[0];
          break;
      }
    }
    switch(patch->typ)
    { case decxor:
        doitval=doitval^patch->value;
        break;
      case decmul:
        doitval=doitval*patch->value;
        break;
      case decadd:
        doitval=doitval+patch->value;
        break;
      case decsub:
        doitval=doitval-patch->value;
        break;
      case decxadd:
        tval=doitval;
        doitval=lval;
        lval=tval;
        doitval=doitval+lval;
        break;
      case decrot:
        switch(plen)
        { case 1:
            doitval=(doitval<<(patch->value&0x07))+(doitval>>(8-(patch->value&0x07)));
            break;
          case 2:
            doitval=(doitval<<(patch->value&0x0f))+(doitval>>(16-(patch->value&0x0f)));
            break;
          case 4:
            doitval=(doitval<<(patch->value&0x1f))+(doitval>>(32-(patch->value&0x1f)));
            break;
        }
        break;
      default:
        break;
    }
    switch(plen)
    { case 1:
        ((byte *)(pseg->data+(cpos-pseg->addr)))[0]=(byte)doitval;
        break;
      case 2:
        ((word *)(pseg->data+(cpos-pseg->addr)))[0]=(word)doitval;
        break;
      case 4:
        ((dword *)(pseg->data+(cpos-pseg->addr)))[0]=doitval;
        break;
    }
    cpos+=plen;
    ctr+=plen;
  }
  if(!loading_db) dsm.undefineblock(patch->dec_start,patch->dec_end);
  dio.updatewindowifwithinrange(patch->dec_start,patch->dec_end);
}

/************************************************************************
* exepatch                                                              *
* - given a uid this steps through a decryptor and writes the patch to  *
*   the exe file.                                                       *
************************************************************************/
void decrypt::exepatch(dword dec_id)
{ declist fnd,*patch;
  dsegitem *pseg;
  lptr cpos;
  int plen;
  dword doitval;
  fnd.uid=dec_id;
  patch=(declist *)find((listitem)&fnd);
  if(patch==NULL) return;
  if(patch->uid!=dec_id) return;
  pseg=dta.findseg(patch->dec_start);
  switch(patch->dlength)
  { case decbyte:
      plen=1;
      break;
    case decword:
      plen=2;
      break;
    case decdword:
      plen=4;
      break;
    case decarray:
      plen=1;
      break;
    default:
      plen=1;
      break;
  }
  cpos=patch->dec_start;
  while(cpos<=patch->dec_end)
  { // check within seg, and move to the next seg if we arent
    while(cpos>pseg->addr+(pseg->size-plen))
    { dta.nextseg(&cpos);
      if(!cpos.segm) break;
      if(cpos>patch->dec_end) break;
      pseg=dta.findseg(cpos);
    }
    if(!cpos.segm) break;
    if(cpos>patch->dec_end) break;
    doitval=floader.fileoffset(cpos);
    // write patch
    switch(plen)
    { case 1:
        floader.patchfile(doitval,1,pseg->data+(cpos-pseg->addr));
        break;
      case 2:
        floader.patchfile(doitval,2,pseg->data+(cpos-pseg->addr));
        break;
      case 4:
        floader.patchfile(doitval,4,pseg->data+(cpos-pseg->addr));
        break;
      default:
        floader.patchfile(doitval,1,pseg->data+(cpos-pseg->addr));
        break;
    }
    cpos+=plen;
  }
}

/************************************************************************
* process_reload                                                        *
* - given a uid this steps through a patch a re-reads the bytes in that *
*   would have been changed. This is used in file reconstruction        *
************************************************************************/
void decrypt::process_reload(dword dec_id)
{ declist fnd,*patch;
  dsegitem *pseg;
  lptr cpos;
  int plen;
  dword doitval;
  fnd.uid=dec_id;
  patch=(declist *)find((listitem)&fnd);
  if(patch==NULL) return;
  if(patch->uid!=dec_id) return;
  pseg=dta.findseg(patch->dec_start);
  switch(patch->dlength)
  { case decbyte:
      plen=1;
      break;
    case decword:
      plen=2;
      break;
    case decdword:
      plen=4;
      break;
    case decarray:
      plen=1;
      break;
    default:
      plen=1;
      break;
  }
  cpos=patch->dec_start;
  while(cpos<=patch->dec_end)
  { // check within seg, and move to the next seg if we arent
    while(cpos>pseg->addr+(pseg->size-plen))
    { dta.nextseg(&cpos);
      if(!cpos.segm) break;
      if(cpos>patch->dec_end) break;
      pseg=dta.findseg(cpos);
    }
    if(!cpos.segm) break;
    if(cpos>patch->dec_end) break;
    doitval=floader.fileoffset(cpos);
    // write patch
    switch(plen)
    { case 1:
        floader.reloadfile(doitval,1,pseg->data+(cpos-pseg->addr));
        break;
      case 2:
        floader.reloadfile(doitval,2,pseg->data+(cpos-pseg->addr));
        break;
      case 4:
        floader.reloadfile(doitval,4,pseg->data+(cpos-pseg->addr));
        break;
      default:
        floader.reloadfile(doitval,1,pseg->data+(cpos-pseg->addr));
        break;
    }
    cpos+=plen;
  }
}

/************************************************************************
* savedb                                                                *
* - this saves the list of decryptors to the database                   *
************************************************************************/
BOOL decrypt::savedb(savefile *sf)
{ dword ndecs;
  declist *currdec;
  ndecs=numlistitems();
  resetiterator();
  if(!sf->swrite(&ndecs,sizeof(dword)))return FALSE;
  while(ndecs)
  { currdec=(declist *)nextiterator();
	 if(!sf->swrite(currdec,sizeof(declist)))return FALSE;
	 ndecs--;
  }
  return TRUE;
}

/************************************************************************
* loaddb                                                                *
* - here we reload the list of decryptors and apply each to the file in *
*   memory. If we find a decryptor which was saved to disk then we      *
*   reload that block from the exe file. In this way after all of the   *
*   decryptors have been loaded we have synchronised the file in memory *
*   to the file on disk, plus any further patches made but not written  *
*   to disk. [Any byte in the file was synchronised to the file at the  *
*   time of the last patch which was written to file. Subsequent        *
*   patches have been made to memory only, and are just redone. So the  *
*   loaded file is in the same state as when the database was saved]    *
************************************************************************/
BOOL decrypt::loaddb(savefile *sf)
{ dword ndecs,num;
  declist *currdec;
  if(!sf->sread(&ndecs,sizeof(dword),&num))return FALSE;
  while(ndecs)
  { currdec=new declist;
	 if(!sf->sread(currdec,sizeof(declist),&num))return FALSE;
	 addto((listitem)currdec);
    nextitemnum=currdec->uid+1;
    loading_db=TRUE;
    if(!currdec->patch)
      process_dec(currdec->uid);
    else
      process_reload(currdec->uid);
    loading_db=FALSE;
	 ndecs--;
  }
  return TRUE;
}

/************************************************************************
* dialg                                                                 *
* - we stop the thread whilst displaying the decryptor dialog and doing *
*   the patch                                                           *
************************************************************************/
void decrypt::dialg(void)
{ if(!blk.checkblock()) return;
  scheduler.stopthread();
  DialogBox(hInst,MAKEINTRESOURCE(Decrypt_Dialog),mainwindow,(DLGPROC)decbox);
  scheduler.continuethread();
}

/************************************************************************
* decbox                                                                *
* - the decryptor dialog, it only allows patching if the file is not    *
*   readonly, and adds the decryptor to the list and calls the process  *
*   and patch functions.                                                *
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par
#endif
BOOL FAR PASCAL decbox(HWND hdwnd,UINT message,WPARAM wParam,LPARAM lParam)
{ static HBRUSH hbrush;
  RECT drect;
  dword dec_id,d_val;
  lptr d_adr;
  switch(message)
  { case WM_COMMAND:
		{	switch(wParam)
		  { case IDOK:
				if(SendDlgItemMessage(hdwnd,idc_xor,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decxor;
				else if(SendDlgItemMessage(hdwnd,idc_mul,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decmul;
				else if(SendDlgItemMessage(hdwnd,idc_add,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decadd;
				else if(SendDlgItemMessage(hdwnd,idc_sub,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decsub;
				else if(SendDlgItemMessage(hdwnd,idc_rot,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decrot;
				else if(SendDlgItemMessage(hdwnd,idc_xadd,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastdec=decxadd;
            else lastdec=decnull;
				if(SendDlgItemMessage(hdwnd,idc_byte,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastditem=decbyte;
				else if(SendDlgItemMessage(hdwnd,idc_word,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastditem=decword;
				else if(SendDlgItemMessage(hdwnd,idc_dword,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastditem=decdword;
				else if(SendDlgItemMessage(hdwnd,idc_array,BM_GETCHECK,(WPARAM)0,(LPARAM)0))
				  lastditem=decarray;
            else lastditem=decbyte;
				SendDlgItemMessage(hdwnd,idc_value,WM_GETTEXT,(WPARAM)18,(LPARAM)lastvalue);
				SendDlgItemMessage(hdwnd,idc_arrayseg,WM_GETTEXT,(WPARAM)18,(LPARAM)lastseg);
				SendDlgItemMessage(hdwnd,idc_arrayoffset,WM_GETTEXT,(WPARAM)18,(LPARAM)lastoffset);
				if(IsDlgButtonChecked(hdwnd,idc_applytoexe)) patchexe=TRUE;
            else patchexe=FALSE;
				sscanf(lastvalue,"%lx",&d_val);
				sscanf(lastseg,"%lx",&d_adr.segm);
				sscanf(lastoffset,"%lx",&d_adr.offs);
            if((options.readonly)&&(patchexe))
            { patchexe=FALSE;
              MessageBox(mainwindow,"File opened readonly - unable to patch","Borg Message",MB_OK);
            }
            dec_id=decrypter.add_decrypted(blk.top,blk.bottom,lastdec,lastditem,d_val,d_adr,patchexe);
            decrypter.process_dec(dec_id);
            if(patchexe)
              decrypter.exepatch(dec_id);
				EndDialog(hdwnd,NULL);
				return TRUE;
			 case IDCANCEL:
				EndDialog(hdwnd,NULL);
				return TRUE;
			 default:
				break;
		  }
		}
		break;
	 case WM_INITDIALOG:
		GetWindowRect(hdwnd,&drect);
		MoveWindow(hdwnd,((mainwnd.right+mainwnd.left)-(drect.right-drect.left))/2,
		  ((mainwnd.bottom+mainwnd.top)-(drect.bottom-drect.top))/2,drect.right-drect.left,drect.bottom-drect.top,TRUE);
		hbrush=CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
      switch(lastdec)
      { case decxor:
			 SendDlgItemMessage(hdwnd,idc_xor,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decmul:
			 SendDlgItemMessage(hdwnd,idc_mul,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decadd:
			 SendDlgItemMessage(hdwnd,idc_add,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decsub:
          SendDlgItemMessage(hdwnd,idc_sub,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decrot:
			 SendDlgItemMessage(hdwnd,idc_rot,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decxadd:
			 SendDlgItemMessage(hdwnd,idc_xadd,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        default:
			 SendDlgItemMessage(hdwnd,idc_xor,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
      }
      switch(lastditem)
      { case decbyte:
			 SendDlgItemMessage(hdwnd,idc_byte,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decword:
			 SendDlgItemMessage(hdwnd,idc_word,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decdword:
			 SendDlgItemMessage(hdwnd,idc_dword,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        case decarray:
			 SendDlgItemMessage(hdwnd,idc_array,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
        default:
			 SendDlgItemMessage(hdwnd,idc_byte,BM_SETCHECK,(WPARAM)1,(LPARAM)0);
          break;
      }
		SendDlgItemMessage(hdwnd,idc_value,WM_SETTEXT,(WPARAM)0,(LPARAM)lastvalue);
		SendDlgItemMessage(hdwnd,idc_arrayseg,WM_SETTEXT,(WPARAM)0,(LPARAM)lastseg);
		SendDlgItemMessage(hdwnd,idc_arrayoffset,WM_SETTEXT,(WPARAM)0,(LPARAM)lastoffset);
      CheckDlgButton(hdwnd,idc_applytoexe,patchexe);
		SetFocus(GetDlgItem(hdwnd,idc_value));
		return 0;
	 case WM_DESTROY:
		DeleteObject(hbrush);
		return 0;
	 case WM_CTLCOLORSTATIC:
	 case WM_CTLCOLORBTN:
	 case WM_CTLCOLORDLG:
	 case WM_SYSCOLORCHANGE:
		SetBkColor((HDC)wParam,GetSysColor(COLOR_BTNFACE));
		return (LRESULT)hbrush;
  }
  return FALSE;
}
#ifdef __BORLANDC__
#pragma warn +par
#endif

