/************************************************************************
*				   gname.cpp                                                *
* generic name class                                                    *
* the basic gname class consists of a name and a location, and these    *
* are the basic management functions for the class. The class is        *
* inherited by names, exports and imports which are all treated very    *
* slightly differently. These are essentially the common routines       *
* The gname class itself inherits the list class for management of the  *
* array of named locations.                                             *
************************************************************************/

#include <windows.h>

#include "gname.h"
#include "disasm.h"
#include "data.h"
#include "mainwind.h"
#include "debug.h"

/************************************************************************
* forward declarations for the list compare and deletion functions      *
************************************************************************/
int gnamecfunc(listitem i,listitem j);
void gnamedfunc(listitem i);

/************************************************************************
* gname constructor function                                            *
* - here we simply set up the compare and deletion functions for the    *
*   underlying list of names                                            *
************************************************************************/
gname::gname()
{ setcomparefunc(gnamecfunc);
  setdelfunc(gnamedfunc);
}

/************************************************************************
* gname destructor function                                             *
* - empty. the deletion of the list is carried out by the list class    *
*   and by the deletion function                                        *
************************************************************************/
gname::~gname()
{
}

/************************************************************************
* gname compare function                                                *
* - the compare function for ordering the list of names                 *
* - the names are kept in location order                                *
************************************************************************/
int gnamecfunc(listitem i,listitem j)
{ if(((gnameitem *)i)->addr==((gnameitem *)j)->addr)
	 return 0;
  if(((gnameitem *)i)->addr>((gnameitem *)j)->addr)
	 return 1;
  return -1;
}

/************************************************************************
* gname delete function                                                 *
* - first we delete any name string, and then the actual name/location  *
*   struct                                                              *
************************************************************************/
void gnamedfunc(listitem i)
{ // delete name.
  delete ((gnameitem *)i)->name;
  // delete gnameitem.
  delete (gnameitem *)i;
}

/************************************************************************
* addname                                                               *
* - this is to add a name to the list of names.                         *
* - if the address is not covered in our list of segments then we       *
*   ignore the request                                                  *
* - we check that it is ok to name the location before naming it. This  *
*   basically ensures that names cannot be added in the middle of a     *
*   disassembled instruction, etc. It should not affect imports/exports *
*   since these will be named prior to disassembly                      *
* - the name is copied and a new string created for it, so the calling  *
*   function must delete any memory created to hold the name            *
* - if the same location is named twice then the first name is deleted  *
* - the name is added to the disassembly so that it appears in the      *
*   listing.                                                            *
* - naming a location with "" results in any name being deleted         *
************************************************************************/
void gname::addname(lptr loc,char *nm)
{ gnameitem *newname,*checkdup;
  dsegitem *l_ds;
  // check for non-existant address added v2.20
  l_ds=dta.findseg(loc);
  if(l_ds==NULL) return;
  if(!dsm.oktoname(loc))return; // check not in the middle of an instruction.
  newname=new gnameitem;
  newname->name=new char[strlen(nm)+1];
  strcpy(newname->name,nm);
  demangle(&newname->name);
  newname->addr=loc;
  checkdup=(gnameitem *)find((listitem)newname);
  // just add it once.
  if(checkdup!=NULL)
	 if(checkdup->addr==loc)
	 { delete checkdup->name;
		checkdup->name=newname->name;
		delete newname;
		dsm.delcomment(loc,dsmnameloc);
		if(strlen(checkdup->name))
		  dsm.discomment(loc,dsmnameloc,(byte *)checkdup->name);
		else
		{ delfrom((listitem)checkdup);
		}
		return;
	 }
  if(!strlen(newname->name))
  { // bugfix by Mark Ogden
	 delete newname->name;
    delete newname;
    return;
  }
  addto((listitem)newname);
  dsm.discomment(loc,dsmnameloc,(byte *)newname->name);
  // NB - no need for this anymore. Need to ensure that
  // printing of names is done as :
  // if names.isname() ...
  // else if imports.isname()...
  // else if export.isexport()...
  // so names aren't printed twice, and import names,etc are retained always
  //if(dta.findseg(loc)!=NULL)name.addname(loc,nm);
}

/************************************************************************
* isname                                                                *
* - returns TRUE if loc has a name                                      *
************************************************************************/
BOOL gname::isname(lptr loc)
{ gnameitem checkit,*findit;
  checkit.addr=loc;
  findit=(gnameitem *)find((listitem)&checkit);
  if(findit!=NULL) if(findit->addr==loc) return TRUE;
  return FALSE;
}

/************************************************************************
* isname                                                                *
* - prints name to last buffer location in mainwindow buffering array   *
* - use with isname, for example:                                       *
*   if(name.isname(loc))name.printname(loc); etc                        *
************************************************************************/
void gname::printname(lptr loc)
{ gnameitem checkit,*findit;
  checkit.addr=loc;
  findit=(gnameitem *)find((listitem)&checkit);
  if(findit!=NULL) LastPrintBuff("%s",findit->name);
}

/************************************************************************
* delname                                                               *
* - used as a simple name deleter for a given location                  *
* - also deletes the name from the disassembly listing                  *
************************************************************************/
void gname::delname(lptr loc)
{ gnameitem dname,*checkdup;
  dname.addr=loc;
  checkdup=(gnameitem *)find((listitem)&dname);
  if(checkdup!=NULL)
	 if(checkdup->addr==loc)
		delfrom((listitem)checkdup);
  dsm.delcomment(loc,dsmnameloc);
}

/************************************************************************
* savedb                                                                *
* - saves all the names in the list to the database file being saved.   *
*   this is in a one-pass compatible loading format. ie number of items *
*   followed by each item, and for strings the length of the string     *
*   followed by the string.                                             *
************************************************************************/
BOOL gname::savedb(savefile *sf)
{ dword nexps,nlen;
  gnameitem *currexp;
  nexps=numlistitems();
  resetiterator();
  if(!sf->swrite(&nexps,sizeof(dword)))return FALSE;
  while(nexps)
  { currexp=(gnameitem *)nextiterator();
	 if(!sf->swrite(&(currexp->addr),sizeof(lptr)))return FALSE;
	 nlen=strlen(currexp->name)+1;
	 if(!sf->swrite(&nlen,sizeof(dword)))return FALSE;
	 if(!sf->swrite(currexp->name,nlen))return FALSE;
	 nexps--;
  }
  return TRUE;
}

/************************************************************************
* loaddb                                                                *
* - loads the names from the database file and reconstructs the names   *
*   list                                                                *
************************************************************************/
BOOL gname::loaddb(savefile *sf)
{ dword nexps,num,nlen;
  gnameitem *currexp;
  if(!sf->sread(&nexps,sizeof(dword),&num))return FALSE;
  while(nexps)
  { currexp=new gnameitem;
	 if(!sf->sread(&(currexp->addr),sizeof(lptr),&num))return FALSE;
	 if(!sf->sread(&nlen,sizeof(dword),&num))return FALSE;
	 currexp->name=new char[nlen];
	 if(!sf->sread(currexp->name,nlen,&num))return FALSE;
	 addto((listitem)currexp);
	 nexps--;
  }
  return TRUE;
}

/************************************************************************
* demangle                                                              *
* - name demangler has currently been implemented as a generic function *
*   which is null, and is overridden by the import class. this enables  *
*   any name to call name.demangle() although this is currently unused  *
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par
#endif
// null function - overridden by import class.
void gname::demangle(char **name)
{
}
#ifdef __BORLANDC__
#pragma warn +par
#endif