/************************************************************************
*              proctab.cpp                                              *
* Tables of processor instructions. The whole of this file is full of   *
* tables that I constructed some time ago. The main instruction tables  *
* include three arguments for instructions, which was necessary to      *
* easily include instructions like shld r/m16,r16,CL. I was maybe       *
* overexcessive with the number of slightly different modr/ms that I    *
* included but the tables have served well, and when you build them up  *
* from scratch instruction by instruction you do actually find a lot of *
* slightly different encodings, and the modrm encodings generally       *
* indicate what is actually being referenced (like a 16:32 pointer in   *
* memory, etc). I included the Z80 processor because I knew the Z80     *
* processor quite well, but mostly just to show it isnt hard to do.     *
* At one point I decided that I needed a uid for each instruction to    *
* enable proper reconstruction of the disassembly from a saved database *
* and so each instruction also has a uid for this purpose.              *
* Flags for modrm may seem excessive, but consider encodings like 0fae  *
* which includes the strange sfence instruction with no arguments and a *
* code of 0fae /7, along with stmxcsr m32 with the encoding 0fae /3     *
* Hence the flag FLAGS_MODRM indicates a modrm byte. Any argument       *
* encodings may indicate usage of that byte, but may not be present at  *
* all.                                                                  *
* Whilst I havent found any errors in here for some time it would need  *
* a very careful analysis to find any which may be present. Certainly   *
* newer instruction encodings (mmx, kni) may need more work, and it is  *
* worth looking at decodings which should not be possible (eg forced    *
* memory operands only may require looking at as some possible          *
* instructions are illegal)                                             *
************************************************************************/

#include <windows.h>
#include "proctab.h"

proctable procnames[]=
{ {PROC_8086,"8086",tables86},
  {PROC_80286,"80286",tables86},
  {PROC_80386,"80386",tables86},
  {PROC_80486,"80486",tables86},
  {PROC_PENTIUM,"Pentium",tables86},
  {PROC_PENTIUMPRO,"Pentium Pro",tables86},
  {PROC_PENTMMX,"Pentium MMX",tables86},
  {PROC_PENTIUM2,"Pentium II with KNI",tables86},
  {PROC_Z80,"Z-80",tablesz80},
  {0,"",NULL}
};

char *reg32ascii[]={"eax","ecx","edx","ebx","esp","ebp","esi","edi"};
char *reg16ascii[]={"ax","cx","dx","bx","sp","bp","si","di"};
char *reg8ascii[]={"al","cl","dl","bl","ah","ch","dh","bh"};
char *regind16ascii[]={"bx+si","bx+di","bp+si","bp+di","si","di","bp","bx"};
char *regfascii[]={"st(0)","st(1)","st(2)","st(3)","st(4)","st(5)","st(6)","st(7)"};
char *regmascii[]={"mm0","mm1","mm2","mm3","mm4","mm5","mm6","mm7"};
char *regxascii[]={"xmm0","xmm1","xmm2","xmm3","xmm4","xmm5","xmm6","xmm7"};
char *regsascii[]={"es","cs","ss","ds","fs","gs","??","??"};
char *regcascii[]={"cr0","cr1","cr2","cr3","cr4","cr5","cr6","cr7"};
char *regdascii[]={"dr0","dr1","dr2","dr3","dr4","dr5","dr6","dr7"};
char *regtascii[]={"tr0","tr1","tr2","tr3","tr4","tr5","tr6","tr7"};
char *regzascii[]={"b","c","d","e","h","l","(hl)","a"};

// to be filled in.
asminstdata asm86[]=
{ {"add",0x00,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,1},
  {"add",0x01,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,2},
  {"add",0x01,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,3},
  {"add",0x02,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,4},
  {"add",0x03,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,5},
  {"add",0x03,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,6},
  {"add",0x04,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,7},
  {"add",0x05,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,8},
  {"add",0x05,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,9},
  {"push",0x06,PROC_FROM8086,0,ARG_REG_ES,ARG_NONE,ARG_NONE,10},
  {"pop",0x07,PROC_FROM8086,0,ARG_REG_ES,ARG_NONE,ARG_NONE,11},
  {"or",0x08,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,12},
  {"or",0x09,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,13},
  {"or",0x09,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,14},
  {"or",0x0a,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,15},
  {"or",0x0b,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,16},
  {"or",0x0b,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,17},
  {"or",0x0c,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,18},
  {"or",0x0d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,19},
  {"or",0x0d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,20},
  {"push",0x0e,PROC_FROM8086,0,ARG_REG_CS,ARG_NONE,ARG_NONE,21},
  {NULL,0x0f,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,22},  //subtable 0x0f
  {"adc",0x10,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,23},
  {"adc",0x11,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,24},
  {"adc",0x11,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,25},
  {"adc",0x12,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,26},
  {"adc",0x13,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,27},
  {"adc",0x13,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,28},
  {"adc",0x14,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,29},
  {"adc",0x15,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,30},
  {"adc",0x15,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,31},
  {"push",0x16,PROC_FROM8086,0,ARG_REG_SS,ARG_NONE,ARG_NONE,32},
  {"pop",0x17,PROC_FROM8086,0,ARG_REG_SS,ARG_NONE,ARG_NONE,33},
  {"sbb",0x18,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,34},
  {"sbb",0x19,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,35},
  {"sbb",0x19,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,36},
  {"sbb",0x1a,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,37},
  {"sbb",0x1b,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,38},
  {"sbb",0x1b,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,39},
  {"sbb",0x1c,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,40},
  {"sbb",0x1d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,41},
  {"sbb",0x1d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,42},
  {"push",0x1e,PROC_FROM8086,0,ARG_REG_DS,ARG_NONE,ARG_NONE,43},
  {"pop",0x1f,PROC_FROM8086,0,ARG_REG_DS,ARG_NONE,ARG_NONE,44},
  {"and",0x20,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,45},
  {"and",0x21,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,46},
  {"and",0x21,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,47},
  {"and",0x22,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,48},
  {"and",0x23,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,49},
  {"and",0x23,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,50},
  {"and",0x24,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,51},
  {"and",0x25,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,52},
  {"and",0x25,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,53},
  {"es:",0x26,PROC_FROM8086,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,54},
  {"daa",0x27,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,55},
  {"sub",0x28,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,56},
  {"sub",0x29,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,57},
  {"sub",0x29,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,58},
  {"sub",0x2a,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,59},
  {"sub",0x2b,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,60},
  {"sub",0x2b,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,61},
  {"sub",0x2c,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,62},
  {"sub",0x2d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,63},
  {"sub",0x2d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,64},
  {"cs:",0x2e,PROC_FROM8086,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,65},
  {"das",0x2f,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,66},
  {"xor",0x30,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,67},
  {"xor",0x31,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,68},
  {"xor",0x31,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,69},
  {"xor",0x32,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,70},
  {"xor",0x33,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,71},
  {"xor",0x33,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,72},
  {"xor",0x34,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,73},
  {"xor",0x35,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,74},
  {"xor",0x35,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,75},
  {"ss:",0x36,PROC_FROM8086,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,76},
  {"aaa",0x37,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,77},
  {"cmp",0x38,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,78},
  {"cmp",0x39,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,79},
  {"cmp",0x39,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,80},
  {"cmp",0x3a,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,81},
  {"cmp",0x3b,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,82},
  {"cmp",0x3b,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,83},
  {"cmp",0x3c,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,84},
  {"cmp",0x3d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,85},
  {"cmp",0x3d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,86},
  {"ds:",0x3e,PROC_FROM8086,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,87},
  {"aas",0x3f,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,88},
  {"inc",0x40,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_NONE,ARG_NONE,89},
  {"inc",0x40,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_NONE,ARG_NONE,90},
  {"inc",0x41,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_CX,ARG_NONE,ARG_NONE,91},
  {"inc",0x41,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_CX,ARG_NONE,ARG_NONE,92},
  {"inc",0x42,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DX,ARG_NONE,ARG_NONE,93},
  {"inc",0x42,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DX,ARG_NONE,ARG_NONE,94},
  {"inc",0x43,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BX,ARG_NONE,ARG_NONE,95},
  {"inc",0x43,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BX,ARG_NONE,ARG_NONE,96},
  {"inc",0x44,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SP,ARG_NONE,ARG_NONE,97},
  {"inc",0x44,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SP,ARG_NONE,ARG_NONE,98},
  {"inc",0x45,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BP,ARG_NONE,ARG_NONE,99},
  {"inc",0x45,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BP,ARG_NONE,ARG_NONE,100},
  {"inc",0x46,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SI,ARG_NONE,ARG_NONE,101},
  {"inc",0x46,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SI,ARG_NONE,ARG_NONE,102},
  {"inc",0x47,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DI,ARG_NONE,ARG_NONE,103},
  {"inc",0x47,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DI,ARG_NONE,ARG_NONE,104},
  {"dec",0x48,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_NONE,ARG_NONE,105},
  {"dec",0x48,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_NONE,ARG_NONE,106},
  {"dec",0x49,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_CX,ARG_NONE,ARG_NONE,107},
  {"dec",0x49,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_CX,ARG_NONE,ARG_NONE,108},
  {"dec",0x4a,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DX,ARG_NONE,ARG_NONE,109},
  {"dec",0x4a,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DX,ARG_NONE,ARG_NONE,110},
  {"dec",0x4b,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BX,ARG_NONE,ARG_NONE,111},
  {"dec",0x4b,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BX,ARG_NONE,ARG_NONE,112},
  {"dec",0x4c,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SP,ARG_NONE,ARG_NONE,113},
  {"dec",0x4c,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SP,ARG_NONE,ARG_NONE,114},
  {"dec",0x4d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BP,ARG_NONE,ARG_NONE,115},
  {"dec",0x4d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BP,ARG_NONE,ARG_NONE,116},
  {"dec",0x4e,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SI,ARG_NONE,ARG_NONE,117},
  {"dec",0x4e,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SI,ARG_NONE,ARG_NONE,118},
  {"dec",0x4f,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DI,ARG_NONE,ARG_NONE,119},
  {"dec",0x4f,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DI,ARG_NONE,ARG_NONE,120},
  {"push",0x50,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_NONE,ARG_NONE,121},
  {"push",0x50,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_NONE,ARG_NONE,122},
  {"push",0x51,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_CX,ARG_NONE,ARG_NONE,123},
  {"push",0x51,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_CX,ARG_NONE,ARG_NONE,124},
  {"push",0x52,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DX,ARG_NONE,ARG_NONE,125},
  {"push",0x52,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DX,ARG_NONE,ARG_NONE,126},
  {"push",0x53,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BX,ARG_NONE,ARG_NONE,127},
  {"push",0x53,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BX,ARG_NONE,ARG_NONE,128},
  {"push",0x54,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SP,ARG_NONE,ARG_NONE,129},
  {"push",0x54,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SP,ARG_NONE,ARG_NONE,130},
  {"push",0x55,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BP,ARG_NONE,ARG_NONE,131},
  {"push",0x55,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BP,ARG_NONE,ARG_NONE,132},
  {"push",0x56,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SI,ARG_NONE,ARG_NONE,133},
  {"push",0x56,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SI,ARG_NONE,ARG_NONE,134},
  {"push",0x57,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DI,ARG_NONE,ARG_NONE,135},
  {"push",0x57,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DI,ARG_NONE,ARG_NONE,136},
  {"pop",0x58,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_NONE,ARG_NONE,137},
  {"pop",0x58,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_NONE,ARG_NONE,138},
  {"pop",0x59,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_CX,ARG_NONE,ARG_NONE,139},
  {"pop",0x59,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_CX,ARG_NONE,ARG_NONE,140},
  {"pop",0x5a,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DX,ARG_NONE,ARG_NONE,141},
  {"pop",0x5a,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DX,ARG_NONE,ARG_NONE,142},
  {"pop",0x5b,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BX,ARG_NONE,ARG_NONE,143},
  {"pop",0x5b,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BX,ARG_NONE,ARG_NONE,144},
  {"pop",0x5c,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SP,ARG_NONE,ARG_NONE,145},
  {"pop",0x5c,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SP,ARG_NONE,ARG_NONE,146},
  {"pop",0x5d,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BP,ARG_NONE,ARG_NONE,147},
  {"pop",0x5d,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BP,ARG_NONE,ARG_NONE,148},
  {"pop",0x5e,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SI,ARG_NONE,ARG_NONE,149},
  {"pop",0x5e,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SI,ARG_NONE,ARG_NONE,150},
  {"pop",0x5f,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DI,ARG_NONE,ARG_NONE,151},
  {"pop",0x5f,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DI,ARG_NONE,ARG_NONE,152},
  {"pusha",0x60,PROC_FROM80286,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,153},
  {"pushad",0x60,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,154},
  {"popa",0x61,PROC_FROM80286,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,155},
  {"popad",0x61,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,156},
  {"bound",0x62,PROC_FROM80286,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,157},
  {"arpl",0x63,PROC_FROM80286,FLAGS_PMODE|FLAGS_16BIT|FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,158},
  {"fs:",0x64,PROC_FROM80386,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,159},
  {"gs:",0x65,PROC_FROM80386,FLAGS_SEGPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,160},
  {"",0x66,PROC_FROM80386,FLAGS_OPERPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,161},
  {"",0x67,PROC_FROM80386,FLAGS_ADDRPREFIX,ARG_NONE,ARG_NONE,ARG_NONE,162},
  {"push",0x68,PROC_FROM80286,FLAGS_OMODE16,ARG_IMM,ARG_NONE,ARG_NONE,163},
  {"push",0x68,PROC_FROM80386,FLAGS_OMODE32,ARG_IMM,ARG_NONE,ARG_NONE,164},
  {"imul",0x69,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_IMM,165},
  {"push",0x6a,PROC_FROM80286,0,ARG_IMM8,ARG_NONE,ARG_NONE,166},
  {"imul",0x6b,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_IMM8,167},
  {"insb",0x6c,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,168},
  {"insw",0x6d,PROC_FROM80286,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,169},
  {"insd",0x6d,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,170},
  {"outsb",0x6e,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,171},
  {"outsw",0x6f,PROC_FROM80286,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,172},
  {"outsd",0x6f,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,173},
  {"jo",0x70,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,174},
  {"jno",0x71,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,175},
  {"jc",0x72,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,176},
  {"jnc",0x73,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,177},
  {"jz",0x74,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,178},
  {"jnz",0x75,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,179},
  {"jbe",0x76,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,180},
  {"ja",0x77,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,181},
  {"js",0x78,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,182},
  {"jns",0x79,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,183},
  {"jpe",0x7a,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,184},
  {"jpo",0x7b,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,185},
  {"jl",0x7c,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,186},
  {"jge",0x7d,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,187},
  {"jle",0x7e,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,188},
  {"jg",0x7f,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,189},
  {NULL,0x80,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,190},  //subtable 0x80
  {NULL,0x81,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,191},  //subtable 0x81
  {NULL,0x82,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,192},  //subtable 0x82
  {NULL,0x83,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,193},  //subtable 0x83
  {"test",0x84,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,194},
  {"test",0x85,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,195},
  {"test",0x85,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,196},
  {"xchg",0x86,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,197},
  {"xchg",0x87,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,198},
  {"xchg",0x87,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,199},
  {"mov",0x88,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,200},
  {"mov",0x89,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG,ARG_NONE,201},
  {"mov",0x89,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG,ARG_NONE,202},
  {"mov",0x8a,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG,ARG_MODRM,ARG_NONE,203},
  {"mov",0x8b,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,204},
  {"mov",0x8b,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,205},
  {"mov",0x8c,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM,ARG_SREG,ARG_NONE,206},
  {"lea",0x8d,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,207},
  {"lea",0x8d,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,208},
  {"mov",0x8e,PROC_FROM8086,FLAGS_MODRM,ARG_SREG,ARG_MODRM,ARG_NONE,209},
  {"pop",0x8f,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,210},
  {"pop",0x8f,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,211},
  {"nop",0x90,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,212},
  {"xchg",0x91,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_CX,ARG_NONE,213},
  {"xchg",0x91,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_CX,ARG_NONE,214},
  {"xchg",0x92,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_DX,ARG_NONE,215},
  {"xchg",0x92,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_DX,ARG_NONE,216},
  {"xchg",0x93,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_BX,ARG_NONE,217},
  {"xchg",0x93,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_BX,ARG_NONE,218},
  {"xchg",0x94,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_SP,ARG_NONE,219},
  {"xchg",0x94,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_SP,ARG_NONE,220},
  {"xchg",0x95,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_BP,ARG_NONE,221},
  {"xchg",0x95,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_BP,ARG_NONE,222},
  {"xchg",0x96,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_SI,ARG_NONE,223},
  {"xchg",0x96,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_SI,ARG_NONE,224},
  {"xchg",0x97,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_REG_DI,ARG_NONE,225},
  {"xchg",0x97,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_REG_DI,ARG_NONE,226},
  {"cbw",0x98,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,227},
  {"cwde",0x98,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,228},
  {"cwd",0x99,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,229},
  {"cdq",0x99,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,230},
  {"callf",0x9a,PROC_FROM8086,FLAGS_CALL|FLAGS_OMODE16,ARG_FADDR,ARG_NONE,ARG_NONE,231},
  {"callf",0x9a,PROC_FROM80386,FLAGS_CALL|FLAGS_OMODE32,ARG_FADDR,ARG_NONE,ARG_NONE,232},
  {"wait",0x9b,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,233},
  {"pushf",0x9c,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,234},
  {"pushfd",0x9c,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,235},
  {"popf",0x9d,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,236},
  {"popfd",0x9d,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,237},
  {"sahf",0x9e,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,238},
  {"lahf",0x9f,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,239},
  {"mov",0xa0,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_MEMLOC,ARG_NONE,240},
  {"mov",0xa1,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_MEMLOC,ARG_NONE,241},
  {"mov",0xa1,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_MEMLOC,ARG_NONE,242},
  {"mov",0xa2,PROC_FROM8086,FLAGS_8BIT,ARG_MEMLOC,ARG_REG_AL,ARG_NONE,243},
  {"mov",0xa3,PROC_FROM8086,FLAGS_OMODE16,ARG_MEMLOC,ARG_REG_AX,ARG_NONE,244},
  {"mov",0xa3,PROC_FROM80386,FLAGS_OMODE32,ARG_MEMLOC,ARG_REG_AX,ARG_NONE,245},
  {"movsb",0xa4,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,246},
  {"movsw",0xa5,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,247},
  {"movsd",0xa5,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,248},
  {"cmpsb",0xa6,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,249},
  {"cmpsw",0xa7,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,250},
  {"cmpsd",0xa7,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,251},
  {"test",0xa8,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,252},
  {"test",0xa9,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,253},
  {"test",0xa9,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,254},
  {"stosb",0xaa,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,255},
  {"stosw",0xab,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,256},
  {"stosd",0xab,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,257},
  {"lodsb",0xac,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,258},
  {"lodsw",0xad,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,259},
  {"lodsd",0xad,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,260},
  {"scasb",0xae,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,261},
  {"scasw",0xaf,PROC_FROM8086,FLAGS_OMODE16,ARG_NONE,ARG_NONE,ARG_NONE,262},
  {"scasd",0xaf,PROC_FROM80386,FLAGS_OMODE32,ARG_NONE,ARG_NONE,ARG_NONE,263},
  {"mov",0xb0,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,264},
  {"mov",0xb1,PROC_FROM8086,FLAGS_8BIT,ARG_REG_CL,ARG_IMM8,ARG_NONE,265},
  {"mov",0xb2,PROC_FROM8086,FLAGS_8BIT,ARG_REG_DL,ARG_IMM8,ARG_NONE,266},
  {"mov",0xb3,PROC_FROM8086,FLAGS_8BIT,ARG_REG_BL,ARG_IMM8,ARG_NONE,267},
  {"mov",0xb4,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AH,ARG_IMM8,ARG_NONE,268},
  {"mov",0xb5,PROC_FROM8086,FLAGS_8BIT,ARG_REG_CH,ARG_IMM8,ARG_NONE,269},
  {"mov",0xb6,PROC_FROM8086,FLAGS_8BIT,ARG_REG_DH,ARG_IMM8,ARG_NONE,270},
  {"mov",0xb7,PROC_FROM8086,FLAGS_8BIT,ARG_REG_BH,ARG_IMM8,ARG_NONE,271},
  {"mov",0xb8,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM,ARG_NONE,272},
  {"mov",0xb8,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM,ARG_NONE,273},
  {"mov",0xb9,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_CX,ARG_IMM,ARG_NONE,274},
  {"mov",0xb9,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_CX,ARG_IMM,ARG_NONE,275},
  {"mov",0xba,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DX,ARG_IMM,ARG_NONE,276},
  {"mov",0xba,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DX,ARG_IMM,ARG_NONE,277},
  {"mov",0xbb,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BX,ARG_IMM,ARG_NONE,278},
  {"mov",0xbb,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BX,ARG_IMM,ARG_NONE,279},
  {"mov",0xbc,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SP,ARG_IMM,ARG_NONE,280},
  {"mov",0xbc,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SP,ARG_IMM,ARG_NONE,281},
  {"mov",0xbd,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_BP,ARG_IMM,ARG_NONE,282},
  {"mov",0xbd,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_BP,ARG_IMM,ARG_NONE,283},
  {"mov",0xbe,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_SI,ARG_IMM,ARG_NONE,284},
  {"mov",0xbe,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_SI,ARG_IMM,ARG_NONE,285},
  {"mov",0xbf,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_DI,ARG_IMM,ARG_NONE,286},
  {"mov",0xbf,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_DI,ARG_IMM,ARG_NONE,287},
  {NULL,0xc0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,288},  //subtable 0xc0
  {NULL,0xc1,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,289},  //subtable 0xc1
  {"ret",0xc2,PROC_FROM8086,FLAGS_16BIT|FLAGS_RET,ARG_IMM,ARG_NONE,ARG_NONE,290},
  {"ret",0xc3,PROC_FROM8086,FLAGS_RET,ARG_NONE,ARG_NONE,ARG_NONE,291},
  {"les",0xc4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,292},
  {"les",0xc4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,293},
  {"lds",0xc5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,294},
  {"lds",0xc5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,295},
  {"mov",0xc6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,296},
  {"mov",0xc7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,297},
  {"mov",0xc7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,298},
  {"enter",0xc8,PROC_FROM80286,FLAGS_16BIT,ARG_IMM16_A,ARG_IMM8,ARG_NONE,299},
  {"leave",0xc9,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,300},
  {"retf",0xca,PROC_FROM8086,FLAGS_16BIT|FLAGS_RET,ARG_IMM,ARG_NONE,ARG_NONE,301},
  {"retf",0xcb,PROC_FROM8086,FLAGS_RET,ARG_NONE,ARG_NONE,ARG_NONE,302},
  {"int 3",0xcc,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,303},
  {"int",0xcd,PROC_FROM8086,FLAGS_8BIT,ARG_IMM8,ARG_NONE,ARG_NONE,304},
  {"into",0xce,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,305},
  {"iret",0xcf,PROC_FROM8086,FLAGS_RET,ARG_NONE,ARG_NONE,ARG_NONE,306},
  {NULL,0xd0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,307},  //subtable 0xd0
  {NULL,0xd1,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,308},  //subtable 0xd1
  {NULL,0xd2,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,309},  //subtable 0xd2
  {NULL,0xd3,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,310},  //subtable 0xd3
  {"aam",0xd4,PROC_FROM8086,FLAGS_8BIT,ARG_IMM8,ARG_NONE,ARG_NONE,311},
  {"aad",0xd5,PROC_FROM8086,FLAGS_8BIT,ARG_IMM8,ARG_NONE,ARG_NONE,312},
  {"setalc",0xd6,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,313}, //UNDOCUMENTED
  {"xlat",0xd7,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,314},
  {NULL,0xd8,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,315},  //subtable 0xd8
  {NULL,0xd9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,316},  //subtable 0xd9
  {NULL,0xda,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,317},  //subtable 0xda
  {NULL,0xdb,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,318},  //subtable 0xdb
  {NULL,0xdc,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,319},  //subtable 0xdc
  {NULL,0xdd,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,320},  //subtable 0xdd
  {NULL,0xde,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,321},  //subtable 0xde
  {NULL,0xdf,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,322},  //subtable 0xdf
  {"loopnz",0xe0,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,323},
  {"loopz",0xe1,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,324},
  {"loop",0xe2,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,325},
  {"jcxz",0xe3,PROC_FROM8086,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,326},
  {"in",0xe4,PROC_FROM8086,FLAGS_8BIT,ARG_REG_AL,ARG_IMM8,ARG_NONE,327},
  {"in",0xe5,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_IMM8,ARG_NONE,328},
  {"in",0xe5,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_IMM8,ARG_NONE,329},
  {"out",0xe6,PROC_FROM8086,FLAGS_8BIT,ARG_IMM8,ARG_REG_AL,ARG_NONE,330},
  {"out",0xe7,PROC_FROM8086,FLAGS_OMODE16,ARG_IMM8,ARG_REG_AX,ARG_NONE,331},
  {"out",0xe7,PROC_FROM80386,FLAGS_OMODE32,ARG_IMM8,ARG_REG_AX,ARG_NONE,332},
  {"call",0xe8,PROC_FROM8086,FLAGS_CALL|FLAGS_OMODE16,ARG_RELIMM,ARG_NONE,ARG_NONE,333},
  {"call",0xe8,PROC_FROM80386,FLAGS_CALL|FLAGS_OMODE32,ARG_RELIMM,ARG_NONE,ARG_NONE,334},
  {"jmp",0xe9,PROC_FROM8086,FLAGS_JMP|FLAGS_OMODE16,ARG_RELIMM,ARG_NONE,ARG_NONE,335},
  {"jmp",0xe9,PROC_FROM80386,FLAGS_JMP|FLAGS_OMODE32,ARG_RELIMM,ARG_NONE,ARG_NONE,336},
  {"jmp",0xea,PROC_FROM8086,FLAGS_JMP|FLAGS_OMODE16,ARG_FADDR,ARG_NONE,ARG_NONE,337},
  {"jmp",0xea,PROC_FROM80386,FLAGS_JMP|FLAGS_OMODE32,ARG_FADDR,ARG_NONE,ARG_NONE,338},
  {"jmp",0xeb,PROC_FROM8086,FLAGS_JMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,339},
  {"in",0xec,PROC_FROM8086,0,ARG_REG_AL,ARG_16REG_DX,ARG_NONE,340},
  {"in",0xed,PROC_FROM8086,FLAGS_OMODE16,ARG_REG_AX,ARG_16REG_DX,ARG_NONE,341},
  {"in",0xed,PROC_FROM80386,FLAGS_OMODE32,ARG_REG_AX,ARG_16REG_DX,ARG_NONE,342},
  {"out",0xee,PROC_FROM8086,0,ARG_16REG_DX,ARG_REG_AL,ARG_NONE,343},
  {"out",0xef,PROC_FROM8086,FLAGS_OMODE16,ARG_16REG_DX,ARG_REG_AX,ARG_NONE,344},
  {"out",0xef,PROC_FROM80386,FLAGS_OMODE32,ARG_16REG_DX,ARG_REG_AX,ARG_NONE,345},
  {"lock:",0xf0,PROC_FROM8086,FLAGS_PREFIX,ARG_NONE,ARG_NONE,ARG_NONE,346},
  {"smi",0xf1,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,347},  //UNDOCUMENTED/AMD ?
  {"repne:",0xf2,PROC_FROM8086,FLAGS_PREFIX,ARG_NONE,ARG_NONE,ARG_NONE,348},
  {"rep:",0xf3,PROC_FROM8086,FLAGS_PREFIX,ARG_NONE,ARG_NONE,ARG_NONE,349},
  {"hlt",0xf4,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,350},
  {"cmc",0xf5,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,351},
  {NULL,0xf6,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,352},  //subtable 0xf6
  {NULL,0xf7,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,353},  //subtable 0xf7
  {"clc",0xf8,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,354},
  {"stc",0xf9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,355},
  {"cli",0xfa,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,356},
  {"sti",0xfb,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,357},
  {"cld",0xfc,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,358},
  {"std",0xfd,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,359},
  {NULL,0xfe,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,360},  //subtable 0xfe
  {NULL,0xff,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,361},  //subtable 0xff
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};
//Subtables needed - 0x0f, 0x80, 0x81, 0x82, 0x83, 0xc0, 0xc1,           ***done!
// 0xd0, 0xd1, 0xd2, 0xd3, 0xf6, 0xf7, 0xfe, 0xff                        ***done!
// 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf - FPU instructions
// 0x0f subtables : 0x00, 0x01, 0x18, 0x71, 0x72, 0x73, 0xae, 0xba, 0xc7 ***done!
//nb some instructions change when they have a segment overrider eg xlat.
// - how will this go in ?
//need to check undocumented instructions/amd insts- args/size/etc
//- setalc, smi

// subtable 0x0f
asminstdata asm86sub0f[]=
{ {NULL,0x00,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,1000}, //subtable 0x0f/0x00
  {NULL,0x01,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,1001}, //subtable 0x0f/0x01
  {"lar",0x02,PROC_FROM80286,FLAGS_PMODE|FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,1002},
  {"lar",0x02,PROC_FROM80386,FLAGS_PMODE|FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,1003},
  {"lsl",0x03,PROC_FROM80286,FLAGS_PMODE|FLAGS_MODRM|FLAGS_OMODE16,ARG_REG,ARG_MODRM,ARG_NONE,1004},
  {"lsl",0x03,PROC_FROM80386,FLAGS_PMODE|FLAGS_MODRM|FLAGS_OMODE32,ARG_REG,ARG_MODRM,ARG_NONE,1005},
  {"clts",0x06,PROC_FROM80286,FLAGS_PMODE,ARG_NONE,ARG_NONE,ARG_NONE,1006},
  {"invd",0x08,PROC_FROM80486,0,ARG_NONE,ARG_NONE,ARG_NONE,1007},
  {"wbinvd",0x09,PROC_FROM80486,0,ARG_NONE,ARG_NONE,ARG_NONE,1008},
  {"cflsh",0x0a,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,1009},
  {"ud2",0x0b,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,1010},
  {"movups",0x10,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1011},
  {"movups",0x11,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XMMMODRM,ARG_XREG,ARG_NONE,1012},
  {"movlps",0x12,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1013},
  {"movlps",0x13,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XMMMODRM,ARG_XREG,ARG_NONE,1014},
  {"unpcklps",0x14,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1015},
  {"unpckhps",0x15,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1016},
  {"movhps",0x16,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1017},
  {"movhps",0x17,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XMMMODRM,ARG_XREG,ARG_NONE,1018},
  {NULL,0x18,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,1019}, // subtable 0x0f/0x18
  {"mov",0x20,PROC_FROM80386,FLAGS_MODRM|FLAGS_32BIT,ARG_MODREG,ARG_CREG,ARG_NONE,1020},
  {"mov",0x21,PROC_FROM80386,FLAGS_MODRM|FLAGS_32BIT,ARG_MODREG,ARG_DREG,ARG_NONE,1021},
  {"mov",0x22,PROC_FROM80386,FLAGS_MODRM|FLAGS_32BIT,ARG_CREG,ARG_MODREG,ARG_NONE,1022},
  {"mov",0x23,PROC_FROM80386,FLAGS_MODRM|FLAGS_32BIT,ARG_DREG,ARG_MODREG,ARG_NONE,1023},
  {"mov",0x24,PROC_80386|PROC_80486,FLAGS_MODRM|FLAGS_32BIT,ARG_MODREG,ARG_TREG_67,ARG_NONE,1024},
  {"mov",0x26,PROC_80386|PROC_80486,FLAGS_MODRM|FLAGS_32BIT,ARG_MODREG,ARG_TREG,ARG_NONE,1025},
  {"movaps",0x28,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1026},
  {"movaps",0x29,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XMMMODRM,ARG_XREG,ARG_NONE,1027},
  {"cvtpi2ps",0x2a,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_MMXMODRM,ARG_NONE,1028},
  {"movntps",0x2b,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XMMMODRM,ARG_XREG,ARG_NONE,1029},
  {"cvttps2pi",0x2c,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_XMMMODRM,ARG_NONE,1030},
  {"cvtps2pi",0x2d,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_XMMMODRM,ARG_NONE,1031},
  {"ucomiss",0x2e,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1032},
  {"comiss",0x2f,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1033},
  {"wrmsr",0x30,PROC_FROMPENTIUM,0,ARG_NONE,ARG_NONE,ARG_NONE,1034},
  {"rdtsc",0x31,PROC_FROMPENTIUM,0,ARG_NONE,ARG_NONE,ARG_NONE,1035},
  {"rdmsr",0x32,PROC_FROMPENTIUM,0,ARG_NONE,ARG_NONE,ARG_NONE,1036},
  {"rdpmc",0x33,PROC_FROMPENTPRO,0,ARG_NONE,ARG_NONE,ARG_NONE,1037},
  {"sysenter",0x34,PROC_FROMPENTIUM2,0,ARG_NONE,ARG_NONE,ARG_NONE,1038},
  {"sysexit",0x35,PROC_FROMPENTIUM2,0,ARG_NONE,ARG_NONE,ARG_NONE,1039},
  {"cmovo",0x40,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1040},
  {"cmovno",0x41,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1041},
  {"cmovc",0x42,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1042},
  {"cmovnc",0x43,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1043},
  {"cmovz",0x44,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1044},
  {"cmovnz",0x45,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1045},
  {"cmovbe",0x46,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1046},
  {"cmova",0x47,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1047},
  {"cmovs",0x48,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1048},
  {"cmovns",0x49,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1049},
  {"cmovpe",0x4a,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1050},
  {"cmovpo",0x4b,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1051},
  {"cmovl",0x4c,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1052},
  {"cmovge",0x4d,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1053},
  {"cmovle",0x4e,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1054},
  {"cmovg",0x4f,PROC_FROMPENTPRO,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1055},
  {"movmskps",0x50,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_REG,ARG_XMMMODRM,ARG_NONE,1056},
  {"sqrtps",0x51,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1057},
  {"rsqrtps",0x52,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1058},
  {"rcpps",0x53,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1059},
  {"andps",0x54,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1060},
  {"andnps",0x55,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1061},
  {"orps",0x56,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1062},
  {"xorps",0x57,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1063},
  {"addps",0x58,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1064},
  {"mulps",0x59,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1065},
  {"subps",0x5c,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1066},
  {"minps",0x5d,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1067},
  {"divps",0x5e,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1068},
  {"maxps",0x5f,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,1069},
  {"punpcklbw",0x60,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1070},
  {"punpcklwd",0x61,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1071},
  {"punpckldq",0x62,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1072},
  {"packsswb",0x63,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1073},
  {"pcmpgtb",0x64,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1074},
  {"pcmpgtw",0x65,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1075},
  {"pcmpgtd",0x66,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1076},
  {"packuswb",0x67,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1077},
  {"punpckhbw",0x68,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1078},
  {"punpckhwd",0x69,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1079},
  {"punpckhdq",0x6a,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1080},
  {"packssdw",0x6b,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1081},
  {"movd",0x6e,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MODRM,ARG_NONE,1082},
  {"movq",0x6f,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1083},
  {"pshuf",0x70,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_IMM8,1084},
  {NULL,0x71,PROC_FROMPENTMMX,0,ARG_NONE,ARG_NONE,ARG_NONE,1085}, //subtable 0x0f/0x71
  {NULL,0x72,PROC_FROMPENTMMX,0,ARG_NONE,ARG_NONE,ARG_NONE,1086}, //subtable 0x0f/0x72
  {NULL,0x73,PROC_FROMPENTMMX,0,ARG_NONE,ARG_NONE,ARG_NONE,1087}, //subtable 0x0f/0x73
  {"pcmpeqb",0x74,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1088},
  {"pcmpeqw",0x75,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1089},
  {"pcmpeqd",0x76,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1090},
  {"emms",0x77,PROC_FROMPENTMMX,0,ARG_NONE,ARG_NONE,ARG_NONE,1091},
  {"movd",0x7e,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MODRM,ARG_MREG,ARG_NONE,1092},
  {"movq",0x7f,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_MREG,ARG_NONE,1093},
  {"jo",0x80,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1094},
  {"jno",0x81,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1095},
  {"jc",0x82,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1096},
  {"jnc",0x83,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1097},
  {"jz",0x84,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1098},
  {"jnz",0x85,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1099},
  {"jbe",0x86,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1100},
  {"ja",0x87,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1101},
  {"js",0x88,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1102},
  {"jns",0x89,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1103},
  {"jpe",0x8a,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1104},
  {"jpo",0x8b,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1105},
  {"jl",0x8c,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1106},
  {"jge",0x8d,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1107},
  {"jle",0x8e,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1108},
  {"jg",0x8f,PROC_FROM80386,FLAGS_CJMP,ARG_RELIMM,ARG_NONE,ARG_NONE,1109},
  {"seto",0x90,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1110},
  {"setno",0x91,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1111},
  {"setc",0x92,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1112},
  {"setnc",0x93,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1113},
  {"setz",0x94,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1114},
  {"setnz",0x95,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1115},
  {"setbe",0x96,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1116},
  {"seta",0x97,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1117},
  {"sets",0x98,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1118},
  {"setns",0x99,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1119},
  {"setpe",0x9a,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1120},
  {"setpo",0x9b,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1121},
  {"setl",0x9c,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1122},
  {"setge",0x9d,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1123},
  {"setle",0x9e,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1124},
  {"setg",0x9f,PROC_FROM80386,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,1125},
  {"push",0xa0,PROC_FROM80386,0,ARG_REG_FS,ARG_NONE,ARG_NONE,1126},
  {"pop",0xa1,PROC_FROM80386,0,ARG_REG_FS,ARG_NONE,ARG_NONE,1127},
  {"cpuid",0xa2,PROC_FROM80486,0,ARG_NONE,ARG_NONE,ARG_NONE,1128},
  {"bt",0xa3,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_NONE,1129},
  {"shld",0xa4,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_IMM8,1130},
  {"shld",0xa5,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_REG_CL,1131},
  {"push",0xa8,PROC_FROM80386,0,ARG_REG_GS,ARG_NONE,ARG_NONE,1132},
  {"pop",0xa9,PROC_FROM80386,0,ARG_REG_GS,ARG_NONE,ARG_NONE,1133},
  {"rsm",0xaa,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,1134},
  {"bts",0xab,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_NONE,1135},
  {"shrd",0xac,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_IMM8,1136},
  {"shrd",0xad,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_REG_CL,1137},
  {NULL,0xae,PROC_FROMPENTIUM2,0,ARG_NONE,ARG_NONE,ARG_NONE,1138}, //subtable 0x0f/0xae
  {"imul",0xaf,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1139},
  {"cmpxchg",0xb0,PROC_FROM80486,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,1140},
  {"cmpxchg",0xb1,PROC_FROM80486,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_NONE,1141},
  {"lss",0xb2,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1142},
  {"btr",0xb3,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_NONE,1143},
  {"lfs",0xb4,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1144},
  {"lgs",0xb5,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1145},
  {"movzx",0xb6,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM8,ARG_NONE,1146},
  {"movzx",0xb7,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM16,ARG_NONE,1147},
  {"ud1",0xb9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,1148},
  {NULL,0xba,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,1149}, //subtable 0x0f/0xba
  {"btc",0xbb,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_REG,ARG_NONE,1150},
  {"bsf",0xbc,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1151},
  {"bsr",0xbd,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM,ARG_NONE,1152},
  {"movsx",0xbe,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM8,ARG_NONE,1153},
  {"movsx",0xbf,PROC_FROM80386,FLAGS_MODRM,ARG_REG,ARG_MODRM16,ARG_NONE,1154},
  {"xadd",0xc0,PROC_FROM80486,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG,ARG_NONE,1155},
  {"xadd",0xc1,PROC_FROM80486,FLAGS_MODRM,ARG_MODRM,ARG_NONE,ARG_NONE,1156},
  {NULL,0xc2,PROC_FROMPENTIUM2,0,ARG_NONE,ARG_NONE,ARG_NONE,1157}, //subtable 0x0f/0xc7
  {"pinsrw",0xc4,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MODRM,ARG_IMM8,1158},
  {"pextrw",0xc5,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_REG,ARG_MMXMODRM,ARG_IMM8,1159},
  {"shufps",0xc6,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_IMM8,1160},
  {NULL,0xc7,PROC_FROMPENTMMX,0,ARG_NONE,ARG_NONE,ARG_NONE,1161}, //subtable 0x0f/0xc7
  {"bswap",0xc8,PROC_FROM80486,FLAGS_32BIT,ARG_REG_AX,ARG_NONE,ARG_NONE,1162},
  {"bswap",0xc9,PROC_FROM80486,FLAGS_32BIT,ARG_REG_CX,ARG_NONE,ARG_NONE,1163},
  {"bswap",0xca,PROC_FROM80486,FLAGS_32BIT,ARG_REG_DX,ARG_NONE,ARG_NONE,1164},
  {"bswap",0xcb,PROC_FROM80486,FLAGS_32BIT,ARG_REG_BX,ARG_NONE,ARG_NONE,1165},
  {"bswap",0xcc,PROC_FROM80486,FLAGS_32BIT,ARG_REG_SP,ARG_NONE,ARG_NONE,1166},
  {"bswap",0xcd,PROC_FROM80486,FLAGS_32BIT,ARG_REG_BP,ARG_NONE,ARG_NONE,1167},
  {"bswap",0xce,PROC_FROM80486,FLAGS_32BIT,ARG_REG_SI,ARG_NONE,ARG_NONE,1168},
  {"bswap",0xcf,PROC_FROM80486,FLAGS_32BIT,ARG_REG_DI,ARG_NONE,ARG_NONE,1169},
  {"psrlw",0xd1,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1170},
  {"psrld",0xd2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1171},
  {"psrlq",0xd3,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1172},
  {"pmullw",0xd5,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1173},
  {"pmovmskb",0xd7,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_REG,ARG_MMXMODRM,ARG_NONE,1174},
  {"psubusb",0xd8,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1175},
  {"psubusw",0xd9,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1176},
  {"pminub",0xda,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1177},
  {"pand",0xdb,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1178},
  {"paddusb",0xdc,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1179},
  {"paddusw",0xdd,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1180},
  {"pmaxub",0xde,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1181},
  {"pandn",0xdf,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1182},
  {"pavgb",0xe0,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1183},
  {"psraw",0xe1,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1184},
  {"psrad",0xe2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1185},
  {"pavgw",0xe3,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1186},
  {"pmulhuw",0xe4,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1187},
  {"pmulhw",0xe5,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1188},
  {"movntq",0xe7,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_MREG,ARG_NONE,1189},
  {"psubsb",0xe8,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1190},
  {"psubsw",0xe9,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1191},
  {"pminsw",0xea,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1192},
  {"por",0xeb,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1193},
  {"paddsb",0xec,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1194},
  {"paddsw",0xed,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1195},
  {"pmaxsw",0xee,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1196},
  {"pxor",0xef,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1197},
  {"psllw",0xf1,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1198},
  {"pslld",0xf2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1199},
  {"psllq",0xf3,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1200},
  {"pmaddwd",0xf5,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1201},
  {"psadbw",0xf6,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1202},
  {"maskmovq",0xf7,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1203},
  {"psubb",0xf8,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1204},
  {"psubw",0xf9,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1205},
  {"psubd",0xfa,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1206},
  {"paddb",0xfc,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1207},
  {"paddw",0xfd,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1208},
  {"paddd",0xfe,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MREG,ARG_MMXMODRM,ARG_NONE,1209},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x80
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub80[]=
{ {"add",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2000},
  {"or",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2001},
  {"adc",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2002},
  {"sbb",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2003},
  {"and",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2004},
  {"sub",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2005},
  {"xor",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2006},
  {"cmp",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,2007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x81
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub81[]=
{ {"add",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3000},
  {"add",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3001},
  {"or",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3002},
  {"or",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3003},
  {"adc",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3004},
  {"adc",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3005},
  {"sbb",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3006},
  {"sbb",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3007},
  {"and",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3008},
  {"and",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3009},
  {"sub",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3010},
  {"sub",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3011},
  {"xor",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3012},
  {"xor",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3013},
  {"cmp",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,3014},
  {"cmp",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,3016},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x82
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub82[]=
{ {"add",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4000},
  {"or",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4001},
  {"adc",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4002},
  {"sbb",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4003},
  {"and",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4004},
  {"sub",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4005},
  {"xor",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4006},
  {"cmp",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_SIMM8,ARG_NONE,4007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x83
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub83[]=
{ {"add",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5000},
  {"add",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5001},
  {"or",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5002},
  {"or",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5003},
  {"adc",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5004},
  {"adc",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5005},
  {"sbb",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5006},
  {"sbb",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5007},
  {"and",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5008},
  {"and",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5009},
  {"sub",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5010},
  {"sub",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5011},
  {"xor",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5012},
  {"xor",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5013},
  {"cmp",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_SIMM8,ARG_NONE,5014},
  {"cmp",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_SIMM8,ARG_NONE,5015},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xc0
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subc0[]=
{ {"rol",0x0,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6000},
  {"ror",0x1,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6001},
  {"rcl",0x2,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6002},
  {"rcr",0x3,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6003},
  {"shl",0x4,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6004},
  {"shr",0x5,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6005},
  {"sal",0x6,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6006},
  {"sar",0x7,PROC_FROM80286,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,6007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xc1
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subc1[]=
{ {"rol",0x0,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7000},
  {"rol",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7001},
  {"ror",0x1,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7002},
  {"ror",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7003},
  {"rcl",0x2,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7004},
  {"rcl",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7005},
  {"rcr",0x3,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7006},
  {"rcr",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7007},
  {"shl",0x4,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7008},
  {"shl",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7009},
  {"shr",0x5,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7010},
  {"shr",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7011},
  {"sal",0x6,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7012},
  {"sal",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7013},
  {"sar",0x7,PROC_FROM80286,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM8,ARG_NONE,7014},
  {"sar",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM8,ARG_NONE,7015},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd0
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd0[]=
{ {"rol",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8000},
  {"ror",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8001},
  {"rcl",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8002},
  {"rcr",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8003},
  {"shl",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8004},
  {"shr",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8005},
  {"sal",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8006},
  {"sar",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM_1,ARG_NONE,8007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd1
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd1[]=
{ {"rol",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9000},
  {"rol",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9001},
  {"ror",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9002},
  {"ror",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9003},
  {"rcl",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9004},
  {"rcl",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9005},
  {"rcr",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9006},
  {"rcr",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9007},
  {"shl",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9008},
  {"shl",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9009},
  {"shr",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9010},
  {"shr",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9011},
  {"sal",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9012},
  {"sal",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9013},
  {"sar",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM_1,ARG_NONE,9014},
  {"sar",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM_1,ARG_NONE,9015},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd2
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd2[]=
{ {"rol",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10000},
  {"ror",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10001},
  {"rcl",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10002},
  {"rcr",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10003},
  {"shl",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10004},
  {"shr",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10005},
  {"sal",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10006},
  {"sar",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_REG_CL,ARG_NONE,10007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd3
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd3[]=
{ {"rol",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11000},
  {"rol",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11001},
  {"ror",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11002},
  {"ror",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11003},
  {"rcl",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11004},
  {"rcl",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11005},
  {"rcr",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11006},
  {"rcr",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11007},
  {"shl",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11008},
  {"shl",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11009},
  {"shr",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11010},
  {"shr",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11011},
  {"sal",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11012},
  {"sal",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11013},
  {"sar",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_REG_CL,ARG_NONE,11014},
  {"sar",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_REG_CL,ARG_NONE,11015},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xf6
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subf6[]=
{ {"test",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,12000},
  {"test",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_IMM8,ARG_NONE,12001},
  {"not",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12002},
  {"neg",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12003},
  {"mul",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12004},
  {"imul",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12005},
  {"div",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12006},
  {"idiv",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,12007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xf7
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subf7[]=
{ {"test",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,13000},
  {"test",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,13001},
  {"test",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_IMM,ARG_NONE,13002},
  {"test",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_IMM,ARG_NONE,13003},
  {"not",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13004},
  {"not",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13005},
  {"neg",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13006},
  {"neg",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13007},
  {"mul",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13008},
  {"mul",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13009},
  {"imul",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13010},
  {"imul",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13011},
  {"div",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13012},
  {"div",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13013},
  {"idiv",0x7,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,13014},
  {"idiv",0x7,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,13015},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xfe
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subfe[]=
{ {"inc",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,14000},
  {"dec",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_8BIT,ARG_MODRM,ARG_NONE,ARG_NONE,14001},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xff
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subff[]=
{ {"inc",0x0,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,15000},
  {"inc",0x0,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,15001},
  {"dec",0x1,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,15002},
  {"dec",0x1,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,15003},
  {"call",0x2,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16|FLAGS_CALL,ARG_MODRM,ARG_NONE,ARG_NONE,15004},
  {"call",0x2,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32|FLAGS_CALL,ARG_MODRM,ARG_NONE,ARG_NONE,15005},
  {"call",0x3,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16|FLAGS_CALL,ARG_MODRM_FPTR,ARG_NONE,ARG_NONE,15006},
  {"call",0x3,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32|FLAGS_CALL,ARG_MODRM_FPTR,ARG_NONE,ARG_NONE,15007},
  {"jmp",0x4,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16|FLAGS_JMP,ARG_MODRM,ARG_NONE,ARG_NONE,15008},
  {"jmp",0x4,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32|FLAGS_JMP,ARG_MODRM,ARG_NONE,ARG_NONE,15009},
  {"jmp",0x5,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16|FLAGS_JMP,ARG_MODRM_FPTR,ARG_NONE,ARG_NONE,15010},
  {"jmp",0x5,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32|FLAGS_JMP,ARG_MODRM_FPTR,ARG_NONE,ARG_NONE,15011},
  {"push",0x6,PROC_FROM8086,FLAGS_MODRM|FLAGS_OMODE16,ARG_MODRM,ARG_NONE,ARG_NONE,15012},
  {"push",0x6,PROC_FROM80386,FLAGS_MODRM|FLAGS_OMODE32,ARG_MODRM,ARG_NONE,ARG_NONE,15013},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x00
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f00[]=
{ {"sldt",0x0,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16000},
  {"str",0x1,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16001},
  {"lldt",0x2,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16002},
  {"ltr",0x3,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16003},
  {"verr",0x4,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16004},
  {"verw",0x5,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,16005},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x01
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f01[]=
{ {"sgdt",0x0,PROC_FROM80286,FLAGS_MODRM,ARG_MODRM_S,ARG_NONE,ARG_NONE,17000},
  {"sidt",0x1,PROC_FROM80286,FLAGS_MODRM,ARG_MODRM_S,ARG_NONE,ARG_NONE,17001},
  {"lgdt",0x2,PROC_FROM80286,FLAGS_MODRM,ARG_MODRM_S,ARG_NONE,ARG_NONE,17002},
  {"lidt",0x3,PROC_FROM80286,FLAGS_MODRM,ARG_MODRM_S,ARG_NONE,ARG_NONE,17003},
  {"smsw",0x4,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,17004},
  {"lmsw",0x6,PROC_FROM80286,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,17005},
  {"invlpg",0x7,PROC_FROM80486,FLAGS_MODRM|FLAGS_16BIT,ARG_MODRM,ARG_NONE,ARG_NONE,17006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x18
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f18[]=
{ {"prefetchnta",0x0,PROC_FROMPENTIUM2,FLAGS_MODRM,ARG_MODRM,ARG_NONE,ARG_NONE,18000},
  {"prefetcht0",0x1,PROC_FROMPENTIUM2,FLAGS_MODRM,ARG_MODRM,ARG_NONE,ARG_NONE,18001},
  {"prefetcht1",0x2,PROC_FROMPENTIUM2,FLAGS_MODRM,ARG_MODRM,ARG_NONE,ARG_NONE,18002},
  {"prefetcht2",0x3,PROC_FROMPENTIUM2,FLAGS_MODRM,ARG_MODRM,ARG_NONE,ARG_NONE,18003},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x71
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f71[]=
{ {"psrlw",0x2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,19000},
  {"psraw",0x4,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,19001},
  {"psllw",0x6,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,19002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x72
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f72[]=
{ {"psrld",0x2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,20000},
  {"psrad",0x4,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,20001},
  {"pslld",0x6,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,20002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0x73
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0f73[]=
{ {"psrlq",0x2,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,21000},
  {"psllq",0x6,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MMXMODRM,ARG_IMM8,ARG_NONE,21001},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0xae
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0fae[]=
{ {"fxsave",0x0,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MODRMM512,ARG_NONE,ARG_NONE,22000},
  {"fxrstor",0x1,PROC_FROMPENTMMX,FLAGS_MODRM|FLAGS_32BIT,ARG_MODRMM512,ARG_NONE,ARG_NONE,22001},
  {"ldmxcsr",0x2,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MODRM,ARG_NONE,ARG_NONE,22002},
  {"stmxcsr",0x3,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_MODRM,ARG_NONE,ARG_NONE,22003},
  {"sfence",0x7,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_NONEBYTE,ARG_NONE,ARG_NONE,22004},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0xba
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0fba[]=
{ {"bt",0x4,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_IMM8,ARG_NONE,23000},
  {"bts",0x5,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_IMM8,ARG_NONE,23001},
  {"btr",0x6,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_IMM8,ARG_NONE,23002},
  {"btc",0x7,PROC_FROM80386,FLAGS_MODRM,ARG_MODRM,ARG_IMM8,ARG_NONE,23003},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0xc2
// -num is the follow up byte
asminstdata asm86sub0fc2[]=
{ {"cmpeqps",0x0,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24000},
  {"cmpltps",0x1,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24001},
  {"cmpleps",0x2,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24002},
  {"cmpunordps",0x3,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24003},
  {"cmpneqps",0x4,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24004},
  {"cmpnltps",0x5,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24005},
  {"cmpnleps",0x6,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24006},
  {"cmpordps",0x7,PROC_FROMPENTIUM2,FLAGS_MODRM|FLAGS_32BIT,ARG_XREG,ARG_XMMMODRM,ARG_NONE,24007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0x0f/0xc7
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86sub0fc7[]=
{ {"cmpxch8b",0x1,PROC_FROMPENTIUM,FLAGS_MODRM,ARG_MODRMQ,ARG_NONE,ARG_NONE,25000},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// FPU Instructions

// subtable 0xd8/ modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd8a[]=
{ {"fadd",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26000},
  {"fmul",0x1,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26001},
  {"fcom",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26002},
  {"fcomp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26003},
  {"fsub",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26004},
  {"fsubr",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26005},
  {"fdiv",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26006},
  {"fdivr",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,26007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd8/ modrm = 0xc0-0xff
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subd8b[]=
{ {"fadd",0x18,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27000},
  {"fmul",0x19,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27001},
  {"fcom",0x1a,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27002},
  {"fcomp",0x1b,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27003},
  {"fsub",0x1c,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27004},
  {"fsubr",0x1d,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27005},
  {"fdiv",0x1e,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27006},
  {"fdivr",0x1f,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,27007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd9/modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subd9a[]=
{ {"fld",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,28000},
  {"fst",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,28001},
  {"fstp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SREAL,ARG_NONE,ARG_NONE,28002},
  {"fldenv",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_PTR,ARG_NONE,ARG_NONE,28003},
  {"fldcw",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WORD,ARG_NONE,ARG_NONE,28004},
  {"fstenv",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_PTR,ARG_NONE,ARG_NONE,28005},
  {"fstcw",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WORD,ARG_NONE,ARG_NONE,28006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd9/ modrm = 0xc0-0xcf
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subd9b[]=
{ {"fld",0x18,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,29000},
  {"fxch",0x19,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,29001},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xd9/ modrm = 0xd0-0xff
asminstdata asm86subd9c[]=
{ {"fnop",0xd0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30000},
  {"fchs",0xe0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30001},
  {"fabs",0xe1,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30002},
  {"ftst",0xe4,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30003},
  {"fxam",0xe5,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30004},
  {"fld1",0xe8,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30005},
  {"fldl2t",0xe9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30006},
  {"fldl2e",0xea,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30007},
  {"fldpi",0xeb,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30008},
  {"fldlg2",0xec,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30009},
  {"fldln2",0xed,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30010},
  {"fldz",0xee,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30011},
  {"f2xm1",0xf0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30012},
  {"fyl2x",0xf1,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30013},
  {"fptan",0xf2,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30014},
  {"fpatan",0xf3,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30015},
  {"fxtract",0xf4,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30016},
  {"fprem1",0xf5,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,30017},
  {"fdecstp",0xf6,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30018},
  {"fincstp",0xf7,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30019},
  {"fprem",0xf8,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30020},
  {"fyl2xp1",0xf9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30021},
  {"fsqrt",0xfa,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30022},
  {"fsincos",0xfb,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,30023},
  {"frndint",0xfc,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30024},
  {"fscale",0xfd,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,30025},
  {"fsin",0xfe,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,30026},
  {"fcos",0xff,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,30027},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xda/ modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdaa[]=
{ {"fiadd",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31000},
  {"fimul",0x1,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31001},
  {"ficom",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31002},
  {"ficomp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31003},
  {"fisub",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31004},
  {"fisubr",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31005},
  {"fidiv",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31006},
  {"fidivr",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,31007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xda/ modrm = 0xc0-0xdf
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subdab[]=
{ {"fmovb",0x18,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,32000},
  {"fmove",0x19,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,32001},
  {"fmovbe",0x1a,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,32002},
  {"fmovu",0x1b,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,32003},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xda/ modrm = 0xe0-0xff
asminstdata asm86subdac[]=
{ {"fucompp",0xe9,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,33000},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdb/modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdba[]=
{ {"fild",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,34000},
  {"fist",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,34001},
  {"fistp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_SINT,ARG_NONE,ARG_NONE,34002},
  {"fld",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_EREAL,ARG_NONE,ARG_NONE,34003},
  {"fstp",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_EREAL,ARG_NONE,ARG_NONE,34004},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdb/ modrm = 0xc0-0xff
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subdbb[]=
{ {"fcmovnb",0x18,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35000},
  {"fcmovne",0x19,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35001},
  {"fcmovnbe",0x1a,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35002},
  {"fcmovnu",0x1b,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35003},
  {"fucomi",0x1d,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35004},
  {"fcomi",0x1e,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,35005},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdb/ modrm = 0xe0-0xff
asminstdata asm86subdbc[]=
{ {"feni",0xe0,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,36000},
  {"fdisi",0xe1,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,36001},
  {"fclex",0xe2,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,36002},
  {"finit",0xe3,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,36003},
  {"fsetpm",0xe4,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,36004},
  {"frstpm",0xe5,PROC_FROM80286,0,ARG_NONE,ARG_NONE,ARG_NONE,36005},
  {"frint2",0xec,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,36006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdc/ modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdca[]=
{ {"fadd",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37000},
  {"fmul",0x1,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37001},
  {"fcom",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37002},
  {"fcomp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37003},
  {"fsub",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37004},
  {"fsubr",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37005},
  {"fdiv",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37006},
  {"fdivr",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,37007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdc/ modrm = 0xc0-0xff
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subdcb[]=
{ {"fadd",0x18,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38000},
  {"fmul",0x19,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38001},
  {"fcom2",0x1a,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38002},
  {"fcomp3",0x1b,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38003},
  {"fsub",0x1c,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38004},
  {"fsubr",0x1d,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38005},
  {"fdiv",0x1e,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38006},
  {"fdivr",0x1f,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,38007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdd/modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdda[]=
{ {"fld",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,39000},
  {"fst",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,39001},
  {"fstp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_DREAL,ARG_NONE,ARG_NONE,39002},
  {"frstor",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_PTR,ARG_NONE,ARG_NONE,39003},
  {"fsave",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_PTR,ARG_NONE,ARG_NONE,39004},
  {"fstsw",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WORD,ARG_NONE,ARG_NONE,39005},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdd/ modrm = 0xc0-0xff
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subddb[]=
{ {"ffree",0x18,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,40000},
  {"fxch4",0x19,PROC_FROM8086,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,40001},
  {"fst",0x1a,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,40002},
  {"fstp",0x1b,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,40003},
  {"fucom",0x1c,PROC_FROM80386,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,40004},
  {"fucomp",0x1d,PROC_FROM80386,0,ARG_FREG,ARG_NONE,ARG_NONE,40005},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xde/ modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdea[]=
{ {"fiadd",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41000},
  {"fimul",0x1,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41001},
  {"ficom",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41002},
  {"ficomp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41003},
  {"fisub",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41004},
  {"fisubr",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41005},
  {"fidiv",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41006},
  {"fidivr",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,41007},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xde/ modrm = 0xc0-0xdf
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subdeb[]=
{ {"faddp",0x18,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42000},
  {"fmulp",0x19,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42001},
  {"fcomp5",0x1a,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42002},
  {"fsubrp",0x1c,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42003},
  {"fsubp",0x1d,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42004},
  {"fdivrp",0x1e,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42005},
  {"fdivp",0x1f,PROC_FROM8086,0,ARG_FREG,ARG_REG_ST0,ARG_NONE,42006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xde/ modrm = 0xd8-0xdf
asminstdata asm86subdec[]=
{ {"fcompp",0xd9,PROC_FROM8086,0,ARG_NONE,ARG_NONE,ARG_NONE,43000},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdf/modrm = 0x00-0xbf
// - num is encoding of modrm bits 5,4,3 only
asminstdata asm86subdfa[]=
{ {"fild",0x0,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,44000},
  {"fist",0x2,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,44001},
  {"fistp",0x3,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_WINT,ARG_NONE,ARG_NONE,44002},
  {"fbld",0x4,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_BCD,ARG_NONE,ARG_NONE,44003},
  {"fild",0x5,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_LINT,ARG_NONE,ARG_NONE,44004},
  {"fbstp",0x6,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_BCD,ARG_NONE,ARG_NONE,44005},
  {"fistp",0x7,PROC_FROM8086,FLAGS_MODRM,ARG_MODRM_LINT,ARG_NONE,ARG_NONE,44006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdf/ modrm = 0xc0-0xff
// - num is mod bits 7,6 only + bits 5,4,3 (ie modrm/8)
asminstdata asm86subdfb[]=
{ {"ffreep",0x18,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,45000},
  {"fxch7",0x19,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,45001},
  {"fstp8",0x1a,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,45002},
  {"fstp9",0x1b,PROC_FROM8086,0,ARG_FREG,ARG_NONE,ARG_NONE,45003},
  {"fucomip",0x1d,PROC_FROMPENTPRO,0,ARG_REG_ST0,ARG_FREG,ARG_NONE,45004},
  {"fcomip",0x1e,PROC_FROMPENTPRO,0,ARG_FREG,ARG_NONE,ARG_NONE,45005},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdf/ modrm = 0xe0-0xff
asminstdata asm86subdfc[]=
{ {"fstsw",0xe0,PROC_FROM8086,FLAGS_16BIT,ARG_REG_AX,ARG_NONE,ARG_NONE,46000},
  {"fstdw",0xe1,PROC_FROM80386,0,ARG_REG_AX,ARG_NONE,ARG_NONE,46001},
  {"fstsg",0xe2,PROC_FROM80386,0,ARG_REG_AX,ARG_NONE,ARG_NONE,46002},
  {"frinear",0xe2,PROC_FROM80386,0,ARG_NONE,ARG_NONE,ARG_NONE,46003},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmz80[]=
{ {"nop",0x00,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100000},
  {"ld",0x01,PROC_Z80,0,ARG_REG_BC,ARG_IMM16,ARG_NONE,100001},
  {"ld",0x02,PROC_Z80,0,ARG_REG_BC_IND,ARG_REG_A,ARG_NONE,100002},
  {"inc",0x03,PROC_Z80,0,ARG_REG_BC,ARG_NONE,ARG_NONE,100003},
  {"inc",0x04,PROC_Z80,0,ARG_REG_B,ARG_NONE,ARG_NONE,100004},
  {"dec",0x05,PROC_Z80,0,ARG_REG_B,ARG_NONE,ARG_NONE,100005},
  {"ld",0x06,PROC_Z80,0,ARG_REG_B,ARG_IMM8,ARG_NONE,100006},
  {"rlca",0x07,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100007},
  {"ex",0x08,PROC_Z80,0,ARG_REG_AF,ARG_REG_AF2,ARG_NONE,100008},
  {"add",0x09,PROC_Z80,0,ARG_REG_HL,ARG_REG_BC,ARG_NONE,100009},
  {"ld",0x0a,PROC_Z80,0,ARG_REG_A,ARG_REG_BC_IND,ARG_NONE,100010},
  {"dec",0x0b,PROC_Z80,0,ARG_REG_BC,ARG_NONE,ARG_NONE,100011},
  {"inc",0x0c,PROC_Z80,0,ARG_REG_C,ARG_NONE,ARG_NONE,100012},
  {"dec",0x0d,PROC_Z80,0,ARG_REG_C,ARG_NONE,ARG_NONE,100013},
  {"ld",0x0e,PROC_Z80,0,ARG_REG_C,ARG_IMM8,ARG_NONE,100014},
  {"rrca",0x0f,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100015},
  {"djnz",0x10,PROC_Z80,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100016},
  {"ld",0x11,PROC_Z80,0,ARG_REG_DE,ARG_IMM16,ARG_NONE,100017},
  {"ld",0x12,PROC_Z80,0,ARG_REG_DE_IND,ARG_REG_A,ARG_NONE,100018},
  {"inc",0x13,PROC_Z80,0,ARG_REG_DE,ARG_NONE,ARG_NONE,100019},
  {"inc",0x14,PROC_Z80,0,ARG_REG_D,ARG_NONE,ARG_NONE,100020},
  {"dec",0x15,PROC_Z80,0,ARG_REG_D,ARG_NONE,ARG_NONE,100021},
  {"ld",0x16,PROC_Z80,0,ARG_REG_D,ARG_IMM8,ARG_NONE,100022},
  {"rla",0x17,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100023},
  {"jr",0x18,PROC_Z80,FLAGS_JMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100024},
  {"add",0x19,PROC_Z80,0,ARG_REG_HL,ARG_REG_DE,ARG_NONE,100025},
  {"ld",0x1a,PROC_Z80,0,ARG_REG_A,ARG_REG_DE_IND,ARG_NONE,100026},
  {"dec",0x1b,PROC_Z80,0,ARG_REG_DE,ARG_NONE,ARG_NONE,100027},
  {"inc",0x1c,PROC_Z80,0,ARG_REG_E,ARG_NONE,ARG_NONE,100028},
  {"dec",0x1d,PROC_Z80,0,ARG_REG_E,ARG_NONE,ARG_NONE,100029},
  {"ld",0x1e,PROC_Z80,0,ARG_REG_E,ARG_IMM8,ARG_NONE,100030},
  {"rra",0x1f,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100031},
  {"jr nz",0x20,PROC_Z80,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100032},
  {"ld",0x21,PROC_Z80,0,ARG_REG_HL,ARG_IMM16,ARG_NONE,100033},
  {"ld",0x22,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_HL,ARG_NONE,100034},
  {"inc",0x23,PROC_Z80,0,ARG_REG_HL,ARG_NONE,ARG_NONE,100035},
  {"inc",0x24,PROC_Z80,0,ARG_REG_H,ARG_NONE,ARG_NONE,100036},
  {"dec",0x25,PROC_Z80,0,ARG_REG_H,ARG_NONE,ARG_NONE,100037},
  {"ld",0x26,PROC_Z80,0,ARG_REG_H,ARG_IMM8,ARG_NONE,100038},
  {"daa",0x27,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100039},
  {"jr z",0x28,PROC_Z80,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100040},
  {"add",0x29,PROC_Z80,0,ARG_REG_HL,ARG_REG_HL,ARG_NONE,100041},
  {"ld",0x2a,PROC_Z80,0,ARG_REG_HL,ARG_MEMLOC16,ARG_NONE,100042},
  {"dec",0x2b,PROC_Z80,0,ARG_REG_HL,ARG_NONE,ARG_NONE,100043},
  {"inc",0x2c,PROC_Z80,0,ARG_REG_L,ARG_NONE,ARG_NONE,100044},
  {"dec",0x2d,PROC_Z80,0,ARG_REG_L,ARG_NONE,ARG_NONE,100045},
  {"ld",0x2e,PROC_Z80,0,ARG_REG_L,ARG_IMM8,ARG_NONE,100046},
  {"cpl",0x2f,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100047},
  {"jr nc",0x30,PROC_Z80,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100048},
  {"ld",0x31,PROC_Z80,0,ARG_REG_SP,ARG_IMM16,ARG_NONE,100049},
  {"ld",0x32,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_A,ARG_NONE,100050},
  {"inc",0x33,PROC_Z80,0,ARG_REG_SP,ARG_NONE,ARG_NONE,100051},
  {"inc",0x34,PROC_Z80,0,ARG_REG_HL_IND,ARG_NONE,ARG_NONE,100052},
  {"dec",0x35,PROC_Z80,0,ARG_REG_HL_IND,ARG_NONE,ARG_NONE,100053},
  {"ld",0x36,PROC_Z80,0,ARG_REG_HL_IND,ARG_IMM8,ARG_NONE,100054},
  {"scf",0x37,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100055},
  {"jr c",0x38,PROC_Z80,FLAGS_CJMP,ARG_RELIMM8,ARG_NONE,ARG_NONE,100056},
  {"add",0x39,PROC_Z80,0,ARG_REG_HL,ARG_REG_SP,ARG_NONE,100057},
  {"ld",0x3a,PROC_Z80,0,ARG_REG_A,ARG_MEMLOC16,ARG_NONE,100058},
  {"dec",0x3b,PROC_Z80,0,ARG_REG_SP,ARG_NONE,ARG_NONE,100059},
  {"inc",0x3c,PROC_Z80,0,ARG_REG_A,ARG_NONE,ARG_NONE,100060},
  {"dec",0x3d,PROC_Z80,0,ARG_REG_A,ARG_NONE,ARG_NONE,100061},
  {"ld",0x3e,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100062},
  {"ccf",0x3f,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100063},
  {"ld",0x40,PROC_Z80,0,ARG_REG_B,ARG_REG_B,ARG_NONE,100064},
  {"ld",0x41,PROC_Z80,0,ARG_REG_B,ARG_REG_C,ARG_NONE,100065},
  {"ld",0x42,PROC_Z80,0,ARG_REG_B,ARG_REG_D,ARG_NONE,100066},
  {"ld",0x43,PROC_Z80,0,ARG_REG_B,ARG_REG_E,ARG_NONE,100067},
  {"ld",0x44,PROC_Z80,0,ARG_REG_B,ARG_REG_H,ARG_NONE,100068},
  {"ld",0x45,PROC_Z80,0,ARG_REG_B,ARG_REG_L,ARG_NONE,100069},
  {"ld",0x46,PROC_Z80,0,ARG_REG_B,ARG_REG_HL_IND,ARG_NONE,100070},
  {"ld",0x47,PROC_Z80,0,ARG_REG_B,ARG_REG_A,ARG_NONE,100071},
  {"ld",0x48,PROC_Z80,0,ARG_REG_C,ARG_REG_B,ARG_NONE,100072},
  {"ld",0x49,PROC_Z80,0,ARG_REG_C,ARG_REG_C,ARG_NONE,100073},
  {"ld",0x4a,PROC_Z80,0,ARG_REG_C,ARG_REG_D,ARG_NONE,100074},
  {"ld",0x4b,PROC_Z80,0,ARG_REG_C,ARG_REG_E,ARG_NONE,100075},
  {"ld",0x4c,PROC_Z80,0,ARG_REG_C,ARG_REG_H,ARG_NONE,100076},
  {"ld",0x4d,PROC_Z80,0,ARG_REG_C,ARG_REG_L,ARG_NONE,100077},
  {"ld",0x4e,PROC_Z80,0,ARG_REG_C,ARG_REG_HL_IND,ARG_NONE,100078},
  {"ld",0x4f,PROC_Z80,0,ARG_REG_C,ARG_REG_A,ARG_NONE,100079},
  {"ld",0x50,PROC_Z80,0,ARG_REG_D,ARG_REG_B,ARG_NONE,100080},
  {"ld",0x51,PROC_Z80,0,ARG_REG_D,ARG_REG_C,ARG_NONE,100081},
  {"ld",0x52,PROC_Z80,0,ARG_REG_D,ARG_REG_D,ARG_NONE,100082},
  {"ld",0x53,PROC_Z80,0,ARG_REG_D,ARG_REG_E,ARG_NONE,100083},
  {"ld",0x54,PROC_Z80,0,ARG_REG_D,ARG_REG_H,ARG_NONE,100084},
  {"ld",0x55,PROC_Z80,0,ARG_REG_D,ARG_REG_L,ARG_NONE,100085},
  {"ld",0x56,PROC_Z80,0,ARG_REG_D,ARG_REG_HL_IND,ARG_NONE,100086},
  {"ld",0x57,PROC_Z80,0,ARG_REG_D,ARG_REG_A,ARG_NONE,100087},
  {"ld",0x58,PROC_Z80,0,ARG_REG_E,ARG_REG_B,ARG_NONE,100088},
  {"ld",0x59,PROC_Z80,0,ARG_REG_E,ARG_REG_C,ARG_NONE,100089},
  {"ld",0x5a,PROC_Z80,0,ARG_REG_E,ARG_REG_D,ARG_NONE,100090},
  {"ld",0x5b,PROC_Z80,0,ARG_REG_E,ARG_REG_E,ARG_NONE,100091},
  {"ld",0x5c,PROC_Z80,0,ARG_REG_E,ARG_REG_H,ARG_NONE,100092},
  {"ld",0x5d,PROC_Z80,0,ARG_REG_E,ARG_REG_L,ARG_NONE,100093},
  {"ld",0x5e,PROC_Z80,0,ARG_REG_E,ARG_REG_HL_IND,ARG_NONE,100094},
  {"ld",0x5f,PROC_Z80,0,ARG_REG_E,ARG_REG_A,ARG_NONE,100095},
  {"ld",0x60,PROC_Z80,0,ARG_REG_H,ARG_REG_B,ARG_NONE,100096},
  {"ld",0x61,PROC_Z80,0,ARG_REG_H,ARG_REG_C,ARG_NONE,100097},
  {"ld",0x62,PROC_Z80,0,ARG_REG_H,ARG_REG_D,ARG_NONE,100098},
  {"ld",0x63,PROC_Z80,0,ARG_REG_H,ARG_REG_E,ARG_NONE,100099},
  {"ld",0x64,PROC_Z80,0,ARG_REG_H,ARG_REG_H,ARG_NONE,100100},
  {"ld",0x65,PROC_Z80,0,ARG_REG_H,ARG_REG_L,ARG_NONE,100101},
  {"ld",0x66,PROC_Z80,0,ARG_REG_H,ARG_REG_HL_IND,ARG_NONE,100102},
  {"ld",0x67,PROC_Z80,0,ARG_REG_H,ARG_REG_A,ARG_NONE,100103},
  {"ld",0x68,PROC_Z80,0,ARG_REG_L,ARG_REG_B,ARG_NONE,100104},
  {"ld",0x69,PROC_Z80,0,ARG_REG_L,ARG_REG_C,ARG_NONE,100105},
  {"ld",0x6a,PROC_Z80,0,ARG_REG_L,ARG_REG_D,ARG_NONE,100106},
  {"ld",0x6b,PROC_Z80,0,ARG_REG_L,ARG_REG_E,ARG_NONE,100107},
  {"ld",0x6c,PROC_Z80,0,ARG_REG_L,ARG_REG_H,ARG_NONE,100108},
  {"ld",0x6d,PROC_Z80,0,ARG_REG_L,ARG_REG_L,ARG_NONE,100109},
  {"ld",0x6e,PROC_Z80,0,ARG_REG_L,ARG_REG_HL_IND,ARG_NONE,100110},
  {"ld",0x6f,PROC_Z80,0,ARG_REG_L,ARG_REG_A,ARG_NONE,100111},
  {"ld",0x70,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_B,ARG_NONE,100112},
  {"ld",0x71,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_C,ARG_NONE,100113},
  {"ld",0x72,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_D,ARG_NONE,100114},
  {"ld",0x73,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_E,ARG_NONE,100115},
  {"ld",0x74,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_H,ARG_NONE,100116},
  {"ld",0x75,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_L,ARG_NONE,100117},
  {"halt",0x76,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100118},
  {"ld",0x77,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_A,ARG_NONE,100119},
  {"ld",0x78,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100120},
  {"ld",0x79,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100121},
  {"ld",0x7a,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100122},
  {"ld",0x7b,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100123},
  {"ld",0x7c,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100124},
  {"ld",0x7d,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100125},
  {"ld",0x7e,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100126},
  {"ld",0x7f,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100127},
  {"add",0x80,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100128},
  {"add",0x81,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100129},
  {"add",0x82,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100130},
  {"add",0x83,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100131},
  {"add",0x84,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100132},
  {"add",0x85,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100133},
  {"add",0x86,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100134},
  {"add",0x87,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100135},
  {"adc",0x88,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100136},
  {"adc",0x89,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100137},
  {"adc",0x8a,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100138},
  {"adc",0x8b,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100139},
  {"adc",0x8c,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100140},
  {"adc",0x8d,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100141},
  {"adc",0x8e,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100142},
  {"adc",0x8f,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100143},
  {"sub",0x90,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100144},
  {"sub",0x91,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100145},
  {"sub",0x92,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100146},
  {"sub",0x93,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100147},
  {"sub",0x94,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100148},
  {"sub",0x95,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100149},
  {"sub",0x96,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100150},
  {"sub",0x97,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100151},
  {"sbc",0x98,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100152},
  {"sbc",0x99,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100153},
  {"sbc",0x9a,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100154},
  {"sbc",0x9b,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100155},
  {"sbc",0x9c,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100156},
  {"sbc",0x9d,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100157},
  {"sbc",0x9e,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100158},
  {"sbc",0x9f,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100159},
  {"and",0xa0,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100160},
  {"and",0xa1,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100161},
  {"and",0xa2,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100162},
  {"and",0xa3,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100163},
  {"and",0xa4,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100164},
  {"and",0xa5,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100165},
  {"and",0xa6,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100166},
  {"and",0xa7,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100167},
  {"xor",0xa8,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100168},
  {"xor",0xa9,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100169},
  {"xor",0xaa,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100170},
  {"xor",0xab,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100171},
  {"xor",0xac,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100172},
  {"xor",0xad,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100173},
  {"xor",0xae,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100174},
  {"xor",0xaf,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100175},
  {"or",0xb0,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100176},
  {"or",0xb1,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100177},
  {"or",0xb2,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100178},
  {"or",0xb3,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100179},
  {"or",0xb4,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100180},
  {"or",0xb5,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100181},
  {"or",0xb6,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100182},
  {"or",0xb7,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100183},
  {"cp",0xb8,PROC_Z80,0,ARG_REG_A,ARG_REG_B,ARG_NONE,100184},
  {"cp",0xb9,PROC_Z80,0,ARG_REG_A,ARG_REG_C,ARG_NONE,100185},
  {"cp",0xba,PROC_Z80,0,ARG_REG_A,ARG_REG_D,ARG_NONE,100186},
  {"cp",0xbb,PROC_Z80,0,ARG_REG_A,ARG_REG_E,ARG_NONE,100187},
  {"cp",0xbc,PROC_Z80,0,ARG_REG_A,ARG_REG_H,ARG_NONE,100188},
  {"cp",0xbd,PROC_Z80,0,ARG_REG_A,ARG_REG_L,ARG_NONE,100189},
  {"cp",0xbe,PROC_Z80,0,ARG_REG_A,ARG_REG_HL_IND,ARG_NONE,100190},
  {"cp",0xbf,PROC_Z80,0,ARG_REG_A,ARG_REG_A,ARG_NONE,100191},
  {"ret nz",0xc0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100192},
  {"pop",0xc1,PROC_Z80,0,ARG_REG_BC,ARG_NONE,ARG_NONE,100193},
  {"jp nz",0xc2,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100194},
  {"jp",0xc3,PROC_Z80,FLAGS_JMP,ARG_IMM16,ARG_NONE,ARG_NONE,100195},
  {"call nz",0xc4,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100196},
  {"push",0xc5,PROC_Z80,0,ARG_REG_BC,ARG_NONE,ARG_NONE,100197},
  {"add",0xc6,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100198},
  {"rst 00h",0xc7,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100199},
  {"ret z",0xc8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100200},
  {"ret",0xc9,PROC_Z80,FLAGS_RET,ARG_NONE,ARG_NONE,ARG_NONE,100201},
  {"jp z",0xca,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100202},
  {NULL,0xcb,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100203}, // subtable 0xcb
  {"call z",0xcc,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100204},
  {"call",0xcd,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100205},
  {"adc",0xce,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100206},
  {"rst 08h",0xcf,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100207},
  {"ret nc",0xd0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100208},
  {"pop",0xd1,PROC_Z80,0,ARG_REG_DE,ARG_NONE,ARG_NONE,100209},
  {"jp nc",0xd2,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100210},
  {"out",0xd3,PROC_Z80,0,ARG_IMM8_IND,ARG_REG_A,ARG_NONE,100211},
  {"call nc",0xd4,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100212},
  {"push",0xd5,PROC_Z80,0,ARG_REG_DE,ARG_NONE,ARG_NONE,100213},
  {"sub",0xd6,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100214},
  {"rst 10h",0xd7,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100215},
  {"ret c",0xd8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100216},
  {"exx",0xd9,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100217},
  {"jp c",0xda,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100218},
  {"in",0xdb,PROC_Z80,0,ARG_REG_A,ARG_IMM8_IND,ARG_NONE,100219},
  {"call c",0xdc,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100220},
  {NULL,0xdd,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100221}, // subtable 0xdd
  {"sbc",0xde,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100222},
  {"rst 18h",0xdf,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100223},
  {"ret po",0xe0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100224},
  {"pop",0xe1,PROC_Z80,0,ARG_REG_HL,ARG_NONE,ARG_NONE,100225},
  {"jp po",0xe2,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100226},
  {"ex",0xe3,PROC_Z80,0,ARG_REG_SP_IND,ARG_REG_HL,ARG_NONE,100227},
  {"call po",0xe4,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100228},
  {"push",0xe5,PROC_Z80,0,ARG_REG_HL,ARG_NONE,ARG_NONE,100229},
  {"and",0xe6,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100230},
  {"rst 20h",0xe7,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100231},
  {"ret pe",0xe8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100232},
  {"jp",0xe9,PROC_Z80,FLAGS_IJMP,ARG_REG_HL_IND,ARG_NONE,ARG_NONE,100233},
  {"jp pe",0xea,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100234},
  {"ex",0xeb,PROC_Z80,0,ARG_REG_DE,ARG_REG_HL,ARG_NONE,100235},
  {"call pe",0xec,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100236},
  {NULL,0xed,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100237}, // subtable 0xed
  {"xor",0xee,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100238},
  {"rst 28h",0xef,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100239},
  {"ret p",0xf0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100240},
  {"pop",0xf1,PROC_Z80,0,ARG_REG_AF,ARG_NONE,ARG_NONE,100241},
  {"jp p",0xf2,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100242},
  {"di",0xf3,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100243},
  {"call p",0xf4,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100244},
  {"push",0xf5,PROC_Z80,0,ARG_REG_AF,ARG_NONE,ARG_NONE,100245},
  {"or",0xf6,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100246},
  {"rst 30h",0xf7,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100247},
  {"ret m",0xf8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100248},
  {"ld",0xf9,PROC_Z80,0,ARG_REG_SP,ARG_REG_HL,ARG_NONE,100249},
  {"jp m",0xfa,PROC_Z80,FLAGS_CJMP,ARG_IMM16,ARG_NONE,ARG_NONE,100250},
  {"ei",0xfb,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100251},
  {"call m",0xfc,PROC_Z80,FLAGS_CALL,ARG_IMM16,ARG_NONE,ARG_NONE,100252},
  {NULL,0xfd,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100253}, // subtable 0xfd
  {"cp",0xfe,PROC_Z80,0,ARG_REG_A,ARG_IMM8,ARG_NONE,100254},
  {"rst 38h",0xff,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,100255},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xcb
// - num = second byte/8
// - reg = second byte&7
asminstdata asmz80subcba[]=
{ {"rlc",0x0,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101000},
  {"rrc",0x1,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101001},
  {"rl",0x2,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101002},
  {"rr",0x3,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101003},
  {"sla",0x4,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101004},
  {"sra",0x5,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101005},
  {"srl",0x7,PROC_Z80,0,ARG_REG,ARG_NONE,ARG_NONE,101006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xcb part2
// - num = second byte bits 7,6
// - bit = second byte bits 5,4,3
// - reg = second byte bits 2,1,0
asminstdata asmz80subcbb[]=
{ {"bit",0x1,PROC_Z80,0,ARG_BIT,ARG_REG,ARG_NONE,102000},
  {"res",0x2,PROC_Z80,0,ARG_BIT,ARG_REG,ARG_NONE,102001},
  {"set",0x3,PROC_Z80,0,ARG_BIT,ARG_REG,ARG_NONE,102002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// subtable 0xdd
asminstdata asmz80subdd[]=
{ {"add",0x09,PROC_Z80,0,ARG_REG_IX,ARG_REG_BC,ARG_NONE,103000},
  {"add",0x19,PROC_Z80,0,ARG_REG_IX,ARG_REG_DE,ARG_NONE,103001},
  {"ld",0x21,PROC_Z80,0,ARG_REG_IX,ARG_IMM16,ARG_NONE,103002},
  {"ld",0x22,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_IX,ARG_NONE,103003},
  {"inc",0x23,PROC_Z80,0,ARG_REG_IX,ARG_NONE,ARG_NONE,103004},
  {"add",0x29,PROC_Z80,0,ARG_REG_IX,ARG_REG_IX,ARG_NONE,103005},
  {"ld",0x2a,PROC_Z80,0,ARG_REG_IX,ARG_MEMLOC16,ARG_NONE,103006},
  {"dec",0x2b,PROC_Z80,0,ARG_REG_IX,ARG_NONE,ARG_NONE,103007},
  {"inc",0x34,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,103008},
  {"dec",0x35,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,103009},
  {"ld",0x36,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_IMM8,ARG_NONE,103010},
  {"add",0x39,PROC_Z80,0,ARG_REG_IX,ARG_REG_SP,ARG_NONE,103011},
  {"ld",0x46,PROC_Z80,FLAGS_INDEXREG,ARG_REG_B,ARG_REG_IX_IND,ARG_NONE,103012},
  {"ld",0x4e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_C,ARG_REG_IX_IND,ARG_NONE,103013},
  {"ld",0x56,PROC_Z80,FLAGS_INDEXREG,ARG_REG_D,ARG_REG_IX_IND,ARG_NONE,103014},
  {"ld",0x5e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_E,ARG_REG_IX_IND,ARG_NONE,103015},
  {"ld",0x66,PROC_Z80,FLAGS_INDEXREG,ARG_REG_H,ARG_REG_IX_IND,ARG_NONE,103016},
  {"ld",0x6e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_L,ARG_REG_IX_IND,ARG_NONE,103017},
  {"ld",0x70,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_B,ARG_NONE,103018},
  {"ld",0x71,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_C,ARG_NONE,103019},
  {"ld",0x72,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_D,ARG_NONE,103020},
  {"ld",0x73,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_E,ARG_NONE,103021},
  {"ld",0x74,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_H,ARG_NONE,103022},
  {"ld",0x75,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_L,ARG_NONE,103023},
  {"ld",0x77,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_REG_A,ARG_NONE,103024},
  {"ld",0x7e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103025},
  {"add",0x86,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103026},
  {"adc",0x8e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103027},
  {"sub",0x96,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103028},
  {"sbc",0x9e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103029},
  {"and",0xa6,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103030},
  {"xor",0xae,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103031},
  {"or",0xb6,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103032},
  {"cp",0xbe,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IX_IND,ARG_NONE,103033},
  {NULL,0xcb,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,103034}, // subtable 0xdd/0xcb
  {"pop",0xe1,PROC_Z80,0,ARG_REG_IX,ARG_NONE,ARG_NONE,103035},
  {"ex",0xe3,PROC_Z80,0,ARG_REG_SP_IND,ARG_REG_IX,ARG_NONE,103036},
  {"push",0xe5,PROC_Z80,0,ARG_REG_IX,ARG_NONE,ARG_NONE,103037},
  {"jp",0xe9,PROC_Z80,FLAGS_IJMP,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,103038},
  {"ld",0xf9,PROC_Z80,0,ARG_REG_SP,ARG_REG_IX,ARG_NONE,103039},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xdd/0xcb
// - num = fourth byte
asminstdata asmz80subddcba[]=
{ {"rlc",0x06,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104000},
  {"rrc",0x0e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104001},
  {"rl",0x16,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104002},
  {"rr",0x1e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104003},
  {"sla",0x26,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104004},
  {"sra",0x2e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104005},
  {"srl",0x3e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IX_IND,ARG_NONE,ARG_NONE,104006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xdd/0xcb part2
// - num = second byte bits 7,6,2,1,0
// - bit = second byte bits 5,4,3
asminstdata asmz80subddcbb[]=
{ {"bit",0x46,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IX_IND,ARG_NONE,105000},
  {"res",0x86,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IX_IND,ARG_NONE,105001},
  {"set",0xc6,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IX_IND,ARG_NONE,105002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmz80subed[]=
{ {"in",0x40,PROC_Z80,0,ARG_REG_B,ARG_REG_C_IND,ARG_NONE,106000},
  {"out",0x41,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_B,ARG_NONE,106001},
  {"sbc",0x42,PROC_Z80,0,ARG_REG_HL,ARG_REG_BC,ARG_NONE,106002},
  {"ld",0x43,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_BC,ARG_NONE,106003},
  {"neg",0x44,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106004},
  {"retn",0x45,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106005},
  {"im 0",0x46,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106006},
  {"ld",0x47,PROC_Z80,0,ARG_REG_I,ARG_REG_A,ARG_NONE,106007},
  {"in",0x48,PROC_Z80,0,ARG_REG_C,ARG_REG_C_IND,ARG_NONE,106008},
  {"out",0x49,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_C,ARG_NONE,106009},
  {"adc",0x4a,PROC_Z80,0,ARG_REG_HL,ARG_REG_BC,ARG_NONE,106010},
  {"ld",0x4b,PROC_Z80,0,ARG_REG_BC,ARG_MEMLOC16,ARG_NONE,106011},
  {"reti",0x4d,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106012},
  {"ld",0x4f,PROC_Z80,0,ARG_REG_R,ARG_REG_A,ARG_NONE,106013},
  {"in",0x50,PROC_Z80,0,ARG_REG_D,ARG_REG_C_IND,ARG_NONE,106014},
  {"out",0x51,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_D,ARG_NONE,106015},
  {"sbc",0x52,PROC_Z80,0,ARG_REG_HL,ARG_REG_DE,ARG_NONE,106016},
  {"ld",0x53,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_DE,ARG_NONE,106017},
  {"im 1",0x56,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106018},
  {"ld",0x57,PROC_Z80,0,ARG_REG_A,ARG_REG_I,ARG_NONE,106019},
  {"in",0x58,PROC_Z80,0,ARG_REG_E,ARG_REG_C_IND,ARG_NONE,106020},
  {"out",0x59,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_E,ARG_NONE,106021},
  {"adc",0x5a,PROC_Z80,0,ARG_REG_HL,ARG_REG_DE,ARG_NONE,106022},
  {"ld",0x5b,PROC_Z80,0,ARG_REG_DE,ARG_MEMLOC16,ARG_NONE,106023},
  {"im 2",0x5e,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106024},
  {"ld",0x5f,PROC_Z80,0,ARG_REG_A,ARG_REG_R,ARG_NONE,106025},
  {"in",0x60,PROC_Z80,0,ARG_REG_H,ARG_REG_C_IND,ARG_NONE,106026},
  {"out",0x61,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_H,ARG_NONE,106027},
  {"sbc",0x62,PROC_Z80,0,ARG_REG_HL,ARG_REG_HL,ARG_NONE,106028},
  {"ld",0x63,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_HL,ARG_NONE,106029},
  {"rrd",0x67,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106030},
  {"in",0x68,PROC_Z80,0,ARG_REG_L,ARG_REG_C_IND,ARG_NONE,106031},
  {"out",0x69,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_L,ARG_NONE,106032},
  {"adc",0x6a,PROC_Z80,0,ARG_REG_HL,ARG_REG_HL,ARG_NONE,106033},
  {"ld",0x6b,PROC_Z80,0,ARG_REG_HL,ARG_MEMLOC16,ARG_NONE,106034},
  {"rld",0x6f,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106035},
  {"in",0x70,PROC_Z80,0,ARG_REG_HL_IND,ARG_REG_C_IND,ARG_NONE,106036},
  {"out",0x71,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_HL_IND,ARG_NONE,106037},
  {"sbc",0x72,PROC_Z80,0,ARG_REG_HL,ARG_REG_SP,ARG_NONE,106038},
  {"ld",0x73,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_SP,ARG_NONE,106039},
  {"in",0x78,PROC_Z80,0,ARG_REG_A,ARG_REG_C_IND,ARG_NONE,106040},
  {"out",0x79,PROC_Z80,0,ARG_REG_C_IND,ARG_REG_A,ARG_NONE,106041},
  {"adc",0x7a,PROC_Z80,0,ARG_REG_HL,ARG_REG_SP,ARG_NONE,106042},
  {"ld",0x7b,PROC_Z80,0,ARG_REG_SP,ARG_MEMLOC16,ARG_NONE,106043},
  {"ldi",0xa0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106044},
  {"cpi",0xa1,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106045},
  {"ini",0xa2,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106046},
  {"outi",0xa3,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106047},
  {"ldd",0xa8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106048},
  {"cpd",0xa9,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106049},
  {"ind",0xaa,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106050},
  {"outd",0xab,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106051},
  {"ldir",0xb0,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106052},
  {"cpir",0xb1,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106053},
  {"inir",0xb2,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106054},
  {"outir",0xb3,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106055},
  {"lddr",0xb8,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106056},
  {"cpdr",0xb9,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106057},
  {"indr",0xba,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106058},
  {"otdr",0xbb,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,106059},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmz80subfd[]=
{ {"add",0x09,PROC_Z80,0,ARG_REG_IY,ARG_REG_BC,ARG_NONE,107000},
  {"add",0x19,PROC_Z80,0,ARG_REG_IY,ARG_REG_DE,ARG_NONE,107001},
  {"ld",0x21,PROC_Z80,0,ARG_REG_IY,ARG_IMM16,ARG_NONE,107002},
  {"ld",0x22,PROC_Z80,0,ARG_MEMLOC16,ARG_REG_IY,ARG_NONE,107003},
  {"inc",0x23,PROC_Z80,0,ARG_REG_IY,ARG_NONE,ARG_NONE,107004},
  {"add",0x29,PROC_Z80,0,ARG_REG_IY,ARG_REG_IX,ARG_NONE,107005},
  {"ld",0x2a,PROC_Z80,0,ARG_REG_IY,ARG_MEMLOC16,ARG_NONE,107006},
  {"dec",0x2b,PROC_Z80,0,ARG_REG_IY,ARG_NONE,ARG_NONE,107007},
  {"inc",0x34,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,107008},
  {"dec",0x35,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,107009},
  {"ld",0x36,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_IMM8,ARG_NONE,107010},
  {"add",0x39,PROC_Z80,0,ARG_REG_IY,ARG_REG_SP,ARG_NONE,107011},
  {"ld",0x46,PROC_Z80,FLAGS_INDEXREG,ARG_REG_B,ARG_REG_IY_IND,ARG_NONE,107012},
  {"ld",0x4e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_C,ARG_REG_IY_IND,ARG_NONE,107013},
  {"ld",0x56,PROC_Z80,FLAGS_INDEXREG,ARG_REG_D,ARG_REG_IY_IND,ARG_NONE,107014},
  {"ld",0x5e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_E,ARG_REG_IY_IND,ARG_NONE,107015},
  {"ld",0x66,PROC_Z80,FLAGS_INDEXREG,ARG_REG_H,ARG_REG_IY_IND,ARG_NONE,107016},
  {"ld",0x6e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_L,ARG_REG_IY_IND,ARG_NONE,107017},
  {"ld",0x70,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_B,ARG_NONE,107018},
  {"ld",0x71,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_C,ARG_NONE,107019},
  {"ld",0x72,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_D,ARG_NONE,107020},
  {"ld",0x73,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_E,ARG_NONE,107021},
  {"ld",0x74,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_H,ARG_NONE,107022},
  {"ld",0x75,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_L,ARG_NONE,107023},
  {"ld",0x77,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_REG_A,ARG_NONE,107024},
  {"ld",0x7e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107025},
  {"add",0x86,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107026},
  {"adc",0x8e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107027},
  {"sub",0x96,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107028},
  {"sbc",0x9e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107029},
  {"and",0xa6,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107030},
  {"xor",0xae,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107031},
  {"or",0xb6,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107032},
  {"cp",0xbe,PROC_Z80,FLAGS_INDEXREG,ARG_REG_A,ARG_REG_IY_IND,ARG_NONE,107033},
  {NULL,0xcb,PROC_Z80,0,ARG_NONE,ARG_NONE,ARG_NONE,107034}, // subtable 0xfd/0xcb
  {"pop",0xe1,PROC_Z80,0,ARG_REG_IY,ARG_NONE,ARG_NONE,107035},
  {"ex",0xe3,PROC_Z80,0,ARG_REG_SP_IND,ARG_REG_IY,ARG_NONE,107036},
  {"push",0xe5,PROC_Z80,0,ARG_REG_IY,ARG_NONE,ARG_NONE,107037},
  {"jp",0xe9,PROC_Z80,FLAGS_IJMP,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,107038},
  {"ld",0xf9,PROC_Z80,0,ARG_REG_SP,ARG_REG_IY,ARG_NONE,107039},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xfd/0xcb
// - num = fourth byte
asminstdata asmz80subfdcba[]=
{ {"rlc",0x06,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108000},
  {"rrc",0x0e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108001},
  {"rl",0x16,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108002},
  {"rr",0x1e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108003},
  {"sla",0x26,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108004},
  {"sra",0x2e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108005},
  {"srl",0x3e,PROC_Z80,FLAGS_INDEXREG,ARG_REG_IY_IND,ARG_NONE,ARG_NONE,108006},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

// z80 subtable 0xfd/0xcb part2
// - num = second byte bits 7,6,2,1,0
// - bit = second byte bits 5,4,3
asminstdata asmz80subfdcbb[]=
{ {"bit",0x46,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IY_IND,ARG_NONE,109000},
  {"res",0x86,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IY_IND,ARG_NONE,109001},
  {"set",0xc6,PROC_Z80,FLAGS_INDEXREG,ARG_BIT,ARG_REG_IY_IND,ARG_NONE,109002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmdword[]=
{ {"dd",0x00,PROC_ALL,0,ARG_IMM32,ARG_NONE,ARG_NONE,200000},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmword[]=
{ {"dw",0x00,PROC_ALL,0,ARG_IMM16,ARG_NONE,ARG_NONE,201000},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asmstr[]=
{ {"ds",0x00,PROC_ALL,0,ARG_STRING,ARG_NONE,ARG_NONE,202000},
  {"ds",0x00,PROC_ALL,0,ARG_PSTRING,ARG_NONE,ARG_NONE,202001},
  {"ds",0x00,PROC_ALL,0,ARG_DOSSTRING,ARG_NONE,ARG_NONE,202002},
  {"ds",0x00,PROC_ALL,0,ARG_CUNICODESTRING,ARG_NONE,ARG_NONE,202003},
  {"ds",0x00,PROC_ALL,0,ARG_PUNICODESTRING,ARG_NONE,ARG_NONE,202004},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asminstdata asm_fp[]=
{ {"dword",0x00,PROC_ALL,0,ARG_IMM_SINGLE,ARG_NONE,ARG_NONE,203000},
  {"qword",0x00,PROC_ALL,0,ARG_IMM_DOUBLE,ARG_NONE,ARG_NONE,203001},
  {"tbyte",0x00,PROC_ALL,0,ARG_IMM_LONGDOUBLE,ARG_NONE,ARG_NONE,203002},
  {NULL,0x00,0,0,ARG_NONE,ARG_NONE,ARG_NONE,0}  //end marker - processor=0 & opcode=0
};

asmtable tables86[]=
{ {asm86,TABLE_MAIN,0,0,0,0xff,0,0xff,0},
  {asm86sub0f,TABLE_EXT,0x0f,0,0,0xff,0,0xff,1},
  {asm86sub80,TABLE_EXT,0x80,0,8,0x07,0,0xff,0},
  {asm86sub81,TABLE_EXT,0x81,0,8,0x07,0,0xff,0},
  {asm86sub82,TABLE_EXT,0x82,0,8,0x07,0,0xff,0},
  {asm86sub83,TABLE_EXT,0x83,0,8,0x07,0,0xff,0},
  {asm86subc0,TABLE_EXT,0xc0,0,8,0x07,0,0xff,0},
  {asm86subc1,TABLE_EXT,0xc1,0,8,0x07,0,0xff,0},
  {asm86subd0,TABLE_EXT,0xd0,0,8,0x07,0,0xff,0},
  {asm86subd1,TABLE_EXT,0xd1,0,8,0x07,0,0xff,0},
  {asm86subd2,TABLE_EXT,0xd2,0,8,0x07,0,0xff,0},
  {asm86subd3,TABLE_EXT,0xd3,0,8,0x07,0,0xff,0},
  {asm86subf6,TABLE_EXT,0xf6,0,8,0x07,0,0xff,0},
  {asm86subf7,TABLE_EXT,0xf7,0,8,0x07,0,0xff,0},
  {asm86subfe,TABLE_EXT,0xfe,0,8,0x07,0,0xff,0},
  {asm86subff,TABLE_EXT,0xff,0,8,0x07,0,0xff,0},
  {asm86sub0f00,TABLE_EXT2,0x0f,0x00,8,0x07,0,0xff,1},
  {asm86sub0f01,TABLE_EXT2,0x0f,0x01,8,0x07,0,0xff,1},
  {asm86sub0f18,TABLE_EXT2,0x0f,0x18,8,0x07,0,0xff,1},
  {asm86sub0f71,TABLE_EXT2,0x0f,0x71,8,0x07,0,0xff,1},
  {asm86sub0f72,TABLE_EXT2,0x0f,0x72,8,0x07,0,0xff,1},
  {asm86sub0f73,TABLE_EXT2,0x0f,0x73,8,0x07,0,0xff,1},
  {asm86sub0fae,TABLE_EXT2,0x0f,0xae,8,0x07,0,0xff,1},
  {asm86sub0fba,TABLE_EXT2,0x0f,0xba,8,0x07,0,0xff,1},
  {asm86sub0fc2,TABLE_EXT2,0x0f,0xc2,0,0x00,0,0xff,1},
  {asm86sub0fc7,TABLE_EXT2,0x0f,0xc7,8,0x07,0,0xff,1},
  {asm86subd8a,TABLE_EXT,0xd8,0,8,0x07,0,0xbf,0},
  {asm86subd8b,TABLE_EXT,0xd8,0,8,0x1f,0xc0,0xff,0},
  {asm86subd9a,TABLE_EXT,0xd9,0,8,0x07,0,0xbf,0},
  {asm86subd9b,TABLE_EXT,0xd9,0,8,0x1f,0xc0,0xff,0},
  {asm86subd9c,TABLE_EXT,0xd9,0,1,0xff,0xc0,0xff,1},
  {asm86subdaa,TABLE_EXT,0xda,0,8,0x07,0,0xbf,0},
  {asm86subdab,TABLE_EXT,0xda,0,8,0x1f,0xc0,0xff,0},
  {asm86subdac,TABLE_EXT,0xda,0,1,0xff,0xc0,0xff,1},
  {asm86subdba,TABLE_EXT,0xdb,0,8,0x07,0,0xbf,0},
  {asm86subdbb,TABLE_EXT,0xdb,0,8,0x1f,0xc0,0xff,0},
  {asm86subdbc,TABLE_EXT,0xdb,0,1,0xff,0xc0,0xff,1},
  {asm86subdca,TABLE_EXT,0xdc,0,8,0x07,0,0xbf,0},
  {asm86subdcb,TABLE_EXT,0xdc,0,8,0x1f,0xc0,0xff,0},
  {asm86subdda,TABLE_EXT,0xdd,0,8,0x07,0,0xbf,0},
  {asm86subddb,TABLE_EXT,0xdd,0,8,0x1f,0xc0,0xff,0},
  {asm86subdea,TABLE_EXT,0xde,0,8,0x07,0,0xbf,0},
  {asm86subdeb,TABLE_EXT,0xde,0,8,0x1f,0xc0,0xff,0},
  {asm86subdec,TABLE_EXT,0xde,0,1,0xff,0xc0,0xff,1},
  {asm86subdfa,TABLE_EXT,0xdf,0,8,0x07,0,0xbf,0},
  {asm86subdfb,TABLE_EXT,0xdf,0,8,0x1f,0xc0,0xff,0},
  {asm86subdfc,TABLE_EXT,0xdf,0,1,0xff,0xc0,0xff,1},
  {NULL,0,0,0,0,0,0,0,0}
};

asmtable tablesz80[]=
{ {asmz80,TABLE_MAIN,0,0,0,0xff,0,0xff,0},
  {asmz80subcba,TABLE_EXT,0xcb,0,8,0x1f,0,0x39,0},
  {asmz80subcbb,TABLE_EXT,0xcb,0,64,0x03,0x40,0xff,0},
  {asmz80subdd,TABLE_EXT,0xdd,0,0,0xff,0,0xff,0},
  {asmz80subddcba,TABLE_EXT2,0xdd,0xcb,0,0xff,0,0xff,0},
  {asmz80subddcbb,TABLE_EXT2,0xdd,0xcb,0,0xc7,0,0xff,0},
  {asmz80subed,TABLE_EXT,0xed,0,0,0xff,0,0xff,0},
  {asmz80subfd,TABLE_EXT,0xfd,0,0,0xff,0,0xff,0},
  {asmz80subfdcba,TABLE_EXT2,0xfd,0xcb,0,0xff,0,0xff,0},
  {asmz80subfdcbb,TABLE_EXT2,0xfd,0xcb,0,0xc7,0,0xff,0},
  {NULL,0,0,0,0,0,0,0,0}
};

asminstdata *reconstruct[]=
{ asm86,asm86sub0f,asm86sub80,asm86sub81,asm86sub82,asm86sub83,asm86subc0,asm86subc1,asm86subd0,
  asm86subd1,asm86subd2,asm86subd3,asm86subf6,asm86subf7,asm86subfe,asm86subff,asm86sub0f00,
  asm86sub0f01,asm86sub0f18,asm86sub0f71,asm86sub0f72,asm86sub0f73,asm86sub0fae,asm86sub0fba,
  asm86sub0fc2,asm86sub0fc7,asm86subd8a,asm86subd8b,asm86subd9a,asm86subd9b,asm86subd9c,
  asm86subdaa,asm86subdab,asm86subdac,asm86subdba,asm86subdbb,asm86subdbc,asm86subdca,asm86subdcb,
  asm86subdda,asm86subddb,asm86subdea,asm86subdeb,asm86subdec,asm86subdfa,asm86subdfb,asm86subdfc,
  asmz80,asmz80subcba,asmz80subcbb,asmz80subdd,asmz80subddcba,asmz80subddcbb,asmz80subed,asmz80subfd,
  asmz80subfdcba,asmz80subfdcbb,asmdword,asmword,asmstr,asm_fp,NULL
};


