/************************************************************************
*                 relocs.cpp                                            *
* This class maintains a list of relocation items from the exe file.    *
* Borg keeps a list of any items which have been or should be relocated *
* when an executable is loaded. The class includes functions to         *
* maintain the list including loading and saving to a database file,    *
* and including relocating the image in memory when the database and    *
* the file are reloaded.                                                *
* The main reason for doing this is that during disassembly we can gain *
* additional information by knowing that a certain location is          *
* relocated on file load. For example with an instruction like          *
* mov eax,400000h, if the location containing 400000h is a relocation   *
* item then we know that 400000h is an offset, and so we can            *
* immediately rewrite it as mov eax, offset 400000h                     *
************************************************************************/

#include <windows.h>

#include "relocs.h"
#include "data.h"
#include "debug.h"
#include "dasm.h"

/************************************************************************
* forward declarations                                                  *
************************************************************************/
int relocscfunc(listitem i,listitem j);
void relocsdfunc(listitem i);

/************************************************************************
* constructor function                                                  *
* - simply sets the compare and deletion function for the relocation    *
*   items list                                                          *
************************************************************************/
relocs::relocs()
{ setcomparefunc(relocscfunc);
  setdelfunc(relocsdfunc);
}

/************************************************************************
* destructor function                                                   *
* - currently null                                                      *
************************************************************************/
relocs::~relocs()
{
}

/************************************************************************
* addreloc                                                              *
* - adds a reloc item to the list of relocs                             *
* - a reloc item consists of only a location which is in the reloc      *
*   table, and a type for the relocation                                *
************************************************************************/
void relocs::addreloc(lptr loc,reloctype type)
{ relocitem *newname;
  newname=new relocitem;
  newname->addr=loc;
  newname->type=type;
  addto((listitem)newname);
#ifdef DEBUG
  DebugMessage("Added Reloc : %04lx:%04lx",loc.segm,loc.offs);
#endif
}

/************************************************************************
* isreloc                                                               *
* - this is the function used throughout the disassembly engine to see  *
*   if somewhere was relocated. It returns true if the location is in   *
*   the table                                                           *
************************************************************************/
BOOL relocs::isreloc(lptr loc)
{ relocitem checkit,*findit;
  checkit.addr=loc;
  findit=(relocitem *)find((listitem)&checkit);
  if(findit!=NULL) if(findit->addr==loc) return TRUE;
  return FALSE;
}

/************************************************************************
* the compare function for the reloc items                              *
* - relocs are sorted by location                                       *
************************************************************************/
int relocscfunc(listitem i,listitem j)
{ if(((relocitem *)i)->addr==((relocitem *)j)->addr)
	 return 0;
  if(((relocitem *)i)->addr>((relocitem *)j)->addr)
	 return 1;
  return -1;
}

/************************************************************************
* the deletion function for reloc items                                 *
* - relocs are simply deleted                                           *
************************************************************************/
void relocsdfunc(listitem i)
{ delete (relocitem *)i;
}

/************************************************************************
* savedb                                                                *
* - this saves the relocs list to the database file.                    *
* - we can simply save the number of items followed by each item        *
************************************************************************/
BOOL relocs::savedb(savefile *sf)
{ dword nrels;
  relocitem *currrel;
  nrels=numlistitems();
  resetiterator();
  // save number of reloc items
  if(!sf->swrite(&nrels,sizeof(dword)))return FALSE;
  while(nrels)
  { currrel=(relocitem *)nextiterator();
	 // save each reloc item
	 if(!sf->swrite(currrel,sizeof(relocitem)))return FALSE;
	 nrels--;
  }
  return TRUE;
}

/************************************************************************
* loaddb                                                                *
* - this reloads the list of relocs from the database file and          *
*   constructs the list again                                           *
************************************************************************/
BOOL relocs::loaddb(savefile *sf)
{ dword nrels,num;
  relocitem *currrel;
  // get number of items
  if(!sf->sread(&nrels,sizeof(dword),&num))return FALSE;
  while(nrels)
  { currrel=new relocitem;
	 // load each and add to list
	 if(!sf->sread(currrel,sizeof(relocitem),&num))return FALSE;
	 addto((listitem)currrel);
	 nrels--;
  }
  return TRUE;
}

/************************************************************************
* relocfile                                                             *
* - this should be called after loading a database file. It goes        *
*   through all of the relocs and relocates the file again since when   *
*   we load a database we simply load the file image and do not work    *
*   our way through the whole file format again.                        *
************************************************************************/
BOOL relocs::relocfile(void)
{ dsegitem *ds;
  relocitem *ri;
  resetiterator();
  ri=(relocitem *)nextiterator();
  while(ri!=NULL)
  { // relocate item.
	 ds=dta.findseg(ri->addr);
	 if(ds!=NULL)   // changed in build 14, used to return FALSE if not found
	 { switch(ri->type)
		{ case RELOC_NONE:
			 break;
		  case RELOC_SEG:
			 ((word *)(&(ds->data[ri->addr.offs-ds->addr.offs])))[0]+=options.loadaddr.segm;
			 break;
		  case RELOC_OFFS16:
			 break;
		  case RELOC_OFFS32:
			 break;
		  case RELOC_SEGOFFS16:
			 break;
		  case RELOC_SEGOFFS32:
			 break;
		  default:
			 return FALSE;
		}
	 }
	 ri=(relocitem *)nextiterator();
  }
  return TRUE;
}
