/************************************************************************
*                   savefile.cpp                                        *
* File I/O for saving to and reading from database files only.          *
* Added in version 2.11. Compression is implemented based on rle, but   *
* it is nibble based (2*nibbles=byte) rather than byte based. I decided *
* on this after closer examination of the database files. It results in *
* around a 30% reduction to the database file size.                     *
************************************************************************/

#include <windows.h>

#include "savefile.h"
#include "dasm.h"
#include "debug.h"

/************************************************************************
* Constructor - sets up some variables used in the compression and save *
************************************************************************/
savefile::savefile()
{	sfile=INVALID_HANDLE_VALUE;
	rbufflen=0;
   rbuffptr=0;
   rbhigh=TRUE;
   rlecount=0;
   rlemode=FALSE;
   rlestart=TRUE;
}

/************************************************************************
* Destructor - closes the database file if it is still open             *
************************************************************************/
savefile::~savefile()
{  sclose();
}

/************************************************************************
* sopen                                                                 *
* - opens the database file and returns TRUE on success                 *
************************************************************************/
BOOL savefile::sopen(LPCTSTR lpFileName,DWORD dwDesiredAccess,DWORD dwShareMode,
			DWORD dwCreationDistribution,DWORD dwFlagsAndAttributes)
{	sfile=CreateFile(lpFileName,dwDesiredAccess,dwShareMode,NULL,
				dwCreationDistribution,dwFlagsAndAttributes,NULL);
   if(sfile==INVALID_HANDLE_VALUE)
   { MessageBox(mainwindow,"File open failed ?",lpFileName,MB_OK|MB_ICONEXCLAMATION);
	  return FALSE;
   }
   if(GetFileType(sfile)!=FILE_TYPE_DISK)
   { MessageBox(mainwindow,"File open failed ?",lpFileName,MB_OK|MB_ICONEXCLAMATION);
	  sclose();
	  return FALSE;
   }
	return TRUE;
}

/************************************************************************
* sclose                                                                *
* - closes the database file if still open                              *
************************************************************************/
void savefile::sclose(void)
{  if(sfile!=INVALID_HANDLE_VALUE) CloseHandle(sfile);
}

/************************************************************************
* getnibble                                                             *
* - This function sets n to the next nibble from the file, it uses      *
*   buffering and reads from the file as required                       *
************************************************************************/
BOOL savefile::getnibble(byte *n)
{  BOOL rval;
	if(rbuffptr<rbufflen)
	{	if(rbhigh)
   	{	(*n)=(byte)(rbuff[rbuffptr]>>4);
      	rbhigh=FALSE;
      	return TRUE;
      }
      (*n)=(byte)(rbuff[rbuffptr]&0x0f);
      rbhigh=TRUE;
      rbuffptr++;
      return TRUE;
   }
   else
   {  rval=ReadFile(sfile,rbuff,RBUFF_MAXLEN,&rbufflen,NULL);
   	rbhigh=TRUE;
      rbuffptr=0;
      if((rval)&&(rbufflen)) return getnibble(n);
      return FALSE;
   }
}

/************************************************************************
* getrlenibble                                                          *
* - this function sets n to the next nibble from the file, taking into  *
*   account rle encoding. So this returns the next uncompressed nibble  *
* - rle encoding is:                                                    *
*   rle_code count nibble                                               *
*   count is number-1 (so it can encode from 2 to 16 nibbles)           *
*   or rle_code 0 signifies a nibble equal to rle_code                  *
* - note that rle_code is a constant specified in the savefile.h which  *
*   indicates an rle encoding, and is currently 0x0f. Do not set this   *
*   constant to 0 as this would be inefficient....                      *
************************************************************************/
BOOL savefile::getrlenibble(byte *n)
{	if(rlemode)
	{	rlecount--;
   	if(!rlecount)rlemode=FALSE;
      (*n)=(byte)(rlebyte);
      return TRUE;
   }
   if(!getnibble(&rlebyte)) return FALSE;
   if(rlebyte==rle_code)
   {	if(!getnibble(&rlebyte)) return FALSE;
   	if(rlebyte)
      {	rlecount=rlebyte;
      	rlemode=TRUE;
         if(!getnibble(&rlebyte)) return FALSE;
   	   (*n)=rlebyte;
         return TRUE;
      }
   	(*n)=rle_code;
	   return TRUE;
   }
   (*n)=rlebyte;
   return TRUE;
}

/************************************************************************
* putnibble                                                             *
* - The opposite function to getnibble, it writes one nibble to the     *
*   file using buffering                                                *
************************************************************************/
BOOL savefile::putnibble(byte n)
{	BOOL rval;
	dword num;
	if(rbuffptr<RBUFF_MAXLEN)
   {	if(rbhigh)
   	{	rbuff[rbuffptr]=(byte)((n&0x0f)<<4);
      	rbhigh=FALSE;
         rbufflen++;
         return TRUE;
      }
      rbuff[rbuffptr]+=(byte)(n&0x0f);
      rbhigh=TRUE;
      rbuffptr++;
      return TRUE;
   }
   else
   {  rval=WriteFile(sfile,rbuff,RBUFF_MAXLEN,&num,NULL);
   	rbhigh=TRUE;
      rbuffptr=0;
      rbufflen=0;
      if(rval) return putnibble(n);
      return FALSE;
   }
}

/************************************************************************
* flushnibble                                                           *
* - A necessity of buffered writing, this flushes the remainder of the  *
*   buffer, writing it out to file                                      *
************************************************************************/
BOOL savefile::flushnibble(void)
{	dword num;
 	if(rbufflen)
		return WriteFile(sfile,rbuff,rbufflen,&num,NULL);
	return TRUE;
}

/************************************************************************
* putrlenibble                                                          *
* - This is the opposite function to getrlenibble. It writes nibbles to *
*   file whilst performing the compression. The rle encoding happens    *
*   here and when nibbles are ready to be written the putnibble         *
*   function is called                                                  *
************************************************************************/
BOOL savefile::putrlenibble(byte n)
{  if(rlestart)
	{	rlestart=FALSE;
   	rlebyte=n;
      return TRUE;
   }
	if(rlemode)
	{	if((rlebyte==n)&&(rlecount<0x0f))
   	{	rlecount++;
      	return TRUE;
      }
      if(!putnibble(rle_code)) return FALSE;
      if(!putnibble(rlecount)) return FALSE;
      if(!putnibble(rlebyte)) return FALSE;
      rlemode=FALSE;
      rlebyte=n;
      return TRUE;
   }
   if(rlebyte==n)
   {	rlemode=TRUE;
   	rlecount=1;
      return TRUE;
   }
   if(!putnibble(rlebyte)) return FALSE;
   if(rlebyte==rle_code)
   	if(!putnibble(0)) return FALSE;
   rlebyte=n;
   return TRUE;
}

/************************************************************************
* flushrlenibble                                                        *
* - This flushes any partial rle at the end of a file and forces it to  *
*   the putnibble function                                              *
************************************************************************/
BOOL savefile::flushrlenibble(void)
{  if(rlemode)
   {  if(!putnibble(rle_code)) return FALSE;
      if(!putnibble(rlecount)) return FALSE;
      if(!putnibble(rlebyte)) return FALSE;
   }
	else
   {	if(!putnibble(rlebyte)) return FALSE;
		if(rlebyte==rle_code)
   		if(!putnibble(0)) return FALSE;
   }
   return TRUE;
}

/************************************************************************
* flushfilewrite                                                        *
* - The function to flush writing which should be called at the end of  *
*   the save. It flushes any partial encoding and then flushes the      *
*   buffered write                                                      *
************************************************************************/
BOOL savefile::flushfilewrite(void)
{	if(!flushrlenibble()) return FALSE;
	return flushnibble();
}

/************************************************************************
* sread                                                                 *
* - This is the external call for reading from a file. Its a similar    *
*   format to ReadFile, and uses the rle compression routines           *
************************************************************************/
BOOL savefile::sread(LPVOID lpBuffer,DWORD nNumberOfBytesToRead,LPDWORD lpNumberOfBytesRead)
{  byte n;
	dword num;
   (*lpNumberOfBytesRead)=0;
   for(num=0;num<nNumberOfBytesToRead;num++)
   {  if(!getrlenibble(&n)) return FALSE;
   	((byte *)lpBuffer)[num]=(byte)(n<<4);
      if(!getrlenibble(&n)) return FALSE;
      ((byte *)lpBuffer)[num]+=n;
      (*lpNumberOfBytesRead)++;
   }
   return TRUE;
}

/************************************************************************
* swrite                                                                *
* - This is the external call for writing to a file. Its a similar      *
*   format to WriteFile, and uses the rle compression routines          *
************************************************************************/
BOOL savefile::swrite(LPCVOID lpBuffer,DWORD nNumberOfBytesToWrite)
{  dword num;
	for(num=0;num<nNumberOfBytesToWrite;num++)
   {	if(!putrlenibble((byte)(((byte *)lpBuffer)[num]>>4))) return FALSE;
   	if(!putrlenibble((byte)(((byte *)lpBuffer)[num]&0x0f)))return FALSE;
   }
   return TRUE;
}


