/*
 * uncc - The ultimate C decompiler
 * Copyright (C) 2003  Megabug <megabug@autistici.org>,
 *                     Little-John <littlejohn@autistici.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>

#include "expressions.h"

//! Fatal output message generator
/*!
*/
void expr_fatal( char *fmt, ... )
{
	va_list ap;
	va_start(ap,fmt);
	fprintf(stderr,"Expression FATAL: ");
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr,"\n");
	exit(1);
}

//! This is issued when a new expression is being generated
/*!
 Here we mallocate the memory needed for the new expression.
 Actually there are 3 kind of expressions:
 - EXPR_OP_SUM : for the add opcode
 - EXPR_OP_SUB : for the sub opcode
 - EXPR_OP_AND : for the and opcode
 - EXPR_OP_MULT : for the mul opcode
 - EXPR_OP_ASSIGN : for the mov opcode
 - EXPR_STRING : this is a general purpose expression type
                 and it handles the ret, left and right values
                 of sum and assignments
 - EXPR_INT : contains an integer
 - EXPR_OP_EQUAL, EXPR_OP_NOT_EQUAL, EXPR_LESS_EQUAL : ==, !=, <= operators
 - EXPR_IF : if->goto command. sx is the condition, dx is the label to goto
 - EXPR_GOTO: goto command, we only get the labels (dx)
 - EXPR_LABEL : label, printed with a : instead ;
 - EXPR_OP_INC : (...)++
 - EXPR_OP_ADDRESS : &(...)
 - EXPR_OP_DEREFERENCE : *(...) statement
 - EXPR_VARIABLE : variable name

 The sx_fill and dx_fill expression structure values are initialized
 to 0.
 \param s - is the string for the EXPR_STRING
 \param t - the type of the expression
 \return The new mallocated expression

*/

Expr *EXPR_new(void *s, ExpressionType t)
{
	Expr *n;
	n = (Expr *)malloc(sizeof(Expr));
	if (!n) expr_fatal("Not enough memory");
	switch (t) {
		case EXPR_OP_SUM:
		case EXPR_OP_SUB:
	        case EXPR_OP_AND:
	        case EXPR_OP_MULT:
	        case EXPR_OP_EQUAL:
	        case EXPR_OP_NOT_EQUAL:
	        case EXPR_OP_LESS_EQUAL:
		case EXPR_OP_ASSIGN:
		case EXPR_OP_INC:
		case EXPR_IF:
		case EXPR_OP_ADDRESS:
		case EXPR_OP_DEREFERENCE:
			break;
		case EXPR_GOTO:
		case EXPR_STRING:
		case EXPR_LABEL:
			n->data._pchar = strdup(s);
			break;
		case EXPR_INT:
			expr_fatal("EXPR_new with EXPR_INT type. Use EXPR_new_int instead");
			break;
		case EXPR_VARIABLE:
			n->data._pvar = s;
			break;
		default:
			expr_fatal("Unknown type");
	};
	n->type=t;
	n->sx_fill = 0;
	n->dx_fill = 0;
	return n;
}

Expr *EXPR_new_int(int number)
{
	Expr *n;
	n = (Expr *)malloc(sizeof(Expr));
	if (!n) expr_fatal("Not enough memory");
	n->data._pint = (int *)malloc(sizeof(int));
	*(n->data._pint) = number;
	n->type=EXPR_INT;
	n->sx_fill = 0;
	n->dx_fill = 0;
	return n;
}

/*! Links the sx_fill to an expression
 \param parent - parent expression
 \param child - child expression
 \return The parent is modified
*/
void EXPR_link_sx(Expr *parent, Expr *child)
{
	if (parent->sx_fill) expr_fatal("SX Fill failure");
	parent->sx_fill=1;
	parent->sx=child;
}

/*! Links the dx_fill to an expression
 \param parent - parent expression
 \param child - child expression
 \return The parent is modified
*/
void EXPR_link_dx(Expr *parent, Expr *child)
{
	if (parent->dx_fill) expr_fatal("DX Fill failure");
	parent->dx_fill=1;
	parent->dx=child;
}

/*! Creates a new expression and sx_links a child to it
 \param data - contents of the new expression
 \param t - the type of the expression
 \param child - the child expression to link
 \return The new expression
*/
Expr *EXPR_new_sx(void *data, ExpressionType t, Expr *child)
{
	Expr *n = EXPR_new(data,t);
	EXPR_link_sx(n,child);
	return n;
}

/*! Creates a new expression and dx_links a child to it
 \param data - contents of the new expression
 \param t - the type of the expression
 \param child - the child expression to link
 \return The new expression
*/
Expr *EXPR_new_dx(void *data, ExpressionType t, Expr *child)
{
	Expr *n = EXPR_new(data,t);
	EXPR_link_dx(n,child);
	return n;
}

/*! Creates a new expression, given the sx_fill and dx_fill
 \param s - contents of the new expression
 \param t - the type of the expression
 \param sx - sx expression to be linked
 \param dx - dx expression to be linked
 \return The new expression
*/
Expr *EXPR_new_linked(void *s, ExpressionType t, Expr *sx, Expr *dx)
{
	Expr *n = EXPR_new_dx(s,t,dx);
	EXPR_link_sx(n,sx);
	return n;
}

//! Check if the expression is an EXPR_LABEL
int EXPR_is_label(Expr *e)
{ return e->type==EXPR_LABEL; }

//! Evaluate arithmetical expressions
int EXPR_eval(Expr *e, int *res)
{
	int a,b;
	switch (e->type) {
		case EXPR_STRING:
		case EXPR_VARIABLE:
		case EXPR_IF:
		case EXPR_GOTO:
		case EXPR_LABEL:
	        case EXPR_OP_INC:
	        case EXPR_OP_AND:
	        case EXPR_OP_EQUAL:
	        case EXPR_OP_LESS_EQUAL:
	        case EXPR_OP_NOT_EQUAL:
		case EXPR_OP_ASSIGN:
	        case EXPR_OP_ADDRESS:
		case EXPR_OP_DEREFERENCE:
			return 0;
		case EXPR_INT:
			*res = *(e->data._pint);
			return 1;
		case EXPR_OP_SUM:
			if ( !EXPR_eval(e->sx, &a) ) return 0;
			if ( !EXPR_eval(e->dx, &b) ) return 0;
			*res = a+b;
			return 1;
		case EXPR_OP_SUB:
			if ( !EXPR_eval(e->sx, &a) ) return 0;
			if ( !EXPR_eval(e->dx, &b) ) return 0;
			*res = a-b;
			return 1;
	        case EXPR_OP_MULT:
			if ( !EXPR_eval(e->sx, &a) ) return 0;
			if ( !EXPR_eval(e->dx, &b) ) return 0;
			*res = a*b;
			return 1;
		default:
			return 0;
	};
}

//! Walks through the expression, and prints each node to a file
void EXPR_print_tree(Expr *parent, FILE *t)
{
	if (parent==NULL)
	{
		fprintf(t,"<NULL>");
		return;
	}

#define p_sx if (parent->sx_fill) EXPR_print_tree(parent->sx,t)
#define p_dx if (parent->dx_fill) EXPR_print_tree(parent->dx,t)
#define p fprintf
	switch (parent->type) {
		case EXPR_STRING:
			p_sx; p(t,"%s",parent->data._pchar); p_dx; break;
		case EXPR_INT:
			p_sx; p(t,"%d",*(parent->data._pint)); p_dx; break;
		case EXPR_VARIABLE:
			p_sx; p(t,"%s",FUNCTIONS_var_name(parent->data._pvar)); p_dx; break;
		case EXPR_OP_INC:
			p(t,"("); p_sx; p_dx; p(t,")++"); break;
		case EXPR_OP_SUM:
			p(t,"("); p_sx; p(t,"+"); p_dx; p(t,")"); break;
		case EXPR_OP_SUB:
			p(t,"("); p_sx; p(t,"-"); p_dx; p(t,")"); break;
	        case EXPR_OP_AND:
			p(t,"("); p_sx; p(t,"&"); p_dx; p(t,")"); break;
	        case EXPR_OP_MULT:
			p(t,"("); p_sx; p(t,"*"); p_dx; p(t,")"); break;
	        case EXPR_OP_EQUAL:
			p(t,"("); p_sx; p(t,"=="); p_dx; p(t,")"); break;
	        case EXPR_OP_NOT_EQUAL:
			p(t,"("); p_sx; p(t,"!="); p_dx; p(t,")"); break;
	        case EXPR_OP_LESS_EQUAL:
			p(t,"("); p_sx; p(t,"<="); p_dx; p(t,")"); break;
		case EXPR_OP_ASSIGN:
			p_sx; p(t,"="); p_dx; break;
		case EXPR_IF:
			p(t,"if ("); p_sx; p(t, ") goto "); p_dx; break;
		case EXPR_GOTO:
			p(t,"goto "); p(t,"%s",parent->data._pchar); break;
		case EXPR_LABEL:
			p(t,"%s:",parent->data._pchar); break;
	        case EXPR_OP_ADDRESS:
			p(t,"&("); p_sx; p(t,")"); break;
		case EXPR_OP_DEREFERENCE:
			p(t,"*("); p_sx; p(t,")"); break;
		default:
			expr_fatal("Unknown type (print_tree)");
	};

}

Expr *EXPR_copy(Expr *o)
{
	Expr *n;
	if (!o) return NULL;

	n = (Expr *)malloc(sizeof(Expr));
	memcpy(n,o,sizeof(Expr));

	switch (o->type) {
		case EXPR_GOTO:
		case EXPR_STRING:
		case EXPR_LABEL:
			n->data._pchar = strdup(o->data._pchar); 
			break;
		case EXPR_INT:
			n->data._pint = (int *)malloc(sizeof(int));
			*(n->data._pint) = *(o->data._pint);
			break;
		case EXPR_VARIABLE:
		case EXPR_OP_INC:
		case EXPR_OP_SUM:
		case EXPR_OP_SUB:
	        case EXPR_OP_AND:
	        case EXPR_OP_MULT:
	        case EXPR_OP_EQUAL:
	        case EXPR_OP_NOT_EQUAL:
	        case EXPR_OP_LESS_EQUAL:
		case EXPR_OP_ASSIGN:
		case EXPR_OP_ADDRESS:
		case EXPR_OP_DEREFERENCE:
		case EXPR_IF:
			break;
		default:
			expr_fatal("Unknown type (copy)");
	};
	if (o->sx_fill) n->sx = EXPR_copy(o->sx);
	if (o->dx_fill) n->dx = EXPR_copy(o->dx);
	return n;
}

//! Deallocates an expression
void EXPR_free(Expr *parent)
{
	if (!parent) return;
	if (parent->sx_fill) EXPR_free(parent->sx);
	if (parent->dx_fill) EXPR_free(parent->dx);

	switch (parent->type) {
		case EXPR_GOTO:
		case EXPR_STRING:
		case EXPR_LABEL:
			free(parent->data._pchar); 
			break;
		case EXPR_INT:
			free(parent->data._pint); 
			break;
		case EXPR_VARIABLE:
		case EXPR_OP_INC:
		case EXPR_OP_SUM:
		case EXPR_OP_SUB:
	        case EXPR_OP_AND:
	        case EXPR_OP_MULT:
	        case EXPR_OP_EQUAL:
	        case EXPR_OP_NOT_EQUAL:
	        case EXPR_OP_LESS_EQUAL:
		case EXPR_OP_ASSIGN:
		case EXPR_OP_ADDRESS:
		case EXPR_OP_DEREFERENCE:
		case EXPR_IF:
			break;
		default:
			expr_fatal("Unknown type (free)");
	};
	free(parent);
}

