/**************************************************
*                                                 *
*             brainfuck Interpreter               *
*                by Kwazy Webbit                  *
*                  18-01-2006                     *
*                                                 *
*    http://reteam.org  &  http://its.mine.nu     *
*                                                 *
**************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

//Enable COMMENTS to skip C-style comments starting with //
//Disabled by default because it is not brainfuck standard

#define COMMENTS

//Enable DEBUG to allow the debug operation using #
//Is implemented in SOME brainfuck implementations, but not all, so also disabled by default

#define DEBUG

#define KILOBYTE 1024

#define MAX_VAR_COUNT 64*KILOBYTE

char aMemory[MAX_VAR_COUNT];
int nCurMem;
int nLastMem;

#define MAX_WHILE_COUNT KILOBYTE	//4*KILOBYTE bytes

struct WHILE_STRUCT
{
	int start;
	int end;
};

WHILE_STRUCT aWhiles[MAX_WHILE_COUNT];
int nCurWhile;

char *aCode;
int nCodeSize;
int nCurPos;

void loadFile(char*);
void initMachine();
void runProgram();
void parseCommand();
void doInc();
void doDec();
void doNext();
void doPrev();
void doStartWhile();
void doEndWhile();
void doInput();
void doOutput();

#ifdef DEBUG
	void doDebug();
#endif

#ifdef COMMENTS
	void doComment();
#endif

int main (int argc, char** args)
{
	if( argc<=1 )
	{
		printf("Error: missing filename parameter");
		exit(1);
	}

	loadFile( args[1] );

	initMachine();

	runProgram();

	free( aCode );

	return 0;
}

void error(char* pErrorMsg)
{
	int nLine=1, nToken=1;
	
	for(int i=0; i<nCurPos; i++)
	{
		nToken++;
		if(aCode[i]=='\n') {
			nLine++;
			nToken = 1;
		}
	}
	
	printf("\n");
	printf(" * Error (line %d, token %d): %s\n", nLine, nToken, pErrorMsg);
	
	exit(1);
}

void loadFile(char* pFilename)
{
	FILE* pFile;
	unsigned long nSize;

	// open file for reading
	pFile = fopen( pFilename, "rb" );
	if (pFile==NULL)
	{
		printf("Error: Could not open file %s", pFilename);
		exit(1);
	}

	// obtain file size.
	fseek( pFile , 0 , SEEK_END );
	nCodeSize = ftell( pFile );
	rewind( pFile );

	// allocate memory for code
	aCode = (char*) malloc( nCodeSize );
	if (aCode == NULL)
	{
		printf("Error: Could not allocate %lu bytes of memory", nCodeSize);
		exit(1);
	}

	// read code into memory
	fread( aCode, 1, nCodeSize, pFile );

	fclose( pFile );
}

void initMachine()
{
	memset( aMemory, 0, MAX_VAR_COUNT );

	nCurMem = 0;
	nLastMem = 0;

	nCurWhile = 0;
}

void runProgram()
{
	// Run all commands
	for( nCurPos=0; nCurPos<nCodeSize; nCurPos++ ) 
		parseCommand();
}

void parseCommand()
{
	switch( aCode[nCurPos] )
	{
		case '+':
			doInc();
			break;
		case '-':
			doDec();
			break;
		case '>':
			doNext();
			break;
		case '<':
			doPrev();
			break;
		case '[':
			doStartWhile();
			break;
		case ']':
			doEndWhile();
			break;
		case ',':
			doInput();
			break;
		case '.':
			doOutput();
			break;
#ifdef DEBUG			
		case '#':
			doDebug();
			break;
#endif
#ifdef COMMENTS
		case '/':
			if(nCurPos<(nCodeSize-1) && aCode[nCurPos+1]=='/') {
				doComment();
			}
			break;
#endif
		default:
			return;
	}
}

void doInc()
{
	aMemory[nCurMem]++;
}

void doDec()
{
	aMemory[nCurMem]--;
}

void doNext()
{
	if( nCurMem==(MAX_VAR_COUNT-1) )
	{
		error("Too many variables used.");
	}

	nCurMem++;
	
	if(nCurMem>nLastMem)
		nLastMem = nCurMem;
}

void doPrev()
{
	if( nCurMem<=0 )
	{
		error("Invalid variable id (-1)");
	}

	nCurMem--;
}

void doStartWhile()
{
	// Encountered a new while loop
	if( aWhiles[nCurWhile].start!=(nCurPos-1) )
	{
		nCurWhile++;
		if(nCurWhile>=MAX_WHILE_COUNT) {
			error("Too many [] loops.");
		}
		
		aWhiles[nCurWhile].start = nCurPos-1;

		// Find matching close tag
		int nOpenWhiles = 1;
		for( int i=(nCurPos+1) ; i<nCodeSize; i++ )
		{
			// Nested loop found
			if( aCode[i] == '[' )
			{
				nOpenWhiles++;
			}
			else if( aCode[i] == ']' )
			{
				// Close (possibly nested) loop
				nOpenWhiles--;

				// This was the main loop
				if(!nOpenWhiles)
				{
					aWhiles[nCurWhile].end = i;
					break;
				}
			}
		}

		// We've run out of code without closing the loop
		if( nOpenWhiles!=0 )
		{
			error("[ without matching ]");
			exit (1);
		}		
	}
	

	// Check the while-condition
	if( !aMemory[nCurMem] )
	{
		nCurPos = aWhiles[nCurWhile].end;
		nCurWhile--;
		return;
	}
}

void doEndWhile()
{
	// We're not in a loop
	if( !nCurWhile )
	{
		error("] found without matching [");
		exit(1);
	}

	nCurPos = aWhiles[nCurWhile].start;
}

void doInput()
{
	aMemory[nCurMem] = getchar();
}

void doOutput()
{
	putchar( aMemory[nCurMem] );
}

#ifdef DEBUG
void doDebug()
{
	printf("\n");
	printf("Debug information:\n");
	printf("------------------\n");
	
	if(nCurWhile!= 0)
		printf("Current while loop: %d - %d\n", aWhiles[nCurWhile].start, aWhiles[nCurWhile].end);
		
	printf("\n");
	
	for(int i=0; i<=nLastMem; i++)
	{
		
		printf("variable[%d]", i);
		
		if(i==nCurMem)
			printf("*");
			
		printf("\t = \'%c\' (0x%X)\n", aMemory[i], aMemory[i]);
	}
	
	printf("------------------\n");
	printf("\n");
}
#endif

#ifdef COMMENTS
void doComment()
{
	//skip second /
	nCurPos++;
	while(nCurPos<nCodeSize && aCode[nCurPos]!='\n')
		nCurPos++;
}
#endif
