/* ----------------------------------------------------------------------------- 
 * _Get_ASM v1.0
 * Plugin module for IDA Pro v4.30/IDASDK430
 * by: J.C. Roberts
 *
 *  NOTES:
 *      Here are two seperate ways to grab the ASM text from a particular
 *      address. The first way, get_asm_text1(), was made so that the resulting
 *      string is not dependent on the user settings in IDA. The second way,
 *      get_asm_text2(), _IS_ dependednt on user settings but uses the standard
 *      SDK API for getting ASM text, so it's a bit more simple.
 *
 * ----------------------------------------------------------------------------- 
 */

/* ----------------------------------------------------------------------------- 
 *
 * Copyright J.C. Roberts all rights reserved. Copyright notices in
 * the code are not to be removed.  If this code is used in a product,
 * J.C. Roberts should be given credit as the author of the parts used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials for products using of this software must
 *    display the following acknowledgement:
 *    "This product includes software developed by: J.C. Roberts"
 * 
 * THIS SOFTWARE IS PROVIDED BY J.C. Roberts ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence such as the GNU Public 
 * Licence or any other license.
 *
 * ----------------------------------------------------------------------------- 
 */


#include <ida.hpp>
#include <idp.hpp>
#include <bytes.hpp>
#include <loader.hpp>
#include <kernwin.hpp>
#include <name.hpp>
#include <entry.hpp>
#include <funcs.hpp>
#include <expr.hpp>
#include <fpro.h>
#include <lines.hpp>
#include <ua.hpp>


#include <diskio.hpp>
#include <string.h>  


//--------------------------------------------------------------------------
// This callback is called for UI notification events
static int sample_callback(void * /*user_data*/, int event_id, va_list /*va*/)
{
  if ( event_id != ui_msg )     // avoid recursion
    if ( event_id != ui_setstate
      && event_id != ui_showauto
      && event_id != ui_refreshmarked ) // ignore uninteresting events
                    msg("ui_callback %d\n", event_id);
  return 0;                     // 0 means "process the event"
                                // otherwise the event would be ignored
}


//--------------------------------------------------------------------------
//
//      Initialize.
//
//      IDA will call this function only once.
//      If this function returns PLGUIN_SKIP, IDA will never load it again.
//      If this function returns PLUGIN_OK, IDA will unload the plugin but
//      remember that the plugin agreed to work with the database.
//      The plugin will be loaded again if the user invokes it by
//      pressing the hotkey or selecting it from the menu.
//      After the second load the plugin will stay on memory.
//      If this function returns PLUGIN_KEEP, IDA will keep the plugin
//      in the memory. In this case the initialization function can hook
//      into the processor module and user interface notification points.
//      See the hook_to_notification_point() function.
//
//      In this example we check the input file format and make the decision.
//      You may or may not check any other conditions to decide what you do:
//      whether you agree to work with the database or not.
//
int init(void)
{
  if ( inf.filetype == f_ELF ) return PLUGIN_SKIP;

// Please uncomment the following line to see how the notification works
//  hook_to_notification_point(HT_UI, sample_callback, NULL);

// Please uncomment the following line to see how the user-defined prefix works
//  set_user_defined_prefix(prefix_width, get_user_defined_prefix);

  return PLUGIN_KEEP;
}

//--------------------------------------------------------------------------
//      Terminate.
//      Usually this callback is empty.
//      The plugin should unhook from the notification lists if
//      hook_to_notification_point() was used.
//
//      IDA will call this function when the user asks to exit.
//      This function won't be called in the case of emergency exits.

void term(void)
{
  unhook_from_notification_point(HT_UI, sample_callback);
  set_user_defined_prefix(0, NULL);
}


#define BUFLEN 1024


//--------------------------------------------------------------------------
//  
// This function is easier but it is dependent on the user settings in
//      Menu:       {Options}{General}
//      TAB:        "Disassembly"
//      CheckBox:   "Auto comments"
// and
//      CheckBox:   "Bad Instructions"
//
// If these options are turned on you end up with a string like:
// "db 8Dh,44h,24h,0 ; <BAD>lea     eax, [esp+514h+CPInfo]; Load Effective Address"
//
char *get_asm_text2 (ea_t sEA) {

    flags_t sFlags;                                 // Flags of EA
    char isn_buf[BUFLEN];                           // Instruction ASM text buffer
    char *isn_ptr;                                  // Instruction ASM text pointer
    isn_ptr = &isn_buf[0];

    sFlags = getFlags(sEA);                         // Get EA Flags
    if(isCode(sFlags)) {                            // Test ea is INS start.
        generate_disasm_line(sEA, isn_ptr);
        tag_remove(isn_ptr, isn_ptr, BUFLEN);       // remove color code tags
        return isn_ptr;
    } else {
        // Do error
        return "";      // error -the EA is not the start of an isntruction.
    }

}

//--------------------------------------------------------------------------
//  
// This function is consistent regardless of user settings. It will produce 
// a string like this:
// "lea eax, [esp+514h+CPInfo]"
// even if the bad instruction or Auto comments options are enabled. (see above)
//
char *get_asm_text1(ea_t sEA)
{
    int i = 0;                                      // Counter
    flags_t sFlags;                                 // Flags of EA

    char isn_buf[BUFLEN];                           // Instruction ASM text buffer
    char *isn_ptr;                                  // Instruction ASM text pointer
    isn_ptr = &isn_buf[0];
    
    char opr_buf[BUFLEN];                           // Operand ASM text buffer
    char *opr_ptr;                                  // Operand ASM text pointer
    opr_ptr = &opr_buf[0];


    sFlags = getFlags(sEA);                         // Get EA Flags
    
    if(isCode(sFlags)) {                            // Test ea is INS start.

        // Get the instruction ASM text
        ua_mnem(sEA, isn_ptr, BUFLEN);              // grab instruction text
        tag_remove(isn_ptr, isn_ptr, BUFLEN);       // remove color code tags
    
        // Get the operand ASM text
        ua_outop((ulong) sEA, opr_ptr, i);          // grab operand text
        tag_remove(opr_ptr, opr_ptr, BUFLEN);       // remove color code tags
        i++;
    
        while (*opr_ptr) {                          // cycle through oprs
            if (i == 1) {
                strncat(isn_ptr, " ", BUFLEN);      // ins <space> operand
            } else {
                strncat(isn_ptr, ", ", BUFLEN);     // opr <comma space> opr
            }
            strncat(isn_ptr, opr_ptr, BUFLEN);      // Make full string...
            ua_outop((ulong) sEA, opr_ptr, i);      // grab next operand text
            tag_remove(opr_ptr, opr_ptr, BUFLEN);   // remove color code tags
            i++;
        }
    
        return isn_ptr;

    } else {
        // Do error
        return "";      // error -the EA is not the start of an isntruction.
    }
}

//--------------------------------------------------------------------------
//
//      The main plugin method
//

void run(int arg)
{
    ea_t sEA;

    char full_buf[BUFLEN];                      // Full ASM text buffer
    char *full_ptr;                             // Full ASM text pointer
    full_ptr = &full_buf[0];    


    // Plugin start message
    msg("\n Start Test \n");

    // Get cursor effective address
    sEA = get_screen_ea();
    
    // Get the full ASM text
    qstrncpy(full_ptr, get_asm_text1(sEA), BUFLEN); 
    msg("#1 at %08Xl: %s\n", sEA, full_ptr);

    qstrncpy(full_ptr, get_asm_text2(sEA), BUFLEN); 
    msg("#2 at %08Xl: %s\n", sEA, full_ptr);

    // Plugin End message
    msg("\n End Test \n");

}

//--------------------------------------------------------------------------
char comment[] = "This is a sample plugin. It does nothing useful";

char help[] =
        "A sample plugin module\n"
        "\n"
        "This module shows you how to create plugin modules.\n"
        "\n"
        "It does nothing useful - just prints a message that is was called\n"
        "and shows the current address.\n";


//--------------------------------------------------------------------------
// This is the preferred name of the plugin module in the menu system
// The preferred name may be overriden in plugins.cfg file

char wanted_name[] = "_Get_ASM";


// This is the preferred hotkey for the plugin module
// The preferred hotkey may be overriden in plugins.cfg file
// Note: IDA won't tell you if the hotkey is not correct
//       It will just disable the hotkey.

char wanted_hotkey[] = "";


//--------------------------------------------------------------------------
//
//      PLUGIN DESCRIPTION BLOCK
//
//--------------------------------------------------------------------------

extern "C" plugin_t PLUGIN = {
  IDP_INTERFACE_VERSION,
  0,                    // plugin flags
  init,                 // initialize

  term,                 // terminate. this pointer may be NULL.

  run,                  // invoke plugin

  comment,              // long comment about the plugin
                        // it could appear in the status line
                        // or as a hint

  help,                 // multiline help about the plugin

  wanted_name,          // the preferred short name of the plugin
  wanted_hotkey         // the preferred hotkey to run the plugin
};
