/*
  HookLib is copyright (c) lallous <lallousx86@yahoo.com>
  You may freely use/modify/distribute this library as long as you keep/give appropriate credits.
*/

#include "hooklib.h"



//--------------------------------------------------------------------------
//
// This function returns a module handle from inside a given process ID       
// This function is very useful when you want to locate the base address of a DLL inside a given process
// 
//
BOOL FindModuleByProcessId(DWORD _ProcessID, LPCSTR modName, PDWORD _hModule)
{
  // load PsAPI
  if (!Load_PsApi())
    return FALSE;

  DWORD nModules, nCount = 1024;

  HANDLE hProcess;

  // buffer for process names
  TCHAR buf[MAX_PATH];

  // allocate memory for modules
  HMODULE *modules   = new HMODULE[nCount];

  BOOL bStopLoop = FALSE;

  // open process for querying only
  hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, _ProcessID);
    
  // now try to enum all modules
  if (!EnumProcessModules(hProcess, modules, nCount * sizeof(DWORD), &nModules))    
  {
    CloseHandle(hProcess);
    return FALSE;
  }

  // because nModules returned from EnumProcessModules() is in bytes, I divid by 4 to return n DWORDs
  nModules /= 4;

  for (DWORD j=0; j< nModules; j++)
  {
    // get module name
    GetModuleBaseName(hProcess, modules[j], buf, sizeof(buf));    

    // is that our desired module ?
    if (_tcsicmp(buf, modName) == 0)
    {
      // return hModule
      *_hModule    = (DWORD)modules[j];

      bStopLoop = TRUE;

      break;
    }
  }

  CloseHandle(hProcess);
  delete modules;
  return bStopLoop;
}

//--------------------------------------------------------------------------
//
// This function, given a process name (can be full path) and a module name that is used
// by the process it will then return both the process' ID and the module handle.
//
// The modulename can be the procName too, and that will return module of processname
// As if you were in the same processes and issued a GetModuleHandle(NULL)
//
//
//
BOOL FindModule(
  LPCTSTR procName,
  LPCTSTR modName,
  PDWORD _uProcessId,
  PDWORD _hModule,
  BOOL BaseNames,
  bool (*match) (LPCTSTR, LPCTSTR))
{
  // load PsAPI
  if (!Load_PsApi())
    return FALSE;

  DWORD uProcessId, nModules, nProcess, nCount = 1024;

  HANDLE hProcess;

  DWORD *processes = new DWORD[nCount];

  // buffer for process names
  CHAR buf[MAX_PATH];

  bool found;

  // enum all processes
  if (!EnumProcesses(processes, nCount * sizeof(DWORD), &nProcess))
  {
    delete processes;
    return FALSE;
  }

  // convert fron bytes count to items count
  nProcess /= 4;

  // allocate memory for modules
  HMODULE *modules   = new HMODULE[nCount], hModule;

  BOOL bStopLoop = FALSE;

  // walk in process list
  for (DWORD i=0; (i < nProcess) && !bStopLoop; i++)
  {
    uProcessId = processes[i];

    // open process for querying only
    hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, uProcessId);
    
    if (!hProcess)
      continue;

    // try to get first module
    if (!EnumProcessModules( hProcess, &hModule, sizeof(hModule), &nModules))
    {
      CloseHandle(hProcess);
      continue;
    }

    if (BaseNames)
    {
      // get module name
      GetModuleBaseName(hProcess, hModule, buf, sizeof(buf));    
    }
    else
    {
      // get module name (base)
      GetModuleFileNameEx(hProcess, hModule, buf, sizeof(buf));
    }

    if (match)
      found = match(buf, procName);
    else
      found = _tcsicmp(buf, procName) == 0;

    // that is not our process
    if (!found)
    {
      CloseHandle(hProcess);
      continue;
    }

    // now try to enum all modules
    if (!EnumProcessModules(hProcess, modules, nCount * sizeof(DWORD), &nModules))    
    {
      CloseHandle(hProcess);
      continue;
    }

    nModules /= 4;

    for (DWORD j=0;j<nModules;j++)
    {
      if (BaseNames)
      {
        // get module name (base)
        GetModuleBaseName(hProcess, modules[j], buf, sizeof(buf));    
      }
      else
      {
        // get module name (full)
        GetModuleFileNameEx(hProcess, modules[j], buf, sizeof(buf));
      }

      // is that our desired module ?
      if (match)
        found = match(buf, modName);
      else
        found = _tcscmp(buf, modName) == 0;

      if (found)
      {
        // return this process id
        *_uProcessId = uProcessId;    

        // return hModule
        *_hModule    = (DWORD)modules[j];

        bStopLoop = TRUE;

        CloseHandle(hProcess);
 
        break;
      }
    }
  }


  delete modules;
  delete processes;
  return bStopLoop;
}




//---------------------------------------------------------------------
//
// Locates import entries inside ".idata" section of a given module
// and then patches the import entries by new entries.
// This function uses the tHookedApiArray struct.
// Specify the 'unhook' = TRUE to restore old import entries valuesf
//
eHookApiErrors HookApiVA(
   const DWORD ProcessID, 
   const DWORD hModule, 
   tHookedApiArray *haa, 
   const DWORD nEntries, 
   const BOOL unhook)
{
  CHAR bufHeader[1024*2];
  HANDLE hProcess;

  PIMAGE_DOS_HEADER pDosheader;
  PIMAGE_FILE_HEADER pFileHeader;
  PIMAGE_OPTIONAL_HEADER pOptionalHeader;
  PIMAGE_SECTION_HEADER pIsh1, pIsh, pIshImport = NULL;
  DWORD nSections, i, j, nbRead, patchedCount;

  CHAR *bufIDATA = NULL;
  eHookApiErrors nRetVal = hava_openprocess;

  hProcess = OpenProcess(PROCESS_ALL_ACCESS, FALSE, ProcessID);

  if (!hProcess)
    return nRetVal;

  DWORD p, p1, fnc;

  p = hModule;

  // Read all headers
  if (!ReadProcessMemory(hProcess, (LPVOID)hModule, bufHeader, sizeof(bufHeader), &nbRead))
  {
    nRetVal = hava_readprocess;
    goto l_ret;
  }

  // set dos header
  pDosheader = (PIMAGE_DOS_HEADER) &bufHeader;

  // set file header
  pFileHeader = (PIMAGE_FILE_HEADER) ((DWORD)pDosheader + pDosheader->e_lfanew + sizeof(DWORD));

  // set optional header
  pOptionalHeader = (PIMAGE_OPTIONAL_HEADER)((DWORD)pFileHeader + sizeof(IMAGE_FILE_HEADER));
  
  // setup first section
  pIsh1 = (PIMAGE_SECTION_HEADER)((DWORD)pOptionalHeader + sizeof(IMAGE_OPTIONAL_HEADER));

  // get number of sections
  nSections = pFileHeader->NumberOfSections;

  // walk in all sections
  for (pIsh = pIsh1, i=0; i < nSections;i++, pIsh++)
  {
    // I hardcoded ".idata" , basically I could look in the Ioh.DataDirectory
    // or use a heuristic method to locate the import entries....
    if ( stricmp((LPCTSTR)pIsh->Name, ".idata") == 0)
    {
      pIshImport = pIsh;  
      break;
    }
  }  

  // failed to find .idata ?
  if (pIshImport == NULL)
  {
    nRetVal = hava_noimport;
    goto l_ret;
  }

  // alloc enough memory to hold import ....
  bufIDATA = new CHAR[pIshImport->Misc.VirtualSize];
  if (bufIDATA == NULL)
  {
    nRetVal = hava_nomemforimports;
    goto l_ret;
  }

  PDWORD ddIDATA;

  // Point to VA of import table
  p = hModule + pIshImport->VirtualAddress;

  // read whole import table
  if (!ReadProcessMemory(hProcess, (LPVOID)p, (LPVOID)bufIDATA, pIshImport->Misc.VirtualSize, &nbRead))
    goto l_ret;


  // change page protection
  if (!VirtualProtectEx(
           hProcess, 
           (LPVOID)((DWORD)hModule + pIshImport->VirtualAddress), 
           pIshImport->Misc.VirtualSize,  
           PAGE_READWRITE,
           &nSections))
  {
    nRetVal = hava_changeprotection;
    goto l_ret;
  }

  // reset patched imports count
  patchedCount = 0;

  for (ddIDATA = (PDWORD)bufIDATA, i=0;
       i<pIshImport->Misc.VirtualSize/4;
       i++, ddIDATA++)
  {
    for (j=0;j<nEntries;j++)
    {
      if (unhook)
        fnc = haa[j].PatchedEP;
      else
        fnc = haa[j].OriginalEP;

      // located an entry ?
      if (*ddIDATA == fnc)
      {
        //
        p1 = hModule + pIshImport->VirtualAddress + ((DWORD)ddIDATA - (DWORD)bufIDATA);

        // save VA position
        haa[j].VA = (DWORD)p1;

        WriteProcessMemory(hProcess, 
                           (LPVOID)p1, 
                           (LPVOID)(unhook ? &haa[j].OriginalEP : &haa[j].PatchedEP), 
                           sizeof(DWORD), &nbRead);

        if (GetLastError() != ERROR_SUCCESS)
        {
          nRetVal = hava_writeprocess;
          goto l_ret;
        }
        patchedCount++;
        break;
      }
    }
  }

  // restore old page protection
  VirtualProtectEx(
      hProcess, 
      (LPVOID)((DWORD)hModule + pIshImport->VirtualAddress), 
      pIshImport->Misc.VirtualSize,  
      nSections,
      &i);

  nRetVal = (patchedCount == nEntries) ? hava_success : hava_notall;

  // clean exit
  l_ret:
  if (bufIDATA != NULL)
    delete bufIDATA;

  CloseHandle(hProcess);

  return nRetVal;
}


//---------------------------------------------------------------------
//
//
BOOL EnumProcessModulesCB(DWORD 
  ProcessID,
  void (*cbEnum)(DWORD, LPCTSTR)
  )
{
  // load PsAPI
  if (!Load_PsApi())
    return FALSE;

  DWORD nModules, nCount = 1024;

  HANDLE hProcess;

  // buffer for process names
  TCHAR buf[MAX_PATH];

  // allocate memory for modules
  HMODULE *modules   = new HMODULE[nCount];

  BOOL bStopLoop = FALSE;

  // open process for querying only
  hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, ProcessID);
    
  // now try to enum all modules
  if (!EnumProcessModules(hProcess, modules, nCount * sizeof(DWORD), &nModules))    
  {
    CloseHandle(hProcess);
    return FALSE;
  }

  // because nModules returned from EnumProcessModules() is in bytes, I divid by 4 to return n DWORDs
  nModules /= 4;

  for (DWORD j=0; j< nModules; j++)
  {
    // get module name
    GetModuleFileNameEx(hProcess, modules[j], buf, sizeof(buf));
    cbEnum((DWORD)modules[j], buf);
  }

  CloseHandle(hProcess);
  delete modules;
  return bStopLoop;
}
