/*********************************************************************
  Bitmap routines
  support OS/2 1.x, Windows 3.x and OS/2 2.x single image bitmaps
  optimize the palette of 256 color images for steganography by dupli-
  cating used entries at the cost of unused and/or pairing similar
  entries

  part of: Hide4PGP by Heinz Repp
  last modified: 05/20/97
*********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include "hide4pgp.h"

#define MAXDIST (UWORD16) 0xFFFE

typedef union
{ UBYTE RGB[3];
  struct
  { UBYTE L;
    SBYTE a;
    SBYTE b;
  } Lab;
} pal_entry;

typedef struct
{ UBYTE first;
  UBYTE second;
  UWORD16 dist1;
  UWORD16 dist2;
} w_entry;

typedef struct
{ int one;
  int other;
} p_entry;

static long lineoff, *frequency;
static WORD32 position;
static size_t width;
static UWORD16 height, *distance;
static UBYTE *old_pos, *new_pos, *scanline;
static pal_entry *palette;
static p_entry *pair;


/*********************************************************************
  sqr:       squares
  parameter: (int) argument
  returns:   (UWORD32) result
*********************************************************************/
static UWORD32 sqr (int x)
{ return (UWORD32) x * x;
} /* sqr */


/*********************************************************************
  thirdroot: iterates the third root of a float value encoded into a
             UWORD16 with 1.0 = 0x8000 (range 0.0 <= x < 2.0)
             using the method of Newton
  parameter: (UWORD16) argument
  returns:   (UWORD16) result (encoded the same)
*********************************************************************/
static UWORD16 thirdroot (UWORD16 y)
{ UWORD16 x = (UWORD16) (((UWORD32) 0x10000 + y) / 3);
  UWORD32 a = (UWORD32) y << 15;

  do
  { y = x;
    x = (UWORD16) ((a / ((UWORD32) x * x >> 15) + ((UWORD32) x << 1)) / 3);
  } while (x < y);
  return x;
} /* thirdroot */


/*********************************************************************
  f:         function needed to convert XYZ to Lab coordinates
  parameter: (UWORD16) argument
  returns:   (UWORD16) result
*********************************************************************/
static UWORD16 f (UWORD16 x)
{ if (x > 290)
    return thirdroot (x);
  else
    return (UWORD16) ((UWORD32) 255164 * x >> 15) + (UWORD16) 4520;
} /* f */


/*********************************************************************
  RGB2Lab:   converts palette entries in monitor RGB coordinates to
             the CIE L*a*b* colorspace
  parameter: pointer to palette entry
  returns:   nothing
*********************************************************************/
static void RGB2Lab (pal_entry *color)
{ register int i,j;
  UWORD16 cRGB[3], XYZ[3];
  static UWORD16 convD65[3][3] =
    { {14110, 11190,   5846},
      {  7275, 23157,  2336},
      {   662,  4243, 30782} };

  /* gamma-correct with gamma = 2.0, transform to 1.0 = 0x8000 */
  for (j = 3; j--; )
    cRGB[j] = (UWORD16) ((((UWORD32) color->RGB[j] * color->RGB[j]) << 15)
                         / 0xFE01u);

  /* convert corrected RGB to CIE XYZ coordinates */
  for (i = 3; i--; )
  { XYZ[i] = 0;
    for (j = 3; j--; )
      XYZ[i] += (UWORD32) convD65[i][j] * cRGB[j] >> 15;
  } /* endfor to CIE XYZ */

  /* convert CIE XYZ to CIE L*a*b* coordinates and transform to
     appropriate encoding : 0 <= L <= 100 in 0x00 - 0xFF
                        -129 <= a,b < 128 in 0x80 - 0x7F  */
  color->Lab.L = (UBYTE) ((1479 * (long) f (XYZ[1]) - 6684672L) / 5 >> 15);
  color->Lab.a = (SBYTE) (125 * ((long) f (XYZ[0]) - f (XYZ[1])) >> 13);
  color->Lab.b = (SBYTE) (25 * ((long) f (XYZ[1]) - f (XYZ[2])) >> 12);
} /* RGB2Lab */


/*********************************************************************
  sdist:     calculates the square 'distance' (delta E) of two palette
             entries in CIE L*a*b* coordinates
  parameter: pointers to two palette entries
  returns:   result limited to UWORD16
*********************************************************************/
static UWORD16 sdist (pal_entry *x, pal_entry *y)
{ UWORD32 d;
  d = sqr ((int) y->Lab.L - x->Lab.L) * 400 / 2601 +
      sqr ((int) y->Lab.a - x->Lab.a) +
      sqr ((int) y->Lab.b - x->Lab.b);
  return d > MAXDIST ? MAXDIST : (UWORD16) d;
} /* sdist */


/*********************************************************************
  trindex:   calculates the index in a triangular matrix stored in a
             single dimension linear array of the form [0] = [1][0],
             [1] = [2][0], [2] = [2][1], [4]= [3][0], [5] = [3][1] ...
  parameter: both indices of triangular matrix, larger first
  returns:   corresponding index in linear array
*********************************************************************/
static int trindex (int i, int j)
{ return (int) ((long) i * (i - 1) >> 1) + j;
} /* trindex */


/*********************************************************************
  trindexs:  same as trindex but without presorted indices
  parameter: both indices of triangular matrix in any order
  returns:   same as trindex
*********************************************************************/
static int trindexs (int i, int j)
{ if (i > j)
    return (int) ((long) i * (i - 1) >> 1) + j;
  else
    return (int) ((long) j * (j - 1) >> 1) + i;
} /* trindexs */


/*********************************************************************
  split_pair: tries to split the distance between the members of one
             pair by pairing every member with one member of another
             pair lying within a sqhere with centre in the middle
             between the members and radius of half of the distance;
             calls itself recursively until no pair is found between;
             uses and modifies the pair[] vector and uses the
             distance[] vector
  parameter: pair to split, actual scope, highest pair
  returns:   nothing
*********************************************************************/
static void split_pair (int work, int bias, int max)
{ int one, other, selected, crossed, i;
  long range, temp;

  one = pair[work].one - bias;
  other = pair [work].other - bias;
  range = (long) distance[trindexs (one, other)];
  selected = FALSE;

  for (i = bias >> 1; i <= max; i++)
  { if (i == work) continue; /* skip examined pair */
    if ((temp = (long)
         distance[trindexs (one, pair[i].one - bias)] +
         distance[trindexs (other, pair[i].other - bias)] -
         distance[trindexs (pair[i].one - bias , pair[i].other - bias)])
        < range)
    { selected = i;
      range = temp;
      crossed = FALSE;
    } /* endif pair in range */
    if ((temp = (long)
         distance[trindexs (one, pair[i].other - bias)] +
         distance[trindexs (other, pair[i].one - bias)] -
         distance[trindexs (pair[i].one - bias , pair[i].other - bias)])
        < range)
    { selected = i;
      range = temp;
      crossed = TRUE;
    } /* endif pair in range */
  } /* endfor all pairs */

  if (selected)
  { other = pair[work].other;
    if (crossed)
    { pair[work].other = pair[selected].other;
      pair[selected].other = other;
    } else
    { pair[work].other = pair[selected].one;
      pair[selected].one = other;
    } /* endif crossed */
    split_pair (work, bias, max);
    split_pair (selected, bias, max);
  } /* endif pair found */
} /* split_pair */


/*********************************************************************
  list_pal:  lists the palette indices in the outer nodes of the
             binary tree stored in the pair array recursively into
             the old_pos vector
  parameter: index of pair in pair[] to work on
             next position index to be set in old_pos[]
  returns:   new next position index
*********************************************************************/
static int list_pal (int pair_index, int pos_index)
{ if (pair_index > 255)
  { old_pos[pos_index] = (UBYTE) (pair_index & 0xFF);
    return ++pos_index;
  } else
  { return list_pal (pair[pair_index].other,
                     list_pal (pair[pair_index].one, pos_index));
  } /* endif criterion */
} /* list_pal */


/*********************************************************************
  bmp_nextblk: positions the read pointer in StegFile to the beginning
             of the next scanline
  parameter: none; uses external variables:
             position: offset of scanline from beginning of file
             width:    length of scanline in bytes
             height:   number of scanlines
             lineoff:  offset to next scanline
  returns:   length of scanline in bytes or 0L if EOF
             changed external variables:
               position: points to next scanline
               height:   number of scanlines left
*********************************************************************/
long bmp_nextblk (void)
{ if (height-- && ! fseek (StegFile, (long) position, SEEK_SET))
  { position += lineoff;
    return (long) width;
  } else return 0L;
} /* bmp_nextblk */


/*********************************************************************
  rewrite_pal: reads the palette into memory and rewrites it in new
             order defined in old_pos, calculates the inverse vector
             in new_pos
  parameter: offset of palette from file start, flag if 4 byte struct
  returns:   nothing
*********************************************************************/
static void rewrite_pal (long pal_off, int ext_stru)
{ int i, j;

  /* allocate dynamic array pal_entry palette[256] */
  if ((palette = malloc (256 * sizeof (pal_entry))) == NULL)
    no_mem (verbose);

  /* read palette */
  fseek (StegFile, pal_off, SEEK_SET);
  for (i = 0; i < 256; i++)
  { for (j = 3; j--; )
      palette[i].RGB[j] = (UBYTE) fgetc (StegFile);
    if (ext_stru) fgetc (StegFile);    /* if 4 byte struct */
  } /* endfor */

  /* write rearranged palette to file */
  fseek (StegFile, pal_off, SEEK_SET);
  for (i = 0; i < 256; i++)
  { for (j = 3; j--; )
      fputc (palette[old_pos[i]].RGB[j], StegFile);
    if (ext_stru) fputc (0, StegFile); /* if 4 byte struct */
  } /* endfor */

  /* set new_pos to the first occurence of every entry in old_pos */
  for (i = 256; i--; )
    new_pos[old_pos[i]] = (UBYTE) i;

  free (palette);
} /* rewrite_pal */


/*********************************************************************
  all_lines: calls a vectorized function for all scanlines
  parameter: function pointer
  returns:   nothing
*********************************************************************/
static void all_lines (void (*do_line) (void))
{ UWORD16 s_height;
  long a_position;
  WORD32 s_position;

  if (verbose)
  { if (verbose > 1)
      fputs (" DONE.\n"
             "Changing Bitmap pixels to match new palette order ... ",
             stderr);
    else
      fputs (". ", stderr);
  } /* endif verbose */

  /* save position and height to restore to initial values afterwards */
  s_position = position;
  s_height = height;
  if ((scanline = malloc (width)) == NULL) no_mem (verbose);

  while (bmp_nextblk())
  { a_position = ftell (StegFile);
    fread (scanline, 1, width, StegFile);
    (*do_line) ();
    fseek (StegFile, a_position, SEEK_SET);
    fwrite (scanline, 1, width, StegFile);
  } /* endwhile scanlines */

  position = s_position;
  height = s_height;
  free (scanline);
  if (verbose) fputs ("DONE.\n", stderr);
} /* all_lines */


/*********************************************************************
  count_col: counts the frequency of each palette entry in the bitmap
  parameter: none
  returns:   nothing
*********************************************************************/
static void count_col (void)
{ register int i;
  for (i = width; i--; )
    frequency[scanline[i]]++;
} /* count_col */


/*********************************************************************
  set_new:   changes one scanline by replacing each pixel with the
             corresponding new_pos entry
  parameter: none
  returns:   nothing
*********************************************************************/
static void set_new (void)
{ register int i;
  for (i = width; i--; )
    scanline[i] = new_pos[scanline[i]];
} /* set_new */


/*********************************************************************
  duplic_pal: count the frequency of each palette entry, fill unused
             entries with copies of more often used; if < 128 used use
             powers of 2
  parameter: offset of palette data from file start, flag if 4 byte struct
  returns:   nothing
*********************************************************************/
static void duplic_pal (long pal_off, int ext_stru)
{ int i, l, m, n;
  long f;

  /* allocate dynamic array long frequency[256] and initialize with 0 */
  if ((frequency = malloc (256 * sizeof (long))) == NULL) no_mem (FALSE);
  for (i = 0; i < 256; ) frequency[i++] = 0;

  if (m = verbose) verbose--;
  if (verbose)
    fputs ("Counting color frequency ..", stderr);

  /* count color frequency */
  all_lines (count_col);
  verbose = m;

  /* allocate dynamic arrays UBYTE old_pos[256], new_pos[256] */
  if ((old_pos = malloc (512 * sizeof (UBYTE))) == NULL)
    no_mem (FALSE);
  new_pos = &old_pos[256];

  /* count colors used and gather them in old_pos */
  for (n = i = 0; i < 256; i++)
  { if (frequency[i])
    { old_pos[n++] = (UBYTE) i;
      new_pos[i] = 1;
    } else
    { new_pos[i] = 0;
      l = i; /* last with 0 frequency */
    } /* endif frequency */
  } /* endfor */

  if (n == 256)
  { if (verbose)
      fputs ("All palette entries are used - no duplication possible.\n",
             stderr);
    goto duplic_bypass; /* goto avoids nested ifs - skips rest */
  } else
  { if (verbose)
    { if (verbose > 1)
        fprintf (stderr, "Bitmap uses %i palette entries - ", n);
      fputs ("Duplicating palette entries ...", stderr);
    } /* endif verbose */
  } /* endif all used */

  /* sort entries in order of frequency */
  old_pos[n] = (UBYTE) l; /* end mark */
  for (i = n - 1; i--; )
  { f = frequency[m = old_pos[l = i]];
    while (f < frequency[old_pos[l + 1]])
    { old_pos[l] = old_pos[l + 1];
      l++;
    } /* endwhile last entries less frequent */
    old_pos[l] = (UBYTE) m;
  } /* endfor insertion sort */

  /* double individual entries successively */
  m = 256 - n;
  l = 1;
  i = 0;
  while (m)
  { new_pos[old_pos[i]] += (UBYTE) l;
    m -= l;
    if (++i >= n)
    { i = 0;
      l <<= 1;
    } /* endif new turn */
  } /* endwhile position left */

  /* assign positions */
  l = 256;
  for (i = n; i--; )
    for (m = new_pos[old_pos[i]]; m--; )
      old_pos[--l] = old_pos[i];

  /* rewrite palette */
  rewrite_pal (pal_off, ext_stru);

  /* set pixels */
  all_lines (set_new);

duplic_bypass:
  free (old_pos);
  free (frequency);
} /* duplic_pal */


/*********************************************************************
  pair_pal:  optimizes palette of a 256 colors bitmap for stegano-
             graphy if not greyscale in strictly ascending order; it
             minimizes the distance in the Lab colorspace between two
             adjacent entries, pairs, pairs of pairs a.s.o.
             handles 3-byte and 4-byte palette structures
  parameter: offset of palette data from file start
             flag, non zero if 4 byte entries
  returns:   nothing; position of file pointers undefined
*********************************************************************/
static void pair_pal (long pal_off, int ext_stru)
{ register int i, j;
  register UWORD16 temp;
	int k, l, m, one, other, greyorder, scope;
  UWORD16 threshold;
  UBYTE *availbl, *availno;
  w_entry *wanted;

  /* read palette into memory */
  /* allocate dynamic array pal_entry palette[512]
     (510 would suffice, but that's more straightforward) */
  if ((palette = malloc (512 * sizeof (pal_entry))) == NULL)
    no_mem (FALSE);
  fseek (StegFile, pal_off, SEEK_SET);
  for (i = 256; i < 512; i++)
  { for (j = 3; j--; )   /* reads B - G - R */
      palette[i].RGB[j] = (UBYTE) fgetc (StegFile);
    if (ext_stru) fgetc (StegFile); /* dummy read if 4 byte struct */
  } /* endfor */

  /* check if ordered greyscale */
  greyorder = TRUE;
  for (i = 256; i--; )
    for (j = 3; j--; )
      if (palette[i + 256].RGB[j] != (UBYTE) i)
        greyorder = FALSE;
  if (greyorder)
  { if (verbose)
      fputs ("Palette is ordered greyscale - pairing not necessary.\n",
             stderr);
    free (palette);
    return;
  } /* endif greyorder */

  if (verbose)
    fputs ("Pairing palette entries ", stderr);

  /* convert palette into CIE L*a*b* colorspace */
  for (i = 256; i < 512; i++)
    RGB2Lab (&palette[i]);
  if (verbose) fputc ('.', stderr);

  /* allocate dynamic array p_entry pair[256] */
  if ((pair = malloc (256 * sizeof (p_entry))) == NULL) no_mem (verbose);

  /* order successive pairs, then pairs of pairs, etc. */
  for (scope = 256; scope >= 4; scope >>= 1)
  { /* allocate dynamic array UWORD16 distance[scope*(scope-1)/2] */
    if ((distance =
				 malloc (((size_t)  (scope * (scope - 1)) >> 1) * sizeof (UWORD16)))
        == NULL) no_mem (verbose);

    /* allocate dynamic array w_entry wanted[scope] */
		if ((wanted = malloc ((size_t) scope * sizeof (w_entry))) == NULL)
			no_mem (verbose);

    /* calculate the square distance between all and search the nearest
       and second nearest of every entry */
    for (i = scope; i--; )
      wanted[i].dist1 = wanted[i].dist2 = MAXDIST + 1;
    for (i = scope; --i; )
    { for (j = i; j--; )
      { distance[trindex (i, j)] = temp =
          sdist (&palette[i + scope], &palette[j + scope]);

        /* check if j is one of the nearest to i */
        if (temp < wanted[i].dist2)
        { if (temp < wanted[i].dist1)
          { wanted[i].dist2 = wanted[i].dist1;
            wanted[i].second = wanted[i].first;
            wanted[i].dist1 = temp;
            wanted[i].first = (UBYTE) j;
          } else
          { wanted[i].dist2 = temp;
            wanted[i].second = (UBYTE) j;
          } /* endif distance < dist2 */
        } /* endif distance < dist1 */

        /* check if i is one of the nearest to j */
        if (temp < wanted[j].dist2)
        { if (temp < wanted[j].dist1)
          { wanted[j].dist2 = wanted[j].dist1;
            wanted[j].second = wanted[j].first;
            wanted[j].dist1 = temp;
            wanted[j].first = (UBYTE) i;
          } else
          { wanted[j].dist2 = temp;
            wanted[j].second = (UBYTE) i;
          } /* endif distance < dist2 */
        } /* endif distance < dist1 */

      } /* endfor j */
    } /* endfor i */

    /* allocate dynamic arrays UBYTE *availbl[scope],
       *availno[scope] */
		if ((availbl = malloc ((size_t) (scope << 1) * sizeof (UBYTE))) == NULL)
      no_mem (verbose);
    availno = &availbl[scope];

    /* search the maximum minimum distance and initialize availbl[] */
    threshold = 0;
    for (i = scope; i--; )
    { availbl[i] = (UBYTE) i;
      if (wanted[i].dist1 > threshold) threshold = wanted[i].dist1;
    } /* endfor all */

    /* sort the availbl[i] entries as a priority queue with the
       criterion wanted[availbl[i]].dist2 */
    i = scope >> 1;
    while (i)
    { temp = wanted[m = availbl[l = --i]].dist2;
      do
      { j = l << 1;
        if (wanted[availbl[j]].dist2 < wanted[availbl[j + 1]].dist2) j++;
        if (wanted[availbl[j]].dist2 > temp)
        { availbl[l] = availbl[j];
          l = j;
        } else break;
      } while (l < (scope >> 1));
      availbl[l] = (UBYTE) m;
    } /* endwhile heapsort */

    /* calc the inverse vector in availno */
    for (i = scope; i--; )
      availno[availbl[i]] = (UBYTE) i;

    /* pair the first pair */
    k = scope >> 1;
    one = availbl[0];
    other = wanted[one].first;
    pair[k].one = one + scope;
    pair[k].other = other + scope;

    /* use availbl[0] as top mark */
    wanted[one].dist2 = MAXDIST + 1;

    /* delete availbl[availno[other]] (= other) */
    temp = wanted[m = availbl[scope - 1]].dist2;
    l = availno[other];
    if (temp > wanted[other].dist2)
    { /* upheap */
      while (wanted[availbl[j = l >> 1]].dist2 < temp)
      { availno[availbl[l] = availbl[j]] = (UBYTE) l;
        l = j;
      } /* endwhile upheap */
    } else
    { /* downheap */
      while (l < k)
      { j = l << 1;
        if (j < scope - 2 &&
            wanted[availbl[j]].dist2 < wanted[availbl[j + 1]].dist2) j++;
        if (wanted[availbl[j]].dist2 > temp)
        { availno[availbl[l] = availbl[j]] = (UBYTE) l;
          l = j;
        } else break;
      } /* endwhile inner node */
    } /* endif up or down */
    availno[availbl[l] = (UBYTE) m] = (UBYTE) l;

    /* pair successively the one on top of the heap */
    for (k++; k < scope - 1; k++)
    { /* check and recalculate every remaining entry that has one
         of the newly paired as first or second nearest */
      for (i = (scope - k << 1) + 1; --i; )
      { if (wanted[m = availbl[i]].first == (UBYTE) one ||
            wanted[m].first == (UBYTE) other ||
            wanted[m].second == (UBYTE) one ||
            wanted[m].second == (UBYTE) other)
        { wanted[m].dist1 = wanted[m].dist2 = MAXDIST + 1;
          for (j = (scope - k << 1) + 1; --j; )
          { if (i == j) continue;
            if ((temp = distance[trindexs (m, availbl[j])])
                      < wanted[m].dist2)
            { if (temp < wanted[m].dist1)
              { wanted[m].dist2 = wanted[m].dist1;
                wanted[m].second = wanted[m].first;
                wanted[m].dist1 = temp;
                wanted[m].first = availbl[j];
              } else
              { wanted[m].dist2 = temp;
                wanted[m].second = availbl[j];
              } /* endif distance < dist2 */
            } /* endif distance < dist1 */
          } /* endfor search nearest */
          /* correct heap */
          temp = wanted[m].dist2;
          l = i;
          while (wanted[availbl[j = l >> 1]].dist2 < temp)
          { availno[availbl[l] = availbl[j]] = (UBYTE) l;
            l = j;
          } /* endwhile upheap */
          availno[availbl[l] = (UBYTE) m] = (UBYTE) l;
        } /* endif search new partners */
      } /* endfor scan all remaining */

      one = availbl[1];
      other = wanted[one].first;
      pair[k].one = one + scope;
      pair[k].other = other + scope;
      /* if newly paired are far apart: try to split the pair */
      if (wanted[one].dist1 > threshold)
        split_pair (k, scope, k);

      /* remove availbl[1] (= one) */
      temp = wanted[m = availbl[i = scope - k << 1]].dist2;
      l = 1;
      do
      { j = l << 1;
        if (wanted[availbl[j]].dist2 < wanted[availbl[j + 1]].dist2) j++;
        if (wanted[availbl[j]].dist2 > temp)
        { availno[availbl[l] = availbl[j]] = (UBYTE) l;
          l = j;
        } else break;
      } while (l < scope - k);
      availno[availbl[l] = (UBYTE) m] = (UBYTE) l;

      /* delete availbl[availno[other]] (= other) */
      temp = wanted[m = availbl[--i]].dist2;
      l = availno[other];
      if (temp > wanted[other].dist2)
      { /* upheap */
        while (wanted[availbl[j = l >> 1]].dist2 < temp)
        { availno[availbl[l] = availbl[j]] = (UBYTE) l;
          l = j;
        } /* endwhile upheap */
      } else
      { /* downheap */
        i--;      /* now points to new last element */
        while (l < scope - k)
        { j = l << 1;
          if (j < i &&
              wanted[availbl[j]].dist2 < wanted[availbl[j + 1]].dist2) j++;
          if (wanted[availbl[j]].dist2 > temp)
          { availno[availbl[l] = availbl[j]] = (UBYTE) l;
            l = j;
          } else break;
        } /* endwhile inner node */
      } /* endif up or down */
      availno[availbl[l] = (UBYTE) m] = (UBYTE) l;

    } /* endfor pairing all but two */

    /* pair the last pair (the last two entries in availbl[]) */
    pair[k].one = availbl[1] + scope;
    pair[k].other = availbl[2] + scope;
    /* if newly paired are far apart: try to split the pair */
    if (distance[trindexs(one, other)] > threshold)
      split_pair (k, scope, k);

    free (availbl);
    free (wanted);
    free (distance);

    /* order pair entries in acscending order of luminosity and
       calculate average color */
    for (i = scope >> 1; i < scope; i++)
    { if (palette[pair[i].one].Lab.L >
          palette[pair[i].other].Lab.L)
      { temp = pair[i].one;
        pair[i].one = pair[i].other;
        pair[i].other = temp;
      } /* endif order pair */
      palette[i].Lab.L =
        (UBYTE) (((int) palette[pair[i].one].Lab.L +
                 palette[pair[i].other].Lab.L) >> 1);
      palette[i].Lab.a =
        (SBYTE) (((int) palette[pair[i].one].Lab.a +
                 palette[pair[i].other].Lab.a) >> 1);
      palette[i].Lab.b =
        (SBYTE) (((int) palette[pair[i].one].Lab.b +
                 palette[pair[i].other].Lab.b) >> 1);
    } /* endfor all pairs */
    if (verbose) fputc ('.', stderr);
  } /* endfor scope >=4 */

  if (palette[2].Lab.L > palette[3].Lab.L)
  { pair[1].one = 3;
    pair[1].other = 2;
  } else
  { pair[1].one = 2;
    pair[1].other = 3;
  } /* endif order top pair */

  /* allocate dynamic arrays UBYTE old_pos[256], new_pos[256] */
  if ((old_pos = malloc (512 * sizeof (UBYTE))) == NULL)
    no_mem (verbose);
  new_pos = &old_pos[256];

  /* list palette indices in new order into old_pos */
  list_pal (1, 0);

  free (pair);
  free (palette);

  /* change palette to new order */
  rewrite_pal (pal_off, ext_stru);

  /* change all pixels to reflect new order */
  all_lines (set_new);

  free (old_pos);
} /* pair_pal */


/*********************************************************************
  bmp_prepare: checks and outputs the characteristics of StegFile;
             aborts, if no recognized bitmap format or not suited for
             steganography: # of planes != 1, # of bits not 8 or 24,
             bitmap compressed
             sets external variables for bmp_nextblk:
               position: offset of first scanline from beginning of file
               width:    length of scanline in bytes
               height:   number of scanlines
               lineoff:  offset to next scanline
             for 8 bit color palettes when hiding:
             1. with newer format and fewer than maximum # of colors
                used or important these are reset to maximum for ste-
                ganography almost certainly changes this number
             2. the palette is optimized it switches are set
  parameter: none
             read pointer set to second byte in file
  returns:   data width flag, always zero indicating 8 bit data
             (24 bit are 3 times 8 bit values too!)
*********************************************************************/
int bmp_prepare (void)
{ int size, col_depth;

  switch (fgetc (StegFile))
  { case 'M':
      break;
    case 'A':
      if (verbose > 1)
        fputs ("Hint: File is probably an OS/2 bitmap ARRAY. "
               "Convert to normal bitmap!\n", stderr);
    default:
      fputs ("Error (BMP): File identification failed.\n", stderr);
      exit (10);
  } /* switch 2. not M */

  fseek (StegFile, 10, SEEK_SET);
  position = getlong (StegFile);

  size = getword (StegFile);
  if (verbose)
  { fputs ("Bitmap format <", stderr);
    switch (size)
    { case 12:
        fputs ("OS/2 1", stderr);
        break;
      case 40:
        fputs ("Windows 3", stderr);
        break;
      case 64:
        fputs ("OS/2 2", stderr);
        break;
      default:
        fputs ("?> unknown.\n", stderr);
    } /* switch header size */
  } /* if verbose */
  switch (size)
  { case 12:
    case 40:
    case 64:
      break;
    default:
      fputs ("Error (BMP): Unsupported Bitmap format.\n", stderr);
      exit (11);
  } /* switch header size */
  if (verbose) fputs (".x> detected", stderr);

  getword (StegFile);
  width = (size_t) getword (StegFile);
  if (size > 12) getword (StegFile);
  height = (UWORD16) getword (StegFile);
  if (size > 12) getword (StegFile);
  if (verbose > 1)
    fprintf (stderr, ".\nBitmap size %i x %i pixels", width, (int) height);

  if (height == 0 || width == 0)
  { if (verbose) fputs (".\n", stderr);
    fputs ("Error (BMP): No picture ?!?\n", stderr);
    exit (12);
  } /* endif no picture */

  if (getword (StegFile) != 1)
  { if (verbose) fputs (".\n", stderr);
    fputs ("Error (BMP): More than one plane is not supported.\n",
           stderr);
    exit (12);
  } /* planes not 1 */

  col_depth = getword (StegFile);
  if (verbose)
  { fputs (", ", stderr);
    switch (col_depth)
    { case 1:
        fputs ("monochrome (2", stderr);
        break;
      case 4:
        fputs ("VGA (16", stderr);
        break;
      case 8:
        fputs ("SVGA or greyscale (256", stderr);
        break;
      case 16:
        fputs ("High Color (65536", stderr);
        break;
      case 24:
        fputs ("True Color (16.7 Mio", stderr);
        width *= 3;
        break;
      default:
        fprintf (stderr, "unknown color depth (? = 2 ^ %i", col_depth);
    } /* switch colors */
    fputs (" colors).\n", stderr);
  } /* endif verbose */
  if (col_depth != 8 && col_depth != 24)
  { fputs ("Error (BMP): Color depth not suited for steganography.\n",
           stderr);
    exit (13);
  } /* endif not 8 or 24 bit */
  lineoff = width + 3 & ~0 << 2;  /* adjust scanline to WORD32 boundary */

  if (size > 12)
  { WORD32 temp;
    if (getlong (StegFile))
    { fputs ("Error (BMP): Bitmap is compressed - store uncompressed!\n",
             stderr);
      exit (14);
    } /* endif compressed */
    if ((temp = getlong (StegFile)) && verbose > 1)
      fprintf (stderr, "Actual size of bitmap data: %li Bytes.\n", (long) temp);
    if ((temp = getlong (StegFile)) && verbose > 1)
      fprintf (stderr, "Bitmap resolution: %li dpi horizontal ",
               (long) (temp * 127 + 2500) / 5000);
    if ((temp = getlong (StegFile)) && verbose > 1)
      fprintf (stderr, "and %li dpi vertical.\n",
               (long) (temp * 127 + 2500) / 5000);

    if (hiding)
    { if ((temp = getlong (StegFile)) & (WORD32) 0xFF)
      { if (verbose)
          fprintf (stderr, "No. of colors used reset to all (was: %li).\n",
                   (long) temp);
        fseek (StegFile, -4L, SEEK_CUR);
        putlong ((WORD32) 0L, StegFile);
        fseek (StegFile, 0L, SEEK_CUR);
      } /* endif colors used */
      if ((temp = getlong (StegFile)) & (WORD32) 0xFF)
      { if (verbose)
          fprintf (stderr, "No. of important colors reset to all (was: %li).\n",
                   (long) temp);
        fseek (StegFile, -4L, SEEK_CUR);
        putlong ((WORD32) 0L, StegFile);
        fseek (StegFile, 0L, SEEK_CUR);
      } /* endif colors used */
    } /* endif hiding */
  } /* endif newer format */

  if (hiding && col_depth == 8)
  { if (duplic)
      duplic_pal (14L + size, size > 12);
    if (pairing)
      pair_pal (14L + size, size > 12);
  }
  else
  { if (verbose)
    { if (duplic)
        fputs ("Requested duplication of palette entries is not applicable.\n",
               stderr);
      if (pairing)
        fputs ("Requested pairing of palette entries is not applicable.\n",
               stderr);
    } /* endif verbose */
  } /* endif hiding and 256 colors */

  return FALSE;
} /* bmp_prepare */
