/*********************************************************************
  Hide for PGP:
  a steganographic program that hides files in BMP, WAV or VOC files.
  If the hidden files resemble random noise - e.g. files encrypted with
  Pretty Good Privacy by Phil Zimmermann and without any header - e.g.
  the header information removed by Stealth by Henry Hastur - the hidden
  information cannot be detected by any means.

  copyright 1996/1997 by Heinz Repp
  last modified: 05/22/97

  This program has been developped with PGP and Stealth in mind so it
  supports the stream mode of Stealth: the file to be hidden or extracted
  must be provided or read through a pipe or stdin/stdout redirection.

  usage:
  hide4pgp [-#] [-d] [-p] [-v|-q] <hostfile> < <file-to-hide>
  hide4pgp -x [-#] [-v|-q] <hostfile> > <extracted-file>

  parameters may be in any order and switches combined

  !!  for non UNIX systems either USE_FREOPEN or USE_SETMODE must be
  !!  defined (see Hide4PGP.Doc / Source for details)

*********************************************************************/

/* includes */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#if defined (USE_SETMODE)
  #include <io.h>
  #include <fcntl.h>
#endif
#include "hide4pgp.h"

/* globals */
FILE *StegFile;                              /* data file to work on */
int hiding = TRUE, verbose = 1;               /* general usage flags */
int pairing = FALSE, duplic = FALSE;          /* 256 color BMP flags */
static int BitCount = 0;                         /* no. of LSBs used */
static int BitMask;                          /* to mask out the LSBs */
static long capacity = 0, occupied = 0;   /* char counters stdin/out */
static size_t data_size;                      /* data width in bytes */
static int (*get_data) (FILE *);   /* funct ptr: read item from file */
static void (*mod_data) (void *, size_t); /* func ptr: modify buffer */

/* extern declarations */
extern int bmp_prepare (void);
extern int wav_prepare (void);
extern int voc_prepare (void);
extern long bmp_nextblk (void);
extern long wav_nextblk (void);
extern long voc_nextblk (void);


/*********************************************************************
  getword:   reads one 16 bit word from opened file
  parameter: Pointer to FILE descriptor
  returns:   next two bytes
*********************************************************************/
int getword (FILE *open_file)
{
  WORD16 value;
  fread (&value, 2, 1, open_file);
  return (int) value;
}


/*********************************************************************
  putword:   writes one 16 bit word to opened file
  parameter: data word, Pointer to FILE descriptor
  returns:   next two bytes
*********************************************************************/
int putword (WORD16 value, FILE *open_file)
{
  return (int) fwrite (&value, 2, 1, open_file);
}


/*********************************************************************
  getlong:   reads one 32 bit word from opened file
  parameter: Pointer to FILE descriptor
  returns:   next four bytes
*********************************************************************/
WORD32 getlong (FILE *open_file)
{
  WORD32 value;
  fread (&value, 4, 1, open_file);
  return value;
}


/*********************************************************************
  putlong:   writes one 32 bit word to opened file
  parameter: data dword, Pointer to FILE descriptor
  returns:   next four bytes
*********************************************************************/
int putlong (WORD32 value, FILE *open_file)
{
  return (int) fwrite (&value, 4, 1, open_file);
}


/*********************************************************************
  no_mem:    is called when malloc fails, outputs an error message
             and aborts
  parameter: flag if current line is not yet finished
  returns:   never
*********************************************************************/
void no_mem (int midline)
{ if (midline) fputc ('\n', stderr);
  fputs ("Error: Not enough memory - aborting.\n",
         stderr);
  exit (8);
} /* no_mem */


/*********************************************************************
  unexp_EOF: is called when read past EOF, outputs an error message
             and aborts
  parameter: none
  returns:   never
*********************************************************************/
void unexp_EOF (void)
{ fputs ("Error: Unexpected end of file encountered - aborting.\n",
         stderr);
  exit (9);
} /* unexp_EOF */


/*********************************************************************
  next_is:   compare string to next bytes in file, allways read the
             whole length
  parameter: pointer to a string to be matched
  returns:   flag, TRUE if equal
             file pointer set behind the characters compared
*********************************************************************/
int next_is (char *pattern)
{ int c, result = TRUE;
  while (*pattern)
  { if ((c = fgetc (StegFile)) == EOF) unexp_EOF ();
    if (c != *pattern++) result = FALSE;
  } /* while length */
  return result;
} /* next_is */


/*********************************************************************
  set_bits:  sets the least significant bits of a byte/word according
             to the next bits of the secret data; keeps a byte wide
             buffer for the secret data and fills it by reading from
             stdin when empty; after EOF read the buffer is filled
             with random data; the global variables capacity and occu-
             pied are set according to total bytes and read up to EOF
  parameter: the (8 or 16 bit) value in whose least significant bits
             the next bits of the secret data are to be stored
  returns:   an int value with the corresponding bits set
*********************************************************************/
static int set_bits (int original_data)
{ static int byte_in_work;
  static int count = 0;
  static int from_file = TRUE;
  register int next_byte;

  if ((count -= BitCount) < 0)
  { if (from_file)
    { if ((next_byte = getchar ()) == EOF)
      { from_file = FALSE;
        occupied = capacity;
        if (count + BitCount > 0) occupied++;
        srand ((unsigned int) clock ());
        next_byte = rand () >> 7 & 0xFF;
      } /* endif EOF */
    } else
    { next_byte = rand () >> 7 & 0xFF;
    } /* endif fromfile */
    byte_in_work = byte_in_work << 8 | next_byte;
    count += 8;
    capacity++;
  } /* endif count < 0 */
  return byte_in_work >> count & BitMask | original_data & ~BitMask;
} /* set_bits */


/*********************************************************************
  get_bits:  extracts the hidden bits out of previously modified data
             and rearranges them to bytes in a local shift register
             that is emptied to stdout when full; the number of bytes
             output is recorded to capacity
  parameter: value with secret data in its least significant bits
  returns:   nothing
*********************************************************************/
static void get_bits (int hiding_data)
{ static int byte_in_work = 0;
  static int count = 0;

  byte_in_work |= hiding_data & BitMask;
  if ((count += BitCount) >= 8)
  { putchar (byte_in_work >> (count -= 8) & 0xFF);
    capacity++;
  } /* endif bitcount >= 8 */
  byte_in_work <<= BitCount;
} /* get_bits *


/*********************************************************************
  mod_bytes: injects the secret data bits into a block of 8 bit data
             in memory
  parameter: pointer to UBYTE array, size of array
  returns:   nothing
*********************************************************************/
static void mod_bytes (UBYTE *block, size_t count)
{ while (count--)
  { *block = (UBYTE) set_bits ((int) *block);
    block++;
  } /* while count */
} /* mod_bytes */


/*********************************************************************
  mod_words: injects the secret data bits into a block of 16 bit data
             in memory
  parameter: pointer to int-array, size of array
  returns:   nothing
*********************************************************************/
static void mod_words (WORD16 *block, size_t count)
{ while (count--)
  { *block = (WORD16) set_bits ((int) *block);
    block++;
  } /* while count */
} /* mod_words */


/*********************************************************************
  set_block: set the secret bits in a block of the disk file; tries to
             read the block into memory; if not successful, splits the
             block into halves and calls itself recursively
  parameter: length of the continous block in bytes, file pointer set
             to first byte
  returns:   nothing
*********************************************************************/
static void set_block (long length)
{ void *block;
  long position;
  size_t count;

  if (length == (long) (count = (size_t) length) &&
      (length ? (block = malloc (count)) : (no_mem (FALSE), NULL)))
       /* if length == 0 no_mem aborts and never returns */
  { count /= data_size;
    position = ftell (StegFile);
    fread (block, data_size, count, StegFile);
    if (feof (StegFile)) unexp_EOF ();
    (*mod_data) (block, count);
    fseek (StegFile, position, SEEK_SET);
    fwrite (block, data_size, count, StegFile);
    free (block);
  } else
  { /* length > size_t or malloc failed */
    set_block (length++ >> 1);
    fseek (StegFile, 0L, SEEK_CUR);
    set_block (length >> 1);
  } /* endif block size */
} /* set_block */


/*********************************************************************
  get_block: get bits from a block of the disk file
  parameter: length of the continous block, read pointer set to first
             byte
  returns:   nothing
*********************************************************************/
static void get_block (long length)
{ length /= data_size;
  do
  { get_bits ((*get_data) (StegFile));
    if (feof (StegFile)) unexp_EOF ();
  } while (--length);
} /* get_block */


/*********************************************************************
  hello:  show initial prompt and copyright
*********************************************************************/
static void hello (void)
{ fputs ("Hide4PGP v1.1 - hides secret files in BMP/WAV/VOC\n"
         "(c) 1996, 1997 by Heinz Repp\n\n", stderr);
} /* hello */


/*********************************************************************
  usage:  show syntax
*********************************************************************/
static void usage (void)
{ hello ();
  fputs (
"usage:\n"
"    Hide4PGP [-#] [-d] [-p] [-v|-q] <datafile> < <file-to-hide>\n"
"or: Hide4PGP -x [-#] [-v|-q] <datafile> > <extracted-file>\n"
"\n"
"    Hide4PGP -h for help.\n", stderr);
} /* usage */


/*********************************************************************
  help:  show detailed help
*********************************************************************/
static void help (void)
{ hello ();
  fputs (
"To hide secret data in a data file:\n"
"  Hide4PGP [-option] <datafile> < <secret.in>          or using pipes:\n"
"  pgp -ef <YourID> < <secret.in> | stealth | Hide4PGP [-option] <datafile>\n"
"\n"
"To extract secret data out of a data file:\n"
"  Hide4PGP -x [-option] <datafile> > <secret.out>      or using pipes:\n"
"  Hide4PGP -x [-option] datafile | stealth -a <MyID> | pgp -f > <secret.out>\n"
"\n"
"  (!)  Secret data are always read from and written to standard in/out,\n"
"                 so the secret file MUST use i/o-redirection or a pipe!\n"
"\n"
"Options may be in arbitrary order or combined:\n"
"   x   eXtract (see above, the default action is hiding)\n"
"   #   any number between 1 and 8 specifying how many least significant bits\n"
"       will be used (default = 1 with 8 bit data and = 4 with 16 bit data)\n"
"   v   Verbose: detailed information     q   Quiet: report only fatal errors\n"
"   h   Help (this screen)\n"
"Options only useful with bitmaps (BMP) with 256 colors:\n"
"   d   Duplicate used palette entries    p   Pair similar palette entries\n",
  stderr);
} /* help */


/*********************************************************************
  main:       parses the command line, reads the first byte in the
              StegFile, decides which file type to assume, and then
              does most of the work by vectorized calls to type and
              data width specific routines
  parameters: one file name (required) and optional switches
*********************************************************************/
int main (int argc, char *argv[])
{ int (*i_prepare) (void);
  long (*i_nextblk) (void);
  void (*i_editblk) (long);
  int steg_pos = 0;
  long block;
  char swch;

  if (! --argc)
  { usage ();
    exit (1);
  } /* endif */

  /* parse command line */
  do
  { switch (*argv[argc])
    { case '-':
      case '/':
        while (swch = *++argv[argc])
        { switch (swch)
          { case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
              BitCount = swch - '0';
              break;
            case '?':
            case 'H':
            case 'h':
              help ();
              exit (0);
            case 'D':
            case 'd':
              duplic = TRUE;
              break;
            case 'P':
            case 'p':
              pairing = TRUE;
              break;
            case 'Q':
            case 'q':
              verbose = 0;
              break;
            case 'V':
            case 'v':
              verbose = 2;
              break;
            case 'X':
            case 'x':
              hiding = FALSE;
              break;
            default:
              if (verbose) hello ();
              fputs ("Error: Invalid switch.\n", stderr);
              exit (2);
          } /* switch swch */
        } /* while swch */
        break;
      default:
        if (steg_pos)
        { if (verbose) hello ();
          fputs ("Error: Only ONE filename allowed as parameter.\n", stderr);
          exit (3);
        } else
        { steg_pos = argc;
        } /* endif */
    } /* switch commandline parameter */
  } while (--argc);

  if (verbose) hello ();
  if (! steg_pos)
  { fputs ("Error: Required filename missing.\n", stderr);
    exit (4);
  } /* endif */

  if ((StegFile = fopen (argv[steg_pos], hiding ? "r+b" : "rb")) == NULL)
  { fprintf (stderr, "Error: Cannot open stegfile %s.\n", argv[steg_pos]);
    exit (5);
  } /* endif */

  /* determine data file type from first byte in file */
  switch (fgetc (StegFile))
  { case 'B':
      i_prepare = bmp_prepare;
      i_nextblk = bmp_nextblk;
      break;
    case 'C':
      i_prepare = voc_prepare;
      i_nextblk = voc_nextblk;
      break;
    case 'R':
      i_prepare = wav_prepare;
      i_nextblk = wav_nextblk;
      break;
    default:
      fputs ("Error: File format not recognized.\n", stderr);
      exit (6);
  } /* switch */

  /* check and initialize data file, set vectorized data routines */
  if ((*i_prepare) ())
  { data_size = 2;
    get_data = getword;
    mod_data = (void (*) (void *, size_t)) mod_words;
    if (! BitCount) BitCount = 4;
  } else
  { data_size = 1;
    get_data = fgetc;
    mod_data = (void (*) (void *, size_t)) mod_bytes;
    switch (BitCount)
    { case 0:
        BitCount = 1;
        break;
      case 8:
        fputs ("Error: 8 bit only useful with 16 bit data.\n", stderr);
        exit (7);
    } /* switch */
  } /* endif 16 bit */
  BitMask = ~(~0 << BitCount);

  i_editblk = hiding ? set_block : get_block;

#if ! defined (UNIX)

  /* set stdin/stdout to binary */
#if defined (USE_FREOPEN)
  if (hiding)
    freopen ("", "rb", stdin);
  else
    freopen ("", "wb", stdout);
#elif defined (USE_SETMODE)
  setmode (fileno (hiding ? stdin : stdout), O_BINARY);
#else
  #error You MUST define a macro to choose the method \
         of setting stdin/out to binary (see doc)!
#endif

#endif

  /* main workhorse loop */
  while (block = (*i_nextblk) ())
    (*i_editblk) (block);

  /* clean up and summarize */
  fclose (StegFile);
  if (verbose)
  { if (hiding)
    { set_bits (0); /* dummy */
      if (occupied)
      { if (verbose > 1)
          fprintf (stderr,
              "Encrypted file uses %li out of %li available bytes.\n",
                   occupied, --capacity);
      } else
      { fprintf (stderr,
              "*** Warning: Encrypted file truncated at %li Bytes"
        " ***\n             will most likely not decode at all!!\n",
                 --capacity);
      } /* endif occupied */
    } else
    { if (verbose > 1)
        fprintf (stderr, "%li bytes extracted - encrypted file may be"
                 " considerably smaller!\n", capacity);
    } /* endif hiding */
    fputs ("Steganographic operation finished.\n", stderr);
  } /* endif verbose */
  return 0;
} /* main */
