/*********************************************************************
  Routines for Creative Labs VOC files
  support for 8 bit sound data

  part of: Hide4PGP by Heinz Repp
  last modified: 05/20/97
*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "hide4pgp.h"

static long position;


/*********************************************************************
  chk_pbyte: checks the 'packbyte' encoding the number of bits one
             data point uses; aborts if not zero i.e. 8 bit
  parameter: none
  returns:   nothing
*********************************************************************/
static void chk_pbyte (void)
{ if (fgetc (StegFile))
  { fputs ("Error (VOC): resolution not 8 bit.\n", stderr);
    exit (31);
  } /* endif */
} /* chk_pbyte */


/*********************************************************************
  mode_out:  outputs mode and sample frequency if verbose > 1
  parameter: stereo flag, sample rate
  returns:   nothing
*********************************************************************/
static void mode_out (int mode, long rate)
{ if (verbose > 1)
    fprintf (stderr, "Mode: %so, Sample frequency: %li Hz.\n",
             mode ? "Stere" : "Mon", rate);
} /* mode_out */


/*********************************************************************
  voc_prepare: checks file characteristics and optionally outputs
             version number
  parameter: none
  returns:   data width flag - always FALSE meaning 8 bit data
*********************************************************************/
int voc_prepare (void)
{ int vers;

  if (! next_is ("reative Voice File\x1A"))
  { fputs ("Error (VOC): File identification failed.\n", stderr);
    exit (30);
  } /* endif */
  position = (unsigned) getword (StegFile);
  if (verbose)
  { fputs ("Creative Labs VOC file", stderr);
    if (verbose > 1)
    { vers = fgetc (StegFile);
      fprintf (stderr, " version %i.%i",
               fgetc (StegFile), vers);
    } /* endif verbose > 1 */
    fputs (" detected.\n", stderr);
  } /* endif verbose */
  return FALSE;
} /* voc_prepare */


/*********************************************************************
  voc_nextblk: positions read pointer to the next data block (if any)
             optionally outputs sound characteristics
  parameter: none
  returns:   length of continous data block, 0L if none
*********************************************************************/
long voc_nextblk (void)
{ int id, extended = FALSE;
  long length;

  while (!fseek (StegFile, position, SEEK_SET) &&
         (id = fgetc (StegFile)) != EOF && id)
  { length =  fgetc (StegFile);
    length += (long) (unsigned) getword (StegFile) << 8;
    position += length + 4;
    if (extended && id != 1)
    { if (verbose > 1)
        fputs ("VOC: Missing data block after extended block.\n", stderr);
      extended = FALSE;
    } /* endif ID 8 w/o 1 */
    switch (id)
    { case 1:                          /* voice data */
        if (extended)
        { extended = FALSE;
          getword (StegFile); /* dummy */
        } else
        { mode_out (0, 1000000L / (256 - fgetc (StegFile)));
          chk_pbyte ();
        } /* endif extended */
        return length - 2;
      case 2:                          /* voice continuation */
        return length;
      case 8:                          /* extended block */
        length = (unsigned) getword (StegFile);
        chk_pbyte ();
        if (fgetc (StegFile))
          mode_out (1, 128000000L / (65536L - length));
        else
          mode_out (0, 256000000L / (65536L - length));
        extended = TRUE;
    } /* switch id */
  } /* while not EOF or terminator */
  return 0L;
} /* voc_nextblk */
