//==============================================================
//= FlowAnalysis.idc : Program Execution Flow Tracer           =
//=                    and Engine for FlowAnalyzer.exe         =
//= Code per mammon_, 1998 All Rights Reversed                 =
//= Portions inspired by cephren and slava                     =
//=                                                            =
//=           A N A L Y S I S    O P T I O N S                 =
//= Trace Into Named Functions............................Fn   =
//= Trace Into UnNamed Functions ('sub_', 'loc_', 'unk')..Fu   =
//= Trace Into Near Jumps.................................Jn   =
//= Trace Into Far Jumps..................................Jf   =
//= Trace Into Near Calls.................................Cn   =
//= Trace Into Far Calls..................................Cf   =
//= Report Externs........................................X    =
//= Produce Output In Call-Trace (Non-FDB) Format.........P    =
//= Redirect Output To IDA Messages Window................S    =
//= Set Maxmimum Number Of XRef Levels To Trace...........L??  =
//==============================================================

#include <idc.idc>

#define FALSE 0
#define TRUE 1
#define OPT_Fu 0
#define OPT_Fn 1
#define OPT_Jn 2
#define OPT_Jf 3
#define OPT_Cn 4
#define OPT_Cf 5
#define OPT_X 6
#define OPT_S 7
#define OPT_P 8
#define OPT_L 9

//======================================================================
//=  OutputLine( OutFileH, parent, obtype, obname, obaddr, reloc,      =
//=              OptArray, nest )                                  =
//======================================================================
static OutputLine(OutFileH, parent, obtype, obname, obaddr, reloc, OptArray, nest){
    auto OutLine;
    if ( GetArrayElement(AR_LONG, OptArray, OPT_P) == FALSE ){
        OutLine = form("%s,%s,%s,%s,%s\n", parent ,obtype,obname,obaddr,reloc);
    } else {
        auto x;
        for ( x=0; x < nest; x = x + 1) {
            if(x !=0) OutLine = form("%s-", OutLine);
        }
        OutLine = form("%s>%s,%s,%s,%s,%s\n", OutLine, parent ,obtype,obname,obaddr,reloc);
    }
    if ( GetArrayElement(AR_LONG, OptArray, OPT_S) == TRUE )
            Message( OutLine );
    else {
        fprintf(OutFileH, OutLine);
    }
}

//======================================================================
//=  Trace_Execution( ea, RecArray, ArrIndex, OutFileH, OptArray )     =
//======================================================================
static Trace_Execution( ea, RecArray, ArrIndex, OutFileH, OptArray){
    auto x, func_end, crossref, flowtype;
    auto obParent, obType, obName, obAddr, obReloc;
    auto is_recursive, trace_Xref, report_line, XMnem, XName;

    // 1) Stay Within Max_Levels
    if (ArrIndex <= GetArrayElement(AR_LONG, OptArray, OPT_L)) {
        obParent = Name(ea);
        if ( obParent == "") obParent = atoa(ea);
    // 2) Add Function to Recursion Array
        ArrIndex = ArrIndex + 1;
        SetArrayLong(RecArray, ArrIndex, ea);
    // 3) Loop Through Addresses In Function
        func_end = FindFuncEnd(ea);
        for ( ea ; ea <= func_end; ea = NextAddr(ea) ) {
        // 3a) Get XRef For Current Line
          crossref = Rfirst0(ea);
          if ( crossref!= BADADDR) {
        // 3b) Check if XRef is in Recursion Array
            is_recursive = FALSE;
            report_line = TRUE;
            x = GetLastIndex(AR_LONG, RecArray);
            while ( x != -1) {
              if (GetArrayElement(AR_LONG, RecArray, x) == crossref) is_recursive = TRUE;
              x = GetPrevIndex(AR_LONG, RecArray, x);
            }
            if ( is_recursive == FALSE) {
              trace_Xref = TRUE;
              obName = Name(crossref);
              if ( obName == " ") obName = atoa(crossref);
        // 3c) Get XRef Flow Type
              flowtype =  XrefType();
            // 3cI)   Far Call
              if ( flowtype == fl_CF && GetArrayElement(AR_LONG, OptArray, OPT_Cf) == FALSE)
                    trace_Xref = FALSE;
            // 3cII)  Near Call
              if ( flowtype == fl_CN && GetArrayElement(AR_LONG, OptArray, OPT_Cn) == FALSE)
                    trace_Xref = FALSE;
            // 3cIII) Far Jmp
              if ( flowtype == fl_JF && GetArrayElement(AR_LONG, OptArray, OPT_Jf) == FALSE)
                    trace_Xref = FALSE;
            // 3cIV)  Near Jmp
              if ( flowtype == fl_JN && GetArrayElement(AR_LONG, OptArray, OPT_Jn) == FALSE)
                    trace_Xref = FALSE;
        // 3d) Get XRef Target Type
            XName = Name(crossref);
            // 3dI)   UnNamed Function
            if ( substr(XName,0,4) == "sub_" || substr(XName,0,4) == "loc_" || substr(XName,0,3) == "unk"){
              if (GetArrayElement(AR_LONG, OptArray, OPT_Fu) == FALSE) trace_Xref = FALSE;
            // 3dII)  Named Function
            } else if (GetArrayElement(AR_LONG, OptArray, OPT_Fn) == FALSE) trace_Xref = FALSE;
            // 3dIII) Extern
            if ( GetSegmentAttr(crossref, SEGATTR_TYPE) == SEG_XTRN || GetSegmentAttr(crossref, SEGATTR_TYPE) == SEG_DATA){
              obType = "EXTERN";
              trace_Xref = FALSE;
              if (GetArrayElement(AR_LONG, OptArray, OPT_X) == FALSE) report_line = FALSE;
            } else obType = "SUB";
        // 3e) Get Operation Mnemonic
            XMnem = GetMnem(ea);
            if( substr(XMnem, 0, 3) == "jmp") obReloc = "JMP";
            else if( substr(XMnem, 0, 4) == "call") obReloc = "CALL";
            else if( substr(XMnem, 0, 1) == "j") obReloc = "JCC";
            else obReloc = "UNK";
        // 3f) OutputLine
            obAddr = atoa(crossref);
            if( report_line == TRUE)
                    OutputLine(OutFileH, obParent, obType, obName, obAddr, obReloc, OptArray, ArrIndex);
        // 3g) Recurse through XRef
            if( trace_Xref == TRUE)
                    Trace_Execution( crossref, RecArray, ArrIndex, OutFileH, OptArray);
            }
          }
        }
        DelArrayElement(AR_LONG, RecArray, ArrIndex);
        ArrIndex = ArrIndex - 1;
    }
}

//======================================================================
//=  PromptOptions( OptArray ) : Set Flow Analysis Options             =
//======================================================================
static PromptOptions( OptArray ){
    auto OptString, OutStr;
    Message("---------------------------------------------------\n");
    OutStr = form( "Analysis Options Reference\n");
    OutStr = form( "%sTrace Named Funcs...Fn  Trace Near Calls.Cn\n", OutStr);
    OutStr = form( "%sTrace UnNamed Funcs.Fu  Trace Far Calls..Cf\n", OutStr);
    OutStr = form( "%sTrace Near Jumps....Jn  Show Externs.....X\n",  OutStr);
    OutStr = form( "%sTrace Far Jumps.....Jf  Screen Output....S\n",  OutStr);
    OutStr = form( "%sNon-FDB Output......P   Max Levels.......L??\n", OutStr);
    // A) Print Options to the Message Window
    Message(OutStr);
    // B) Show dialog with Options
    Warning (OutStr);
    // C) Prompt User for Analysis Options
    OptString = AskStr( "XFnFuJnJfCnCfL05","Options:");
    Message("---------------------------------------------------\n");
    if (strstr(OptString, "Fn") != -1){
        Message("Option: <Trace Named Functions> Enabled\n");
        SetArrayLong(OptArray, OPT_Fn, TRUE);        }
    if (strstr(OptString, "Fu") != -1){
        Message("Option: <Trace UnNamed Functions> Enabled\n");
        SetArrayLong(OptArray, OPT_Fu, TRUE);}
    if (strstr(OptString, "Jn") != -1){
        Message("Option: <Trace Near Jumps> Enabled\n");
        SetArrayLong(OptArray, OPT_Jn, TRUE);}
    if (strstr(OptString, "Jf") != -1){
        Message("Option: <Trace Far Jumps> Enabled\n");
        SetArrayLong(OptArray, OPT_Jf, TRUE);}
    if (strstr(OptString, "Cn") != -1){
        Message("Option: <Trace Near Calls> Enabled\n");
        SetArrayLong(OptArray, OPT_Cn, TRUE);}
    if (strstr(OptString, "Cf") != -1){
        Message("Option: <Trace Far Calls> Enabled\n");
        SetArrayLong(OptArray, OPT_Cf, TRUE);}
    if (strstr(OptString, "X") != -1){
        Message("Option: <Display Externs> Enabled\n");
        SetArrayLong(OptArray, OPT_X, TRUE); }
    if (strstr(OptString, "S") != -1){
        Message("Option: <Output To Screen Only> Enabled\n");
        SetArrayLong(OptArray, OPT_S, TRUE);}
    if (strstr(OptString, "P") != -1){
        Message("Option: <Output in Trace/Non-FDB Format> Enabled\n");
        SetArrayLong(OptArray, OPT_P, TRUE);}
    if ( strstr(OptString, "L") != -1){
        auto i, num_levels;
        i = strstr(OptString, "L");
        num_levels = substr(OptString, i + 1, i + 3);
        SetArrayLong(OptArray, OPT_L, atol(num_levels));
        Message("Option: <Maximum Trace Levels> Set To "+ num_levels +"\n"); }
    Message("---------------------------------------------------\n");
}

//======================================================================
//=  main()                                                            =
//======================================================================
static main(){
    auto ea, x, OutFileH, RecArray, OptArray, FDB_Trace;
    // A) Set up Arrays for Tracking Recursion and Passing Options
    RecArray =  CreateArray("RecursionArray");
    OptArray =  CreateArray("OptionsArray");
        // A1) Set Array Defaults to FALSE (Array Elements Match #defines above)
    for ( x=0; x<9; x = x + 1) {
        SetArrayLong(OptArray, x, FALSE);
    }
        // A2) Set Default MAX_LEVELS to 5
    SetArrayLong(OptArray, 9, 5);
        // A3) Initialize Recursion
    SetArrayLong(RecArray, 0, 256);
    // B) Prompt User For Starting Point
    FDB_Trace = AskIdent("F", "Start Analysis From: (F)Function (E)ntry Point (C)ursor Position");
    if ( FDB_Trace == "E") {
        auto EPChoice;
        EPChoice = AskStr( "0", "Entry Point (0 - " + ltoa(GetEntryPointQty(), 10) + ") ?" );
        ea = GetEntryPoint(GetEntryOrdinal(atol(EPChoice)));
    } else if (FDB_Trace == "C") ea = ScreenEA();
    else ea = ChooseFunction("Choose Function to Trace");
    // C) Prompt User For Analysis Options
    PromptOptions( OptArray );
    // D) Open Output File If Necessary
    if ( GetArrayElement(AR_LONG, OptArray, OPT_S) ==FALSE ) OutFileH = fopen(AskStr("flow_analysis.fdb", "Enter output filename: "), "wt");
    else OutFileH = "null";
    // E) Set Root Od FDB Tree
    OutputLine(OutFileH, "0", "ENTRY", Name(ea),  atoa(ea), "ENTRY", OptArray, 0);
    // F) Begin Recursice Tracing
    Trace_Execution(ea, RecArray, 0, OutFileH, OptArray);
    // G) Cleanup
    if (OutFileH != "null") fclose (OutFileH);
    DeleteArray(RecArray);
    DeleteArray(OptArray);
    Message("End of output. \n");
}
