.MODEL	MEM_MODEL,C

INCLUDE	sochalib.inc

;-----------------------------------------------------------------------;
; Copyright (C) 1992 by John Socha					;
;-----------------------------------------------------------------------;

;-----------------------------------------------------------------------;
; This file contains a number of procedure that work with the screen	;
; cursor.								;
;									;
;	SetCursor		Set the size of the hardware cursor	;
;	GetVisibleCursor	Get size of a visible cursor		;
;	GetCursor		Get the current size of the cursor	;
;	CursorOff		Turns the cursor off			;
;	CursorOn		Turns the cursor back on		;
;	CursorBlock		Changes the cursor to a block cursor	;
;	CursorUnderscore	Changes the cursor to an underscore	;
;									;
;	goto_xy			Move the cursor position		;
;	get_xy			Returns the current cursor position	;
;	get_active_page		Return the number of the active page	;
;	set_cursor		Sets the size of the hardware cursor	;
;	fix_cursor (private)	Makes sure the cursor size is valid	;
;-----------------------------------------------------------------------;

.CODE

;-----------------------------------------------------------------------;
; This procedure moves the cursor to another position on the screen.	;
; It doesn't do any error checking in case you want to do something	;
; off the screen.  But be careful!					;
;									;
; On Entry:	DH	Row (Y)						;
;		DL	Column (X)					;
;									;
; Uses:		GET_ACTIVE_PAGE						;
;-----------------------------------------------------------------------;
goto_xy		proc	uses ax bx si di bp
	call	get_active_page		;Get the number of the active page
	mov	bh,al			;Put this page number into BH
	mov	ah,2			;Call for SET CURSOR POSITION
	int	10h
	ret
goto_xy		endp


;-----------------------------------------------------------------------;
; This procedure is the reverse of GOTO_XY.  Instead of moving the	;
; cursor to a new position, it returns the current cursor position.	;
;									;
; Returns:	AH	Row (Y)						;
;		AL	Column (X)					;
;									;
; Uses:		GET_ACTIVE_PAGE						;
;-----------------------------------------------------------------------;
get_xy	proc	uses bx dx si di bp
	call	get_active_page		;Return active page in AL
	mov	bh,al			;Put this page number into BH
	mov	ah,3			;Call for read cursor position
	int	10h
	mov	ax,dx
	ret
get_xy	endp


;-----------------------------------------------------------------------;
; This procedure gets the page number for the page shown on the screen	;
; and returns it in the AL register.					;
;									;
; Returns:	AL	Active page number				;
;-----------------------------------------------------------------------;
get_active_page		proc	uses bx si di bp
	mov	ah,15			;Ask for current video state
	int	10h
	mov	al,bh			;Put active page number in AL
	ret
get_active_page		endp


;-----------------------------------------------------------------------;
; This variable is used by CursorOn and CursorOff to keep track of the	;
; size of the cursor.							;
;-----------------------------------------------------------------------;
.DATA
old_cursor_type		dw	0	;Old start and end scan lines
.CODE


;-----------------------------------------------------------------------;
; This procedure allows you to set the cursor to a new size.  Usually	;
; you'll use this to restore a cursor that you obtained with		;
; GetVisibleCursor().							;
;									;
;	void SetCursor(int size);					;
;-----------------------------------------------------------------------;
SetCursor	proc	cursorSize:Word
	mov	dx, cursorSize
	call	set_cursor
	ret
SetCursor	endp



;-----------------------------------------------------------------------;
; This procedure sets the end and start lines of the cursor.		;
;									;
; On entry:	DH	First scan-line of cursor.			;
;		DL	Last scan-line of cursor.			;
;									;
; Writes:	OLD_CURSOR_TYPE						;
;-----------------------------------------------------------------------;
set_cursor	proc	private uses ax cx si di bp
	call	GetCursor		;Get the old cursor shape
	mov	old_cursor_type, ax	;And save it
	mov	ch,dh			;Put start line into CH
	mov	cl,dl			;Put end line into CL
	mov	ah,1			;Set cursor type
	int	10h
	ret
set_cursor	endp


;-----------------------------------------------------------------------;
; This procedure gets the start and end scan lines of the cursor.  If	;
; the cursor was not visible, it returns a visible cursor.  Call	;
; GetCursor if you want the exact information on the old cursor.	;
;									;
; Returns:	AH	First scan line of cursor			;
;		AL	Last scan line of cursor			;
;-----------------------------------------------------------------------;
GetVisibleCursor	proc
	call	GetCursor		;Get cursor size into AX
	cmp	ah,0Fh			;Is the cursor off?
	jb	cursor_not_off		;No, then we're all done
	mov	ax,607h			;Yes, set to 607 for CGA
	call	fix_cursor		;Make sure CX contains a legal value
cursor_not_off:
	ret
GetVisibleCursor	endp


;-----------------------------------------------------------------------;
; This procedure gets the start and end scan lines of the cursor.  You	;
; can then use SET_CURSOR to restore the cursor's shape.		;
;									;
; NOTE: There is a bug in the ROM BIOS.  Everytime you set the mode, it	;
;	stores 0607h into the cursor type, which is the wrong value for	;
;	a monochrome display.  So...this routine translates 0607h into	;
;	0B0Ch if the display is a monochrome display.			;
;									;
; Returns:	AH	First scan line of cursor.			;
;		AL	Last scan line of cursor.			;
;-----------------------------------------------------------------------;
GetCursor	proc	uses bx cx dx si di bp
	xor	bh,bh			;Make sure page has reasonable value
	mov	ah,3			;Get current cursor type
	int	10h			;CX = cursor type
	mov	ax,cx			;Put cursor type into AX
	call	fix_cursor		;Make sure CX contains a legal value
	ret
GetCursor	endp


;-----------------------------------------------------------------------;
; This procedure "fixes" the cursor start and end scan lines in case	;
; the ROM BIOS gave us bogus information:				;
;									;
; On entry:	AH	Starting scan line of the cursor		;
;		AL	Ending scan line of the cursor			;
;									;
;	67h		COMPAQ returns this incorrect result		;
;	607h		ROM BIOS returns this for monochrome display	;
;									;
; Returns:	AX	Valid cursor size				;
;-----------------------------------------------------------------------;
	extrn	GetDisplayRows:procRef
fix_cursor	proc	private
	cmp	ax,67h			;Is this a bogus COMPAQ cursor?
	jne	not_bogus_compaq	;No, then check for monochrome
	mov	ax,607h			;Make it a legal type
not_bogus_compaq:
	push	ax
	int	11h			;Get the equipment flag
	and	al,30h			;Leave only the display type
	cmp	al,30h			;Is this a monochrome display?
	pop	ax
	jne	cursor_type_valid	;No, then accept cursor type

is_monochrome:
	push	ax
	call	GetDisplayRows		;Get number of lines on screen
	cmp	ax,25			;Are we in 43-line mode?
	pop	ax
	ja	cursor_type_valid	;Yes, keep cursor at 607h

	cmp	ax,607h			;Yes, is this a bogus cursor?
	jne	cursor_type_valid	;No, then continue
	mov	AX,0B0Ch		;Set to value for underscore cursor
cursor_type_valid:
	ret
fix_cursor	endp


;-----------------------------------------------------------------------;
; This procedure turns the cursor off.					;
;									;
; Uses:		SET_CURSOR						;
;-----------------------------------------------------------------------;
CursorOff	proc	uses ax dx
	mov	dh,0Fh			;Set start line for cursor
	mov	dl,0			;Set end line for cursor
	call	set_cursor
	ret
CursorOff	endp


;-----------------------------------------------------------------------;
; This procedure turns the cursor off.					;
;									;
; Uses:		set_cursor						;
;									;
; Note: This procedure preserves the registers so you can call it from	;
;	assembly language.						;
;-----------------------------------------------------------------------;
CursorOn	proc	uses dx
	mov	dx,old_cursor_type	;Restore old cursor type
	call	set_cursor		;And set it.
	ret
CursorOn	endp


;-----------------------------------------------------------------------;
; This procedure sets the cursor to a block cursor.  Since the color-	;
; graphics and monochrome-diplay adapter use a different number of	;
; lines for the character cell, this procedure has to check the display	;
; type.									;
;-----------------------------------------------------------------------;
	extrn	GetDisplayType:procRef

CursorBlock	proc	uses ax bx
	xor	dh,dh			;Set start line for cursor to 0
	mov	dl,7			;Set for color-graphics adapter
	call	GetDisplayType
	cmp	al,3			;Is this a monochrome display?
	jne	block_is_graphics	;No, then don't change CL
	mov	dl,13			;Set for monochrome display.
block_is_graphics:
	call	set_cursor		;Change the cursor's shape
	ret
CursorBlock	endp


;-----------------------------------------------------------------------;
; This procedure sets the cursor to a block cursor.  Since the color-	;
; graphics and monochrome-diplay adapter use a different number of	;
; lines for the character cell, this procedure has to check the display	;
; type.									;
;-----------------------------------------------------------------------;
	extrn	GetDisplayType:procRef

CursorUnderscore	proc	uses ax dx
	mov	dh,6			;Set start line for cursor to 0
	mov	dl,7			;Set for color-graphics adapter
	call	GetDisplayType
	cmp	al,3			;Is this a monochrome display?
	jne	underscore_is_graphics	;No, then don't change CL
	mov	dh,11			;Set for monochrome display.
	mov	dl,12
underscore_is_graphics:
	call	set_cursor
	ret
CursorUnderscore	endp

	end
