.MODEL	MEM_MODEL,C

INCLUDE	sochalib.inc

;-----------------------------------------------------------------------;
; Copyright (C) 1992 by John Socha					;
;-----------------------------------------------------------------------;

BELL	equ	7
TAB	equ	9
LF	equ	10
CR	equ	13

;-----------------------------------------------------------------------;
; The procedures in this module are all designed for very fast writes	;
; to the screen.  DOS is very slow, and so is the ROM BIOS.  So these	;
; procedures poke characters directly, and very quickly, into memory.	;
;									;
; Because these procedures need special information, and it would be	;
; ineffecient to constantly update local variables, there is one	;
; procedure called INIT_FAST_DISPLAY_MODULE that initializes various	;
; variables.  Call this procedure any time you change the display mode	;
; or screen.								;
;									;
;	InitFastDisplayModule	Call this before calling other procs	;
;	FastFlush		Flush unwritten chars to the screen	;
;	FastWriteRawChar	Doesn't check for special characters	;
;	FastWriteChar		Much faster version of WriteChar	;
;	FastReadAttr		Returns attribute of char under cursor	;
;	FastWriteString		Much faster version of WriteString	;
;	FastWriteUDecimal	Write an unsigned-decimal number	;
;	FastWriteNChars		Writes N copies of a character		;
;	FastWriteSpaces		Fill an are with N spaces		;
;	FastGotoXY		Version of GOTO_XY for FAST module	;
;	FastSetCursor		Moves real cursor to screen_x, screen_y	;
;	FastGetXY		Version of GET_XY for FAST module	;
;									;
;	move_to_screen		Copies chars in the buffer to screen	;
;	fast_write_raw_char	The low-level version			;
;	fast_write_char		The low-level version			;
;	calc_display_offset	(private) low-level function		;
;	fast_goto_xy		(private) low-level function		;
;									;
; Public variables:							;
;	char_attribute		This is a public variable		;
;	clear_attr		Used for new lines when scrolling	;
;	display_lines		Number of lines on the screen		;
;-----------------------------------------------------------------------;


.DATA
	public	char_attribute, clear_attr, display_lines
char_attribute		db	7, 0	;Attribute for characters
clear_attr		db	7, 0	;Attribute for areas to clear
wait_retrace_flag	db	0, 0	;1 if we should wait for horiz. retrace
topview_flag		db	0, 0	;We if we need to make update calls

display_base		dw	0B800h	;For color graphics adapter
display_offset		dw	0	;The display offset (used for page > 0)
display_page		db	0	;The currently active page
display_lines		dw	25	;Number of screen lines

screen_x		dw	0
screen_y		dw	0
screen_ptr		dw	0	;Points to character under cursor

line_buffer		dw	80 DUP (0)
line_ptr		dw	line_buffer	;Pointer to next char
line_count		dw	0	;Number of characters to write
line_start		dw	0	;Offset in screen memory of line start


max_lines	equ	50		;Maximum number of screen lines
screen_offset		dw	0,	80,	160,	240,	320
			dw	400,	480,	560,	640,	720
			dw	800,	880,	960,	1040,	1120
			dw	1200,	1280,	1360,	1440,	1520
			dw	1600,	1680,	1760,	1840,	1920
			dw	2000,	2080,	2160,	2240,	2320
			dw	2400,	2480,	2560,	2640,	2720
			dw	2800,	2880,	2960,	3040,	3120
			dw	3200,	3280,	3360,	3440,	3520
			dw	3600,	3680,	3760,	3840,	3920
			dw	4000,	4080,	4160,	4240,	4320
			dw	4400,	4480,	4560,	4640,	4720

.CODE

	extrn	GetDisplayType:procRef		;In HARDWARE module
	extrn	EgaActive:procRef		;In HARDWARE module
	extrn	GetDisplayRows:procRef		;In HARDWARE module
	extrn	get_xy:procRef			;In CURSOR   module
	extrn	get_active_page:procRef		;In CURSOR   module
;-----------------------------------------------------------------------;
; Call this procedure at least once before you call *any* other		;
; procedures.  This procedure initializes variables that other		;
; procedures use.							;
;-----------------------------------------------------------------------;
InitFastDisplayModule	proc
	push	ax
	push	dx
	mov	display_base,0B000h	;Set up for monochrome card
	mov	display_lines,25	;Set to 25 lines on the screen
	mov	wait_retrace_flag,0	;Don't wait for retraces

	call	GetDisplayType		;See which card we have
	cmp	al,3			;Do we have a monochrome card?
	je	is_monochrome		;Yes, then we're done
	mov	display_base,0B800h	;Set up for color graphics card
	call	EgaActive		;See if we have EGA card (-1)
	or	ax,ax			;Is this EGA card?
	jnz	is_ega			;Yes, don't wait for retrace
	mov	wait_retrace_flag,1	;Wait for retraces before writing chars
	jmp	Short finish_init

is_monochrome:
	;-----------------------------------------------;
	; The display adapter is attached to an IBM	;
	; monochrome display, but it could be an EGA	;
	; card.						;
	;-----------------------------------------------;
	call	EgaActive		;See if EGA card (-1)
	or	ax,ax			;Is this an EGA/VGA card?
	Jz	finish_init		;No, then we're done
					;Yes, set display lines

is_ega:
	;-----------------------------------------------;
	; This section of code gets the number of	;
	; screen lines for EGA and VGA monitors.	;
	;-----------------------------------------------;
	call	GetDisplayRows		;AX = Number of display lines
	cmp	al,max_lines		;Too many lines?
	jbe	set_display_lines	;No, then set
	mov	al,max_lines		;Yes, set to max_lines
set_display_lines:
	mov	Byte Ptr display_lines,al

finish_init:
	;-----------------------------------------------;
	; This section of code gets the current page	;
	; and calculates the offset.			;
	;-----------------------------------------------;
	push	cx
	call	get_active_page		;Get current page number in AL
	mov	display_page,al		;Save the display page
	mov	ah,al			;Put display page into AH
	xor	al,al			;AX = page * 256
	mov	cl,4			;Multiply AX by 16
	shl	ax,cl			;AX = page * 4096 (offset to page)
	mov	display_offset,ax	;Save this page offset
	pop	cx
	


	;-----------------------------------------------;
	; Check TopView interface (also windows) for	;
	; address of the display buffer.		;
	;-----------------------------------------------;
	push	bx
	push	di
	push	es
	mov	bx,display_base		;Get what we think is the display base
	mov	es,bx
	xor	di,di			;ES:DI - where we think display base is
	mov	ah,0FEh			;Make call to TopView
	int	10h			;Get Video Buffer
	mov	ax,es			;Get the new? display base
	cmp	ax,bx			;Are we running under TopView?
	je	not_in_topview		;No, then' we're done here
	mov	topview_flag,1		;Yes we are, set flag
	mov	display_base,ax		;Save new display base
	mov	display_offset,di	;Save offset to the buffer
	mov	display_page,0		;Use page 0 for TopView
	mov	wait_retrace_flag,0	;No need to wait since TopView updates
not_in_topview:
	pop	es
	pop	di
	pop	bx

	;-----------------------------------------------;
	; Now read the default character attribute.	;
	;-----------------------------------------------;
	call	get_xy			;Get the current cursor position
	mov	dx,ax			;Set up for Fast_goto_xy call
	call	fast_goto_xy		;Set the character offset
	call	FastReadAttr		;Read the current attribute
	mov	char_attribute,al	;Set character attribute to normal
	mov	clear_attr,al		;And set for cleared reagions
	pop	dx
	pop	ax
	ret
InitFastDisplayModule	endp

;-----------------------------------------------------------------------;
; This procedure writes any characters left in the buffer to the	;
; screen.								;
;									;
;	void FastFlush(void);						;
;									;
; Note: this subroutine preserves all the registers so you can call it	;
;	from assembly-language.						;
;-----------------------------------------------------------------------;
FastFlush	proc uses ax cx si di es
	mov	cx,line_count		;Number of characters to display
	jcxz	done_fast_flush		;If no chars, we're all done

	lea	ax,line_buffer		;Pointer to off-screen buffer
	mov	line_ptr,ax		;Reset line_ptr to start of buffer
	mov	si,ax			;DS:SI points to line buffer

	mov	ax,line_start		;Where to start drawing the line
	mov	di,ax
	mov	ax,display_base		;Segment for the screen buffer
	mov	es,ax			;ES:DI points to where we'll draw

	call	move_to_screen		;Copy the buffer to the screen
	mov	ax,line_start		;Get the start of the line
	mov	cx,line_count		;Get num of char/attr pairs in buffer
	shl	cx,1			;Convert to number of bytes
	add	ax,cx			;Point to first char after line
	mov	screen_ptr,ax		;Point to next location
	mov	line_start,ax
	mov	line_count,0		;Reset the line count

done_fast_flush:
	ret
FastFlush	endp


	extrn	HideMouse:procRef
	extrn	UnhideMouse:procRef
;-----------------------------------------------------------------------;
; This procedure moves a section of memory directly to screen memory.	;
; It waits for horizontal retrace periods, where needed, to avoid	;
; snow.									;
;									;
; On entry:	DS:SI		Pointer to data to move to screen	;
;		DI		Offset in screen of first char		;
;		CX		Number of words to transfer to screen	;
;									;
; Destroys:	CX, SI, DI						;
;-----------------------------------------------------------------------;
move_to_screen	proc	private uses ax es
	call	HideMouse		;Hide cursor while we write
	jcxz	done_move_to		;CX == 0, we're all done

	mov	ax,display_base		;Seg register of frame buffer
	mov	es,ax			;Set segment to start of frame buffer

	push	cx			;Save the character count
	push	di			;Save start of screen area
	cld				;Set for increment
	test	wait_retrace_flag,1	;Should we wait for horizontal retrace?
	jnz	move_to_wait		;Yes.

rep	movsw				;No, just move to the screen
	jmp	Short finish_move_to	;We're all done

move_to_wait:				;We need to wait for hoizontal retrace.
	push	bx
	push	dx

	mov	dx,03DAh		;Status register on color card
move_to_loop:
	lodsw				;Get character from off-screen buffer
	mov	bx,ax			;Save character in BX register
to_still_in_retrace:
	in	al,dx			;Get status byte
	test	al,1			;In horizontal retrace?
	jnz	to_still_in_retrace	;Yes, wait until it's done
	cli				;Turn interrupts off for now
to_wait_for_retrace:
	in	al,dx			;Get status byte
	test	al,1			;In horizontal retrace?
	jz	to_wait_for_retrace	;Not yet, wait
	mov	ax,bx			;Recover character and attribute
	stosw				;Save it to the screen
	sti				;Turn interrupts back on
	loop	move_to_loop		;Copy another word.

	pop	dx
	pop	bx

finish_move_to:
	pop	di			;Get back start of screen area
	pop	cx			;Get back number of characters

	test	topview_flag,1		;Are we in TopView?
	jz	done_move_to		;No, we're all done
	mov	ah,0FFh			;Tell TopView to update screen
	int	10h

done_move_to:
	call	UnhideMouse		;Show the mouse again
	ret
move_to_screen	endp


;-----------------------------------------------------------------------;
; This procedure provides the C-language interface to the function	;
; below.								;
;-----------------------------------------------------------------------;
FastWriteRawChar	proc	char:Word
	mov	dl, Byte Ptr char	; Get the character to display
	call	fast_write_raw_char	; Display the character
	ret
FastWriteRawChar	endp


;-----------------------------------------------------------------------;
; This procedure is like WRITE_CHAR, except that it writes characters	;
; directly to the screen and it doesn't attach special meaning to any	;
; characters, so CR, LF, etc. all appear on the screen as characters	;
;									;
; NOTE: This procedure will set SCREEN_X to 80 if you try to write off	;
;	the right side of the screen.  This way you can tell when you	;
;	tried to write off the screen.					;
;									;
; Note: This procedure is used by other assembly-language procedures in	;
;	this file.							;
;									;
; On entry:	DL	Character you want to print.			;
;									;
; Reads:	CHAR_ATTRIBUTE						;
;-----------------------------------------------------------------------;
fast_write_raw_char	proc	private uses ax di es
	push	ds
	pop	es			;Set ES so it points to data segment
	cmp	screen_x,79		;Are we still on screen?
	ja	done_write_raw		;No, don't write this char
	mov	di,line_ptr		;Get pointer to the next char
	mov	ah,char_attribute	;Get current character attribute
	mov	al,dl			;Get the character to write
	stosw				;Save this character
	inc	screen_x		;Move cursor to next column
	inc	line_count		;Number of characters in buffer
	mov	line_ptr,di		;Save new character pointer

done_write_raw:
	ret
fast_write_raw_char	endp



;-----------------------------------------------------------------------;
; This procedure provides the C-language interface to fast_write_char:	;
;									;
;	void FastWriteChar(int c);					;
;-----------------------------------------------------------------------;
FastWriteChar	proc	char:Word
	mov	dl, Byte Ptr char	; Get the character to display
	call	fast_write_char		; Display the character
	ret
FastWriteChar	endp


;-----------------------------------------------------------------------;
; This is a version of WRITE_CHAR that writes characters directly to	;
; the screen.  It uses attribute 7 as the default attribute.		;
;									;
; The following characters all need special treatment:			;
;	7	Bell							;
;	9	Tab							;
;	10	Line Feed						;
;	13	Carriage Return						;
;									;
; On entry:	DL	Character you want to print.			;
;-----------------------------------------------------------------------;
fast_write_char		proc	private
	test	dl,0F0h			;Special character?
	jz	special			;Yes, then handle them

not_special:
	call	fast_write_raw_char	;No, the write character to screen
	ret

special:
	cmp	dl,CR			;Is it a carriage return?
	je	do_cr			;Yes, then take care of CR
	cmp	dl,LF			;Is it a line feed?
	je	do_lf			;Yes, then take care of LF
	cmp	dl,TAB			;Is it a tab?
	je	do_tab			;Yes, then take care of tab
	cmp	dl,BELL			;Is it a bell?
	je	do_bell			;Yes, the sound the bell
	jmp	Short not_special	;Not a special charcter
					;(It's below all the DO routines)

do_cr:					;Move the cursor to the left side
	push	dx
	mov	dh,Byte Ptr screen_y
	xor	dl,dl			;Move to start of this line
	call	fast_goto_xy
	pop	dx
	ret

do_lf:
	push	ax
	mov	ax,display_lines	;Get number of lines on screen
	dec	ax			;AX == last line on screen
	cmp	screen_y,ax		;At or past bottom of screen?
	pop	ax
	jae	passed_bottom		;Yes, then scroll the screen
	push	dx
	mov	dh,Byte Ptr screen_y
	inc	dh			;Move to the next line
	mov	dl,Byte Ptr screen_x	;But same column
	call	fast_goto_xy		;Move the cursor
	pop	dx
	ret

passed_bottom:
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	bp
	mov	ax,0601h		;Scroll up by one line
	mov	bh,char_attribute	;Use char. attribute in scrolled lines
	mov	cx,0			;Upper left corner at (0,0)
	mov	dh,Byte Ptr display_lines ;Lower right at (display_lines, 79)
	dec	dh			;Convert to last screen line
	mov	dl,4Fh
	int	10h			; ***Replace with CALL to SCROLL_WINDOW
	pop	bp
	pop	di
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	ret

do_bell:				;Ring the bell.
	ret

do_tab:					;Expand the tab into spaces.
	push	cx
	push	dx
	mov	dl,' '			;Write out N spaces
	mov	cx,screen_x
	and	cx,7			;CX := SCREEN_X MOD 8
	neg	cx
	add	cx,8			;CX := 8 - (SCREEN_X MOD 8)
tab_loop:
	call	fast_write_raw_char
	loop	tab_loop
	pop	dx
	pop	cx
	ret
fast_write_char		endp


;-----------------------------------------------------------------------;
; This procedure reads the attribute of the character under the cursor	;
;									;
; Returns:								;
;	AX	Attribute of character under cursor			;
;-----------------------------------------------------------------------;
FastReadAttr	proc	uses dx di es
	mov	ax,display_base
	mov	es,ax
	mov	di,screen_ptr
	inc	di			;Point to the attribute

;-----------------------------------------------;
; This section of code waits for the horizontal	;
; retrace period before writing characters so	;
; you won't see snow on an IBM color-graphics	;
; adapter.  Since the horizontal retrace period	;
; is only about 2 microseconds, there's only	;
; time to read one character/attribute.		;
;-----------------------------------------------;
	test	wait_retrace_flag,1	;Should we wait for horizontal retrace?
	jz	read_dont_wait		;Nope

	mov	dx,03DAh		;Status register on color card
read_still_in_retrace:
	in	al,dx			;Get status byte
	test	al,1			;In horizontal retrace?
	jnz	read_still_in_retrace	;Yes, wait until it's done
	cli				;Turn interrupts off for now
read_wait_for_retrace:
	in	al,dx			;Get status byte
	test	al,1			;In horizontal retrace?
	jz	read_wait_for_retrace	;Not yet, wait
read_dont_wait:
	mov	al,es:[di]		;Get the attribute
	sti				;Turn interrupts back on again
	xor	ah,ah			;Return attribute in a word

	ret
FastReadAttr	endp


;-----------------------------------------------------------------------;
; This procedure writes an ASCIIZ string to the screen.  It uses the	;
; FAST_WRITE_CHAR so it's very fast, and only a few characters are	;
; treated specially.							;
;									;
; On entry:	DS:DX	Address of ASCIIZ string.			;
; Uses:		FAST_WRITE_CHAR						;
;-----------------------------------------------------------------------;
FastWriteString	proc	uses ax dx si, string:Ptr Byte
	lodES	si,string		;Get the address of the string
write_string_loop:
	mov	al,refES [si]		;Get one character
	inc	si			;Point to next character
	or	al,al			;Is the the end of the string?
	jz	end_of_string		;Yes, then stop writing
	mov	dl,al			;No, then print the character
	call	fast_write_char
	jmp	Short write_string_loop
end_of_string:
	ret
FastWriteString	endp


;-----------------------------------------------------------------------;
; This procedure writes an unsigned decimal number.			;
;									;
; On entry:	DX	Number you want to write in decimal		;
; Uses:		fast_write_char						;
;-----------------------------------------------------------------------;
FastWriteUDecimal	proc	num:Word
	mov	ax, num			;Put number in AX, where we want it
	mov	bx,10			;We divide by 10 to pick off digits
	xor	cx,cx			;Start with 0 digits on the stack
not_zero:
	xor	dx,dx			;Set upper word of 32-bit num to 0
	div	bx			;(N mod 10) --> DX, (N div 10) -- AX
	push	dx			;Save this digit on the stack
	inc	cx			;Keep track of num. of digits on stack
	or	ax,ax			;N == 0 yet?
	jnz	not_zero		;No, put more digits on stack
write_digit_loop:
	pop	dx			;Get most recent digit
	add	dl, '0'			;Convert to '0'..'9'
	call	fast_write_char		;Display this character
	loop	write_digit_loop	;Until no more digits on stack
	ret
FastWriteUDecimal	endp


;-----------------------------------------------------------------------;
; This procedure writes N characters, starting at the cursor position.	;
;									;
;	void FastWriteNChars(int char, int count);			;
;									;
; Calls:	fast_write_raw_char					;
;-----------------------------------------------------------------------;
FastWriteNChars	proc	uses cx, char:Word, count:Word
	mov	dl, Byte Ptr char	;Get the character to display
	mov	cx, count		;Number of chars to show
	jcxz	done_fast_write_chars	;Count == 0, nothing to write
chars_loop:
	call	fast_write_raw_char	;Write out one character
	loop	chars_loop		;Keep writing until CX == 0
done_fast_write_chars:
	ret
FastWriteNChars	endp


;-----------------------------------------------------------------------;
; This procedure writes N spaces, starting at the cursor position.	;
;									;
; On entry:	CX	Number of spaces we should write		;
; Calls:	FAST_WRITE_N_CHARS					;
;-----------------------------------------------------------------------;
FastWriteSpaces	proc	count:Word
	INVOKE	FastWriteNChars, ' ', count
	ret
FastWriteSpaces	endp


;-----------------------------------------------------------------------;
; This procedure calculates the offset into the screen buffer of a	;
; given character on the screen.  (Note: this procedure handles screen	;
; pages and TopView correctly.)						;
;									;
; On entry:	DH	Row	Y					;
;		DL	Column	X					;
; Returns:	Offset, in bytes, from the start of the screen segment	;
;		for the character at (X, Y).				;
;-----------------------------------------------------------------------;
calc_display_offset	proc	private
	push	bx
	mov	bl,dh			;Get the line number
	xor	bh,bh			;Convert to a word
	shl	bx,1			;Multiply line by 2 for lookup table
	mov	ax,screen_offset[bx]	;Get character offset for this line

	mov	bl,dl			;Get the X position
	xor	bh,bh			;Convert to a word
	add	ax,bx			;Character offset of cursor
	shl	ax,1			;Byte offset of character/attribute
	add	ax,display_offset	;Add in offset from non-zero page
	pop	bx
	ret
calc_display_offset	endp


;-----------------------------------------------------------------------;
; This procedure provides the C interface to fast_goto_xy:		;
;									;
;	void FastGotoXY(x, y);						;
;-----------------------------------------------------------------------;
FastGotoXY	proc	x:Word, y:Word
	mov	dh, Byte Ptr y
	mov	dl, Byte Ptr x
	call	fast_goto_xy
	ret
FastGotoXY	endp

	extrn	goto_xy:procRef		;In CURSOR module
;-----------------------------------------------------------------------;
; This is a version of GOTO_XY that works with the other fast screen	;
; procedures.  This procedure moves the cursor to another position on	;
; the screen.  It doesn't do any error checking in case you want to do	;
; something off the screen.  But be careful!				;
;									;
; On entry:	DH	Row(Y)						;
;		DL	Column (X)					;
; Uses:		GOTO_XY							;
;-----------------------------------------------------------------------;
fast_goto_xy	proc	private uses ax bx dx
	call	FastFlush		;Flush any unwritten characters
	mov	bl,Byte Ptr display_lines ;Temporary storage for Num lines
	dec	bl			;Convert to last line on screen
	cmp	dh,bl			;Are we off the screen?
	jbe	do_goto_xy		;No, then move the cursor
	cmp	dh,bl			;Are we off the bottom?
	mov	dh,bl			; (Set DH to the last line)
	jg	do_goto_xy		;Yes, then set to last line
	mov	dh,0			;No, we're off the top, set to line 0
do_goto_xy:
	call	goto_xy			;Move the real cursor
	mov	Byte Ptr screen_x,dl	;Save coordinates of cursor
	mov	Byte Ptr screen_y,dh
	call	calc_display_offset	;Get byte offset into AX
	mov	screen_ptr,ax		;Save this offset
	mov	line_start,ax		;Remember where next line starts
	ret
fast_goto_xy	endp


;-----------------------------------------------------------------------;
; This procedure will move the real cursor to the position of the	;
; fast cursor, which is just a pair of numbers stored in memory.	;
;									;
; Note: This procedure preserves the registers so you can call it from	;
;	assembly language.						;
;-----------------------------------------------------------------------;
FastSetCursor	proc	uses dx
	mov	dl,Byte Ptr screen_x
	mov	dh,Byte Ptr screen_y
	call	goto_xy			;Move the real cursor
	ret
FastSetCursor	endp


;-----------------------------------------------------------------------;
; This is a version of FAST_GET_XY that works with the fast screen	;
; module procedures.  It's the reverse of FAST_GOTO_XY.  Instead of	;
; of moving the cursor to a new position, it returns the current cursor	;
; position.								;
;									;
;	void FastGetXY(int *x, int *y);					;
;									;
; Returns:	x	Column						;
;		y	Row						;
;-----------------------------------------------------------------------;
FastGetXY	proc	x:Ptr Word, y:Ptr Word
	lodES	bx, x			; Get pointer to x into es:[bx]
	mov	ax, screen_x		; Get current x position
	mov	refES [bx], ax		; Save into C's x variable

	lodES	bx, y			; Get pointer to y into es:[bx]
	mov	ax, screen_y		; Get current y position
	mov	refES [bx], ax		; Save into C's y variable
	ret
FastGetXY	endp


if 0


	PUBLIC	FAST_CLEAR_TO_EOL
	EXTRN	CLEAR_WINDOW:NEAR
;-----------------------------------------------------------------------;
; This procedure clears the line from the current cursor position to	;
; the end of the line.							;
;-----------------------------------------------------------------------;
	PUBLIC	_FAST_CLEAR_TO_EOL
_FAST_CLEAR_TO_EOL	LABEL	NEAR
FAST_CLEAR_TO_EOL	PROC	NEAR
	PUSH	AX
	PUSH	BX
	MOV	BL,BYTE PTR SCREEN_X	;Get position of cursor
	MOV	BH,BYTE PTR SCREEN_Y
	MOV	AL,79			;Last column of the line
	CMP	BL,AL			;Was last char on screen?
	JA	DONT_CLEAR_TO_EOL	;No, the don't try to clear
	MOV	AH,BH			;Stay on same line
	CALL	CLEAR_WINDOW
DONT_CLEAR_TO_EOL:
	POP	BX
	POP	AX
	RET
FAST_CLEAR_TO_EOL	ENDP

endif

	end
