.MODEL	MEM_MODEL,C

INCLUDE	sochalib.inc

;-----------------------------------------------------------------------;
; Copyright (C) 1992 by John Socha					;
;-----------------------------------------------------------------------;

;-----------------------------------------------------------------------;
; This file contains the following procedures:				;
;									;
; ReadKey()		Reads a character from the ROM BIOS and returns	;
;			either an ASCII code, or 0x1?? where ?? is the	;
;			scan code of the non-ASCII key.			;
; scan_to_extended	Convert to extended ASCII code (1xxh)		;
; ReadRawKey()		Returns scan code in high byte and character	;
;			code in the low byte.				;
; ClearKbdBuffer()	Clear the keyboard buffer			;
; KeyWaiting()		Returns code of next character if the buffer is	;
;			not empty, and 0 if it is empty.		;
; ShiftKeys()		Returns the current state of the shift keys.	;
;-----------------------------------------------------------------------;

.CODE

	public ReadKey
;-----------------------------------------------------------------------;
; This procedure uses the ROM BIOS routines to read a single character.	;
; It returns the ASCII code for keys that generate an ASCII character.	;
; But if you push one of the key-pad or function keys, this procedure	;
; will return the scan code in the lower byte and set the upper byte	;
; to 1.									;
;									;
; Returns:	ASCII code for ASCII characters				;
;		1xx (hex) where xx is the scan code of the special key	;
;-----------------------------------------------------------------------;
ReadKey	proc
	xor	ah,ah			;Ask for a character from BIOS
	int	16h
	call	scan_to_extended	;Convert to 1xx for special keys
	cld				;Must be cleared for MSC
	ret
ReadKey	endp


;-----------------------------------------------------------------------;
; This procedure converts a scan-code/ASCII-code pair into an extended	;
; character.  If the lower byte is 0, this procedure puts the scan	;
; code into the lower byte and sets the upper byte to 1.  Otherwise, it	;
; sets the upper byte to 0.						;
;									;
; On entry:	AH	Scan code for the key pushed			;
;		AL	ASCII code, or 0 for special keys		;
;									;
; Returns:	ASCII code for ASCII characters				;
;		1xx (hex) where xx is the scan code of the special key	;
;									;
; Note:		This procedure now works also for the grey keys on the	;
;		extended keyboard (which have E0h in the lower byte	;
;		rather than 00h).					;
;-----------------------------------------------------------------------;
scan_to_extended	proc	private
	or	al,al			;Is this a special key?
	je	special_key		;Yes, then handle it
	cmp	al,0E0h			;Is this a special key?
	je	special_key		;Yes, then handle it
	xor	ah,ah			;No, return the ASCII code
	ret

special_key:
	xchg	al,ah			;Put the scan code into AL
	mov	ah,1			;And set AH = 1
	ret
scan_to_extended	endp


;-----------------------------------------------------------------------;
; This procedure reads one character (or one special function key) from	;
; the keyboard and returns the ASCII code, or scan code.		;
;									;
; This procedure uses the ROM BIOS call so it can read both the		;
; character code AND the scan code.  The DOS call only returns the	;
; ASCII code.								;
;									;
; Returns:	AH	The scan code of the key you pushed.  This is	;
;			useful for special keys that don't have ASCII	;
;			codes, like the function and cursor keys.	;
;		AL	The ASCII code for the key you pushed.		;
;			0 for all special keys.				;
;-----------------------------------------------------------------------;
ReadRawKey	proc
	mov	ah,0			;Ask for a character
	int	16h
	cld				;Must be cleared for MSC
	ret
ReadRawKey	endp


;-----------------------------------------------------------------------;
; This procedure clears the keyboard buffer of any characters that may	;
; still be around.							;
;-----------------------------------------------------------------------;
ClearKbdBuffer	proc
	push	dx
	mov	ax,0C06h		;Clear input buffer
	mov	dl,0FFh			;Check status the return
	int	21h
	pop	dx
	cld				;Must be cleared for MSC
	ret
ClearKbdBuffer	endp


;-----------------------------------------------------------------------;
; This procedure checks the keyboard buffer to see if there are any	;
; characters waiting to be read from the ROM BIOS:			;
;									;
; Returns:	Character code of next character waiting.		;
;		-1 if there are no characters waiting.			;
;									;
; NOTE:	The ROM BIOS returns a scan code and character code of 0 after	;
;	you've hit ^break, so it is possible to detect a ^break by	;
;	checking to see if you read a 0 from the keyboard.		;
;									;
; NOTE: This procedure issues an INT 28h before checking the keyboard	;
;	status to allow background processes more frequent access to	;
;	system resources.						;
;-----------------------------------------------------------------------;
KeyWaiting	proc
	int	28h			;Allow background processes a slice

	mov	ah,1			;Check for waiting character
	int	16h
	jnz	done_kbd_hit		;Return the character code
	xor	ax,ax			;Return -1 for no character
	not	ax
done_kbd_hit:
	cld				;Must be clear for MSC compiler
	ret
KeyWaiting	endp


;-----------------------------------------------------------------------;
; This procedure returns the current state of the shift keys:		;
;									;
; INS_STATE	80h	Insert state is turned on			;
; CAPS_STATE	40h	Caps lock key is on				;
; NUM_STATE	20h	Num lock is turned on				;
; SCROLL_STATE	10h	Scroll lock is on				;
; ALT_SHIFT	08h	The alt key is down				;
; CTL_SHIFT	04h	The control key is down				;
; LEFT_SHIFT	02h	The left-hand shift key is down			;
; RIGHT_SHIFT	01h	The right-hand shift key is down		;
;-----------------------------------------------------------------------;
ShiftKeys	proc
	mov	ah,2
	int	16h			;Get the current flags
	xor	ah,ah			;Set upper byte to zero
	cld				;Must be cleared for MSC
	ret
ShiftKeys	endp


	end
