.MODEL	MEM_MODEL,C

INCLUDE	sochalib.inc

;-----------------------------------------------------------------------;
; Copyright (C) 1992 by John Socha					;
;									;
; This file contains some of the mouse interface routines.		;
;									;
; InitMouse();		Do a full initlize on mouse hardware and sw.	;
; InitMouseSw();	Initialize the mouse software, and return	;
;				-1	Mouse installed			;
;				0	No mouse installed		;
; HideMouser();		Removes the mouse cursor from the screen	;
; UnhideMouse();	Makes the mouse cursor visible again		;
;									;
; Get_mouse_position(&x, &y);		Returns character coords	;
; Set_mouse_position(x, y);		Returns character coords	;
; Mouse_buttons(&x, &y);		Which and where buttons pushed	;
;-----------------------------------------------------------------------;

.DATA
	public	mouse_installed, swap_buttons, mouse_visible
mouse_installed		DB	0		;TRUE if a mouse is installed
swap_buttons		DW	0		;TRUE to swap mouse buttons
mouse_visible		DB	0		;Visible if > 0
			DB	0		;Make it a word

mouse_cursor	DW	0		;Storage for mouse cursor char


.CODE


;-----------------------------------------------------------------------;
; This procedure tests to see if the mouse driver is installed, or if	;
; some driver is installed on INT 33h.					;
;									;
; Returns:	AX	0	There is no mouse installed		;
;			-1	There is a driver on INT 33h		;
;-----------------------------------------------------------------------;
check_int_33	proc	private uses bx es
	mov	ax,3533h		;Get the current INT 33h vector
	int	21h
	mov	ax,es			;Put the segment into AX
	or	ax,ax			;Is anything installed?
	jz	no_int_33		;No, then return 0
	or	bx,bx			;Is anything installed?
	jz	no_int_33		;No, then return 0
	mov	ax,0FFFFh		;Return -1, there is INT 33h vector
	jmp	Short done_check_int_33

no_int_33:
	xor	ax,ax
done_check_int_33:
	ret
check_int_33	endp


;-----------------------------------------------------------------------;
; This procedure initializes the mouse by calling function 0, which is	;
; a full reset of the moues software AND hardware.  In some cases this	;
; can take several seconds (a serial mouse or an IBM mouse), which may	;
; not be acceptable.  Init_mouse() can be much faster in such cases.	;
; Here are the rules on which of these two procedures to call:		;
;									;
;	Once								;
;		Init_mouse() must be called at least to initialize	;
;		the mouse hardware the first time since not all mouse	;
;		drivers initialize the mouse proplerly when you do	;
;		a software mouse reset.					;
;									;
;	After Init_mouse()						;
;		From then on you can call Init_mouse_sw() for a fast	;
;		reset.  The Commander calls Init_mouse() once, then	;
;		Init_mouse_sw() each time you return to the Commander	;
;		after running a DOS command.				;
;-----------------------------------------------------------------------;
InitMouse	proc
	mov	bx,0			;Ask for a full reset
	call	do_init_mouse		;Initialize the mouse
	ret
InitMouse	endp


;-----------------------------------------------------------------------;
; This procedure initializes the mouse.  It tries to do a software	;
; reset first.  Only if the software reset doesn't work does it try to	;
; do a hardware reset.  In some cases (such as a serial mouse or the	;
; IBM mouse) a hardware reset can take as much as 5 seconds, which is	;
; not an acceptable delay when returning from the Commander.		;
;-----------------------------------------------------------------------;
InitMouseSw	proc
	mov	bx,1			;Ask for a software reset
	call	do_init_mouse		;Initialize the mouse
	ret
InitMouseSw		endp


;-----------------------------------------------------------------------;
; This procedure resets the mouse software, and checks to see if the	;
; mouse and mouse software is installed.  It returns the following:	;
;									;
; On entry:	BX	0	Do a hardware reset			;
;			1	Do a soft reset				;
;									;
; Returns:	AX	-1	The mouse and software are installed	;
;			0	There is no mouse, driver not installed	;
;-----------------------------------------------------------------------;
	extrn	EgaActive:procRef
do_init_mouse	proc	private uses bx cx dx es bp
	call	check_int_33		;See INT 33 has vector before we call
	or	ax,ax			;Is anything installed?
	jz	no_mouse		;No, report that there's no mouse

	or	bx,bx			;Yes, Is this a full hardware reset?
	jz	do_hardware_reset	;Yes, do a full hardware reset

do_reset:				;No, try to reset the mouse software
	mov	ax,33			;Request a software reset
	int	33h			;Call the mouse routine
	cmp	ax,33			;Did mouse ignore this function?
	je	do_hardware_reset	;Yes, do a hardware reset
	cmp	ax,-1			;Is the mouse installed?
	jne	do_hardware_reset	;No, then do a hardware reset
	cmp	bx,2			;Is the mouse installed?
	je	finish_init		;Yes, then finish initializing

do_hardware_reset:
	mov	ax,0			;Ask for the mouse reset function
	int	33h			;Call the mouse routine
	or	ax,ax			;Is the mouse installed?
	jz	no_mouse		;No, report that mouse is not installed

finish_init:
	mov	ax,10			;Set the text cursor mask and type
	xor	bx,bx			;Select the software text cursor
	mov	cx,0FFFFh		;Set the mask to all ones
	mov	dx,7700h		;And set cursor to inverting cursor
	int	33h			;Returns -1 if if the mouse installed
	mov	mouse_visible,0		;The mouse is now hidden

	call	EgaActive		;See if an EGA card is active
	or	ax,ax			;Is EGA active?
	jz	done_init_is_mouse	;No, then we're all done
	mov	ax,1130h		;Get EGA information
	xor	bh,bh			;Ask for the default information
	int	10h			;DL = #rows-1, CX = char height, pixels
	inc	dl			;DL = # rows
	cmp	dl,25			;Is this normal 25 line mode?
	je	done_init_is_mouse	;Yes, don't change mouse settings
	mov	al,dl			;Put number of rows into AX
	mov	cl,8			;Microsoft assumes chars 8 pixels high
	mul	cl			;AX = screen height, in pixels
	dec	ax			;AX = screen height - 1
	mov	dx,ax			;DX = screen height - 1
	mov	ax,8			;Set max and min vertical positions
	mov	cx,0
	int	33h

done_init_is_mouse:
	xor	ax,ax			;Return -1 to report mouse installed
	not	ax

done_init_mouse:
	mov	mouse_installed,al	;Save the mouse installed state
	ret

no_mouse:
	xor	ax,ax			;The mouse is not installed
	jmp	done_init_mouse
do_init_mouse	endp


;-----------------------------------------------------------------------;
; This procedure puts the mouse cursor back on the screen.  I unhide	;
; the mouse only when Mouse_visible is 0 when you make this call.  In	;
; other words, only when you cross the visiblity threshold.		;
;									;
; You can nest calls to Hide and Unhide_mouse_cursor.			;
;-----------------------------------------------------------------------;
UnhideMouse	proc
	test	mouse_installed,0FFh	;Is the mouse installed?
	jz	done_unhide		;No, don't do anything
	inc	mouse_visible
	cmp	mouse_visible,1		;Did mouse just become visible?
	jne	done_unhide		;No, then don't unhide again
	push	ax			;Yes, make the INT 33 call
	mov	ax,1			;Ask for the show-cursor function
	int	33h
	pop	ax
done_unhide:
	ret
UnhideMouse	endp


;-----------------------------------------------------------------------;
; This procedure removes the mouse cursor from the screen.		;
;-----------------------------------------------------------------------;
HideMouse	proc
	test	mouse_installed,0FFh	;Is the mouse installed?
	jz	done_hide		;No, don't do anything
	dec	mouse_visible		;Decrement the count
	cmp	mouse_visible,0		;Did mouse just become invisible?
	jne	done_hide		;No, then don't hide mouse again
	push	ax
	mov	ax,2			;Ask for the hide-cursor function
	int	33h
	pop	ax
done_hide:
	ret
HideMouse	endp


;-----------------------------------------------------------------------;
; This procedure returns the current mouse position in character	;
; coordinates.								;
;									;
;	GetMousePosition(int *x, int *y);				;
;-----------------------------------------------------------------------;
GetMousePosition	proc	x:Ptr Word, y:Ptr Word
	test	mouse_installed,0FFh	;Is the mouse installed?
	jz	done_get_mouse		;No, then don't do anything
	mov	ax,3			;Ask for get position function
	int	33h			;Ask for the mouse positoin
	lodES	bx,x			;Get address of X
	shr	cx,1			;Divide result by 8 to get char. coord.
	shr	cx,1
	shr	cx,1
	mov	refES [bx],cx		;Save X coordinate
	shr	dx,1			;Divide by 8 to get char. coord.
	shr	dx,1
	shr	dx,1
	lodES	bx,y			;Get address of Y
	mov	refES [bx],dx		;Save Y coordinate
done_get_mouse:
	ret
GetMousePosition	endp


;-----------------------------------------------------------------------;
; This procedure sets the mouse position on the screen in character	;
; coordinates.								;
;									;
;	SetMousePosition(int x, int y);					;
;-----------------------------------------------------------------------;
SetMousePosition	proc	x:Word, y:Word
	test	mouse_installed,0FFh	;Is the mouse installed?
	jz	done_set_mouse		;No, then don't do anything
	mov	ax,4			;Ask for set mouse position function
	mov	cx,x			;Get the new X coordinate
	shl	cx,1			;Multiply by 8 to get pixel coords
	shl	cx,1
	shl	cx,1
	mov	dx,y			;Get the new Y coordinate
	shl	dx,1			;Multiply by 8 to get pixel coords
	shl	dx,1
	shl	dx,1
	int	33h
done_set_mouse:
	ret
SetMousePosition	endp


;-----------------------------------------------------------------------;
; This procedure checks the status of the mouse buttons:		;
;									;
;	int MouseButtons(int *x, int *y);				;
;									;
; It returns the following information:					;
;									;
;		0	Both buttons up					;
;		1	Left button down				;
;		2	Right button down				;
;		3	Both buttons down				;
;									;
; The X and Y coordinates returned are character coordinates to make	;
; things easier since my stuff is currently character based.		;
;									;
; NOTE: This procedure treats the middle button on three-button mice	;
;	as the same as both buttons.					;
;-----------------------------------------------------------------------;
	extrn	FastFlush:procRef
MouseButtons	proc	x:Ptr Word, y:Ptr Word
	xor	ax,ax			;Return 0 if mouse not installed
	test	mouse_installed,0FFh	;Is the mouse installed?
	jz	done_mouse_buttons	;No, then return 0
	call	FastFlush		;Flush unwritten chars to screen
	mov	ax,3			;Ask for button status
	int	33h
	mov	ax,bx			;Put button status into AX
	cmp	ax,3			;Is center or both buttons down?
	jb	middle_not_down		;No, then continue.
	mov	ax,3			;Yes, set to both buttons
	jmp	Short finish_buttons	;Now calculate coordinates
middle_not_down:
	or	ax,ax			;Are both buttons up?
	jz	finish_buttons		;Yes, then calculate coordinates
	cmp	swap_buttons,0		;Should we swap mouse buttons?
	jz	finish_buttons		;No, calculate coordinates
	xor	ax,3			;Swap the mouse buttons
finish_buttons:
	lodES	bx,x			;Get address for X
	shr	cx,1			;Divide by 8 to return char. coords.
	shr	cx,1
	shr	cx,1
	mov	refES [bx],cx		;Save X coordinate of mouse
	lodES	bx,y			;Get address for Y
	shr	dx,1			;Divide by 8 to return char. coords.
	shr	dx,1
	shr	dx,1
	mov	refES [bx],dx		;Save Y coordinate of mouse
done_mouse_buttons:
	ret
MouseButtons	endp

	end
