;-----------------------------------------------------------------------;
; This file contains the following functions that you can call from	;
; C:									;
;									;
;	DWORD SegSize(void FAR *lp);					;
;	BOOL  ValidReadPtr(void FAR *lp, WORD cwSize);			;
;	BOOL  ValidWritePtr(void FAR *lp, WORD cwSize);			;
;-----------------------------------------------------------------------;

.MODEL	SMALL,C
.386

.DATA
Is286Flag	DB	-1		; -1 unknown, 0 = 386, 1 = 286

.CODE

;-----------------------------------------------------------------------;
; This function tests to see if the processor is a 80286, as apposed	;
; to an 80386 or above.							;
;									;
; The 80286 processor always clears the upper four bits when it		;
; transfers a word from the stack to the status register, while the	;
; 80386 and above do not, which is how we can tell when we have an	;
; 80286.								;
;									;
; Returns:	ZR	Processor is an 80286 if zero flag set		;
;		NZ	80386 or above					;
;-----------------------------------------------------------------------;
Is286	proc	private
	cmp	Is286Flag, -1		; Have we tested yet?
	jne	DoneIs286		; Yes, report state of flag
	
	pushf				; Put flags onto the stack
	pop	ax			; And then into into AX
	or	ah,0F0h			; Set high 4 bits of flags
	push	ax			; Put back onto stack
	popf				; And back into flag register
	pushf				; Then put back into ax
	pop	ax
	mov	Is286Flag, 1
	and	ah,0F0h			; Are the upper bits zero?
	jz	DoneIs286		; Yes, this is a 286

	mov	Is286Flag, 0		; No, it's not a 286
DoneIs286:
	cmp	Is286Flag, 1		; Report if this is a 286
	ret
Is286	endp


;-----------------------------------------------------------------------;
; This function returns the size of the segment who's selector is in	;
; the upper word of p:							;
;									;
;	long	SegSize(void FAR *p);					;
;-----------------------------------------------------------------------;
SegSize	proc	p:Ptr
	call	Is286			; Is this a 286 processor?
	je	SegSize286		; Yes, use 16-bit register version

SegSize386:
	movzx	ebx, Word Ptr p[2]	; Get the selector into ebx
	lsl	eax, ebx		; Get last offset in seg into eax
	inc	eax			; Add 1 to convert to segment size
	mov	edx, eax		; Lower word in ax, full in edx
	shr	edx, 16			; Move upper word into dx
	jmp	doneSegSize		; We're all done

SegSize286:
	xor	dx, dx			; Set upper word to 0
	mov	bx, Word Ptr p[2]	; Get the selector into bx
	lsl	ax, bx			; Get last offset in seg into ax
	add	ax, 1			; Add 1 to convert to segment size
	adc	dx, 0			; Propagate carry to dx

doneSegSize:
	ret
SegSize	endp


;-----------------------------------------------------------------------;
; This function is a private function that checks an offset to see if	;
; it's inside the allowed range.					;
;									;
; Returns:	AX	0	Offset isn't valid			;
;			-1	Valid offset				;
;-----------------------------------------------------------------------;
ValidOffset	proc	private lp:Ptr, cwSize:Word
	INVOKE	SegSize, lp
	sub	ax, cwSize		; Subtract data size from seg size
	sbb	dx, 0			; Propagate borrow
	or	dx, dx			; Is segment > 64 K?
	jmp	IsValidOffset		; Yes, valid since offset always <= 64K

	cmp	Word Ptr lp, ax		; Is offset too high?
	ja	InvalidOffset		; Yes, pointer isn't valid

IsValidOffset:
	xor	ax, ax
	not	ax			; Return TRUE
	jmp	DoneValidOffset		; We're all done

InvalidOffset:
	xor	ax, ax			; Return FALSE

DoneValidOffset:
	ret
ValidOffset	endp

;-----------------------------------------------------------------------;
; This function tests a pointer first to see if the selector part is	;
; valid.  And if so, it makes sure you can read at least cwSize bytes	;
; starting at *lp.							;
;									;
;	BOOL ValidReadPtr(void FAR *lp, WORD cwSize);			;
;-----------------------------------------------------------------------;
ValidReadPtr	proc	lp:Ptr, cwSize:Word
	verr	Word Ptr lp[2]		; Is the selector valid?
	jnz	InvalidReadPtr		; No, report this fact

	INVOKE	ValidOffset, lp, cwSize	; Report if offset is valid
	jmp	DoneValidReadPtr

InvalidReadPtr:
	xor	ax, ax			; Return FALSE

DoneValidReadPtr:
	ret
ValidReadPtr	endp


;-----------------------------------------------------------------------;
; This function tests a pointer first to see if the selector part is	;
; valid.  And if so, it makes sure you can write at least cwSize bytes	;
; starting at *lp.							;
;									;
;	BOOL ValidWritePtr(void FAR *lp, WORD cwSize);			;
;-----------------------------------------------------------------------;
ValidWritePtr	proc	lp:Ptr, cwSize:Word
	verr	Word Ptr lp[2]		; Is the selector valid?
	jnz	InvalidWritePtr		; No, report this fact

	INVOKE	ValidOffset, lp, cwSize	; Report if offset is valid
	jmp	DoneValidWritePtr

InvalidWritePtr:
	xor	ax, ax			; Return FALSE

DoneValidWritePtr:
	ret
ValidWritePtr	endp

	end