;================================================
; CHMOD    Version 1.1    5/3/84
;
;         by David Whitman
;
; Utility to examine and modify the read/write
; mode of a file under DOS 2.0.
;
; Syntax:
;
; CHMOD [d:][path] [filespec] [/N] [/R] [/S] [/H]
;
; If no file is specified, a help message is printed,
; summarizing the CHMOD command syntax.
;
; The attribute byte of the specified file is set
; according to the selected options:
;
;     /N - normal file
;     /R - read only
;     /S - system file
;     /H - hidden file
;
; Multiple options may be selected.  /N will cancel
; options preceding it in the command line.
;
; If no options are selected, a report on the current
; mode of the specified file is sent to the standard output.
;
; On exit, ERRORLEVEL is set to reflect the current file
; mode, and/or the success of the change, as follows:
;
;   normal file      --> 0
;   read only file   --> 1
;   hidden file      --> 2
;   system file      --> 4
;   failed operation --> 8
;
; Requires DOS 2.0, will abort under earlier versions.
;
; This source file is in CHASM assembler syntax.
;======================================================

;==========
; EQUATES
;==========

@chmod     equ    43H        ;change file mode
@dosver    equ    30H        ;get DOS version number
@exit      equ    4CH        ;set ERRORLEVEL and exit
@prnstr    equ    09H        ;print string

normal     equ    00H
readonly   equ    01H
hidden     equ    02H
system     equ    04H
failure    equ    08H

true       equ    0FFH       ;boolean values
false      equ    00H

cr         equ    0DH        ;carriage return
lf         equ    0AH        ;line feed
beep       equ    07H        ;bell

param_area  equ   [81H]      ;unformatted parameter area
param_count equ   [80H]      ;number of characters in above

;===========================================================

chmod      proc   near
           call   chkdos              ;test for proper DOS
           call   parsefile           ;parse path/filename
           call   options             ;parse options, set flags
           call   doit                ;perform specified action
           call   cleanup             ;final processing, exit
           mov    al, opflag          ;set errorlevel with current attributes
           mov    ah, @exit           ;and exit
           int    21H
           endp

;=================================================
; SUBROUTINE CHKDOS
; Checks for proper DOS, exits if not 2.0 or above
;=================================================
chkdos     proc   near
           mov    ah, @dosver         ;get dos version number
           int    21H                 ;with dos call
           cmp    al, 2               ;2.0 or over?
           jae    a1                  ;yes, skip

           mov    ah, @prnstr         ;no, bitch
           mov    dx, offset(baddos)  ;point to message
           int    21H                 ;and print it
           pop    ax                  ;reset stack
           int    20H                 ;and exit
a1
           ret

baddos     db     beep, cr, lf, 'This program requires DOS 2.0!' cr, lf, '$'
           endp

;===================================================
; SUBROUTINE PARSEFILE
; Scans the parameter area for a path/filename.
; Sets PATHPTR to point to the name, and terminates
; it with a 0, to make an ASCIIZ string.
;
; If no name is found, ERRORLEVEL is set to 8,
; and control is passed back to DOS.
;===================================================
parsefile  proc   near

           xor    ch,ch               ;cx <== # of parameter characters
           mov    cl, param_count     ;    "
           mov    di, offset(param_area)
           mov    al, ' '             ;search for first non-blank
           rep
           scasb
           jcxz   berror              ;nothing? bitch

           dec    di                  ;back up to character found
           inc    cx                  ; "
           cmpb   [di], '/'           ;are we into the options?
           jne    b1                  ;no, skip
berror     mov    ah, @prnstr         ;yes, print help message
           mov    dx, offset(nofile)
           int    21H
           pop    ax                  ;reset stack
           mov    ah, @exit           ;set error level and exit
           mov    al, failure
           int    21H

b1
           mov    pathptr, di         ;otherwise point to pathname

           repne                      ;now search for next blank
           scasb
           jcxz   b2                  ;reached end of data?
           dec    di                  ;nope, back up to last non-blank
           inc    cx                  ; "
b2         movb   [di], 00H           ;ASCIIZ string terminator
           ret

pathptr    db   00H, 00H            ;pointer to path/filename
nofile     db   cr, lf
           db   'CHMOD version 1.1' cr lf
           db   cr lf
           db   'Syntax:  CHMOD [d:][path] [filespec] [/N] [/R] [/S] [/H]'
           db   cr lf cr lf
           db   'The attribute byte of the specified file is set' cr lf
           db   'according to the selected options:' cr lf
           db   cr lf
           db   '     /N - normal file' cr lf
           db   '     /R - read only' cr lf
           db   '     /S - system file' cr lf
           db   '     /H - hidden file' cr lf
           db   cr lf
           db   'Multiple options may be selected.' cr lf
           db   cr lf
           db   'If no options are selected, a report on the current mode' cr lf
           db   'of the specified file is sent to STDOUT and ERRORLEVEL'cr lf
           db   'is set to the value of the file attribute byte. ' cr lf cr lf
           db   '$'
           endp
;=====================================================
; SUBROUTINE OPTIONS
; Scans the command line for options, and builds
; the new attribute byte in OPFLAG.
;
; If options are found, OPFOUND is set true, otherwise
; it remains false.
;======================================================
options    proc near
                                      ;cx still has number of chars. left,
                                      ;di points to current position.

c1         mov   al, '/'              ;options marked with '/'
           repnz                      ;scan for options
           scasb
           jcxz  cexit                ;reached end

           mov   al, [di]             ;get option character
           and   al, 0DFH             ;guarantees upper case

           cmp   al, 'N'              ;normal file?
           jne   c2                   ;no, skip
           movb  opflag, normal       ;yes, clear all bits
           movb  opfound, true        ;set found flag
           jmps  c1                   ;and loop

c2         cmp   al, 'R'              ;read only?
           jne   c3                   ;no, skip
           orb   opflag, readonly     ;yes, set option flag
           movb  opfound, true        ;set found flag
           jmps  c1                   ;and loop

c3         cmp   al, 'S'              ;system?
           jne   c4                   ;no, skip
           orb   opflag, system       ;yes, set option flag
           movb  opfound, true        ;set found flag
           jmps  c1                   ;and loop

c4         cmp   al, 'H'              ;hidden?
           jne   c1                   ;no, just loop
           orb   opflag, hidden       ;yes, set option flag
           movb  opfound, true        ;set found flag
           jmps  c1                   ;and loop

cexit      ret

opflag     db    00H                  ;options selected - new attribute byte
opfound    db    00H                  ;if non-zero, an option was decoded
           endp

;========================================================
; SUBROUTINE DOIT
; Does the actual work.  Either sets new file attribute,
; or reads the existing attribute.
;========================================================
doit       proc  near
           mov   dx, pathptr          ;point DX at ASCIIZ path/filename
           cmpb  opfound, false       ;any options?
           je    d1                   ;no, skip
           xor   ch,ch                ;CX <== new attribute byte
           mov   cl, opflag
           mov   al, 01H              ;enable attribute setting
           jmps  d2

d1         mov   al, 00H              ;enable attribute read

d2         mov   ah, @chmod           ;change file mode dos call
           int   21H

           jc    derr                 ;carry flag means error
           and   cl, 07H              ;mask off high bits
           mov   opflag, cl           ;save attribute byte
           jmps  dexit                ;and exit gracefully

derr       cmp   al, 02H              ;file not found error
           jne   d3
           mov   dx, offset(fnferror)
           mov   ah, @prnstr
           int   21H
           jmps  dabort

d3         cmp   al, 03H              ;path not found error
           jne   d4
           mov   dx, offset(pnferror)
           mov   ah, @prnstr
           int   21h
           jmps  dabort

d4         cmp   al, 05H              ;access denied error
           jne   dabort
           mov   dx, offset(aderror)
           mov   ah, @prnstr
           int   21H


dabort     pop   ax                   ;reset stack
           mov   al, failure          ;set errorlevel
           mov   ah, @exit            ;and abort to dos
           int   21H

dexit      ret                        ;normal return

fnferror   db    cr, lf, 'File not found.' cr, lf, '$'
pnferror   db    cr, lf, 'Path not found.' cr, lf, '$'
aderror    db    cr, lf, 'Access denied.'  cr, lf, '$'
           endp

;==================================================
; SUBROUTINE CLEANUP
; If no options were specified, a report of current
; attributes is printed.
;==================================================
cleanup    proc  near
           cmpb  opfound, true        ;were options specified?
           je    eexit                ;yes, exit

           mov   ah, @prnstr          ;print report header
           mov   dx, offset(header)
           int   21H

           testb opflag, readonly     ;read only?
           jz    e1
           mov   dx, offset(roattrib) ;print attribute message
           int   21H

e1         testb opflag, system       ;system?
           jz    e2
           mov   dx, offset(sattrib)  ;print attribute message
           int   21H

e2         testb opflag, hidden       ;hidden?
           jz    e3
           mov   dx, offset(hattrib)  ;print attribute message
           int   21H

e3         cmpb  opflag, normal       ;normal?
           jne   eexit
           mov   dx, offset(nattrib)  ;print attribute message
           int   21H

eexit      ret

header     db    cr, lf, 'Current Attributes:' cr, lf, cr, lf, '$'
roattrib   db            '   Read only' cr, lf, '$'
sattrib    db            '   System' cr, lf, '$'
hattrib    db            '   Hidden' cr, lf, '$'
nattrib    db            '   Normal' cr, lf, '$'
           endp
