unit Main;

interface

uses
  Forms, StdCtrls, Menus, Classes, Controls, ComCtrls, ExtCtrls, ToolWin, ImgList, ResourceInfo,
  SysUtils;

type
  TfmMain = class(TForm)
    Splitter1: TSplitter;
    ResourceInfo1: TResourceInfo;
    TreeView1: TTreeView;
    Panel1: TPanel;
    Memo1: TMemo;
    tbOpen: TToolButton;
    ImageList1: TImageList;
    tbExit: TToolButton;
    tbSave: TToolButton;
    tbUpdateModifications: TToolButton;
    ToolBar1: TToolBar;
    tbCancelModifications: TToolButton;
    Panel2: TPanel;
    tbAbout: TToolButton;
    Label1: TLabel;

    procedure TreeView1Change(Sender: TObject; Node: TTreeNode);
    procedure tbOpenClick(Sender: TObject);
    procedure tbSaveClick(Sender: TObject);
    procedure tbUpdateModificationsClick(Sender: TObject);
    procedure tbExitClick(Sender: TObject);
    procedure Memo1Change(Sender: TObject);
    procedure TreeView1Changing(Sender: TObject; Node: TTreeNode; var AllowChange: Boolean);
    procedure ResourceInfo1Activate(Sender: TObject);
    procedure ResourceInfo1Deactivate(Sender: TObject);
    procedure tbCancelModificationsClick(Sender: TObject);
    procedure tbAboutClick(Sender: TObject);

  private
    FCurrentFile: TFileName;
    FEditing: Boolean;
    FModified: Boolean;

    function GetCurrentDfm: TDfm;
    procedure SetModified(const Value: Boolean);

    function CheckModifications: Boolean;
    procedure UpdateTreeView;

    function GetTitle: string;

    property CurrentDfm: TDfm read GetCurrentDfm;
    property Title: string read GetTitle;
    property Modified: Boolean read FModified write SetModified;

  public
    constructor Create(AOwner: TComponent); override;
  end; {TfmMain}

var
  fmMain: TfmMain;

implementation

uses
  AppData, About, Dialogs;

{$R *.DFM}

resourcestring
  rsFileOpen               = 'Open';
  rsFileSave               = 'Save';
  rsUpdateModifications    = 'Update modifications in memory';
  rsCancelModifications    = 'Cancel modifications';
  rsExit                   = 'Exit';
  rsAbout                  = 'About';

  rsExeFiles               = 'Exe files|*.EXE;*.DLL;*.BPL;*.DPL';
  rsRCDATAASCIIFileFilter  = 'RCDATA ASCII File (*.rc)|*.rc';
  rsRCDATABinaryFileFilter = 'RCDATA Binary File (*.dat)|*.dat';

  rsNoDfmDataFound         = 'No DFM RCDATA found';


procedure TfmMain.UpdateTreeView;
var
  i: Cardinal;
  tmpTreeNode: TTreeNode;
begin
  FEditing := False;
  TreeView1.Items.Clear;
  Memo1.Lines.Clear;
  with ResourceInfo1 do
    if DfmCount > 0 then
      for i := 0 to Pred(DfmCount) do
        begin
          tmpTreeNode := TreeView1.Items.AddChild(nil, Dfms[i].Name);
          tmpTreeNode.Data := Pointer(i);
        end;
  Modified := False;
  with TreeView1 do
    if Items.Count > 0 then
      begin
        FEditing := True;
        Selected := Items[0];
        TreeView1Change(TreeView1, TreeView1.Selected);
      end;
  Caption := Format('%s - %s', [Title, ExtractFileName(FCurrentFile)]);
end; {TfmMain.UpdateTreeView}

procedure TfmMain.TreeView1Change(Sender: TObject; Node: TTreeNode);
begin
  if FEditing then
    with ResourceInfo1.Dfms[Cardinal(Node.Data)] do
      begin
        Data.Seek(0, 0);
        Memo1.Lines.LoadFromStream(Data);
        Modified := False;
      end;
end; {TfmMain.TreeView1Change}

function TfmMain.GetCurrentDfm: TDfm;
begin
  if TreeView1.Selected <> nil then
    Result := ResourceInfo1.Dfms[Cardinal(TreeView1.Selected.Data)]
  else
    Result := nil;
end; {TfmMain.GetCurrentDfm}

procedure TfmMain.tbOpenClick(Sender: TObject);
begin
  with TOpenDialog.Create(Self) do
    try
      Filter := rsExeFiles;
      if Execute then
        if FileName <> '' then
          with ResourceInfo1 do
            begin
              Active := False;
              ExeFileName := FileName;
              Active := True;
              EnumDfmNames;
              UpdateTreeView;
              if DfmCount = 0 then
                ShowMessage(rsNoDfmDataFound);
            end;
    finally
      FCurrentFile := FileName;
      Free;
    end;
end; {TfmMain.tbOpenClick}

procedure TfmMain.tbSaveClick(Sender: TObject);
begin
  if CheckModifications then
    with TSaveDialog.Create(Self) do
      try
        FileName := CurrentDfm.Name;
        Filter := Format('%s|%s', [rsRCDATAASCIIFileFilter, rsRCDATABinaryFileFilter]);
        if Execute then
          case FilterIndex of
            1: CurrentDfm.SaveToFile(ChangeFileExt(FileName, '.rc'), dfmASCII);
            2: CurrentDfm.SaveToFile(ChangeFileExt(FileName, '.dat'), dfmBinary);
          end;
      finally
        Free;
      end;
end; {TfmMain.tbSaveClick}

procedure TfmMain.tbUpdateModificationsClick(Sender: TObject);
begin
  TMemoryStream(CurrentDfm.Data).Clear;
  Memo1.Lines.SaveToStream(TMemoryStream(CurrentDfm.Data));

  Modified := False;
end; {TfmMain.tbUpdateClick}

procedure TfmMain.tbExitClick(Sender: TObject);
begin
  Close;
end; {TfmMain.tbExitClick}

constructor TfmMain.Create(AOwner: TComponent);
begin
  inherited;

  Caption := Title;

  tbOpen.Hint := rsFileOpen;
  tbSave.Hint := rsFileSave;
  tbUpdateModifications.Hint := rsUpdateModifications;
  tbCancelModifications.Hint := rsCancelModifications;
  tbExit.Hint := rsExit;
  tbAbout.Hint := rsAbout;

  Modified := False;

  tbSave.Enabled := ResourceInfo1.Active;
end; {TfmMain.Create}

procedure TfmMain.Memo1Change(Sender: TObject);
begin
  Modified := True;
end; {TfmMain.SetModified}

procedure TfmMain.SetModified(const Value: Boolean);
begin
  FModified := Value;
  tbUpdateModifications.Enabled := Value;
  tbCancelModifications.Enabled := Value;
end; {TfmMain.SetModified}

procedure TfmMain.TreeView1Changing(Sender: TObject; Node: TTreeNode; var AllowChange: Boolean);
begin
  CheckModifications;
end; {TfmMain.TreeView1Changing}

procedure TfmMain.ResourceInfo1Activate(Sender: TObject);
begin
  tbSave.Enabled := True;
end; {TfmMain.ResourceInfo1Activate}

procedure TfmMain.ResourceInfo1Deactivate(Sender: TObject);
begin
  tbSave.Enabled := False;
end; {TfmMain.ResourceInfo1Deactivate}

function TfmMain.CheckModifications: Boolean;
begin
  if FEditing and Modified then
    begin
      if MessageDlg('Current DFM object has been modified, do you want to save them?',
        mtConfirmation, [mbYes, mbNo], 0) = mrYes then
          tbUpdateModificationsClick(Self)
      else
        Result := False;
    end;
end; {TfmMain.CheckModifications}

procedure TfmMain.tbCancelModificationsClick(Sender: TObject);
begin
  UpdateTreeView;
end; {TfmMain.tbCancelModificationsClick}

procedure TfmMain.tbAboutClick(Sender: TObject);
begin
  with TfmAbout.Create(Self) do
    try
      ShowModal;
    finally
      Free;
    end;
end; {TfmMain.tbAboutClick}

function TfmMain.GetTitle: string;
begin
  Result := Format('%s v%s', [rsAppName, rsAppVersion]);
end; {TfmMain.GetTitle}

end.
