// ----------------------------------- //
//            APISpy32 v2.1            //
//     Copyright 1999 Yariv Kaplan     //
//          WWW.INTERNALS.COM          //
// ----------------------------------- //

#include <windows.h>
#include <stdio.h>
#include <commctrl.h>
#include "resource.h"
#include "..\dll\logapi.h"
#include "htmlhelp.h"
#include "apispy9x.h"
#include "apispynt.h"
#include "general.h"
#include "errors.h"
#include "apispy32.h"

// Event handles

HANDLE hOverflowEvent = NULL;
HANDLE hLogEvent = NULL;
HANDLE hInjectThreadExitEvent = NULL;
HANDLE hLogThreadExitEvent = NULL;

// Thread handles

HANDLE hLogThread = NULL;
HANDLE hInjectDLLThread = NULL;

// Mutex handles

HANDLE hListViewMutex = NULL;
HANDLE hLogMutex = NULL;

// Flags

bool IsNT = false;
bool WindowOnTop = false;
bool AutoScroll = true;
bool CaptureEvents = true;

DWORD dwListViewRow = 0;
HANDLE hMailslot = NULL;
HWND hWndMain;
HWND hWndToolbar;
HWND hWndListView;
HINSTANCE hInst;
DWORD dwHelpCookie;
bool DeletingListView = false;

// Exported functions 

GETLOGPARAMETERS_PROC pGetLogParameters;
SETCAPTUREEVENTSFLAG_PROC pSetCaptureEventsFlag;
ADVANCETONEXTLOGENTRY_PROC pAdvanceToNextLogEntry;

// Toolbar buttons

TBBUTTON tbButtons[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 5, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
};

TBBUTTON tbButtonsOld[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 5, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0}
};						 

// Forward declarations

LRESULT CALLBACK MainWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);
bool ShutdownAPISpy32();


void ErrorHandler(PSTR pszErrorMsg, bool FatalFlag)
{
  char szErrorMsg[256];

  if (FatalFlag)
  {
    strcpy(szErrorMsg, pszErrorMsg);
    strcat(szErrorMsg, ERRORMSG);

    MessageBox(hWndMain, szErrorMsg, "APISpy32", MB_OK | MB_ICONERROR);
    ShutdownAPISpy32();
    ExitProcess(0);
  }
  else
    MessageBox(hWndMain, pszErrorMsg, "APISpy32", MB_OK | MB_ICONWARNING);
}


void InitApplication(HINSTANCE hInstance)
{
  WNDCLASS wc;
  ATOM WndClassAtom;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)MainWndProc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, "ICON");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = (HBRUSH)(COLOR_INACTIVEBORDER + 1);
  wc.lpszMenuName = "LISTMENU";
  wc.lpszClassName = "APISpy32Class";

  WndClassAtom = RegisterClass(&wc);

  if (WndClassAtom == 0)
    ErrorHandler(ERROR_REGISTERCLASS, true);
}


bool GetWindowsVersion()
{
  OSVERSIONINFO VersionInfo;
  BOOL Result;

  VersionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

  Result = GetVersionEx(&VersionInfo);

  if (Result == FALSE)
    return false;

  if (VersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT)
    IsNT = true;

  return true;
}


void ReadSettings()
{
  HKEY hKey;
  DWORD dwParamSize;

  Settings.dwLeft = CW_USEDEFAULT;
  Settings.dwTop = CW_USEDEFAULT;
  Settings.dwWidth = CW_USEDEFAULT;
  Settings.dwHeight = CW_USEDEFAULT;
  Settings.WindowOnTop = false;
  Settings.Maximized = false;

  Settings.dwLVColumnWidth[0] = COLUMN_WIDTH1;
  Settings.dwLVColumnWidth[1] = COLUMN_WIDTH2;
  Settings.dwLVColumnWidth[2] = COLUMN_WIDTH3;
  Settings.dwLVColumnWidth[3] = COLUMN_WIDTH4;

  RegCreateKey(HKEY_CURRENT_USER, APISPY32_REG_KEY, &hKey);

  dwParamSize = sizeof(Settings);

  RegQueryValueEx(hKey, "Settings", NULL, NULL, (LPBYTE)&Settings, &dwParamSize);

  RegCloseKey(hKey);

  WindowOnTop = Settings.WindowOnTop;  
}


bool WriteSettings(HWND hWnd)
{
  HKEY hKey;
  RECT Rect;
  LONG Result;
  DWORD dwIndex;

  Result = GetWindowRect(hWnd, &Rect);

  if (Result == 0)
    return false;

  if (!IsIconic(hWnd))
  {
    Settings.dwLeft = Rect.left;
    Settings.dwTop = Rect.top;
    Settings.dwWidth = Rect.right - Rect.left;
    Settings.dwHeight = Rect.bottom - Rect.top;
  }

  Settings.WindowOnTop = WindowOnTop;
  Settings.Maximized = IsZoomed(hWnd);

  for (dwIndex = 0; dwIndex < NUMCOLUMNS; dwIndex++)
    Settings.dwLVColumnWidth[dwIndex] = ListView_GetColumnWidth(hWndListView, dwIndex);

  Result = RegOpenKey(HKEY_CURRENT_USER, "Software\\Internals\\APISpy32", &hKey);

  if (Result != ERROR_SUCCESS)
    return false;

  Result = RegSetValueEx(hKey, "Settings", 0, REG_BINARY, (LPBYTE)&Settings, sizeof(Settings));

  if (Result != ERROR_SUCCESS)
    return false;

  Result = RegCloseKey(hKey);

  if (Result != ERROR_SUCCESS)
    return false;

  return true;
}


HWND InitInstance(HINSTANCE hInstance, int nCmdShow)
{

  ReadSettings();

  hInst = hInstance;

  hWndMain = CreateWindow("APISpy32Class", "APISpy32 - http://www.internals.com", WS_OVERLAPPEDWINDOW, Settings.dwLeft, Settings.dwTop, Settings.dwWidth, Settings.dwHeight, NULL, NULL, hInstance, NULL);

  if (hWndMain == NULL)
    return NULL;

  ShowWindow(hWndMain, nCmdShow);
  UpdateWindow(hWndMain);

  if (Settings.Maximized)
    ShowWindow(hWndMain, SW_SHOWMAXIMIZED);

  if (WindowOnTop)
  {
    SetWindowPos(hWndMain, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
    CheckMenuItem(GetMenu(hWndMain), IDM_ONTOP, MF_BYCOMMAND | (WindowOnTop ? MF_CHECKED:MF_UNCHECKED));
  }

  return hWndMain;
}


HWND CreateListView(HWND hWndParent)
{
  RECT Rect;
  BOOL Result;
  LV_COLUMN	LVColumn;
  DWORD dwIndex;

  struct tagListViewColumn
  {
    PSTR pszText;
    DWORD dwWidth;
  }ListViewColumn[] =
  
  {
    {"Process",      COLUMN_WIDTH1},
    {"API",          COLUMN_WIDTH2},
    {"Return Value", COLUMN_WIDTH3},
    {"Origin",       COLUMN_WIDTH4}
  };
  
  InitCommonControls();

  for (dwIndex = 0; dwIndex < NUMCOLUMNS; dwIndex++)
    ListViewColumn[dwIndex].dwWidth = Settings.dwLVColumnWidth[dwIndex];

  Result = GetClientRect(hWndParent, &Rect);

  if (Result == FALSE)
    return NULL;

  hWndListView = CreateWindowEx(0L, WC_LISTVIEW, "", WS_VISIBLE | WS_CHILD | WS_BORDER | LVS_REPORT | LVS_SINGLESEL, 0, TOOLBARHEIGHT, Rect.right - Rect.left, Rect.bottom - Rect.top - TOOLBARHEIGHT, hWndParent, (HMENU)ID_LIST, hInst, NULL);

  if (hWndListView == NULL)
    return NULL;

  LVColumn.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
  LVColumn.fmt = LVCFMT_LEFT;

  for (dwIndex = 0; dwIndex < sizeof(ListViewColumn) / sizeof(ListViewColumn[0]); dwIndex++)
  {
    LVColumn.iSubItem = dwIndex;
    LVColumn.cx = ListViewColumn[dwIndex].dwWidth;
    LVColumn.pszText = ListViewColumn[dwIndex].pszText;
    
    if (ListView_InsertColumn(hWndListView, dwIndex, &LVColumn) == -1)
      return NULL;
  }

  SendMessage(hWndListView, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_FULLROWSELECT, LVS_EX_FULLROWSELECT);

  return hWndListView;
}


DWORD GetDLLVersion(PSTR pszDLLName)
{
  HRESULT hResult;
  DWORD dwVersion;
  HINSTANCE hInstDll;
  DLLVERSIONINFO DllVersionInfo;
  DLLGETVERSIONPROC pDllGetVersion;


  hInstDll = LoadLibrary(pszDLLName);

  if (hInstDll == NULL)
    return 0;

  pDllGetVersion = (DLLGETVERSIONPROC)GetProcAddress(hInstDll, "DllGetVersion");

  if (pDllGetVersion)
  {
    ZeroMemory(&DllVersionInfo, sizeof(DllVersionInfo));

    DllVersionInfo.cbSize = sizeof(DllVersionInfo);

    hResult = (*pDllGetVersion)(&DllVersionInfo);

    if (SUCCEEDED(hResult))
      dwVersion = PACKVERSION(DllVersionInfo.dwMajorVersion, DllVersionInfo.dwMinorVersion);
  }

  FreeLibrary(hInstDll);

  return dwVersion;
}


DWORD SplitItems(PSTR pszItems, char *pItems[])
{
  DWORD dwItemCount = 0;

  pszItems = strtok(pszItems, ";");

  while (pszItems)
  {
    pItems[dwItemCount] = pszItems;

    dwItemCount++;

    pszItems = strtok(NULL, ";");
  }

  return dwItemCount;
}


void ListView_Append(PSTR pszItems)
{
  char *pItems[NUMCOLUMNS];
  DWORD dwItemCount;
  LVITEM LVItem;
  DWORD dwIndex;

  dwItemCount = SplitItems(pszItems, pItems);

  LVItem.mask = LVIF_TEXT;
  LVItem.iItem = dwListViewRow;
  LVItem.iSubItem = 0;
  LVItem.pszText = pItems[0];
  LVItem.cchTextMax = strlen(pItems[0] + 1);
  ListView_InsertItem(hWndListView, &LVItem);

  if (AutoScroll)
    ListView_EnsureVisible(hWndListView, ListView_GetItemCount(hWndListView) - 1, FALSE);

  if (dwItemCount > 1)
    for (dwIndex = 0; dwIndex < dwItemCount - 1; dwIndex++)
      ListView_SetItemText(hWndListView, dwListViewRow, dwIndex + 1, pItems[dwIndex + 1]);

  dwListViewRow++;
}


LRESULT CALLBACK AboutBoxWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg)
  {
    case WM_INITDIALOG:

      return TRUE;

    break;

    case WM_COMMAND:

      if (LOWORD(wParam) == IDOK)
      {
        EndDialog(hWnd, TRUE);

        return TRUE;
      }

    break;

    case WM_CLOSE:

      EndDialog(hWnd, TRUE);

      return TRUE;

    break;
  }

  return FALSE;
}


bool OutputLogBuffer()
{
  bool Result;
  char szItems[1024];
  DWORD dwIndex;
  DWORD dwNumLogEntries;
  DWORD dwReadLogIndex;
  tagLogEntry *pLogEntry;
  char szReturnValue[20];
  char szOriginAddress[20];

  Result = pGetLogParameters(&dwNumLogEntries, &dwReadLogIndex, &pLogEntry);

  if (Result == false)
    return false;

  if (dwNumLogEntries)
  {
    WaitForSingleObject(hListViewMutex, INFINITE);

    SendMessage(hWndListView, WM_SETREDRAW, FALSE, 0);
  }
  
  for (dwIndex = 0; dwIndex < dwNumLogEntries; dwIndex++)
  {

    Result = GetProcessName9x(pLogEntry[dwReadLogIndex].dwProcessId, szItems);

    if (Result == false)
      strcpy(szItems, "?");

    strcat(szItems, ";");
    strcat(szItems, pLogEntry[dwReadLogIndex].szAPIName);

    wsprintf(szReturnValue, "%#.8x", pLogEntry[dwReadLogIndex].dwReturnValue);
    wsprintf(szOriginAddress, "%#.8x", pLogEntry[dwReadLogIndex].pvOriginAddress);

    strcat(szItems, ";");
    strcat(szItems, szReturnValue);
    strcat(szItems, ";");
    strcat(szItems, szOriginAddress);
          
    ListView_Append(szItems);

    pAdvanceToNextLogEntry();
  }

  if (dwNumLogEntries)
  {
    SendMessage(hWndListView, WM_SETREDRAW, TRUE, 0);
    InvalidateRect(hWndListView, NULL, FALSE);

    ReleaseMutex(hListViewMutex);
  }
  
  return true;
}


DWORD WINAPI LogThread9x(PVOID pvParameter)
{
  DWORD Result;
  HANDLE hEventArray[2];

  hEventArray[0] = hOverflowEvent;
  hEventArray[1] = hLogThreadExitEvent;

  while (1)
  {
    Result = WaitForMultipleObjects(2, &hEventArray[0], FALSE, 100);

    switch (Result)
    {
      case WAIT_OBJECT_0:

        // Buffer overflow

        OutputLogBuffer();

        ResetEvent(hOverflowEvent);
        SetEvent(hLogEvent);

      case WAIT_OBJECT_0 + 1:

        // Exit event

        ResetEvent(hLogThreadExitEvent);

        return TRUE;

      break;

      case WAIT_TIMEOUT:

        OutputLogBuffer();

      break;

      case WAIT_FAILED:

        return FALSE;

      break;
    }
  }

  return TRUE;
}


DWORD WINAPI LogThreadNT(PVOID pvParameter)
{
  DWORD Result;
  DWORD dwIndex;
  DWORD dwBytesRead;
  char szItems[1024];
  char szReturnValue[20];
  char szOriginAddress[20];
  tagLogEntry LogEntry;
  DWORD dwMessageSize;
  DWORD dwMessageCount;

  while (1)
  {
    Result = WaitForSingleObject(hLogThreadExitEvent, 100);

    switch (Result)
    {
      case WAIT_OBJECT_0:

        // Exit event

        ResetEvent(hLogThreadExitEvent);
      
        return TRUE;

      break;

      case WAIT_TIMEOUT:

        Result = GetMailslotInfo(hMailslot, 0, &dwMessageSize, &dwMessageCount, NULL);

        if (Result && dwMessageCount)
        {

          WaitForSingleObject(hListViewMutex, INFINITE);

          for (dwIndex = 0; dwIndex < dwMessageCount; dwIndex++)
          {
            Result = ReadFile(hMailslot, &LogEntry, sizeof(tagLogEntry), &dwBytesRead, NULL);

            if (Result)
            {
              Result = GetProcessNameNT(LogEntry.dwProcessId, szItems);
  
              if (Result == false)
                strcpy(szItems, "?");

              strcat(szItems, ";");
              strcat(szItems, LogEntry.szAPIName);

              wsprintf(szReturnValue, "%#.8x", LogEntry.dwReturnValue);
              wsprintf(szOriginAddress, "%#.8x", LogEntry.pvOriginAddress);

              strcat(szItems, ";");
              strcat(szItems, szReturnValue);
              strcat(szItems, ";");
              strcat(szItems, szOriginAddress);
          
              ListView_Append(szItems);
            }
          }

          ReleaseMutex(hListViewMutex);
        }

      break;

      case WAIT_FAILED:

        return FALSE;

      break;
    }
  }

  return TRUE;
}


DWORD WINAPI InjectDLLThread(PVOID pvParameter)
{
  DWORD Result;

  while (1)
  {
    Result = WaitForSingleObject(hInjectThreadExitEvent, DLL_INJECTION_TIMEOUT);

    switch (Result)
    {
      case WAIT_OBJECT_0:

        ResetEvent(hInjectThreadExitEvent);

        return TRUE;

      break;

      case WAIT_TIMEOUT:

        InjectDLL(LOAD_DLL);

      break;

      case WAIT_FAILED:

        return FALSE;

      break;
    }
  }

  return TRUE;
}


DWORD WINAPI ClearListViewThread(PVOID pvParameter)
{
  DWORD dwIndex;

  WaitForSingleObject(hListViewMutex, INFINITE);

  pSetCaptureEventsFlag(false);

  if (IsNT == false)
  {
    SendMessage(hWndListView, WM_SETREDRAW, FALSE, 0);

    DeletingListView = true;

    for (dwIndex = 0; dwIndex < dwListViewRow; dwIndex++)
      ListView_DeleteItem(hWndListView, 0);

    DeletingListView = false;

    SendMessage(hWndListView, WM_SETREDRAW, TRUE, 0);
    InvalidateRect(hWndListView, NULL, FALSE);
  }
  else
    ListView_DeleteAllItems(hWndListView);

  dwListViewRow = 0;

  pSetCaptureEventsFlag(CaptureEvents);

  ReleaseMutex(hListViewMutex);

  return TRUE;
}


void InitAPISpy32()
{
  SETAPISPY32PROCESSID_PROC pSetAPISpy32ProcessId;
  HINSTANCE hInstDLL;
  DWORD dwLogThreadId;
  DWORD dwInjectDLLThreadId;
  bool Result;

  if (IsNT)
  {
    Result = ObtainSeDebugPrivilege();

    if (Result == false)
      ErrorHandler(WARNING_DEBUGPRIVILEGE, false);

    hMailslot = CreateMailslot("\\\\.\\mailslot\\APISpy32_Mailslot", 0, MAILSLOT_WAIT_FOREVER, 0);

    if (hMailslot == INVALID_HANDLE_VALUE)
      ErrorHandler(ERROR_MAILSLOT, true);

    hInstDLL = LoadLibrary(APISPY32DLL_NT);

    if (hInstDLL == NULL)
      ErrorHandler(ERROR_APISPY32DLL_NT, true);
  }
  else
  {
    hInstDLL = LoadLibrary(APISPY32DLL_9X);

    if (hInstDLL == NULL)
      ErrorHandler(ERROR_APISPY32DLL_9X, true);
  }

  if (IsNT)
  {
    Result = InitPSAPI();

    if (Result == false)
      ErrorHandler(ERROR_PSAPI, true);
  }
  else
  {
    Result = InitToolhelp32();

    if (Result == false)
      ErrorHandler(ERROR_TOOLHELP32, true);
  }

  hLogMutex = CreateMutex(NULL, FALSE, "APISpy32_LogMutex");

  if (hLogMutex == NULL)
    ErrorHandler(ERROR_LOGMUTEX, true);

  hOverflowEvent = CreateEvent(NULL, TRUE, FALSE, "APISpy32_OverflowEvent");

  if (hOverflowEvent == NULL)
    ErrorHandler(ERROR_OVERFLOWEVENT, true);

  hLogEvent = CreateEvent(NULL, TRUE, FALSE, "APISpy32_LogEvent");

  if (hLogEvent == NULL)
    ErrorHandler(ERROR_LOGEVENT, true);

  hLogThreadExitEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

  if (hLogThreadExitEvent == NULL)
    ErrorHandler(ERROR_LOGTHREADEXITEVENT, true);

  hInjectThreadExitEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

  if (hInjectThreadExitEvent == NULL)
    ErrorHandler(ERROR_INJECTTHREADEXITEVENT, true);

  hListViewMutex = CreateMutex(NULL, FALSE, NULL);

  if (hListViewMutex == NULL)
    ErrorHandler(ERROR_LISTVIEWMUTEX, true);

  if (!IsNT)
  {
    pGetLogParameters = (GETLOGPARAMETERS_PROC)GetProcAddress(hInstDLL, "GetLogParameters");

    if (pGetLogParameters == NULL)
      ErrorHandler(ERROR_GETLOGPARAMETERS, true);

    pAdvanceToNextLogEntry = (ADVANCETONEXTLOGENTRY_PROC)GetProcAddress(hInstDLL, "AdvanceToNextLogEntry");

    if (pAdvanceToNextLogEntry == NULL)
      ErrorHandler(ERROR_ADVANCETONEXTLOGENTRY, true);
  }

  pSetCaptureEventsFlag = (SETCAPTUREEVENTSFLAG_PROC)GetProcAddress(hInstDLL, "SetCaptureEventsFlag");

  if (pSetCaptureEventsFlag == NULL)
    ErrorHandler(ERROR_SETCAPTUREEVENTSFLAG, true);

  pSetAPISpy32ProcessId = (SETAPISPY32PROCESSID_PROC)GetProcAddress(hInstDLL, "SetAPISpy32ProcessId");

  if (pSetAPISpy32ProcessId == NULL)
    ErrorHandler(ERROR_SETAPISPY32PROCESSID, true);

  pSetAPISpy32ProcessId(GetCurrentProcessId());

  if (IsNT)
  {
    hInjectDLLThread = CreateThread(NULL, 0, InjectDLLThread, 0, 0, &dwInjectDLLThreadId);

    if (hInjectDLLThread == NULL)
      ErrorHandler(ERROR_INJECTDLLTHREAD, true);

    hLogThread = CreateThread(NULL, 0, LogThreadNT, 0, 0, &dwLogThreadId);
  }
  else
    hLogThread = CreateThread(NULL, 0, LogThread9x, 0, 0, &dwLogThreadId);

  if (hLogThread == NULL)
    ErrorHandler(ERROR_LOGTHREAD, true);

  // Start capturing events

  pSetCaptureEventsFlag(true);
}


bool ShutdownAPISpy32()
{
  if (pSetCaptureEventsFlag)
    pSetCaptureEventsFlag(false);

  if (dwHelpCookie)
  {
    HtmlHelp(NULL, NULL, HH_CLOSE_ALL, 0);
    HtmlHelp(NULL, NULL, HH_UNINITIALIZE, dwHelpCookie);
  }

  if (IsNT && hInjectThreadExitEvent && hInjectDLLThread)
  {
    // Wait until the injection thread resumes

    SetEvent(hInjectThreadExitEvent);
    WaitForSingleObject(hInjectDLLThread, THREAD_EXIT_TIMEOUT);
  }

  // Wait until the logging thread resumes

  if (hLogThreadExitEvent && hLogThread)
  {
    SetEvent(hLogThreadExitEvent);
    WaitForSingleObject(hLogThread, THREAD_EXIT_TIMEOUT);
  }

  // Remove the interception DLL from all address spaces

  if (IsNT)
    InjectDLL(FREE_DLL);

  // Close handles of event objects

  if (hOverflowEvent)
    CloseHandle(hOverflowEvent);

  if (hLogEvent)
    CloseHandle(hLogEvent);

  if (hInjectThreadExitEvent)
    CloseHandle(hInjectThreadExitEvent);

  if (hLogThreadExitEvent)
    CloseHandle(hLogThreadExitEvent);

  // Close handles of mutex objects

  if (hListViewMutex)
    CloseHandle(hListViewMutex);

  if (hLogMutex)
    CloseHandle(hLogMutex);
  
  // Close handles of threads

  if (hLogThread)
    CloseHandle(hLogThread);

  if (hInjectDLLThread)
    CloseHandle(hInjectDLLThread);

  // Close handle of mailslot

  if (IsNT && hMailslot)
    CloseHandle(hMailslot);

  return true;
}


DWORD WINAPI SaveLogFileThread(PVOID SaveAs)
{
  OPENFILENAME OpenFileName;
  BOOL Result;
  DWORD dwRow;
  DWORD dwColumn;
  FILE *hLogFile;
  char szItemText[256] = "";
  char szListViewText[1024];
  static bool LogFileChosen = false;
  static char szLogFileName[MAX_PATH];

  if (SaveAs || !LogFileChosen)
  {
    OpenFileName.lStructSize       =  sizeof(OPENFILENAME);
    OpenFileName.hwndOwner         =  hWndMain;
    OpenFileName.hInstance         =  (HINSTANCE)hInst;
    OpenFileName.lpstrFilter       =  "Log Files (*.log)\0*.log\0All Files (*.*)\0*.*\0";
    OpenFileName.lpstrCustomFilter =  (LPTSTR)NULL;
    OpenFileName.nMaxCustFilter    =  0L;
    OpenFileName.nFilterIndex      =  1L;
    OpenFileName.lpstrFile         =  szLogFileName;
    OpenFileName.nMaxFile          =  256;
    OpenFileName.lpstrFileTitle    =  NULL;
    OpenFileName.nMaxFileTitle     =  0;
    OpenFileName.lpstrInitialDir   =  NULL;
    OpenFileName.lpstrTitle        =  "Save File Info...";
    OpenFileName.nFileOffset       =  0;
    OpenFileName.nFileExtension    =  0;
    OpenFileName.lpstrDefExt       =  "*.log";
    OpenFileName.lpfnHook          =  NULL;
    OpenFileName.Flags             =  OFN_LONGNAMES | OFN_HIDEREADONLY;

    Result = GetSaveFileName(&OpenFileName);

    if (Result == FALSE)
      return FALSE;
  }

  hLogFile = fopen(szLogFileName, "w");

  if (hLogFile == NULL)
  {
    ErrorHandler(WARNING_OPENLOGFILE, false);
    return FALSE;
  }

  WaitForSingleObject(hListViewMutex, INFINITE);

  for (dwRow = 0; dwRow < dwListViewRow; dwRow++)
  {
    strcpy(szListViewText, "");

    for (dwColumn = 0; dwColumn < NUMCOLUMNS; dwColumn++)
    {
      ListView_GetItemText(hWndListView, dwRow, dwColumn, szItemText, sizeof(szItemText)); 

      strcat(szListViewText, szItemText);
      strcat(szListViewText, "\t");
    }

    fprintf(hLogFile, "%s\n", szListViewText);
  }

  fclose(hLogFile);

  ReleaseMutex(hListViewMutex);

  LogFileChosen = true;

  return TRUE;
}


LRESULT CALLBACK MainWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  LPTOOLTIPTEXT pToolTipText;
  char szToolTipText[128];
  PAINTSTRUCT PaintStruct;
  DWORD dwClearListViewThreadId;
  HANDLE hClearListViewThread;
  DWORD dwSaveLogFileThreadId;
  HANDLE hSaveLogFileThread;
  DWORD DllVersion;
  DWORD Result;
  HDC hDC;

  switch (uMsg)
  {
    case WM_CREATE:

      DllVersion = GetDLLVersion("comctl32.dll");

      if (DllVersion >= PACKVERSION(4, 71))
        hWndToolbar = CreateToolbarEx(hWnd, TOOLBAR_FLAT | WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS, ID_TOOLBAR, 10, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtons, NUM_TOOLBAR_BUTTONS, 16, 16, 16, 15, sizeof(TBBUTTON));
      else
        hWndToolbar = CreateToolbarEx(hWnd, WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS, ID_TOOLBAR, 10, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtonsOld, NUM_OLD_TOOLBAR_BUTTONS, 16, 16, 16, 15, sizeof(TBBUTTON));
      
      if (hWndToolbar == NULL)
        ErrorHandler(WARNING_TOOLBAR, false);

      hWndListView = CreateListView(hWnd);

      if (hWndListView == NULL)
        ErrorHandler(ERROR_LISTVIEW, true);

      HtmlHelp(NULL, NULL, HH_INITIALIZE, (DWORD)&dwHelpCookie);
 
      // Initialize the API interception engine

      InitAPISpy32();

    break;

    case WM_NOTIFY:

      if (((LPNMHDR)lParam)->code == TTN_NEEDTEXT)
      {
        pToolTipText = (LPTOOLTIPTEXT)lParam;
        LoadString(hInst, pToolTipText->hdr.idFrom, szToolTipText, sizeof(szToolTipText));
        pToolTipText->lpszText = szToolTipText;
      }
      
    break;

    case WM_SIZE:

      MoveWindow(hWndToolbar, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
      MoveWindow(hWndListView, 0, TOOLBARHEIGHT, LOWORD(lParam), HIWORD(lParam) - TOOLBARHEIGHT, TRUE);

    break;

    case WM_CLOSE:

      ShutdownAPISpy32();

      WriteSettings(hWnd);

      return DefWindowProc(hWnd, uMsg, wParam, lParam);

    break;

    case WM_SETFOCUS:

      SetFocus(hWndListView);

    break;

    case WM_PAINT:

      if (IsNT == false && DeletingListView)
      {
        hDC = BeginPaint(hWnd, &PaintStruct);
        EndPaint(hWnd, &PaintStruct);

        return TRUE;
      }

      return DefWindowProc(hWnd, uMsg, wParam, lParam);
    
    break;

    case WM_DESTROY:

      PostQuitMessage(0);

    break;

    case WM_COMMAND:

      switch (LOWORD(wParam))
      {
        case IDM_CLEAR:

          hClearListViewThread = CreateThread(NULL, 0, ClearListViewThread, NULL, 0, &dwClearListViewThreadId);
          
          if (hClearListViewThread)
            CloseHandle(hClearListViewThread);

        break;

        case IDM_HELP:

          if (dwHelpCookie)
            HtmlHelp(hWnd, "APISpy32.chm", HH_DISPLAY_TOPIC, NULL);
          else
          {
            Result = (DWORD)ShellExecute(hWnd, "open", "APISpy32.htm", NULL, NULL, SW_SHOWNORMAL);

            if (Result <= 32)
              ErrorHandler(WARNING_SHELLEXECUTE, false);
          }

        break;

        case IDM_AUTOSCROLL:

          AutoScroll = !AutoScroll;
          CheckMenuItem(GetMenu(hWnd), IDM_AUTOSCROLL, MF_BYCOMMAND | (AutoScroll ? MF_CHECKED:MF_UNCHECKED));
          SendMessage(hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, (AutoScroll ? 4:3));
          InvalidateRect(hWndToolbar, NULL, TRUE);
          
        break;

        case IDM_ONTOP:

          WindowOnTop = !WindowOnTop;

          if (WindowOnTop)
            SetWindowPos(hWnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
          else
            SetWindowPos(hWnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);

          CheckMenuItem(GetMenu(hWnd), IDM_ONTOP, MF_BYCOMMAND | (WindowOnTop ? MF_CHECKED:MF_UNCHECKED));

        break;

        case IDM_CAPTURE:

          CaptureEvents = !CaptureEvents;
          CheckMenuItem(GetMenu(hWnd), IDM_CAPTURE, MF_BYCOMMAND | (CaptureEvents ? MF_CHECKED:MF_UNCHECKED));
          SendMessage(hWndToolbar, TB_CHANGEBITMAP, IDM_CAPTURE, (CaptureEvents ? 2:1));
          InvalidateRect(hWndToolbar, NULL, TRUE);

          // Notify the DLL about the change

          pSetCaptureEventsFlag(CaptureEvents);

        break;

        case IDM_SAVE:

          hSaveLogFileThread = CreateThread(NULL, 0, SaveLogFileThread, FALSE, 0, &dwSaveLogFileThreadId);

          if (hSaveLogFileThread)
            CloseHandle(hSaveLogFileThread);
          
        break;

        case IDM_SAVEAS:

          hSaveLogFileThread = CreateThread(NULL, 0, SaveLogFileThread, (PVOID)TRUE, 0, &dwSaveLogFileThreadId);

          if (hSaveLogFileThread)
            CloseHandle(hSaveLogFileThread);

        break;

        case IDM_ABOUT:

          DialogBox(hInst, "AboutBox", hWnd, (DLGPROC)AboutBoxWndProc);

        break;

        case IDM_EXIT:

          SendMessage(hWnd, WM_CLOSE, 0, 0);

        break;

        default:

          return DefWindowProc(hWnd, uMsg, wParam, lParam);

        break;
      }

    break;

    default:

      return DefWindowProc(hWnd, uMsg, wParam, lParam);
  }

  return FALSE;
}


int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
  bool Result;
  HWND hWnd;
  HWND hWndPrevInst;
  HACCEL hAccel;
  MSG msg;

  // Is this a second instance ?

  hWndPrevInst = FindWindow("APISpy32Class", "APISpy32 - http://www.internals.com");

  // If so, activate the previous one and terminate

  if (hWndPrevInst)
  {
    OpenIcon(hWndPrevInst);
    SetForegroundWindow(hWndPrevInst);
    return 0;
  }

  Result = GetWindowsVersion();

  if (Result == false)
    ErrorHandler(ERROR_GETWINDOWSVERSION, true);

  InitApplication(hInstance);

  hWnd = InitInstance(hInstance, nCmdShow); 

  if (hWnd == NULL)
    ErrorHandler(ERROR_INITINSTANCE, true);

  hAccel = LoadAccelerators(hInstance, "ACCELERATORS");

  if (hAccel == NULL)
    ErrorHandler(WARNING_ACCELERATORS, false);

  while (GetMessage(&msg, NULL, 0, 0))
  {
    if (!TranslateAccelerator(hWnd, hAccel, &msg))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }

  return msg.wParam;
}
