/******************************************************************************
*
*       FileMon - File System Monitor for Windows NT/9x
*		
*		Copyright (c) 1996-1998 Mark Russinovich and Bryce Cogswell
*
*		See readme.txt for terms and conditions.
*
*    	PROGRAM: FileMon.c
*
*    	PURPOSE: Communicates with the FileMon driver to display 
*		file system activity information.
*
******************************************************************************/
#include <windows.h>    // includes basic windows functionality
#include <windowsx.h>
#include <tchar.h>
#include <commctrl.h>   // includes the common control header
#include <stdio.h>
#include <string.h>
#include <winioctl.h>
#include "resource.h"
#include "ioctlcmd.h"

#include "instdrv.h"

// this typedef, present in newer system include files,
// supports the building filemon on older systems
typedef struct 
{
    DWORD cbSize;
    DWORD dwMajorVersion;
    DWORD dwMinorVersion;
    DWORD dwBuildNumber;
    DWORD dwPlatformID;
} DLLVERSIONINFO_, *PDLLVERSIONINFO_;

HRESULT (CALLBACK *pDllGetVersionProc)( PDLLVERSIONINFO_ pdvi );

// Application name for message boxes
#define APPNAME			_T("Filemon")

// Delay for popping listview subitem tooltips
#define BALLOONDELAY    10

// toolbar height plus the borders
#define TOOLBARHEIGHT	28

// Number of columns in the listview
#define NUMCOLUMNS	7

// Variables/definitions for the driver that performs the actual monitoring.
#define				SYS_FILE		_T("FILEMON.SYS")
#define				SYS_NAME		_T("FILEMON")

#define				VXD_FILE		"\\\\.\\FILEVXD.VXD"
#define				VXD_NAME		"FILEVXD"

static HANDLE		SysHandle		= INVALID_HANDLE_VALUE;

// Drive type names
#define DRVUNKNOWN		0
#define DRVFIXED		1
#define DRVREMOTE		2
#define DRVRAM			3
#define DRVCD			4
#define DRVREMOVE		5
TCHAR DrvNames[][32] = {
	_T("UNKNOWN"),
	_T("FIXED"),
	_T("REMOTE"),
	_T("RAM"),
	_T("CD"),
	_T("REMOVEABLE"),
};	

// must be > 26 (max drive letters)
#define PIPE_DRIVE		64
#define SLOT_DRIVE		65

#define POSVERSION 400

// Position settings data structure 
typedef struct {
	int		posversion;
	int		left;
	int		top;
	int		width;
	int		height;
	DWORD	column[NUMCOLUMNS];
	DWORD	curdriveset;
	DWORD	historydepth;
	BOOLEAN maximized;
	BOOLEAN timeduration;
	FILTER  filter;
	BOOLEAN	ontop;
	BOOLEAN hookpipes;
	BOOLEAN	hookslots;
} POSITION_SETTINGS;

// typedef for balloon popup
typedef struct {
	TCHAR	itemText[1024];
	RECT	itemPosition;
} ITEM_CLICK, *PITEM_CLICK;

// drives that are hooked
DWORD	CurDriveSet;

// The variable that holds the position settings
POSITION_SETTINGS	PositionInfo;

// toolbar constants
#define ID_TOOLBAR         1

// defined for comtl32.dll version 4.7
#define TOOLBAR_FLAT		0x800

// button definitions

// for installations that support flat style
TBBUTTON tbButtons[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 6, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 9, IDM_TIME, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0 },	
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 5, IDM_FILTER, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
	{ 7, IDM_FIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 8, 0, 0, TBSTYLE_BUTTON, 0L, 0},
};
#define NUMBUTTONS		13

// for older installations
TBBUTTON tbButtonsOld[] = {
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 0, IDM_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 2, IDM_CAPTURE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 4, IDM_AUTOSCROLL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 6, IDM_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},	
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 9, IDM_TIME, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0 },	
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 5, IDM_FILTER, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
	{ 0, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0L, 0},
	{ 7, IDM_FIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0L, 0},
};

#define NUMBUTTONSOLD	12


// Buffer into which driver can copy statistics
char				Stats[ MAX_STORE ];
// Current fraction of buffer filled
DWORD				StatsLen;

// Search string
TCHAR				FindString[256];
FINDREPLACE			FindTextInfo;
DWORD				FindFlags = FR_DOWN;
BOOLEAN				PrevMatch;
TCHAR				PrevMatchString[256];				

// Application instance handle
HINSTANCE			hInst;

// Are we running on NT or 9x?
BOOLEAN				IsNT;

// Misc globals
HWND				hWndMain;
HWND				hWndFind = NULL;
UINT				findMessageID;
HWND				hWndList;
WNDPROC 			ListViewWinMain;
HWND				hBalloon = NULL;
BOOLEAN				Capture = TRUE;
BOOLEAN				Autoscroll = TRUE;
BOOLEAN				Deleting = TRUE;
BOOLEAN				OnTop = FALSE;
BOOLEAN				HookPipes = FALSE;
BOOLEAN				HookSlots = FALSE;

// listview size limiting
DWORD				MaxLines = 0;
DWORD				LastRow = 0;

// is time absolute or duration?
BOOLEAN				TimeIsDuration;

// Filter-related
FILTER				FilterDefinition;

// For info saving
TCHAR				szFileName[256];
BOOLEAN				FileChosen = FALSE;

// General buffer for storing temporary strings
static TCHAR		msgbuf[ 257 ];

// General cursor manipulation
HCURSOR 			hSaveCursor;
HCURSOR 			hHourGlass;

// performance counter frequency
LARGE_INTEGER		PerfFrequency;


/******************************************************************************
*
*	FUNCTION:	Abort:
*
*	PURPOSE:	Handles emergency exit conditions.
*
*****************************************************************************/
DWORD Abort( HWND hWnd, TCHAR * Msg )
{
	LPVOID	lpMsgBuf;
	TCHAR	errmsg[256];
	DWORD	error = GetLastError();
 
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
					NULL, GetLastError(), 
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
					(LPTSTR) &lpMsgBuf, 0, NULL );
	if( IsNT ) UnloadDeviceDriver( SYS_NAME );
	_stprintf(errmsg, _T("%s: %s"), Msg, lpMsgBuf );
	if( error == ERROR_INVALID_HANDLE && IsNT ) 
		_stprintf(errmsg, _T("%s\nMake sure that you are an administrator and that ")
			_T("Filemon is located on a local drive."), errmsg  );
	MessageBox( hWnd, errmsg, _T("Filemon"), MB_OK|MB_ICONERROR );
	PostQuitMessage( 1 );
	LocalFree( lpMsgBuf );
	return -1;
}

/******************************************************************************
*
*	FUNCTION:	BalloonDialog
*
*	PURPOSE:	Dialog function for home-brewed balloon help.
*
******************************************************************************/
LONG APIENTRY BalloonDialog( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	static ITEM_CLICK	ctx;
	static RECT			rect;
	static HFONT		hfont;
	LPCREATESTRUCT		lpcs;
	HDC					hdc;
	NONCLIENTMETRICS	ncm;
	POINTS				pts;
	POINT				pt;


	switch (message) {
		case WM_CREATE:

			lpcs = (void *)lParam;
			ctx = *(PITEM_CLICK) lpcs->lpCreateParams;
			hdc = GetDC( hDlg );

			// Get font
			if ( hfont == NULL )  {
				memset(&ncm, 0, sizeof(ncm));
				ncm.cbSize = sizeof(ncm);
				SystemParametersInfo( SPI_GETNONCLIENTMETRICS, 
										0, &ncm, 0);
 				hfont = CreateFontIndirect( &ncm.lfMessageFont ); 
			} 

			// is the app the focus?
			if( !GetFocus()) return -1;

			// Compute size of required rectangle
			rect.left	= 0;
			rect.top	= 0;
			rect.right	= lpcs->cx;
			rect.bottom	= lpcs->cy;
			SelectObject( hdc, hfont );
			DrawText( hdc, ctx.itemText, -1, &rect, 
						DT_NOCLIP|DT_LEFT|DT_NOPREFIX|DT_CALCRECT );

			// if the bounding rectangle of the subitem is big enough to display
			// the text then don't pop the balloon
			if( ctx.itemPosition.right > rect.right + 11 ) {

				return -1;
			}

			// Move and resize window
			if( ctx.itemPosition.left - 5 + rect.right + 10 >
				 GetSystemMetrics(SM_CXFULLSCREEN) ) {

				 ctx.itemPosition.left = GetSystemMetrics(SM_CXFULLSCREEN) -
							(rect.right+10);
			}
			MoveWindow( hDlg, 
						ctx.itemPosition.left+3, ctx.itemPosition.top, 
						rect.right + 6, 
						rect.bottom + 1,
						TRUE );

			// Adjust rectangle so text is centered
			rect.left	+= 2;
			rect.right	+= 2;
			rect.top	-= 1; 
			rect.bottom	+= 0;

			// make it so this window doesn't get the focus
			ShowWindow( hDlg, SW_SHOWNOACTIVATE );
			break;

		case WM_LBUTTONDBLCLK:
		case WM_RBUTTONDBLCLK:
		case WM_MBUTTONDBLCLK:
		case WM_LBUTTONDOWN:
		case WM_RBUTTONDOWN:
		case WM_MBUTTONDOWN:

			pts = MAKEPOINTS( lParam );
			pt.x = (LONG) pts.x;
			pt.y = (LONG) pts.y;
			ClientToScreen( hDlg, &pt );

			DestroyWindow( hDlg );

			// pass this through to the listview
			if( ScreenToClient( hWndList, &pt )) {

				PostMessage( hWndList, message, wParam, (SHORT) pt.y<<16 | (SHORT) pt.x );
			}
			break;

		case WM_LBUTTONUP:
		case WM_RBUTTONUP:
		case WM_MBUTTONUP:

			DestroyWindow( hDlg );
			break;

		case WM_PAINT:
			hdc = GetDC( hDlg );

			// Set colors
			SetTextColor( hdc, 0x00000000 );
			SetBkMode( hdc, TRANSPARENT );
			SelectObject( hdc, hfont );
			DrawText( hdc, ctx.itemText, -1, &rect, 
						DT_NOCLIP|DT_LEFT|DT_NOPREFIX|DT_WORDBREAK );
			break;

		case WM_DESTROY:
			hBalloon = NULL;
			break;

		case WM_CLOSE:	
			DestroyWindow( hDlg );
			break;
	}

    return DefWindowProc( hDlg, message, wParam, lParam );
}


/******************************************************************************
*
*	FUNCTION:	FindInListview:
*
*	PURPOSE:	Searches for a string in the listview. Note: its okay if
*				items are being added to the list view or the list view
*				is cleared while this search is in progress - the effect
*				is harmless.
*
*****************************************************************************/
BOOLEAN FindInListview(HWND hWnd, LPFINDREPLACE FindInfo )
{
	int		currentItem;
	DWORD	i;
	int		subitem, numItems;
	TCHAR	fieldtext[256];
	BOOLEAN match = FALSE;
	TCHAR	errmsg[256];
	BOOLEAN	goUp;

	// get the search direction
	goUp = ((FindInfo->Flags & FR_DOWN) == FR_DOWN);

	// initialize stuff
	if( !(numItems = ListView_GetItemCount( hWndList ))) {

		MessageBox( hWnd, _T("No items to search"), _T("Filemon"), 
			MB_OK|MB_ICONWARNING );
		SetForegroundWindow( hWndFind );
		return FALSE;
	}

	// find the item with the focus
	currentItem = ListView_GetNextItem( hWndList, 0, LVNI_SELECTED );

	// if no current item, start at the top or the bottom
	if( currentItem == -1 ) {
		if( goUp )
			currentItem = 0;
		else {
			if( PrevMatch ) {
				_stprintf(errmsg, _T("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
				MessageBox( hWnd, errmsg, _T("Filemon"), MB_OK|MB_ICONWARNING );
				SetForegroundWindow( hWndFind );
				return FALSE;
			}
			currentItem = numItems;
		}
	}

	// if we're continuing a search, start with the next item
	if( PrevMatch && !_tcscmp( FindString, PrevMatchString ) ) {
		if( goUp ) currentItem++;
		else currentItem--;

		if( (!goUp && currentItem < 0) ||
			(goUp && currentItem >= numItems )) {

			_stprintf(errmsg, _T("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
			MessageBox( hWnd, errmsg, APPNAME, MB_OK|MB_ICONWARNING );
			SetForegroundWindow( hWndFind );
			return FALSE;
		}
	}

	// loop through each item looking for the string
	while( 1 ) {

		// get the item text
		for( subitem = 0; subitem < NUMCOLUMNS; subitem++ ) {
			fieldtext[0] = 0;
			ListView_GetItemText( hWndList, currentItem, subitem, fieldtext, 256 );

			// make sure enought string for a match
			if( _tcslen( fieldtext ) < _tcslen( FindInfo->lpstrFindWhat ))
				continue;

			// do a scan all the way through for the substring
			if( FindInfo->Flags & FR_WHOLEWORD ) {

				i = 0;
				while( fieldtext[i] ) {
					while( fieldtext[i] && fieldtext[i] != ' ' ) i++;
					if( FindInfo->Flags & FR_MATCHCASE ) 
						match = !_tcscmp( fieldtext, FindInfo->lpstrFindWhat );
					else
						match = !_tcsicmp( fieldtext, FindInfo->lpstrFindWhat );
					if( match) break;
					i++;
				}	
			} else {
				for( i = 0; i < _tcslen( fieldtext ) - _tcslen(FindInfo->lpstrFindWhat)+1; i++ ) {
					if( FindInfo->Flags & FR_MATCHCASE ) 
						match = !_tcsncmp( &fieldtext[i], FindInfo->lpstrFindWhat, 
											_tcslen(FindInfo->lpstrFindWhat) );
					else
						match = !_tcsnicmp( &fieldtext[i], FindInfo->lpstrFindWhat,
											_tcslen(FindInfo->lpstrFindWhat) );
					if( match ) break;
				}		
			}

			if( match ) {

				_tcscpy( PrevMatchString, FindInfo->lpstrFindWhat );
				PrevMatch = TRUE;
				ListView_SetItemState( hWndList, currentItem, 
							LVIS_SELECTED|LVIS_FOCUSED,
							LVIS_SELECTED|LVIS_FOCUSED );
				ListView_EnsureVisible( hWndList, currentItem, FALSE ); 
				SetFocus( hWndList );
				return TRUE;
			}
		}
		currentItem = currentItem + (goUp ? 1:-1);
		if( !currentItem || currentItem == numItems+1 ) {
			// end of the road
			break;
		}
	}
	_stprintf(errmsg, _T("Cannot find string \"%s\""), FindInfo->lpstrFindWhat );
	MessageBox( hWnd, errmsg, APPNAME, MB_OK|MB_ICONWARNING );
	SetForegroundWindow( hWndFind );
	return FALSE;
}


/******************************************************************************
*
*	FUNCTION:	PopFindDialog:
*
*	PURPOSE:	Calls the find message dialog box.
*
*****************************************************************************/
void PopFindDialog(HWND hWnd)
{
	_tcscpy( FindString, PrevMatchString );
    FindTextInfo.lStructSize = sizeof( FindTextInfo );
    FindTextInfo.hwndOwner = hWnd;
    FindTextInfo.hInstance = (HANDLE)hInst;
    FindTextInfo.lpstrFindWhat = FindString;
    FindTextInfo.lpstrReplaceWith = NULL;
    FindTextInfo.wFindWhatLen = sizeof(FindString);
    FindTextInfo.wReplaceWithLen = 0;
    FindTextInfo.lCustData = 0;
    FindTextInfo.Flags =  FindFlags;
    FindTextInfo.lpfnHook = (LPFRHOOKPROC)(FARPROC)NULL;
    FindTextInfo.lpTemplateName = NULL;

    if ((hWndFind = FindText(&FindTextInfo)) == NULL)
		MessageBox( hWnd, _T("Unable to create Find dialog"), APPNAME, MB_OK|MB_ICONERROR );      
}

/****************************************************************************
*
*	FUNCTION:	FilterProc
*
*	PURPOSE:	Processes messages for "Filter" dialog box
*
****************************************************************************/
BOOL APIENTRY FilterProc( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	int				nb;
	DWORD			newMaxLines;
	FILTER			upcaseFilter;
	char			history[64];

	switch ( message )  {
	case WM_INITDIALOG:

		// initialize the controls to reflect the current filter
		SetDlgItemTextA( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter );
		SetDlgItemTextA( hDlg, IDC_PROCEXCLUDE, FilterDefinition.excludeprocess );
		SetDlgItemTextA( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter );
		SetDlgItemTextA( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter );
		CheckDlgButton( hDlg, IDC_FILTERREADS, FilterDefinition.logreads );
		CheckDlgButton( hDlg, IDC_FILTERWRITES,FilterDefinition.logwrites );
		sprintf( history, "%d", MaxLines );
		SetDlgItemTextA( hDlg, IDC_HISTORY, history );
		return TRUE;

	case WM_COMMAND:              
		if ( LOWORD( wParam ) == IDOK )	 {

			// make sure that max lines is legal
			GetDlgItemTextA( hDlg, IDC_HISTORY, history, 64 );
			if( !sscanf( history, "%d", &newMaxLines )) {

				MessageBox(	NULL, _T("Invalid History Depth."),
						_T("Filter Error"), MB_OK|MB_ICONWARNING );
				return TRUE;
			} 
			MaxLines = newMaxLines;

			// read the values that were set
			GetDlgItemTextA( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter, MAXFILTERLEN );
			GetDlgItemTextA( hDlg, IDC_PROCEXCLUDE, FilterDefinition.excludeprocess, MAXFILTERLEN );
			GetDlgItemTextA( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter, MAXFILTERLEN );
			GetDlgItemTextA( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter, MAXFILTERLEN );
			FilterDefinition.logreads  = IsDlgButtonChecked( hDlg, IDC_FILTERREADS );
			FilterDefinition.logwrites = IsDlgButtonChecked( hDlg, IDC_FILTERWRITES );

			// make an upcase version for the driver
			upcaseFilter = FilterDefinition;
			_strupr(upcaseFilter.processfilter);
			_strupr(upcaseFilter.excludeprocess);
			_strupr(upcaseFilter.pathfilter);
			_strupr(upcaseFilter.excludefilter);
 
			// tell the driver the new filter
			if ( ! DeviceIoControl(	SysHandle, FILEMON_setfilter,
									&upcaseFilter, sizeof(FILTER), NULL, 
									0, &nb, NULL ) )
			{
				Abort( hDlg, _T("Couldn't access device driver") );
				return TRUE;
			}

			EndDialog( hDlg, TRUE );
			return TRUE;

		} else if( LOWORD( wParam ) == IDCANCEL ) {

			EndDialog( hDlg, TRUE );
		} else if( LOWORD( wParam ) == IDRESET ) {

			// reset filter to default of none
			sprintf( FilterDefinition.processfilter, "*" );
			sprintf( FilterDefinition.excludeprocess, "" );
			sprintf( FilterDefinition.pathfilter, "*" );
			sprintf( FilterDefinition.excludefilter, "");
			FilterDefinition.logreads = TRUE;
			FilterDefinition.logwrites = TRUE;
			MaxLines = 0;

			// initialize the controls to reflect the current filter
			SetDlgItemTextA( hDlg, IDC_PROCFILTER, FilterDefinition.processfilter );
			SetDlgItemTextA( hDlg, IDC_PROCEXCLUDE, FilterDefinition.excludeprocess );
			SetDlgItemTextA( hDlg, IDC_PATHFILTER, FilterDefinition.pathfilter );
			SetDlgItemTextA( hDlg, IDC_EXCLUDEFILTER, FilterDefinition.excludefilter );
			CheckDlgButton( hDlg, IDC_FILTERREADS, FilterDefinition.logreads );
			CheckDlgButton( hDlg, IDC_FILTERWRITES, FilterDefinition.logwrites );
			SetDlgItemTextA( hDlg, IDC_HISTORY, "0" );
		}
		break;

	case WM_CLOSE:
		EndDialog( hDlg, TRUE );
		return TRUE;
	}
	return FALSE;   
}

/******************************************************************************
*
*	FUNCTION:	Get_Position_Settings
*
*	PURPOSE:	Reads the Registry to get the last-set window position.
*
******************************************************************************/
VOID Get_Position_Settings()
{
	HKEY	hKey;
	DWORD	ParamSize;
	POSITION_SETTINGS	regPositionInfo;

	// Fist, set the default settings
	PositionInfo.top	= CW_USEDEFAULT;
	PositionInfo.left	= CW_USEDEFAULT;
	PositionInfo.width	= CW_USEDEFAULT;
	PositionInfo.height	= CW_USEDEFAULT;
	PositionInfo.maximized = FALSE;
	PositionInfo.ontop = FALSE;
	PositionInfo.hookpipes = FALSE;
	PositionInfo.hookslots = FALSE;

	// set the default listview widths
	PositionInfo.column[0] = 35;  // seq 
	PositionInfo.column[1] = 90;  // time
	PositionInfo.column[2] = 90;  // process
	PositionInfo.column[3] = 130; // irp
	PositionInfo.column[4] = 200; // path
	PositionInfo.column[5] = 70;  // result
	PositionInfo.column[6] = 150; // other

	// intialize the hooked drives
	PositionInfo.curdriveset = (DWORD) -1;

	// duration is default
	PositionInfo.timeduration = FALSE;

	// initialize history depth
	PositionInfo.historydepth = 0;

	// initialize filter
	strcpy( PositionInfo.filter.processfilter, "*" );
	strcpy( PositionInfo.filter.excludeprocess, "" );
	strcpy( PositionInfo.filter.pathfilter, "*" );
	strcpy( PositionInfo.filter.excludefilter, "" );
	PositionInfo.filter.logreads = TRUE;
	PositionInfo.filter.logwrites = TRUE;

	// first, get the last-entered params from the registry
	RegCreateKey(HKEY_CURRENT_USER, 
			_T("Software\\Systems Internals\\Filemon"),
			&hKey );

	// get the params and ignore errors
	ParamSize = sizeof( PositionInfo );
	regPositionInfo.posversion = 0;
	RegQueryValueEx( hKey,_T("Settings"), NULL, NULL, (LPBYTE) &regPositionInfo,
				&ParamSize );
	RegCloseKey( hKey );

	// only use the registry settings if the version matches
	if( regPositionInfo.posversion == POSVERSION ) PositionInfo = regPositionInfo;

	// extract global settings from the value returned from the Registry (or the default)
	CurDriveSet			= PositionInfo.curdriveset;
	MaxLines			= PositionInfo.historydepth;
	TimeIsDuration		= PositionInfo.timeduration;
	FilterDefinition    = PositionInfo.filter;
	OnTop				= PositionInfo.ontop;
	HookPipes			= PositionInfo.hookpipes;
	HookSlots			= PositionInfo.hookslots;
}


/******************************************************************************
*
*	FUNCTION:	Save_Position_Settings
*
*	PURPOSE:	Saves the current window settings to the Registry.
*
******************************************************************************/
VOID Save_Position_Settings( HWND hWnd )
{
	RECT		rc;
	int			i;
	HKEY		hKey;

	// set version #
	PositionInfo.posversion = POSVERSION;

	// get the position of the main window
	GetWindowRect( hWnd, &rc );
	if( !IsIconic( hWnd )) {

		PositionInfo.left = rc.left;
		PositionInfo.top = rc.top;
		PositionInfo.width = rc.right - rc.left;
		PositionInfo.height = rc.bottom - rc.top;
	} 
	PositionInfo.maximized = IsZoomed( hWnd );
	PositionInfo.ontop = OnTop;
	PositionInfo.hookpipes = HookPipes;
	PositionInfo.hookslots = HookSlots;

	// get the history depth
	PositionInfo.historydepth = MaxLines;

	// get time format
	PositionInfo.timeduration = TimeIsDuration;

	// get the widths of the listview columns
	for( i = 0; i < NUMCOLUMNS; i++ ) {
		PositionInfo.column[i] = ListView_GetColumnWidth( hWndList, i );
	}

	// save filters
	PositionInfo.filter = FilterDefinition;

	// get the current drive set
	PositionInfo.curdriveset = CurDriveSet;

	// save connection info to registry
	RegOpenKey(HKEY_CURRENT_USER, 
			_T("Software\\Systems Internals\\Filemon"),
			&hKey );
	RegSetValueEx( hKey, _T("Settings"), 0, REG_BINARY, (LPBYTE) &PositionInfo,
			sizeof( PositionInfo ) );
	RegCloseKey( hKey );
}


/******************************************************************************
*
*	FUNCTION:	Hook_Drives
*
*	PURPOSE:	Hook the currently selected drives, updating menu checks
*
******************************************************************************/
DWORD Hook_Drives( HMENU DriveMenu, DWORD MaxDriveSet, DWORD CurDriveSet ) 
{
	DWORD nb;
	DWORD drive;

	// Tell device driver which drives to monitor
	if ( ! DeviceIoControl(	SysHandle, FILEMON_setdrives,
							&CurDriveSet, sizeof CurDriveSet,
							&CurDriveSet, sizeof CurDriveSet,
							&nb, NULL ) )
		return 0;

	// Update menu items
	for ( drive = 0; drive < 32; ++drive )
		if ( MaxDriveSet & (1<<drive) )  {
			if ( CurDriveSet & (1<<drive) )
				CheckMenuItem( DriveMenu, IDC_DRIVE+drive, MF_BYCOMMAND|MF_CHECKED );
			else
				CheckMenuItem( DriveMenu, IDC_DRIVE+drive, MF_BYCOMMAND|MF_UNCHECKED );
		}
	return CurDriveSet;
}


/******************************************************************************
*
*	FUNCTION:	Split
*
*	PURPOSE:	Split a delimited line into components
*
******************************************************************************/
int Split( char * line, char delimiter, char * items[] )
{
	int		cnt = 0;

	for (;;)  {
		// Add prefix to list of components		
		items[cnt++] = line;

		// Check for more components
		line = strchr( line, delimiter );
		if ( line == NULL )
			return cnt;

		// Terminate previous component and move to next
		*line++ = '\0';
	}		
}


/******************************************************************************
*
*	FUNCTION:	ListAppend
*
*	PURPOSE:	Add a new line to List window
*
******************************************************************************/
BOOL List_Append( HWND hWndList, DWORD seq, LONGLONG time, char * line )
{
	LV_ITEM		lvI;	// list view item structure
	int			row;
	char	*	items[20];
	int			itemcnt = 0;
	FILETIME	localTime;
	SYSTEMTIME	systemTime;

	// Split line into columns
	itemcnt = Split( line, '\t', items );
	if ( itemcnt == 0 )
		return FALSE;

	// Determine row number for request
	if ( *items[0] )  {
		// Its a new request.  Put at end.
		row = 0x7FFFFFFF;
	} else {
		// Its a status.  Locate its associated request.
		lvI.mask = LVIF_PARAM;
		lvI.iSubItem = 0;
		for ( row = ListView_GetItemCount(hWndList) - 1; row >= 0; --row )  {
			lvI.iItem = row;
			if ( ListView_GetItem( hWndList, &lvI )  &&  (DWORD)lvI.lParam == seq )
				break;
		}
		if ( row == -1 )
			// No request associated with status.
			return FALSE;
	}

	// Sequence number if a new item
	if ( *items[0] )  {
		_stprintf( msgbuf, _T("%d"), seq );
		lvI.mask		= LVIF_TEXT | LVIF_PARAM;
		lvI.iItem		= row;
		lvI.iSubItem	= 0;
		lvI.pszText		= msgbuf;
		lvI.cchTextMax	= lstrlen( lvI.pszText ) + 1;
		lvI.lParam		= seq;
		row = ListView_InsertItem( hWndList, &lvI );
		if ( row == -1 )  {
			_stprintf( msgbuf, _T("Error adding item %d to list view"), seq );
			MessageBox( hWndList, msgbuf, APPNAME, MB_OK|MB_ICONERROR );
			return FALSE;
		}
        LastRow = row;
	}

	if( IsNT ) {

		// Time - note: in order to save space in the log records
		// I convert all times read to date/times and, based on whether
		// the year is reasonable, chose to interpret the time as 
		// a timestamp or a duration.
		FileTimeToLocalFileTime( (PFILETIME) &time, &localTime );
		FileTimeToSystemTime( &localTime, &systemTime );

		if( systemTime.wYear < 1998 || systemTime.wYear > 2100 ) {

			_stprintf( msgbuf, _T("%10.8f"), (((float) time) /
									  ((float) PerfFrequency.QuadPart)) );
		} else {

			GetTimeFormat( LOCALE_USER_DEFAULT, 0,
							&systemTime, NULL, msgbuf, 64 );
		}
	} else {

		if( TimeIsDuration ) {

			_stprintf( msgbuf, _T("%10.8f"), (float) time * 0.8 / 1e6 );

		} else {

			DosDateTimeToFileTime( (WORD) (time >> 16), (WORD) time, &localTime );
			FileTimeToSystemTime( &localTime, &systemTime );
			GetTimeFormat( LOCALE_USER_DEFAULT, 0,
							&systemTime, NULL, msgbuf, 64 );
		}
	}
	ListView_SetItemText( hWndList, row, 1, msgbuf );

	// Process name
	if ( itemcnt>0 && *items[0] ) {
		OemToChar( items[0], msgbuf );
		ListView_SetItemText( hWndList, row, 2, msgbuf );
	}

	// Request type
	if ( itemcnt>1 && *items[1] )  {
		OemToChar( items[1], msgbuf );
		ListView_SetItemText( hWndList, row, 3, msgbuf );
	}

	// Path
	if ( itemcnt>2 && *items[2] )  {
		OemToChar( items[2], msgbuf );
		ListView_SetItemText( hWndList, row, 4, msgbuf );
	}

	// Result
	if ( itemcnt>4 && *items[4] )  {
		OemToChar( items[4], msgbuf );
		ListView_SetItemText( hWndList, row, 5, msgbuf );
	}

	// Additional
	if ( itemcnt>3 && *items[3] )  {
		OemToChar( items[3], msgbuf );
		ListView_SetItemText( hWndList, row, 6, msgbuf );
	}

	return TRUE;
}


/******************************************************************************
*
*	FUNCTION:	UpdateStatistics
*
*	PURPOSE:	Clear the statistics window and refill it with the current 
*				contents of the statistics buffer.  Does not refresh the 
*				buffer from the device driver.
*
******************************************************************************/
void UpdateStatistics( HWND hWnd, HWND hWndList, BOOL Clear )
{
	PENTRY	ptr;
	BOOLEAN itemsAdded = FALSE;
	int		totitems, i;

	// Just return if nothing to do
	if ( !Clear  &&  StatsLen < sizeof(int)+2 )
		return;

	// post hourglass icon
	if( !IsNT ) {
		
		hSaveCursor = SetCursor(hHourGlass);
		SendMessage(hWndList, WM_SETREDRAW, FALSE, 0);
	}

	// Start with empty list
	if ( Clear ) {

		if( IsNT ) {

			ListView_DeleteAllItems( hWndList );
		} else {

			totitems = ListView_GetItemCount( hWndList );
			Deleting = TRUE;
			for(i = 0; i < totitems; i++)
				ListView_DeleteItem( hWndList, 0 );
			Deleting = FALSE;
		}
		LastRow = 0;
	}

	// Add all List items from Stats[] data
	for ( ptr = (void *)Stats; (char *)ptr < min(Stats+StatsLen,Stats + sizeof (Stats)); )  {
	 	// Add to list
		ULONG len = strlen(ptr->text);
        
		itemsAdded |= List_Append( hWndList, ptr->seq, ptr->time.QuadPart, ptr->text );

		if( IsNT ) {
			
			len += 4; len &= 0xFFFFFFFC; // +1 for null-terminator +3 for 32bit alignment
			ptr = (void *)(ptr->text + len);
		} else 
			ptr = (void *)(ptr->text + len + 1);
	}

	// Empty the buffer
	StatsLen = 0;

	// only do stuff if we added stuff
	if( itemsAdded ) {

		// limit number of lines saved
		if (MaxLines) {
			SendMessage(hWndList, WM_SETREDRAW, FALSE, 0);
			while ( LastRow > MaxLines ) {
				ListView_DeleteItem ( hWndList, 0 );
				LastRow--;
			}
			SendMessage(hWndList, WM_SETREDRAW, TRUE, 0);
		}

		// Scroll so newly added items are visible
		if ( Autoscroll ) {
			if( hBalloon ) DestroyWindow( hBalloon );
			ListView_EnsureVisible( hWndList, ListView_GetItemCount(hWndList)-1, FALSE ); 
		}
	}

	if( !IsNT) {
		SendMessage(hWndList, WM_SETREDRAW, TRUE, 0);
		InvalidateRect( hWndList, NULL, FALSE );
		SetCursor( hSaveCursor );
	}
}

/****************************************************************************
* 
*    FUNCTION: ListViewSubclass(HWND,UINT,WPARAM)
*
*    PURPOSE:  Subclasses the listview so that we can do tooltips
*
****************************************************************************/
LRESULT CALLBACK ListViewSubclass(HWND hWnd, UINT uMsg, WPARAM wParam,
        LPARAM lParam)
{
	ITEM_CLICK		itemClick;
	LVHITTESTINFO	hitItem;
	static initTrack = FALSE;
	POINT           hitPoint, topleftPoint, bottomrightPoint;
	RECT			listRect;
	static POINTS  mousePosition;

	if( !initTrack ) {

		SetTimer( hWnd,	2, BALLOONDELAY, NULL );
		initTrack = TRUE;
	}

    switch (uMsg) {

	case WM_LBUTTONDBLCLK:
	case WM_RBUTTONDBLCLK:
	case WM_MBUTTONDBLCLK:
	case WM_LBUTTONUP:
	case WM_RBUTTONUP:
	case WM_MBUTTONUP:
	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_MOUSEMOVE:

		// delete any existing balloon
		if( hBalloon ) DestroyWindow( hBalloon );

		// save mouse position and reset the timer
		mousePosition = MAKEPOINTS( lParam );
		SetTimer( hWnd,	2, BALLOONDELAY, NULL );
		break;

	case WM_VSCROLL:
	case WM_HSCROLL:
		if( hBalloon ) DestroyWindow( hBalloon );
		break;

	case WM_RBUTTONDOWN:
		mousePosition = MAKEPOINTS( lParam );
		SetTimer( hWnd,	2, BALLOONDELAY, NULL );
		// fall-through

	case WM_TIMER:

		// are we currently in the listview?
		GetCursorPos( &hitPoint );
		GetClientRect( hWnd, &listRect );
		topleftPoint.x = listRect.left;
		topleftPoint.y = listRect.top;
		ClientToScreen( hWnd, &topleftPoint );
		bottomrightPoint.x = listRect.right;
		bottomrightPoint.y = listRect.bottom;
		ClientToScreen( hWnd, &bottomrightPoint );
		if( hitPoint.x < topleftPoint.x ||
			hitPoint.x > bottomrightPoint.x ||
			hitPoint.y < topleftPoint.y ||
			hitPoint.y > bottomrightPoint.y ||
			!GetFocus() || (hWndFind && GetFocus() != hWndList) ) {

			// delete any existing balloon
			if( hBalloon ) DestroyWindow( hBalloon );
			break;
		}

		hitItem.pt.x = mousePosition.x;
		hitItem.pt.y =  mousePosition.y;
		if(	ListView_SubItemHitTest( hWndList, &hitItem ) != -1 ) {

			itemClick.itemText[0] = 0;
			ListView_GetItemText( hWndList, hitItem.iItem,
					hitItem.iSubItem, itemClick.itemText, 1024 );

			// delete any existing balloon
			if( hBalloon ) DestroyWindow( hBalloon );

			if( _tcslen( itemClick.itemText ) ) {

				if( hitItem.iSubItem ) {

					ListView_GetSubItemRect( hWndList, hitItem.iItem, hitItem.iSubItem,
							LVIR_BOUNDS, &itemClick.itemPosition);

					itemClick.itemPosition.bottom -= itemClick.itemPosition.top;
					itemClick.itemPosition.right  -= itemClick.itemPosition.left;

				} else {

					ListView_GetSubItemRect( hWndList, hitItem.iItem, 1,
							LVIR_BOUNDS, &itemClick.itemPosition);

					itemClick.itemPosition.bottom -= itemClick.itemPosition.top;
					itemClick.itemPosition.right  = itemClick.itemPosition.left;
					itemClick.itemPosition.left   = 0;
				}

				hitPoint.y = itemClick.itemPosition.top;
				hitPoint.x = itemClick.itemPosition.left;

				ClientToScreen( hWnd, &hitPoint );

				itemClick.itemPosition.left = hitPoint.x;
				itemClick.itemPosition.top  = hitPoint.y;

				// pop-up a balloon (tool-tip like window)
				hBalloon = CreateWindowEx( 0, _T("BALLOON"), 
								_T("balloon"), 
								WS_POPUP|WS_BORDER|WS_EX_TOOLWINDOW,
								100, 100,
								200, 200,
								hWndMain, NULL, hInst, 
								&itemClick );
				if( hBalloon) SetFocus( hWnd );
			}
		}
		break;
    }

	// pass-through to real listview handler
    return CallWindowProc(ListViewWinMain, hWnd, uMsg, wParam, 
            lParam);
}

/****************************************************************************
* 
*    FUNCTION: CreateListView(HWND)
*
*    PURPOSE:  Creates the statistics list view window and initializes it
*
****************************************************************************/
HWND CreateList( HWND hWndParent )                                     
{
	HWND		hWndList;    	  	// handle to list view window
	RECT		rc;         	  	// rectangle for setting size of window
	LV_COLUMN	lvC;				// list view column structure
	DWORD		j;
	static struct {
		TCHAR *	Label;	// title of column
		DWORD	Width;	// width of column in pixels
		DWORD	Fmt;
	} column[] = {
		{	_T("#"),			35		},
		{   _T("Time"),       50      },
		{	_T("Process"),	90		},
		{	_T("Request"),	130		},
		{	_T("Path"),		200		},
		{	_T("Result"),		70		},
		{	_T("Other"),		150		},
	};

	// Ensure that the common control DLL is loaded.
	InitCommonControls();

	// Set the column widths according to the user-settings
	for( j = 0; j < NUMCOLUMNS; j++ ) {
		column[j].Width = PositionInfo.column[j];
	}

	// Get the size and position of the parent window.
	GetClientRect( hWndParent, &rc );

	// Create the list view window
	hWndList = CreateWindowEx( 0L, WC_LISTVIEW, _T(""), 
								WS_VISIBLE | WS_CHILD | WS_BORDER | LVS_REPORT | 
								   LVS_SINGLESEL | WS_EX_CLIENTEDGE,	// styles
								0, TOOLBARHEIGHT, rc.right - rc.left, rc.bottom - rc.top - TOOLBARHEIGHT,
								hWndParent,	(HMENU)ID_LIST, hInst, NULL );
	if ( hWndList == NULL )
		return NULL;

	// Initialize columns
	lvC.mask	= LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
	lvC.fmt		= LVCFMT_LEFT;	// left-align column

	// Add the columns.
	for ( j = 0; j < sizeof column/sizeof column[0]; ++j )  {
		lvC.iSubItem	= j;
		lvC.cx			= column[j].Width;
	 	lvC.pszText		= column[j].Label;
		if ( ListView_InsertColumn( hWndList, j, &lvC ) == -1 )
			return NULL;
	}

	// set full-row selection
	SendMessage( hWndList, LVM_SETEXTENDEDLISTVIEWSTYLE,
			LVS_EX_FULLROWSELECT, LVS_EX_FULLROWSELECT );

	// sub-class
	ListViewWinMain = (WNDPROC) SetWindowLong(hWndList, 
                                              GWL_WNDPROC, 
                                              (DWORD)ListViewSubclass); 
	return hWndList;
}


/****************************************************************************
* 
*    FUNCTION: SaveFile()
*
*    PURPOSE:  Lets the user go select a file.
*
****************************************************************************/
void SaveFile( HWND hWnd, HWND ListBox, BOOLEAN SaveAs )
{
	OPENFILENAME	SaveFileName;
	TCHAR			szFile[256] = _T(""), fieldtext[256], output[1024];
	FILE			*hFile;
	int				numitems;
	int				row, subitem;

	if( SaveAs || !FileChosen ) {
		SaveFileName.lStructSize       = sizeof (SaveFileName);
		SaveFileName.hwndOwner         = hWnd;
		SaveFileName.hInstance         = (HANDLE) hInst;
		SaveFileName.lpstrFilter       = _T("File Info (*.FIL)\0*.FIL\0All (*.*)\0*.*\0");
		SaveFileName.lpstrCustomFilter = (LPTSTR)NULL;
		SaveFileName.nMaxCustFilter    = 0L;
		SaveFileName.nFilterIndex      = 1L;
		SaveFileName.lpstrFile         = szFile;
		SaveFileName.nMaxFile          = 256;
		SaveFileName.lpstrFileTitle    = NULL;
		SaveFileName.nMaxFileTitle     = 0;
		SaveFileName.lpstrInitialDir   = NULL;
		SaveFileName.lpstrTitle        = _T("Save File Info...");
		SaveFileName.nFileOffset       = 0;
		SaveFileName.nFileExtension    = 0;
		SaveFileName.lpstrDefExt       = _T("*.fil");
		SaveFileName.lpfnHook		   = NULL;
 		SaveFileName.Flags = OFN_LONGNAMES|OFN_HIDEREADONLY;

		if( !GetSaveFileName( &SaveFileName )) 
			return;
	} else 
		// open previous szFile
		_tcscpy( szFile, szFileName );

	// open the file
	hFile = _tfopen( szFile, _T("w") );
	if( !hFile ) {
		MessageBox(	NULL, _T("Create File Failed."),
				_T("Save Error"), MB_OK|MB_ICONSTOP );
		return;
	}

	// post hourglass icon
	SetCapture(hWnd);
	hSaveCursor = SetCursor(hHourGlass);

	numitems = ListView_GetItemCount(ListBox);
	for ( row = 0; row < numitems; row++ )  {
		output[0] = 0;
		for( subitem = 0; subitem < NUMCOLUMNS; subitem++ ) {
			fieldtext[0] = 0;
			ListView_GetItemText( ListBox, row, subitem, fieldtext, 256 );
			_tcscat( output, fieldtext );
			_tcscat( output, _T("\t") );
		}
		_ftprintf( hFile, _T("%s\n"), output );
	}
	fclose( hFile );
	_tcscpy( szFileName, szFile );
	FileChosen = TRUE;
	SetCursor( hSaveCursor );
	ReleaseCapture(); 
}


/****************************************************************************
*
*	FUNCTION:	About
*
*	PURPOSE:	Processes messages for "About" dialog box
*
****************************************************************************/
BOOL APIENTRY About( HWND hDlg, UINT message, UINT wParam, LONG lParam )
{
	switch ( message )  {
		case WM_INITDIALOG:
			return TRUE;

		case WM_COMMAND:              
			if ( LOWORD( wParam ) == IDOK )	 {
			  EndDialog( hDlg, TRUE );
			  return TRUE;
			}
			break;

		case WM_CLOSE:
			
			EndDialog( hDlg, TRUE );
			return TRUE;			
	}
	return FALSE;   
}


/******************************************************************************
*
*	FUNCTION:	GetDLLVersion
*
*	PURPOSE:	Gets the version number of the specified DLL.
*
******************************************************************************/
HRESULT GetDLLVersion( PTCHAR DllName, LPDWORD pdwMajor, LPDWORD pdwMinor)
{
	HINSTANCE			hDll;
	HRESULT				hr = S_OK;
	DLLVERSIONINFO_		dvi;

	*pdwMajor = 0;
	*pdwMinor = 0;

	//Load the DLL.
	hDll = LoadLibrary(DllName);

	if( hDll ) {

	   pDllGetVersionProc = (PVOID)GetProcAddress(hDll, "DllGetVersion");

	   if(pDllGetVersionProc) {
  
		  ZeroMemory(&dvi, sizeof(dvi));
		  dvi.cbSize = sizeof(dvi);

		  hr = (*pDllGetVersionProc)(&dvi);
  
		  if(SUCCEEDED(hr)) {

			 *pdwMajor = dvi.dwMajorVersion;
			 *pdwMinor = dvi.dwMinorVersion;
		  }
 	  } else {

		  // If GetProcAddress failed, the DLL is a version previous to the one 
		  // shipped with IE 3.x.
		  *pdwMajor = 4;
		  *pdwMinor = 0;
      }
   
	  FreeLibrary(hDll);
	  return hr;
	}

	return E_FAIL;
}


/****************************************************************************
*
*    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)
*
*    PURPOSE:  Processes messages for the statistics window.
*
****************************************************************************/
LONG APIENTRY MainWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam) 
{
	static DWORD	MaxDriveSet = 0;
	static HMENU	DriveMenu;
    static HWND		hWndTT;
	static HWND		hWndToolbar;
	static POINTS   hoverPoints;
	DWORD			newDriveSet;
	LPTOOLTIPTEXT	lpToolTipText;
	LPFINDREPLACE	findMessageInfo;
#if _DEBUG
	ULONG			irpcount;
#endif
	POINT			hitPoint;
	RECT			listRect;
	DWORD			nb, versionNumber;
	DWORD			drive, drivetype;
	TCHAR			Path[ 256 ];
	static TCHAR	szBuf[128];
	TCHAR 			name[32];
	TCHAR			*File;
	DWORD			majorver, minorver;
	WIN32_FIND_DATA findData;
	HANDLE			findHandle;
	DWORD			startTime;
	HDC				hDC;
	FILE_SYSTEM_TYPE fsType;
	PAINTSTRUCT		Paint;

	switch ( message ) {

		case WM_CREATE:

			// get hourglass icon ready
			hHourGlass = LoadCursor( NULL, IDC_WAIT );

			// post hourglass icon
			SetCapture(hWnd);
			hSaveCursor = SetCursor(hHourGlass);

			// determine performance counter frequency
			QueryPerformanceFrequency( &PerfFrequency );

			// Create the toolbar control - use modern style if available.
			GetDLLVersion( _T("comctl32.dll"), &majorver, &minorver );
			if( majorver > 4 || (majorver == 4 && minorver >= 70) ) {
				hWndToolbar = CreateToolbarEx( 
					hWnd, TOOLBAR_FLAT | WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS,  
					ID_TOOLBAR, 11, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtons,
					NUMBUTTONS, 16,16,16,15, sizeof(TBBUTTON)); 
			} else {
				hWndToolbar = CreateToolbarEx( 
					hWnd, WS_CHILD | WS_BORDER | WS_VISIBLE | TBSTYLE_TOOLTIPS,  
					ID_TOOLBAR, 11, hInst, IDB_TOOLBAR, (LPCTBBUTTON)&tbButtonsOld,
					NUMBUTTONSOLD, 16,16,16,15, sizeof(TBBUTTON)); 
			}
			if (hWndToolbar == NULL )
				MessageBox (NULL, _T("Toolbar not created!"), NULL, MB_OK );

			// Create the ListBox within the main window
			hWndList = CreateList( hWnd );
			if ( hWndList == NULL )
				MessageBox( NULL, _T("List not created!"), NULL, MB_OK );

		    // open the handle to the device
			if( IsNT ) {

				GetCurrentDirectory( sizeof Path, Path );
				_stprintf( Path+lstrlen(Path), _T("\\%s"), SYS_FILE );

				findHandle = FindFirstFile( Path, &findData );
				if( findHandle == INVALID_HANDLE_VALUE ) {

					if( !SearchPath( NULL, SYS_FILE, NULL, sizeof(Path), Path, &File ) ) {

						_stprintf( msgbuf, _T("%s was not found."), SYS_NAME );
						return Abort( hWnd, msgbuf );
					}

				} else FindClose( findHandle );

				if ( ! LoadDeviceDriver( SYS_NAME, Path, &SysHandle ) )  {

					_stprintf( msgbuf, _T("Opening %s (%s): error %d"), SYS_NAME, Path,
									GetLastError( ) );
					return Abort( hWnd, msgbuf );
				}

				// Correct driver version?
				if ( ! DeviceIoControl(	SysHandle, FILEMON_version,
										NULL, 0, &versionNumber, sizeof(DWORD), &nb, NULL ) ||
						versionNumber != FILEMONVERSION )
				{
					MessageBox( hWnd, _T("Filemon located a driver with the wrong version.\n")
						_T("If you just installed a new version you must reboot before you are\n")
						_T("able to use it."), _T("Filemon"), MB_ICONERROR);
					PostQuitMessage( 1 );
					return 0;
				}
			} else {

				// connect with VxD
				SysHandle = CreateFile( VXD_FILE, 0, 0, NULL,
								0, FILE_FLAG_OVERLAPPED|
								FILE_FLAG_DELETE_ON_CLOSE,
								NULL );
				if ( SysHandle == INVALID_HANDLE_VALUE )  {
					_stprintf( msgbuf, "%s is not loaded properly.", VXD_NAME );
					return Abort( hWnd, msgbuf );
				}
			}

			// Have driver zero information
			if ( ! DeviceIoControl(	SysHandle, FILEMON_zerostats,
									NULL, 0, NULL, 0, &nb, NULL ) )
			{
				Abort( hWnd, _T("Couldn't access device driver") );
				return 0;
			}

			// Give the user to change initial filter
			if( strcmp(FilterDefinition.processfilter, "*") ||
				strcmp(FilterDefinition.excludeprocess, "") ||
				strcmp(FilterDefinition.pathfilter, "*")    ||
				strcmp(FilterDefinition.excludefilter, "") ||
				!FilterDefinition.logreads ||
				!FilterDefinition.logwrites ) {

				DialogBox( hInst, _T("InitFilter"), hWnd, (DLGPROC) FilterProc );
			
			} else {

				// tell the driver the initial filter
				if ( ! DeviceIoControl(	SysHandle, FILEMON_setfilter,
										&FilterDefinition, sizeof(FILTER), NULL, 
										0, &nb, NULL ) )
				{
					return Abort( hWnd, _T("Couldn't access device driver") );
				}	
			}
			
			// tell the driver the timing type
			if ( !DeviceIoControl(	SysHandle, FILEMON_timetype,
									(PVOID) &TimeIsDuration, sizeof(BOOLEAN), 
									NULL, 0, &nb, NULL ) )
			{
				return Abort( hWnd, _T("Couldn't access device driver") );
			}
			CheckMenuItem( GetMenu(hWnd), IDM_TIME,
							MF_BYCOMMAND|(TimeIsDuration?MF_CHECKED:MF_UNCHECKED) ); 

			// Tell driver to start filtering
			if ( ! DeviceIoControl(	SysHandle, FILEMON_startfilter,
									NULL, 0, NULL, 0, &nb, NULL ) )
			{
				return Abort( hWnd, _T("Couldn't access device driver") );
			}	

			// Create a pop-up menu item with the drives
			if( IsNT ) {

				DriveMenu = CreateMenu();

				// Add the special file systems menu items
				InsertMenu( DriveMenu, 0xFFFFFFFF, MF_BYPOSITION|MF_STRING,
							IDC_DRIVE+PIPE_DRIVE, _T("Named Pipes") );
				InsertMenu( DriveMenu, 0xFFFFFFFF, MF_BYPOSITION|MF_STRING,
							IDC_DRIVE+SLOT_DRIVE, _T("Mail Slots") );
				InsertMenu( DriveMenu, 0xFFFFFFFF, MF_BYPOSITION|MF_SEPARATOR,
							0, NULL );

				// Get available drives we can monitor
				MaxDriveSet = GetLogicalDrives();
				if( PositionInfo.curdriveset != (DWORD) -1 )
					CurDriveSet = PositionInfo.curdriveset;
				else
					CurDriveSet = MaxDriveSet;
				for ( drive = 0; drive < 32; ++drive )  {
					if ( MaxDriveSet & (1 << drive) )  {
						_stprintf( name, _T("%c:\\"), 'A'+drive );
						switch ( GetDriveType( name ) )  {
							// We don't like these: remove them
							case 0:					// The drive type cannot be determined.
							case 1:					// The root directory does not exist.
								drivetype = DRVUNKNOWN;
								CurDriveSet &= ~(1 << drive);
								break;
							case DRIVE_REMOVABLE:	// The drive can be removed from the drive.
								drivetype = DRVREMOVE;
								CurDriveSet &= ~(1 << drive);
								break;
							case DRIVE_CDROM:		// The drive is a CD-ROM drive.
								drivetype = DRVCD;
								CurDriveSet &= ~(1 << drive);
								break;

							// We like these types
							case DRIVE_FIXED:		// The disk cannot be removed from the drive.
								drivetype = DRVFIXED;
								break;
							case DRIVE_REMOTE:		// The drive is a remote (network) drive.
								drivetype = DRVREMOTE;
								break;
							case DRIVE_RAMDISK:		// The drive is a RAM disk.
								drivetype = DRVRAM;
								break;
						}
						_stprintf( name, _T("Drive &%c: (%s)"), 'A'+drive, DrvNames[drivetype] );
						InsertMenu( DriveMenu, 0xFFFFFFFF, MF_BYPOSITION|MF_STRING,
									IDC_DRIVE+drive, name );
					}
				}
				// Insert into top-level menu
				InsertMenu( GetMenu( hWnd ), 3, MF_BYPOSITION|MF_POPUP|MF_STRING,
							(UINT)DriveMenu, _T("&Drives") );

				// Have driver hook the selected drives
				CurDriveSet = Hook_Drives( DriveMenu, MaxDriveSet, CurDriveSet );

				if( HookPipes ) {
					fsType = NPFS;
					if( !DeviceIoControl( SysHandle, HookPipes ? FILEMON_hookspecial :
												FILEMON_unhookspecial,
										&fsType, sizeof(fsType), NULL, 0, &nb, NULL )) {
				
						MessageBox( hWnd, _T("Filemon could not attach to Named Pipe File System"), 
							APPNAME, MB_ICONWARNING|MB_OK );
						HookPipes = FALSE;
					}
				}
				if( HookSlots ) {
					fsType = MSFS;
					if( !DeviceIoControl( SysHandle, HookSlots ? FILEMON_hookspecial :
												FILEMON_unhookspecial,
										&fsType, sizeof(fsType), NULL, 0, &nb, NULL )) {
				
						MessageBox( hWnd, _T("Filemon could not attach to Mail Slot File System"), 
							APPNAME, MB_ICONWARNING|MB_OK );
						HookSlots = FALSE;
					}
				}

				CheckMenuItem( DriveMenu, IDC_DRIVE+PIPE_DRIVE, MF_BYCOMMAND|
						(HookPipes ? MF_CHECKED:MF_UNCHECKED ));
				CheckMenuItem( DriveMenu, IDC_DRIVE+SLOT_DRIVE, MF_BYCOMMAND|
						(HookSlots ? MF_CHECKED:MF_UNCHECKED ));
			}

			// set the time type bitmap
			SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_TIME, (TimeIsDuration?10:9) );

			// Start up timer to periodically update screen
			SetTimer( hWnd,	1, 500/*ms*/, NULL );

			// Initialization done
			SetCursor( hSaveCursor );
			ReleaseCapture();
			return 0;

		case WM_NOTIFY:
			// Make sure its intended for us
			if ( wParam == ID_LIST )  {
				NM_LISTVIEW	* pNm = (NM_LISTVIEW *)lParam;
				switch ( pNm->hdr.code )  {

			        case LVN_BEGINLABELEDIT:
						// Don't allow editing of information
						return TRUE;
				}
			} else {

				switch (((LPNMHDR) lParam)->code) 
				{
					case TTN_NEEDTEXT:    
						// Display the ToolTip text.
						lpToolTipText = (LPTOOLTIPTEXT)lParam;
    					LoadString (hInst, lpToolTipText->hdr.idFrom, szBuf, sizeof(szBuf));
				    	lpToolTipText->lpszText = szBuf;
						break;

					default:
						return FALSE;
				}
			}
			return FALSE;

		case WM_COMMAND:

			switch ( LOWORD( wParam ) )	 {

				// stats related commands to send to driver
				case IDM_CLEAR:
					// Have driver zero information
					if ( ! DeviceIoControl(	SysHandle, FILEMON_zerostats,
											NULL, 0, NULL, 0, &nb, NULL ) )
					{
						Abort( hWnd, _T("Couldn't access device driver") );
						return 0;
					}
					// Update statistics windows
					UpdateStatistics( hWnd, hWndList, TRUE );
					return 0;

				case IDM_HELP:
					WinHelp(hWnd, _T("filemon.hlp"), HELP_CONTENTS, 0L);
					return 0;

				case IDM_CAPTURE:
					// Disable filtering
					Capture = !Capture;
					if ( ! DeviceIoControl(	SysHandle, 
								Capture ? FILEMON_startfilter: FILEMON_stopfilter,
								NULL, 0, NULL, 0, &nb, NULL ) )
					{
						Abort( hWnd, _T("Couldn't access device driver") );
						return 0;
					}
					CheckMenuItem( GetMenu(hWnd), IDM_CAPTURE,
									MF_BYCOMMAND|(Capture?MF_CHECKED:MF_UNCHECKED) );
					SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_CAPTURE, (Capture?2:1) );
					InvalidateRect( hWndToolbar, NULL, TRUE );
					return 0;

				case IDM_AUTOSCROLL:
					Autoscroll = !Autoscroll;
					CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
									MF_BYCOMMAND|(Autoscroll?MF_CHECKED:MF_UNCHECKED) ); 
					SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, (Autoscroll?4:3) );
					InvalidateRect( hWndToolbar, NULL, TRUE );
					return 0;

				case IDM_TIME:
					TimeIsDuration = !TimeIsDuration;
					CheckMenuItem( GetMenu(hWnd), IDM_TIME,
									MF_BYCOMMAND|(TimeIsDuration?MF_CHECKED:MF_UNCHECKED) ); 
					// tell the driver
					if ( !DeviceIoControl(	SysHandle, FILEMON_timetype,
											(PVOID) &TimeIsDuration, sizeof(BOOLEAN), 
											NULL, 0, &nb, NULL ) )
					{
						Abort( hWnd, _T("Couldn't access device driver") );
						return 0;
					}
					SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_TIME, (TimeIsDuration?10:9) );
					InvalidateRect( hWndToolbar, NULL, TRUE );
					return 0;

				case IDM_FILTER:
					DialogBox( hInst, _T("Filter"), hWnd, (DLGPROC) FilterProc );
					return 0;

				case IDM_EXIT:
					// Close ourself
					SendMessage( hWnd, WM_CLOSE, 0, 0 );
					return 0;

				case IDM_FIND:
					// search the listview
					if( !hWndFind ) {
						PrevMatch = FALSE;
						PopFindDialog( hWnd );
					} else if( PrevMatch ) {

						// treat this like a find-next
						SetCapture(hWndFind);
						hSaveCursor = SetCursor(hHourGlass);
						EnableWindow( hWndFind, FALSE );
						if (FindInListview( hWnd, &FindTextInfo ) ) {
							Autoscroll = FALSE;
							CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
											MF_BYCOMMAND|MF_UNCHECKED ); 
							SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, 3 );
						}
						EnableWindow( hWndFind, TRUE );
						SetCursor( hSaveCursor );
						ReleaseCapture(); 
					}
					return 0;

				case IDM_ONTOP:
					OnTop = !OnTop;
					if( OnTop ) SetWindowPos( hWnd, HWND_TOPMOST, 0, 0, 0, 0, 
									SWP_NOMOVE|SWP_NOSIZE );
					else  SetWindowPos( hWnd, HWND_NOTOPMOST, 0, 0, 0, 0, 
									SWP_NOMOVE|SWP_NOSIZE );
					CheckMenuItem( GetMenu(hWnd), IDM_ONTOP,
							MF_BYCOMMAND|(OnTop?MF_CHECKED:MF_UNCHECKED) );
					return 0;

				case IDM_ABOUT:
					// Show the names of the authors
					DialogBox( hInst, _T("AboutBox"), hWnd, (DLGPROC)About );
					return 0;

				case IDM_SAVE:
					SaveFile( hWnd, hWndList, FALSE );
					return 0;

				case IDM_SAVEAS:
					SaveFile( hWnd, hWndList, TRUE );
					return 0;

				default:

					// selection in drive menu?
					if( IsNT ) {

						drive = LOWORD( wParam ) - IDC_DRIVE;
						if ( drive < 32 )  {
							// Toggle status
							newDriveSet = CurDriveSet ^ (1 << drive);
							// Have driver hook the selected drives
							CurDriveSet = Hook_Drives( DriveMenu, MaxDriveSet, newDriveSet );
							if( newDriveSet != CurDriveSet ) {
								_stprintf( msgbuf, _T("Filemon could not attach to drive %C:"),
									drive + L'A' );
								MessageBox( hWnd, msgbuf, 
									APPNAME, MB_ICONWARNING|MB_OK );
							}
							return FALSE;
						} else if( drive == PIPE_DRIVE ) {

							HookPipes = !HookPipes;
							fsType = NPFS;
							if( !DeviceIoControl( SysHandle, HookPipes ? FILEMON_hookspecial :
														FILEMON_unhookspecial,
												&fsType, sizeof(fsType), NULL, 0, &nb, NULL )) {
						
								_stprintf( msgbuf, _T("Filemon could not attach to Named Pipe File System"));
								MessageBox( hWnd, msgbuf, 
									APPNAME, MB_ICONWARNING|MB_OK );
								HookPipes = FALSE;
							}
							CheckMenuItem( DriveMenu, IDC_DRIVE+PIPE_DRIVE, MF_BYCOMMAND|
								(HookPipes ? MF_CHECKED:MF_UNCHECKED ));
							
						} else if( drive == SLOT_DRIVE ) {

							HookSlots = !HookSlots;
							fsType = MSFS;
							if( !DeviceIoControl( SysHandle, HookSlots ? FILEMON_hookspecial :
														FILEMON_unhookspecial,
												&fsType, sizeof(fsType), NULL, 0, &nb, NULL )) {
						
								_stprintf( msgbuf, _T("Filemon could not attach to Mail Slot File System"));
								MessageBox( hWnd, msgbuf, 
									APPNAME, MB_ICONWARNING|MB_OK );
								HookSlots = FALSE;
							}
							CheckMenuItem( DriveMenu, IDC_DRIVE+SLOT_DRIVE, MF_BYCOMMAND|
								(HookSlots ? MF_CHECKED:MF_UNCHECKED ));
						} else {

							// Default behavior
							return DefWindowProc( hWnd, message, wParam, lParam );
						}
					}
				}
			break;

		case WM_TIMER:
			// Time to query the device driver for more data
			if ( Capture )  {

				// don't process for more than a second without pausing
				startTime = GetTickCount();
				for (;;)  {
					// Have driver fill Stats buffer with information
					if ( ! DeviceIoControl(	SysHandle, FILEMON_getstats,
											NULL, 0, &Stats, sizeof Stats,
											&StatsLen, NULL ) )
					{
						Abort( hWnd, _T("Couldn't access device driver") );
						return TRUE;
					}
					if ( StatsLen == 0 )
						break;
					// Update statistics windows
					UpdateStatistics( hWnd, hWndList, FALSE );

					if( GetTickCount() - startTime > 1000 ) break;
				}
			}

			// delete balloon if necessary
			if( hBalloon ) {
				GetCursorPos( &hitPoint );
				GetWindowRect( hWndList, &listRect );
				if( hitPoint.x < listRect.left ||
					hitPoint.x > listRect.right ||
					hitPoint.y < listRect.top ||
					hitPoint.y > listRect.bottom ) {
	
					DestroyWindow( hBalloon );
				}
			}
			return 0;

		case WM_SIZE:
			// Move or resize the List
            MoveWindow( hWndList, 0, TOOLBARHEIGHT, LOWORD(lParam), HIWORD(lParam)-TOOLBARHEIGHT, TRUE );
			MoveWindow( hWndToolbar, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE );
			if( hBalloon ) DestroyWindow( hBalloon );
            return 0;

		case WM_SETFOCUS:
			SetFocus( hWndList );
			break;

		case WM_MOUSEMOVE:
		case WM_MOVE:
			if( hBalloon ) DestroyWindow( hBalloon );
			return 0;

		case WM_CLOSE:
			
#if _DEBUG
			if( IsNT ) {
				// see if the driver can unload
				if ( ! DeviceIoControl(	SysHandle, FILEMON_unloadquery,
										NULL, 0, NULL, 0,
										&irpcount, NULL ) ) {
					Abort( hWnd, _T("Couldn't access device driver") );
					return 0;
				}
				if( irpcount ) {

					_stprintf( msgbuf, 	_T("The Filemon device driver cannot unload\n")
										_T("at this time due to oustanding requests.\n\n")
										_T("Do you wish to exit the GUI now?"));
					if( MessageBox( hWnd, msgbuf, APPNAME, MB_ICONSTOP|MB_YESNO ) == IDNO )
						return 0;
				} else {
					if ( ! UnloadDeviceDriver( SYS_NAME ) )  {
						_stprintf( msgbuf, _T("Error unloading \"%s\""), SYS_NAME );
						MessageBox( hWnd, msgbuf, APPNAME, MB_OK );
					}
				}
			} 
#endif
			CloseHandle( SysHandle );
			Save_Position_Settings( hWnd );
			return DefWindowProc( hWnd, message, wParam, lParam );

		case WM_PAINT:
			if( !IsNT && Deleting ) {
				hDC = BeginPaint( hWnd, &Paint );
				EndPaint( hWnd, &Paint );
				return TRUE;
			}
			return DefWindowProc( hWnd, message, wParam, lParam );

		case WM_DESTROY:
			PostQuitMessage(0);
			return 0;

		default:

			// is it a find-string message?
			if (message == findMessageID ){ 

				// get a pointer to the find structure
				findMessageInfo = (LPFINDREPLACE)lParam;

				// If the FR_DIALOGTERM flag is set, invalidate the find window handle
				if( findMessageInfo->Flags & FR_DIALOGTERM) {
					hWndFind = NULL;
					PrevMatch = FALSE;
				    FindFlags = FindTextInfo.Flags & (FR_DOWN|FR_MATCHCASE|FR_WHOLEWORD);
					return 0;
				}

				// if the FR_FINDNEXT flag is set, go do the search
				if( findMessageInfo->Flags & FR_FINDNEXT ) {
					SetCapture(hWndFind);
					hSaveCursor = SetCursor(hHourGlass);
					EnableWindow( hWndFind, FALSE );
					if( FindInListview( hWnd, findMessageInfo ) ) {
						Autoscroll = FALSE;
						CheckMenuItem( GetMenu(hWnd), IDM_AUTOSCROLL,
										MF_BYCOMMAND|MF_UNCHECKED ); 
						SendMessage( hWndToolbar, TB_CHANGEBITMAP, IDM_AUTOSCROLL, 3 );
					}
					EnableWindow( hWndFind, TRUE );
					ReleaseCapture(); 
					return 0;
				}
				return 0;
			}

			// Default behavior
			return DefWindowProc( hWnd, message, wParam, lParam );
	}
	return 0;
}


/****************************************************************************
*
*    FUNCTION: InitApplication(HANDLE)
*
*    PURPOSE: Initializes window data and registers window class
*
****************************************************************************/
BOOL InitApplication( HANDLE hInstance )
{
	WNDCLASS  wc;
	
	// Fill in window class structure with parameters that describe the
	// main (statistics) window. 
	wc.style			= 0;                     
	wc.lpfnWndProc		= (WNDPROC)MainWndProc; 
	wc.cbClsExtra		= 0;              
	wc.cbWndExtra		= 0;              
	wc.hInstance		= hInstance;       
	wc.hIcon			= LoadIcon( hInstance, _T("APPICON") );
	wc.hCursor			= LoadCursor( NULL, IDC_ARROW );
	wc.hbrBackground	= (HBRUSH)(COLOR_INACTIVEBORDER + 1); // Default color
	wc.lpszMenuName		= _T("LISTMENU");  
	wc.lpszClassName	= _T("FileMonClass");
	if ( ! RegisterClass( &wc ) )
		return FALSE;

	wc.lpszMenuName	  = NULL;
 	wc.lpfnWndProc    = (WNDPROC) BalloonDialog;
	wc.hbrBackground  = CreateSolidBrush( 0x00E0FFFF );
	wc.lpszClassName  = _T("BALLOON");
	RegisterClass( &wc );
	
	return TRUE;
}


/****************************************************************************
*
*    FUNCTION:  InitInstance(HANDLE, int)
*
*    PURPOSE:  Saves instance handle and creates main window
*
****************************************************************************/
HWND InitInstance( HANDLE hInstance, int nCmdShow )
{
	// get the window position settings from the registry
	Get_Position_Settings();

	// create the main window
	hInst = hInstance;
	hWndMain = CreateWindow( _T("FileMonClass"), 
							_T("File Monitor - Systems Internals: http://www.sysinternals.com"), 
							WS_OVERLAPPEDWINDOW,
							PositionInfo.left, PositionInfo.top, 
							PositionInfo.width, PositionInfo.height,
							NULL, NULL, hInstance, NULL );

	// if window could not be created, return "failure" 
	if ( ! hWndMain )
		return NULL;
	
	// make the window visible; update its client area; and return "success"
	ShowWindow( hWndMain, nCmdShow );

	// maximize it if necessary
	if( PositionInfo.maximized ) {

		ShowWindow( hWndMain, SW_SHOWMAXIMIZED );
	}

	UpdateWindow( hWndMain ); 
	if( OnTop ) {
		
		SetWindowPos( hWndMain, HWND_TOPMOST, 0,0,0,0, SWP_NOMOVE|SWP_NOSIZE );
		CheckMenuItem( GetMenu(hWndMain), IDM_ONTOP,
						MF_BYCOMMAND|(OnTop?MF_CHECKED:MF_UNCHECKED) );
	}
	return hWndMain;      
}


/****************************************************************************
*
*	FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)
*
*	PURPOSE:	calls initialization function, processes message loop
*
****************************************************************************/
int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
						LPSTR lpCmdLine, int nCmdShow )
{
	MSG 	msg;      
	HWND	hWnd;
	HACCEL	hAccel;
 	DWORD	NTVersion;
        
	if ( ! InitApplication( hInstance ) )
		return FALSE;   
	
	// get NT version
	NTVersion = GetVersion();
	if( NTVersion >= 0x80000000 ) {

		IsNT = FALSE;
	} else {

		IsNT = TRUE;
	}

	// initializations that apply to a specific instance 
	if ( (hWnd = InitInstance( hInstance, nCmdShow )) == NULL )
		return FALSE;

	// load accelerators
	hAccel = LoadAccelerators( hInstance, _T("ACCELERATORS"));

	// register for the find window message
    findMessageID = RegisterWindowMessage( FINDMSGSTRING );

	// acquire and dispatch messages until a WM_QUIT message is received.
	while ( GetMessage( &msg, NULL, 0, 0 ) )  {
		if( !TranslateAccelerator( hWnd, hAccel, &msg ) &&
			(!hWndFind || !IsWindow(hWndFind) || !IsDialogMessage( hWndFind, &msg ))) {
			TranslateMessage( &msg );
			DispatchMessage( &msg ); 
		}
	}
	return msg.wParam;										 
}