%include "icedump.inc"

; ------------------------------------------------------------------------------
; this is the actual dumper code that's executed in ring-0.
; it can be entered from three different modes: V86, ring-3 PM and ring-0 PM.

; for the first two modes we allocated callbacks during Winice initialization
; and use them in the parser to direct execution flow from the client here.

; no matter where we're coming from though, the client registers will be
; set up the same way:
;    EDI: flat start address of block
;    ECX: length of block (or 0 if emulating old PAGEIN)
;    ESI: filename (or 0 if emulating old PAGEIN)

; when entering through one of the callbacks, the real registers will be setup
; by the VMM as usual. we will make use of EBP (pointer to the client registers)
; and EDX (reference value specified when we allocated the callbacks, it's the
; flat start address of Winice itself) only. the ring-0 entry point will set up
; the registers this way as well.

; the dumper routine has to return to an INT3 instruction which will give
; control back to Winice. the address of this instruction was already written
; into an internal Winice variable by the parser (so that Winice's INT3 handler
; would recognize it as a special INT3).

; handling V86 mode is a bit more tricky since we need to return to an INT3
; somewhere in the first 1 megabyte of the linear address space. our first
; attempt is to find one such instruction in the ROM BIOS (actually we couldn't
; care less if a 0xCC there was ever meant to be executed or not ;-). if this
; fails (i really doubt that there's no single 0xCC among 64k bytes...) then
; we put a INT3 into the client's stack ourselves. this is a rather bad method
; since we may be overwriting something important (even though we write below
; the client SS:SP, some clever protection scheme could easily detect it).
; ------------------------------------------------------------------------------

	BITS 32

%if WINICE_VERSION = 0x322
	ORG 0xD14C8
%elif WINICE_VERSION = 0x323
	ORG 0xD64D8
%else
	ORG NotSupportedVersion
%endif

; ------------------------------------------------------------------------------
; this entry point will be called from V86 mode.
; we need to find an INT3 instruction to return to after the dumper has finished
; its job. 
; ------------------------------------------------------------------------------
entryV86CB
	pushad
	mov	edi,0xF0000		; start of ROM BIOS
	mov	ecx,0x10000		; length of area we search through
	mov	al,0xCC			; opcode of INT3
	cld
	repnz	scasb			; may cause page fault
	jecxz	.useClientStack		; damn, let's hack the stack instead

	dec	edi			; adjust EDI
	shld	ecx,edi,28		; ECX: client CS
	shrd	ebx,edi,4		; EBX: client IP
	shr	ebx,28
	jmp short .setPAGEIN_INT3

.useClientStack
	movzx	ecx,word [ebp+Client_SS]	; get client SS
        shrd    edi,ecx,28              ; convert to linear address
        movzx   ebx,word [ebp+Client_ESP]       ; get client SP
	dec	bx			; 1 byte below the stack
	add	edi,ebx			; linear address of our INT3
	stosb				; may cause page fault

;-------------------------------------------------------------------------------
; store the address (IP only) of our INT3 into an internal Winice variable. this
; has already been done in the parser for the PM clients.
; ------------------------------------------------------------------------------
.setPAGEIN_INT3
	mov	[edx+oPAGEIN_INT3],ebx	; store address of our INT3
	jmp short setCSEIP

; ------------------------------------------------------------------------------
; this entry point will be called from ring-3 PM.

; we set CS:EIP to an INT3 which is found inside Winice itself.

; WINICE_DELTA is the relativ base of the special ring-3 code segment
; (whose selector is stored at wWINICE_CODE) that points to Winice itself and
; covers almost all of it. this descriptor is allocated and used by Winice
; itself, nevertheless it comes really handy now ;-)
; ------------------------------------------------------------------------------
entryPMCB
	pushad
	mov	ecx,[edx+wWINICE_CODE]	; set client CS:EIP to an INT3
        mov     ebx,oINT3-WINICE_DELTA

; ------------------------------------------------------------------------------
; set the client CS:(E)IP where it will resume execution at.
; ------------------------------------------------------------------------------
setCSEIP
        mov     [ebp+Client_CS],ecx
	mov	[ebp+Client_EIP],ebx
	jmp short dump

; ------------------------------------------------------------------------------
; this entry point will be "called" from ring-0 PM (it's not a real call since
; the parser simply rewrites the client's EIP to have it resume execution here).

; we will return to an INT3 which is found inside Winice itself.
; ------------------------------------------------------------------------------
entryPMR0
	push	dword oINT3		; reloc, set return point to an INT3
	pushad				; simulate Client Register Structure
	mov	ebp,esp			; on the stack

; ------------------------------------------------------------------------------
; old code from previous version (beta 3), left here for educational purposes.
; it simply simulates a RETF on the client's stack (pops CS:EIP from the stack).
; note that 16 bit stack handling is not really consistent, but it's left as an
; exercise for the reader instead. yeah, you could call me lazy as well ;-)
;
;	push      ds
;	mov       ds,[ebp+Client_SS]
;	mov       esi,[ebp+Client_ESP]
;	lar       eax,[ebp+Client_SS]
;	test      eax,0x400000
;	jne      .a
;	movzx     esi,si
;.a
;	mov       eax,[esi]
;	mov       edx,[esi+4]
;	mov       [ebp+Client_EIP],eax
;	mov       [ebp+Client_CS],dx
;	add       dword [ebp+Client_ESP],8
;	pop       ds
; ------------------------------------------------------------------------------


; ------------------------------------------------------------------------------
; we always dump memory belonging to a single page (at most 4k).

; before dumping a page (or part of it) we read in a single byte from it
; (usually the first one in the page), and let the VMM handle the potential
; page fault (this is the trick used in the original PAGEIN by the way).
; since there always are users who don't heed the warnings in the manual, i
; check if the page has been committed at all before i touch it.

; we use the client registers as temporary variables holding values for the
; start address of the current page and the remaining length.

; because the total area to be dumped may not cover full pages (at its beginning
; and end), we have to make sure that we don't dump more than requested nor do
; we access memory not explicitly specified in the request.

; if we didn't have to care about paging issues (e.g. the user didn't use a page
; file or IFSMgr tolerated page faults itself) then we could write out one big
; block... oh well.
; even trying to page in all pages at once before dumping may not always work
; as nothing prevents the VMM from freeing up pages for the new guys from the
; already brought in ones. locking all pages into physical memory could also
; fail if the block is too large.
; ------------------------------------------------------------------------------
dump
	mov	esi,[ebp+Client_ESI]	; get pointer to file name
	test	esi,esi			; emulate old PAGEIN?
	jz	.emulate

	mov	eax,R0_OPENCREATFILE
	mov	ebx,0x2012		; read/write|share:deny all|no INT24
;	mov	ecx,0x20		; archive
	movzx	ecx,bh
;	mov	edx,0x12		; replace/open|create
	movzx	edx,bl
	db 0xCD,0x20,0x32,0x00,0x40,0x00; VxDcall IFSMGR.Ring0_FileIO
	jb	.return

	mov	ebx,eax			; store file handle

.emulate
	mov	esi,[ebp+Client_EDI]	; get start address of block

; ------------------------------------------------------------------------------
; first compute length of block to dump at once. it's normally 4 kbytes except
; for at the beginning and end where it could be a fraction only.
; ------------------------------------------------------------------------------
.loop
	mov	edi,esi
	shr	edi,12			; get current page number
	mov	eax,edi			; save page number for later use
	inc	edi			; get next page number
	shl	edi,12			; get linear start address of next page
	sub	edi,esi			; get length of block on current page
	mov	ecx,[ebp+Client_ECX]	; get remaining length
	cmp	edi,ecx			; EDI: bytes to dump from current page
	ja	.a

	mov	ecx,edi			; dump from one page at a time
.a

; ------------------------------------------------------------------------------
; alternative code to check the committed flag of current page, of course much
; slower and bigger ;-). never really tested/used, just left here for comparing
; its size to that of the other method and for educational purposes as well.
; after all, this should be the preferred way.
; ------------------------------------------------------------------------------
;	sub	esp,0x1C
;	mov	eax,esp
;	push	dword 0x1C
;	push	eax
;	push	edi
;	db 0xCD,0x20,0x??,0x??,0x01,0x00; VMMcall _PageQuery
;	add	esp,0xC+0x1C
;;	test	[esp+MBI.mbi_State],MEM_COMMIT
;	test	word [esp-0x1C+0x10],0x200
;	jz	.skipPage
; ------------------------------------------------------------------------------

; ------------------------------------------------------------------------------
; the following piece of code directly accesses the page tables and directories
; to figure out if a specific page has been committed or not. credits go to
; LiuTaoTao for the short code to calculate the offsets into the tables.

; other details of the Win9x specific page and directory table layout and the
; flags used in the table entries can be figured out from the _PageQuery code
; itself.
; ------------------------------------------------------------------------------
	mov	edi,0xFF800000		; linear start address of page tables
	lea	edx,[edi+4*eax]		; EAX: current page number
	shr	edx,12			; EDX: current directory number
	test	byte [edi+4*edx],0x1	; is page directory present?
	jz	.skipPage

	test	byte [edi+4*eax+1],0x2	; is page committed?
	jz	.skipPage

	mov	al,[esi]		; PAGEIN!

	jecxz	.return			; emulate old PAGEIN?

	mov	eax,R0_WRITEFILE
	push	ebx			; save file handle
	mov	edx,esi
	sub	edx,[ebp+Client_EDI]	; EDX: absolut file pointer
	db 0xCD,0x20,0x32,0x00,0x40,0x00; VxDcall IFSMGR.Ring0_FileIO
	pop	ebx			; restore file handle
	jb	.close

.skipPage
	jecxz	.return			; emulate old PAGEIN?

	add	esi,ecx
	sub	[ebp+Client_ECX],ecx	; are we done?
	jnz	.loop

.close
	mov	eax,R0_CLOSEFILE
	db 0xCD,0x20,0x32,0x00,0x40,0x00; VxDcall IFSMGR.Ring0_FileIO

.return
	popad
	retn			; client will resume execution at an INT3
