/*
 *  Signature finder by amante4 of Immortal Descendants. 10/26/2000
 *
 */

#include <ida.hpp>
#include <idp.hpp>
#include <loader.hpp>
#include <kernwin.hpp>
#include <bytes.hpp>
#include <fpro.h>

//--------------------------------------------------------------------------
//
//      initialize plugin
//
//      IDA will call this function only once.
//      If this function returns PLGUIN_SKIP, IDA will never load it again.
//      If this function returns PLUGIN_OK, IDA will unload the plugin but
//      remember that the plugin agreed to work with the database.
//      The plugin will be loaded again if the user invokes it by
//      pressing the hotkey or selecting it from the menu.
//      After the second load the plugin will stay on memory.
//
//      In this example we check the input file format and make the decision.
//      You may or may not check any other conditions to decide what you do:
//      whether you agree to work with the database or not.
//
int init(void)
{	
  return PLUGIN_OK;
}

//--------------------------------------------------------------------------
//      terminate
//      usually this callback is empty
//
//      IDA will call this function when the user asks to exit.
//      This function won't be called in the case of emergency exits.

void term(void)
{
}

//--------------------------------------------------------------------------
//
//      The plugin method
//
//      This is the main function of plugin.
//
//      It will be called when the user selects the plugin.
//
//              arg - the input argument, it can be specified in
//                    plugins.cfg file. The default is zero.
//
//

long byte_count = 0; // global for number of bytes counted

// function to search for a signature in a file
int find_code_sig(FILE *fp,int num_bytes,unsigned char *signature)
{
  int index = 0; // index into signature
  unsigned char read_byte;
  int value = 1;
  
  show_wait_box("Searching for code signature....\n");
  clearBreak(); // enable the cancel button
  
  while (value) {
    if (wasBreak()) { // check to see if cancel was pressed.
      beep();
      msg ("Action Cancelled by the User...\n");
      hide_wait_box();
      return -1;
    }
      		
    if (index == num_bytes) { // if we've found a matching signature
      hide_wait_box();
      return 1;
    }
      	  
    value = qfread(fp,&read_byte,1); // read a byte from the file
    ++byte_count;   
      	  
    if (read_byte == signature[index]) { // if we find a matching char
      ++index; // increment to check next char of signature      
    }
    else {      
      index = 0; // reset index if ever a mismatch
    }
  }
  hide_wait_box();
  return 0;
}

// This is the main function that gets called by IDA
void run(int arg)
{ 
  char init_dialog[] = "Signature options\n"
                       "\n"
                       "Select the starting address and the number of bytes of the signature\n\n"
                       "<~S~tarting address:$:32:16::>\n"
                       "<~N~umber of bytes:D:32:16::>\n\n";
                       
  ea_t starting_address = get_screen_ea(); // initialize this to the current screen address
  int num_bytes = 1; // the number of bytes to use for signature
  char *file;
  int i;
  unsigned char *signature;
  FILE *fp;  
  int result;
  int cont = 1;
  
  // start off by displaying the init dialog to get options for this run
  if (AskUsingForm_c(init_dialog,&starting_address,&num_bytes)) { // as long as use doesn't hit cancel or error    
    if (num_bytes > 0) { // make sure there's more than 1 byte in sig before continuing
      // gather the signature
      signature = (unsigned char *) qalloc(num_bytes * sizeof(unsigned char)); // allocate enough for signature
      for (i=0;i<num_bytes;++i) { 
      	signature[i] = get_byte(starting_address + i); // load a byte at a time from starting_address into array
      }      
      file = askfile_c(0,NULL,"Please select the file to find signature in...");      
      
      // now open the file requested and search for signature
      if((fp = qfopen(file,"r+b")) == NULL) { // open the file for binary read
        msg("Error: Can't open %s for reading\n",file);
      }
      else {
      	byte_count = 0;
      	while(cont == 1) {
      	  result = find_code_sig(fp,num_bytes,signature);
      	  if (result == 1) {     
      	    cont = askyn_c(1,"Found selected code signature in\n%s\nat RVA 0x%x\n\nWould you like to search the rest of the file?\n",file,byte_count-num_bytes);      	  
      	      
      	  }
      	  else {
      	    beep();
      	    msg("INFO: Could not find code signature in %s\n",file);
      	    cont = 0;
      	  }
      	}
      	qfclose(fp);
      }
    }
    else {
      beep();
      warning("Error: Need to specify 1 or more bytes\n");          
    }
  }
  else {
    beep();
    msg("Action cancelled by user\n");
  }

}

//--------------------------------------------------------------------------
char comment[] = "Signature finder - By Amante4 of Immortal Descendants";

char help[] =
        "This plugin is a signature finder. It asks for a starting address\n"
        "\n"
        "and a number of bytes to use for the signature and searches in\n"
        "\n"
        "other files for that signature\n";


//--------------------------------------------------------------------------
// This is the preferred name of the plugin module in the menu system
// The preferred name may be overriden in plugins.cfg file

char wanted_name[] = "Find Signature";


// This is the preferred hotkey for the plugin module
// The preferred hotkey may be overriden in plugins.cfg file
// Note: IDA won't tell you if the hotkey is not correct
//       It will just disable the hotkey.

char wanted_hotkey[] = "Shift-F";


//--------------------------------------------------------------------------
//
//      PLUGIN DESCRIPTION BLOCK
//
//--------------------------------------------------------------------------

extern "C" plugin_t PLUGIN = {
  IDP_INTERFACE_VERSION,
  0,                    // plugin flags
  init,                 // initialize

  term,                 // terminate. this pointer may be NULL.

  run,                  // invoke plugin

  comment,              // long comment about the plugin
                        // it could appear in the status line
                        // or as a hint

  help,                 // multiline help about the plugin

  wanted_name,          // the preferred short name of the plugin
  wanted_hotkey         // the preferred hotkey to run the plugin
};
