#ifndef DEBUGCLIENT_H
#define DEBUGCLIENT_H

#include <string>
#include <vector>
#include <cstdlib>

#include <zycon/src/zycon.h>

#include "errors.hpp"
#include "defs.hpp"
#include "logger.hpp"
#include "InformationProvider.hpp"
#include "DebuggerOptions.hpp"

//! Creates the debugger options XML string that is sent to BinNavi
std::string getDebuggerOptionsString(const DebuggerOptions& options);

//! Creates the register values string that is sent to BinNavi
std::string getRegisterString(const std::vector<RegisterDescription>& registers);

//! Creates the thread information string that is sent to BinNavi
std::string getThreadIdString(const std::vector<Thread>& tids);

//! Creates the address size string that is sent to BinNavi
std::string getAddressSizeString(unsigned int size);

//! Creates the information string that is sent to BinNavi
std::string getInformationString(const DebuggerOptions& options, unsigned int addressSize, const std::vector<RegisterDescription>& registerNames);
	
/**
* DebugClient objects are used to communicate with BinNavi.
*
* The template parameter ConnectionPolicy specifies the kind of connection
* that is used to communicate between BinNavi and the debug client.
* The template parameter SystemPolicy specifies the platform on which
* the target process is executed.
**/
template<typename ConnectionPolicy, typename SystemPolicy>
class DebugClient
{
	private:
		
		/**
		* Connection policy that is used to connect the debug client with BinNavi.
		**/
		ConnectionPolicy* connection;
		
		/**
		* System policy that implements the platform dependent debugging code.
		**/
		SystemPolicy* process;
		
		/**
		* Generates and sends the target information string to BinNavi.
		*
		* @return A NaviError code that describes whether the operation was successful or not.
		**/
		NaviError sendInformationString()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
			
			std::vector<RegisterDescription> registerNames = process->getRegisterNames();
			
			DebuggerOptions options = process->getDebuggerOptions();
			
			unsigned int addressSize = process->getAddressSize();
			
			std::string infoString = getInformationString(options, addressSize, registerNames);

			unsigned int infoResult = connection->sendInfoString(infoString);

			if (infoResult)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't send target information to BinNavi");
				return infoResult;
			}

			msglog->log(LOG_VERBOSE, "Successfully sent the Information String to BinNavi");
				
			return NaviErrors::SUCCESS;
		}

		/**
		* Retrieves the debugger event settings packet from BinNavi. This needs to be done before the debuggee is started,
		* otherwise the debugger is unable to react to certain events accordingly.
		*
		* @return The packet on success, NULL otherwise.
		**/
		NaviError handleEventSettingsPacket() const
		{
			Packet settingsPacket;
			
			NaviError error = connection->readPacket(&settingsPacket);
			
			if (error)
			{
				msglog->log(LOG_ALWAYS, "Failed to retrieve debugger settings packet from BinNavi");
				return error;
			}

			if (settingsPacket.hdr.command != cmd_set_debugger_event_settings)
			{
				msglog->log(LOG_ALWAYS, "Expected Debugger Event Settings packet, but received packet with command id: %d\n", settingsPacket.hdr.command);
				return NaviErrors::INVALID_PACKET;
			}

			InformationProvider provider;
			return process->processPacket(&settingsPacket, provider);
		}
		
	public:
	
		/**
		* Creates a new debug client object.
		*
		* @param port The local port which accepts BinNavi connections.
		**/
		DebugClient(unsigned int port)
			: connection(new ConnectionPolicy(port)), process(new SystemPolicy())
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
		}
		
		/**
		* Creates a new debug client object.
		*
		* @param port The local port which accepts BinNavi connections.
		* @param pid The process ID of the target program.
		**/
		DebugClient(unsigned int port, unsigned int pid)
			: connection(new ConnectionPolicy(port)), process(new SystemPolicy(pid))
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
		}
		
		/**
		* Creates a new debug client object.
		*
		* @param port The local port which accepts BinNavi connections.
		* @param path The path to the target process executable.
		**/
		DebugClient(unsigned int port, const NATIVE_STRING path, const std::vector<const NATIVE_STRING>& commands)
			: connection(new ConnectionPolicy(port)), process(new SystemPolicy(path, commands))
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
		}

		/**
		* Creates a new debug client object.
		*
		* @param pipe The path of the target pipe to connect to.
		* @param port The local port which accepts BinNavi connections.
		**/
		DebugClient(unsigned int port, const NATIVE_STRING pipe)
			: connection(new ConnectionPolicy(port)), process(new SystemPolicy(pipe))
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
		}
		
		/**
		* Returns the system policy of the debug client.
		*
		* @return The system policy of the debug client.
		**/
		SystemPolicy* getSystemPolicy() const
		{
			return process;
		}
		
		/**
		* Initializes the connection between the debug client and BinNavi.
		*
		* @return A NaviError code that describes whether the operation was successful or not.
		**/
		NaviError initializeConnection()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
			
			connection->printIpAddress();
	
			return connection->initializeConnection();
		}
		
		/**
		* Closes the connection to BinNavi.
		*
		* @return A NaviError code that describes whether the operation was successful or not.
		**/
		NaviError closeConnection()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
			return connection->closeConnection();
		}
	
		/**
		* Waits for a connection from BinNavi.
		* 
		* @return A NaviError code that describes whether the operation was successful or not.
		**/
		NaviError waitForConnection()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
			
			NaviError connectionResult = connection->waitForConnection();
			
			if (!connectionResult)
			{
				connectionResult = connection->sendAuthentication();
			}
			
			return connectionResult;
		}
		
		/**
		* Tries to start a process
		* 
		* @return A NaviError code that describes whether the operation was successful or not.
		**/
		NaviError attachToProcess()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);

			// send debugger_event_settings_reply
			connection->sendSimpleReply(resp_query_debugger_event_settings, 0);
			handleEventSettingsPacket();

			NaviError infoResult = sendInformationString();
			if (infoResult)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't send message information string (Code %d)", infoResult);
				return infoResult;
			}
			
			// Try to attach to the process.
			NaviError attachResult = process->start();
			
			// Send the appropriate message to BinNavi
			commandtype_t reply = attachResult ? resp_attach_error : resp_attach_success;
			
			NaviError sendResult = connection->sendSimpleReply(reply, 0);
				
			if (sendResult)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't send message (Code %d)", sendResult);
				return sendResult;
			}
			
			if (attachResult)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't attach to target process");
				return attachResult;
			}
			
			// Attach operation was successful
			msglog->log(LOG_VERBOSE, "Attaching to the target process succeeded");

			return NaviErrors::SUCCESS;
		}
		
		/**
		* Sends BinNavi a message that a target was not specified when the debug client was
		* started and therefore must be selected manually now.
		**/
		NaviError requestTarget()
		{
			NaviError sendResult = connection->sendSimpleReply(resp_request_target, 0);
			
			if (sendResult)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't send message (Code %d)", sendResult);
			}
			
			return sendResult;
		}
		
		/**
		* Processes incoming packets from BinNavi.
		**/
		NaviError processPackets()
		{
			msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
			
			bool loop = true;
			NaviError ret = NaviErrors::SUCCESS;

			do
			{
				// There are two important race conditions here that must be handled.
				// 
				// Race Condition I:
				//   0. Precondition: The target process is running (not suspended)
				//   1. RemoveBreakpoint at offset X packet has arrived => connection->hasData() == true
				//   2. The breakpoint at offset X is hit while processing the packet
				//   3. The breakpoint handler is invoked after the packet is processed (= the breakpoint was removed)
				//   4. The breakpoint handler believes that the breakpoint wasn't set by the debug client => Error
				// 
				// Race Condition II:
				//   0. Precondition: The target process is running (not suspended)
				//   1. Breakpoint at offset X is hit
				//   2. RemoveBreakpoint at offset X packet arrives
				//   3. Bad things might happen; to be investigated
				
				// TODO: All other debug commands must be checked for race conditions
				// All commands which modify the process (have side effects) might lead
				// to race conditions.
				
				if (connection->hasData())
				{
					Packet* p = new Packet;
					
					// Read the next packet.
					NaviError error = connection->readPacket(p);
					
					msglog->log(LOG_VERBOSE, "Processing command %s", commandToString(p->hdr.command));
					
					if (!error)
					{
						// The packet was read successfully. It is now possible
						// to perform the desired debugging operation.
						
						msglog->log(LOG_VERBOSE, "Received packet with command %s from BinNavi", commandToString(p->hdr.command));
						
						InformationProvider provider;
						
						// trying to work on the debug events which just came in before we introduce any other things.
						process->readDebugEvents();
						
						NaviError procResult = process->processPacket(p, provider);
						
						if (procResult)
						{
							msglog->log(LOG_VERBOSE, "Couldn't process packet (Code %d)", procResult);
							
							// The debug operation failed. Notify BinNavi of the problem.
							connection->sendErrorReply(p, procResult);
						}
						else
						{
							msglog->log(LOG_VERBOSE, "Packet successfully processed. Sending reply to BinNavi now. ...");
							
							// The debug operation was successful. Notify BinNavi about that.
							connection->sendSuccessReply(p, provider);
						}
						
						if (p->hdr.command == cmd_terminate)
						{
							delete p;
							return NaviErrors::SUCCESS;
						}
						else if (p->hdr.command == cmd_detach)
						{
							delete p;
							return NaviErrors::SUCCESS;
						}
						else if (p->hdr.command == cmd_cancel_target_selection)
						{
							msglog->log(LOG_VERBOSE, "Canceling target selection");
							delete p;
							return NaviErrors::SUCCESS;
						}
						else if (p->hdr.command == cmd_select_process || p->hdr.command == cmd_select_file)
						{
							msglog->log(LOG_VERBOSE, "Selecting target process");
							
							NaviError attachResult = attachToProcess();
							
							if (attachResult)
							{
								return attachResult;
							}
						}
					}
					else if (error == NaviErrors::CONNECTION_CLOSED || error == NaviErrors::CONNECTION_ERROR)
					{
						// Connection broken
						ret = error;
						loop = false;
					}
					else
					{
						msglog->log(LOG_ALWAYS, "Error: Reading packet failed (%s)", commandToString(p->hdr.command));
					}
					
					delete p;
				}
				else if (process->hasTarget() && (process->isDebugEventAvailable() || !process->readDebugEvents()))
				{
					while (process->isDebugEventAvailable())
					{
						DBGEVT* evt = new DBGEVT;
						
						NaviError dbgResult = process->getDebugEvent(evt);
						
						if (dbgResult)
						{
							msglog->log(LOG_ALWAYS, "Error: Couldn't retrieve debug event (Code %d)", dbgResult);
						}
						
						msglog->log(LOG_VERBOSE, "Sending debug event to BinNavi (%s)", debugEventToString(evt->type));

						NaviError sendResult = connection->sendDebugEvent(evt);
						
						if (sendResult)
						{
							msglog->log(LOG_ALWAYS, "Error: Couldn't send debug event (Code %d)", sendResult);
						}
						else
						{
							// If BinNavi was notified of the debug event, the debug
							// event can now be removed from the list of debug events.
							
							unsigned int popResult = process->popDebugEvent();
							
							if (popResult)
							{
								msglog->log(LOG_ALWAYS, "Error: Couldn't remove debug event (Code %d)", popResult);
							}
							
							if (evt->type == dbgevt_process_closed)
							{
								delete evt;
								
								return NaviErrors::SUCCESS;
							}
						}
						
						delete evt;
					}
				}
				else if (process->hasTarget())
				{

				// facilitates debugging - less noise
				#ifdef _DEBUG
					continue;
				#endif

					InformationProvider provider;
					
					Packet* p = new Packet();
					p->hdr.command = cmd_read_memory;
					
					NaviError procResult = process->reloadMemory(p, provider);
					
					if (procResult)
					{
						if (procResult != NaviErrors::NOTHING_TO_REFRESH)
						{
							msglog->log(LOG_VERBOSE, "Couldn't update memory (Code %d)", procResult);
						}
					}
					else
					{
						msglog->log(LOG_VERBOSE, "Packet successfully processed. Sending reply to BinNavi now. ...");
						
						// The debug operation was successful. Notify BinNavi about that.
						connection->sendSuccessReply(p, provider);
					}
					
					delete p;
				}
				
			} while (loop);
			
			return ret;
		}
		
		virtual ~DebugClient()
		{
			delete connection;
			delete process;
		}
};

#endif
