#include <algorithm>
#include <boost/foreach.hpp>
#include <iostream>


#include "WindowsSystem.hpp"
#include <tlhelp32.h>

#include <zycon/src/zycon.h>

#include <zywin/src/toolhelp.h>

#include "../errors.hpp"
#include "../logger.hpp"
#include "../DebuggerOptions.hpp"
#include "NTHeader.h"

#include "../windowscommon/WindowsCommon.hpp"

const unsigned int PROTECTION_CHANGE_FAILED = 1;
const unsigned int PROTECTION_CHANGED = 2;
const unsigned int PROTECTION_CHANGE_UNNECESSARY = 3;

/**
* Comparator class that compares two threads by thread ID.
**/
class ThreadComparator
{
	private:
		//! The TID of the first thread
		unsigned int tid;
		
	public:
		/**
		* Creates new new thread comparator.
		*
		* @param tid The TID of the first thread
		**/
		ThreadComparator(unsigned int tid) : tid(tid) { }
		
		/**
		* Compares two threads by thread ID.
		*
		* @param t2 The second thread
		*
		* @return True, if the two TIDs are equal. False, otherwise.
		**/
		bool operator()(const Thread& t2) { return tid == t2.tid; }
};

void printLastError()
{
	unsigned int err = GetLastError();
	
	LPVOID message;
	FormatMessage(
		FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
		NULL, err,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		(LPTSTR) &message,
		0, NULL);

	msglog->log(LOG_ALWAYS, "Last error: %s (Error Code: %d)", message, err); 

	LocalFree(message);
}

void dbg_echo_MEMORY_BASIC_INFORMATION(MEMORY_BASIC_INFORMATION *m)
{
	msglog->log(LOG_VERBOSE, "Base: 				%lx", m->BaseAddress);
	msglog->log(LOG_VERBOSE, "AllocationBase: 		%lx", m->AllocationBase);
	msglog->log(LOG_VERBOSE, "AllocationProtect: 	%lx", m->AllocationProtect);
	msglog->log(LOG_VERBOSE, "RegionSize: 			%lx", m->RegionSize);
	msglog->log(LOG_VERBOSE, "state:				%lx", m->State);
	msglog->log(LOG_VERBOSE, "Protect:				%lx", m->Protect);
	msglog->log(LOG_VERBOSE, "Type:					%lx", m->Type);
}

NaviError WindowsSystem::fillModules(HANDLE hProcess, std::vector<Module>& modules) const
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);

	std::vector<std::pair<HMODULE, std::string> > internalModules;
	if (!listProcessModules(hProcess, internalModules))
	{
		msglog->log(LOG_ALWAYS, "Error: Listing modules of the target process failed - aborting.");

		return NaviErrors::COULDNT_LIST_PROCESSES;
	}

	msglog->log(LOG_VERBOSE, "Found %d modules in the process", internalModules.size());

	for (std::vector<std::pair<HMODULE, std::string> >::iterator Iter = internalModules.begin(); Iter != internalModules.end(); ++Iter)
	{
		HMODULE hModule = Iter->first;
		std::string path = Iter->second;
		std::string name = boost::filesystem::path(Iter->second).filename().string();

		MODULEINFO moduleInfo;

		CPUADDRESS size = 0;
		CPUADDRESS base = (CPUADDRESS) hModule;

		if (GetModuleInformation(hProcess, hModule, &moduleInfo, sizeof(MODULEINFO)))
		{
			base = (CPUADDRESS) moduleInfo.lpBaseOfDll;
			size = moduleInfo.SizeOfImage;
		}
		else
		{
			msglog->log(LOG_VERBOSE, "Could not determine module size of module %s", name.c_str());
		}

		Module m(name, path, base, size);

		msglog->log(LOG_VERBOSE, "Found module %s loaded from %s (%X : %X)", name.c_str(), path.c_str(), hModule, size);

		modules.push_back(m);
	}

	return NaviErrors::SUCCESS;
}

BOOL isWindowsXPOrLater() 
{
	OSVERSIONINFOEX osvi;
	DWORDLONG dwlConditionMask = 0;
	BYTE op = VER_GREATER_EQUAL;

	// Initialize the OSVERSIONINFOEX structure.
	ZeroMemory(&osvi, sizeof(OSVERSIONINFOEX));
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
	osvi.dwMajorVersion = 5;
	osvi.dwMinorVersion = 1;
	osvi.wServicePackMajor = 0;
	osvi.wServicePackMinor = 0;

	// Initialize the condition mask.

	VER_SET_CONDITION( dwlConditionMask, VER_MAJORVERSION, op );
	VER_SET_CONDITION( dwlConditionMask, VER_MINORVERSION, op );
	VER_SET_CONDITION( dwlConditionMask, VER_SERVICEPACKMAJOR, op );
	VER_SET_CONDITION( dwlConditionMask, VER_SERVICEPACKMINOR, op );

	// Perform the test.

	return VerifyVersionInfo(
		&osvi, 
		VER_MAJORVERSION | VER_MINORVERSION | 
		VER_SERVICEPACKMAJOR | VER_SERVICEPACKMINOR,
		dwlConditionMask
	);
}

/**
* Makes a page of the target process memory writable.
*
* @param hProcess The handle of the target process.
* @param offset The offset whose page should be made writable.
* @param oldProtection The protection level of the memory before it is made writable.
*
* @return A value that indicates whether the protection changed or not.
**/
unsigned int makePageWritable(HANDLE hProcess, CPUADDRESS offset, DWORD& oldProtection, bool silent)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	MEMORY_BASIC_INFORMATION mem;
	memset(&mem, 0, sizeof(mem));
	
	// Check if the location is read-only 
	int returned = VirtualQueryEx(hProcess, (void *)offset, &mem, sizeof(mem));
	
	if (returned != sizeof(mem))
	{
		if (!silent)
		{
			msglog->log(LOG_ALWAYS, "Error: Invalid return value of VirtualQueryEx");
		}
			
		return PROTECTION_CHANGE_FAILED;
	}
	
	if((!(mem.Protect & PAGE_READWRITE)) || (mem.Protect & PAGE_GUARD))
	{
		if(!VirtualProtectEx(hProcess, (void*)offset, 4, PAGE_READWRITE, &oldProtection))
		{
			msglog->log(LOG_VERBOSE, "%s: VirtualProtectEx on offset %x failed (%d) !", __FUNCTION__, offset, GetLastError());
			return PROTECTION_CHANGE_FAILED;
		}
		else
		{
			return PROTECTION_CHANGED;
		}
	}
	
	return PROTECTION_CHANGE_UNNECESSARY;
}

bool isReadable(DWORD protect)
{
	DWORD lower = protect & 0xFF;
	
	return
		(lower & PAGE_EXECUTE_READ) ||
		(lower & PAGE_EXECUTE_READWRITE) ||
		(lower & PAGE_READONLY) ||
		(lower & PAGE_READWRITE);
}

DWORD getReadableProtection(DWORD protect)
{
	switch(protect & 0xFF)
	{
		case PAGE_EXECUTE: return PAGE_EXECUTE_READ;
		case PAGE_NOACCESS: return PAGE_READONLY;
		case PAGE_WRITECOPY: return PAGE_READWRITE;
		case PAGE_EXECUTE_WRITECOPY: return PAGE_EXECUTE_READWRITE;
	}
	
	return protect & 0xFF;
}

unsigned int makePageReadable(HANDLE hProcess, CPUADDRESS offset, DWORD& oldProtection, bool silent)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	MEMORY_BASIC_INFORMATION mem;
	memset(&mem, 0, sizeof(mem));
	
	// Check if the location is read-only 
	int returned = VirtualQueryEx(hProcess, (void *)offset, &mem, sizeof(mem));
	
	if (returned != sizeof(mem))
	{
		if (!silent)
		{
			msglog->log(LOG_ALWAYS, "Error: Invalid return value of VirtualQueryEx");
		}
			
		return PROTECTION_CHANGE_FAILED;
	}
	
	if(!isReadable(mem.Protect) || (mem.Protect & PAGE_GUARD))
	{
		if(!VirtualProtectEx(hProcess, (void*)offset, 4, getReadableProtection(mem.Protect), &oldProtection))
		{
			msglog->log(LOG_ALL, "%s: VirtualProtectEx on offset %x failed (%d) !", __FUNCTION__, offset, GetLastError());
			return PROTECTION_CHANGE_FAILED;
		}
		else
		{
			return PROTECTION_CHANGED;
		}
	}
	
	return PROTECTION_CHANGE_UNNECESSARY;
}

/**
* Restores the old protection level of a page of the target process memory.
*
* @param hProcess The handle of the target process.
* @param offset The offset whose page should be made restored.
* @param oldProtection The original protection level of the page.
**/
NaviError restoreMemoryProtection(HANDLE hProcess, CPUADDRESS offset, DWORD oldProtection)
{
	msglog->log(LOG_ALL, "%08X", oldProtection);
	
	DWORD oldProtection2 = 0;
	
	if(!VirtualProtectEx(hProcess, (void*)offset, 4, oldProtection, &oldProtection2))
	{
		msglog->log(LOG_VERBOSE, "%s: Resetting VirtualProtectEx state failed (%d) !", __FUNCTION__, GetLastError());
		return false;
	}
	
	return true;
}

/**
* Helper function that reads a single byte from the process memory of the target process.
*
* @param hProcess Process handle of the target process.
* @param offset Memory offset to read from.
* @param b The read byte is stored here.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError readByte(HANDLE hProcess, CPUADDRESS offset, char& b)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	SIZE_T writtenbytes = 0;
	
	return ReadProcessMemory(hProcess, (void *)offset, &b, 1, &writtenbytes) ?
		NaviErrors::SUCCESS :  NaviErrors::COULDNT_READ_MEMORY;
}

/**
* Helper function that writes a single byte to the process memory of the target process.
*
* @param hProcess Process handle of the target process.
* @param offset Memory offset to write to.
* @param b Byte to write to the target process memory.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError writeByte(HANDLE hProcess, CPUADDRESS offset, char b)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	DWORD oldProtection = 0;
	
	// Try to make the page writable for us
	unsigned int result = makePageWritable(hProcess, offset, oldProtection, false);
	
	if (result == PROTECTION_CHANGE_FAILED)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't make page read/writable");
		return NaviErrors::PAGE_NOT_WRITABLE;
	}
	
	SIZE_T writtenbytes = 0;
	
	// Write the byte to the target process memory
	if (!WriteProcessMemory(hProcess, (void *)offset, &b, 1, &writtenbytes))
	{
		msglog->log(LOG_VERBOSE, "WriteProcessMemory failed in %s ...", __FUNCTION__);
		return NaviErrors::COULDNT_WRITE_MEMORY;
	}
	
	if (result == PROTECTION_CHANGED)
	{
		restoreMemoryProtection(hProcess, offset, oldProtection);
	}
		
	return NaviErrors::SUCCESS;
}

NaviError writeBytes(HANDLE hProcess, CPUADDRESS offset, std::vector<char> data)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	DWORD oldProtection = 0;
	
	// Try to make the page writable for us
	unsigned int result = makePageWritable(hProcess, offset, oldProtection, false);
	
	if (result == PROTECTION_CHANGE_FAILED)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't make page read/writable");
		return NaviErrors::PAGE_NOT_WRITABLE;
	}
	
	SIZE_T writtenbytes = 0;
	
	// Write the byte to the target process memory
	if(!WriteProcessMemory(hProcess, (void *)offset, &data[0], data.size(), &writtenbytes))
	{
		msglog->log(LOG_VERBOSE, "WriteProcessMemory failed in %s ...", __FUNCTION__);
		return NaviErrors::COULDNT_WRITE_MEMORY;
	}
	
	if (result == PROTECTION_CHANGED)
	{
		restoreMemoryProtection(hProcess, offset, oldProtection);
	}
		
	return NaviErrors::SUCCESS;
}

/**
* Changes the step mode of the target process.
*
* @param threadID The thread ID of the thread whose step mode is changed.
* @param enterSingleStep If true, the trap flag is set. Else, it is cleared.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError changeStepMode(unsigned int threadId, bool enterSingleStep)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	CONTEXT	threadContext;

	// Get a handle to the thread we want to modify
	HANDLE thread = OpenThread(THREAD_ALL_ACCESS, FALSE, threadId);
	
	if (!thread)
	{
		msglog->log(LOG_ALWAYS, "Error: Opening the thread handle of thread %d failed.", threadId);
		
		return NaviErrors::COULDNT_READ_REGISTERS;
	}
	
	threadContext.ContextFlags = CONTEXT_FULL;
	
	// Try to get the thread context of the thread
	if (GetThreadContext(thread, &threadContext) == FALSE)
	{
		msglog->log(LOG_ALWAYS, "Error: GetThreadContext failed %s:%d.", __FUNCTION__, __LINE__);
		
		printLastError();
		
		CloseHandle(thread);
		
		return NaviErrors::COULDNT_READ_REGISTERS;
	}
	
	const unsigned int TRAP_FLAG = 0x100;
	
	// Modify the thread context depending on the parameter
	if (enterSingleStep)
	{
		threadContext.EFlags |= TRAP_FLAG;
	}
	else
	{
		threadContext.EFlags &= ~TRAP_FLAG;
	}

	// Write the new thread context back to the thread
	NaviError result = SetThreadContext(thread, &threadContext) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_WRITE_REGISTERS;
	
	CloseHandle(thread);
	
	return result;
}

/**
* @return True, if the function succeeds. False, otherwise.
**/
bool SetDebugPrivilege(HANDLE hProcess)
{
	// Function to set priviledges stolen from Dan Kurc

	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	LUID luid;
	TOKEN_PRIVILEGES privs;
	HANDLE hToken = NULL;
	DWORD dwBufLen = 0;
	char buf[1024] = {0};

	ZeroMemory(&luid,sizeof(luid));

	msglog->log(LOG_VERBOSE, "Looking up privilege value");
	
	if(!LookupPrivilegeValue(NULL, SE_DEBUG_NAME, &luid))
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't look up privilege value");
		return false;
	}

	privs.PrivilegeCount = 1;
	privs.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
	
	memcpy(&privs.Privileges[0].Luid, &luid, sizeof(privs.Privileges[0].Luid));

	msglog->log(LOG_VERBOSE, "Opening process token");
	
	if (!OpenProcessToken(hProcess, TOKEN_ALL_ACCESS,&hToken))
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't open process token");
		return false;
	}

	msglog->log(LOG_VERBOSE, "Adjusting token privilege");
	
	if (!AdjustTokenPrivileges(hToken, FALSE, &privs, sizeof(buf),(PTOKEN_PRIVILEGES)buf, &dwBufLen))
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't adjust token privileges");
		return false;
	}

	msglog->log(LOG_VERBOSE, "Successfully set debug privilege on target process");
	
	return true;
}

/**
* @return True, if the function succeeds. False, otherwise.
**/
bool WindowsSystem::Win32_resume(unsigned int tid, DWORD mode)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	msglog->log(LOG_VERBOSE, "Resuming thread 0x%X in process 0x%X with mode 0x%X", tid, getPID(), mode);
	
	if (!ContinueDebugEvent(getPID(), tid, mode))
	{
		LPVOID message;
		FormatMessage(
			FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, GetLastError(),
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
			(LPTSTR) &message,
			0, NULL);

		msglog->log(LOG_ALWAYS, "Error: ContinueDebugEvent() failed with error: %s\n", message); 

		LocalFree(message);
		return false;
	}
	
	return true;
}

/**
* Update internal list of threads and generate debug event accordingly.
**/
NaviError WindowsSystem::addThread(const Thread& thread)
{
	tids.push_back(thread);
	
	return threadCreated(thread.tid, RUNNING);
}

/**
* Handle the THREAD_CREATED debug event.
**/
void WindowsSystem::handleCreateThread(const DEBUG_EVENT& dbg_evt)
{
	msglog->log(LOG_VERBOSE, "Created new thread with TID %d", dbg_evt.dwThreadId);
	
	NaviError ctResult = addThread(Thread(dbg_evt.dwThreadId, RUNNING));
	
	if (ctResult)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't handle thread created event");
	}

	Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
}

void WindowsSystem::handleExitThread(const DEBUG_EVENT& dbg_evt)
{
	// Remove the thread from the internal lists
	tids.erase(std::remove_if(tids.begin(), tids.end(), ThreadComparator(dbg_evt.dwThreadId)), tids.end());
	
	// Tell the base system about the exited thread
	NaviError ctResult = threadExit(dbg_evt.dwThreadId);

	if (ctResult)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't handle thread exit event");
	}

	Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
}

/**
* Handle the CREATE_PROCESS_DEBUG_EVENT event and collect the initial information about the target process,
* i.e. the id of the main thread and the module of the process image.
*
* @param dbg_evet The debug event obtained from the debugging API
**/
void WindowsSystem::handleProcessStart(const DEBUG_EVENT& dbg_evt)
{
	Module module = getProcessModule(&dbg_evt.u.CreateProcessInfo);
	Thread thread = Thread(dbg_evt.dwThreadId, SUSPENDED);
	
	msglog->log(LOG_VERBOSE, "Created process at image base address 0x%X; initial thread has id 0x%X", module.baseAddress, thread.tid);
	
	processStart(module, thread);

	// also update the state of the module map and the thread list
	moduleMap.insert(std::make_pair<CPUADDRESS, Module>(module.baseAddress, module));
	tids.push_back(thread);
}

/**
* Handle a dll load event. The process needs to be suspended since BinNavi possibly needs to relocate the corresponding module
* or set breakpoints within the loaded dll.
**/
void WindowsSystem::handleLoadDll(const DEBUG_EVENT& dbg_evt)
{
	HANDLE handle = dbg_evt.u.LoadDll.hFile;
	
	std::string filename;
	
	if (GetFileNameFromHandle(handle, filename))
	{
		DWORD high;
		DWORD low = GetFileSize(handle, &high);
		
		CPUADDRESS size = (high << 8) + low;
	
		size_t lastSlash = filename.find_last_of('\\');
		
		Module module(lastSlash == std::string::npos ? filename : filename.substr(lastSlash + 1), filename, (CPUADDRESS)dbg_evt.u.LoadDll.lpBaseOfDll, size);
		
		addModule(module, dbg_evt.dwThreadId);
	}
	else
	{
		msglog->log(LOG_ALWAYS, "Error: Could not determine name of DLL file");
	}
	
	msglog->log(LOG_VERBOSE, "Breaking on dll load (%s) in thread 0x%X", filename.c_str(), dbg_evt.dwThreadId);
}

void WindowsSystem::handleUnloadDll(const DEBUG_EVENT& dbg_evt)
{
	std::map<CPUADDRESS, Module>::const_iterator cit = moduleMap.find((CPUADDRESS)dbg_evt.u.UnloadDll.lpBaseOfDll);
	
	if (cit == moduleMap.end())
	{
		// Note: this is not an error condition. On Win7 X64 systems, some modules are unloaded during process creation
		// which did not have a corresponding module loaded event.
		moduleUnloaded(Module("", "", (CPUADDRESS)dbg_evt.u.UnloadDll.lpBaseOfDll, 0));
	}
	else
	{
		moduleUnloaded(cit->second);
	}
}

/**
* Determines how to handle a given exception
*
* @return The debug handling action to be performed by the debugger
**/
DebugExceptionHandlingAction WindowsSystem::getNextExceptionAction(const DEBUG_EVENT& event) const
{
	std::map<CPUADDRESS, DebugExceptionHandlingAction>::const_iterator cit = exceptionSettings.find(event.u.Exception.ExceptionRecord.ExceptionCode);
	
	if (cit != exceptionSettings.end())
	{
		return cit->second;
	}
	
	// default action is to stop the debugger and let the user decide what to do
	return Halt;
}

/**
* Finds out whether debug events occurred in the target process.
*
* @return True if a debug event has been received. False if an error occurred.
**/
bool WindowsSystem::Win32_is_dbg_event_available()
{	
	DEBUG_EVENT dbg_evt;
	// Now process all debug events we can get
	while (WaitForDebugEvent(&dbg_evt, 10))
	{	
		nextContinue = DBG_CONTINUE;
		
		// Store the last thread that caused a debug event
		setActiveThread(dbg_evt.dwThreadId);
		
		// Find out what kind of event we're dealing with
		switch(dbg_evt.dwDebugEventCode)
		{
		case CREATE_PROCESS_DEBUG_EVENT:
			{
				msglog->log(LOG_VERBOSE, "\nReceived CREATE_PROCESS_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);

				try
				{
					handleProcessStart(dbg_evt);
				}
				catch (const std::runtime_error& e)
				{
					msglog->log(LOG_ALWAYS, "Error while extracting module information for the debuggee: %s", e.what());
				}
								
				CloseHandle(dbg_evt.u.CreateProcessInfo.hFile);
				
				// At this point, the debuggee is suspended and we wait until the cmd_resume message from BinNavi arrives.
				return true;
			}
		case EXCEPTION_DEBUG_EVENT:
			{
				// Handle exceptions
				
				msglog->log(LOG_VERBOSE, "\nReceived EXCEPTION_DEBUG_EVENT: code %08X at %08X from thread 0x%X.", dbg_evt.u.Exception.ExceptionRecord.ExceptionCode, dbg_evt.u.Exception.ExceptionRecord.ExceptionAddress, dbg_evt.dwThreadId);

				DWORD code = dbg_evt.u.Exception.ExceptionRecord.ExceptionCode;
				CPUADDRESS address = (CPUADDRESS)dbg_evt.u.Exception.ExceptionRecord.ExceptionAddress;			

				if (code == EXCEPTION_SINGLE_STEP)
				{
					// We are not handling this one quite yet. This workaround was introduced when
					// single-stepping in multi-threaded applications. The following thing can
					// happen there:
					//
					// 1. Single step Thread I -> Resume process to do so
					// 2. Thread II hits a breakpoint
					// 3. Thread II is single-stepped
					//
					// This must be fixed in future releases. What needs to happen is probably to
					// turn single-stepping from a synchronous operation into an asynchronous one.
					
					Win32_resume(dbg_evt.dwThreadId, DBG_CONTINUE);
					
					return true;
				}
				else if (code == EXCEPTION_BREAKPOINT)
				{
					if (!initialDebugBreakPassed)
					{						
						msglog->log(LOG_VERBOSE, "\nInitial debug event received - continuing process.");
					
						initialDebugBreakPassed = true;

						Win32_resume(dbg_evt.dwThreadId, DBG_CONTINUE);

						return false;
					}
					
					// Tell the base system that a breakpoint was hit
					NaviError hitResult = breakpointHit((CPUADDRESS)dbg_evt.u.Exception.ExceptionRecord.ExceptionAddress, dbg_evt.dwThreadId);
					
					if (hitResult)
					{
						// What could have happened here? Maybe a breakpoint was hit that wasn't
						// set by the debugger. TODO: Care more about error codes.
						
						msglog->log(LOG_ALWAYS, "Error: Breakpoint handler failed");
						
						// Tell the program that we couldn't handle the breakpoint exception
						Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
						
						continue;
					}
					
					return true;
				}
				else
				{
					switch (getNextExceptionAction(dbg_evt))
					{
					case Continue:
						// we pass the exception back to the process without sending a notification about the exception to BinNavi
						Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
						return false;

					case Halt:
						// we send a notification to BinNavi about this exception; moreover we need to set the continue state
						// for the current thread to DBG_EXCEPTION_NOT_HANDLED so the continue mode is correct when the user resumes the process
						exceptionRaised(dbg_evt.dwThreadId, address, code);
						threadContinueState.setThreadState(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
						return true;

					case Ignore:
						// we simply ignore, i.e. "swallow" the exception and let the process continue -> this only works
						// if the exception has no side effects, otherwise the exception occurs again as a 2nd chance exception and the process is killed
						Win32_resume(dbg_evt.dwThreadId, DBG_CONTINUE);
						return false;
					}
				}
			}
			case CREATE_THREAD_DEBUG_EVENT:
			{
				// Handle thread creation events
				msglog->log(LOG_VERBOSE, "\nReceived CREATE_THREAD_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);
				handleCreateThread(dbg_evt);
				return true;
			}
			case LOAD_DLL_DEBUG_EVENT:
			{
				// Handle load dll events
				msglog->log(LOG_VERBOSE, "\nReceived LOAD_DLL_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);
				handleLoadDll(dbg_evt);
				return true;
			}
			case UNLOAD_DLL_DEBUG_EVENT:
			{
				// Handle unload dll events
				msglog->log(LOG_VERBOSE, "\nReceived UNLOAD_DLL_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);
				handleUnloadDll(dbg_evt);
				return true;
			}
			case EXIT_THREAD_DEBUG_EVENT:
			{
				// Handle thread exit events
				msglog->log(LOG_VERBOSE, "\nReceived EXIT_THREAD_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);
				handleExitThread(dbg_evt);
				return true;
			}
			case EXIT_PROCESS_DEBUG_EVENT:
			{
				// Handle the event that the target process exited
				msglog->log(LOG_VERBOSE, "\nReceived EXIT_PROCESS_DEBUG_EVENT from thread 0x%X.", dbg_evt.dwThreadId);
				Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
				processExit();
				return true;
			}
			default:
			{
				// Unknown debug event
				msglog->log(LOG_VERBOSE, "\nReceived unknown debug event 0x%X from thread 0x%X.", dbg_evt.dwDebugEventCode, dbg_evt.dwThreadId);
				Win32_resume(dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
			}
		}
	}
	
	return false;
}

/**
* Attaches to the target process.
*
* @param tids The thread IDs of the threads that belong to the target process.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::attachToProcess()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	wasAttached = true;

	if(!SetDebugPrivilege(GetCurrentProcess()))
 	{
		msglog->log(LOG_ALWAYS, "Error: could not set debug privilege - aborting.");
		return NaviErrors::COULDNT_ENTER_DEBUG_MODE;
	}
	
	msglog->log(LOG_VERBOSE, "Opening process (PID: %d)", getPID());
		
	// Try to open the target process
	hProcess = OpenProcess(PROCESS_ALL_ACCESS , 0, getPID());
	
	if(hProcess == NULL)
	{
		msglog->log(LOG_ALWAYS, "Error: failed to OpenProcess() - aborting... (GetLastError: %d)", GetLastError());
		return NaviErrors::COULDNT_OPEN_TARGET_PROCESS;
	}
	
	msglog->log(LOG_VERBOSE, "Debugging process (PID: %d)", getPID());
	
	// Try to get debugging rights for the target process
	if (!DebugActiveProcess(getPID()))	
	{
		CloseHandle(hProcess);
		msglog->log(LOG_ALWAYS, "Error: DebugActiceProcess() failed - aborting.");
		return NaviErrors::COULDNT_DEBUG_TARGET_PROCESS;
	}

	return NaviErrors::SUCCESS;
}

/**
* Starts a new process for debugging.
*
* @param path The path to the executable of the process.
* @param tids The thread IDs of the threads that belong to the target process.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::startProcess(const NATIVE_STRING path, const std::vector<const char*>& arguments)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	wasAttached = false;

	if(!SetDebugPrivilege(GetCurrentProcess()))
 	{
		msglog->log(LOG_ALWAYS, "Error: could not set debug privilege - aborting.");
		return NaviErrors::COULDNT_ENTER_DEBUG_MODE;
	}
	
	msglog->log(LOG_VERBOSE, "Getting startup information");
	
	STARTUPINFO startupInfo;
	GetStartupInfo(&startupInfo);
	
	PROCESS_INFORMATION pi;
	
	std::string commandLine = path;
	
	commandLine += " ";
	
	for (unsigned int i=0;i<arguments.size();i++)
	{
		commandLine += arguments[i];
		commandLine += " ";
	}
	
	char* cmdl = new char[commandLine.length() + 1];
	
	strcpy(cmdl, commandLine.c_str());
	
	msglog->log(LOG_VERBOSE, "Starting %s with arguments %s", path, cmdl);
	
	// Create the target process
	if (!CreateProcess(0, cmdl, NULL, NULL, FALSE, DEBUG_PROCESS | DEBUG_ONLY_THIS_PROCESS, NULL, NULL, &startupInfo, &pi))
	{
		delete[] cmdl;
	
		msglog->log(LOG_ALWAYS, "Error: Couldn't start target process (Code: %d)", GetLastError());
		return NaviErrors::COULDNT_OPEN_TARGET_PROCESS;
	}
	
	delete[] cmdl;
	
	// Keep track of the process handle of the target process
	hProcess = pi.hProcess;
	setPID(pi.dwProcessId);
	
	// Keep track of the initial thread of the target process
	Thread ts(pi.dwThreadId, RUNNING);
	tids.push_back(ts);
	this->tids.push_back(ts);

	// Keep track of the last thread that caused a debug event
	setActiveThread(pi.dwThreadId);
	
	return NaviErrors::SUCCESS;
}

/**
* Detaches from the target process.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::detach()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	// Make sure to resume suspended processes before detaching
	// Otherwise it's going to crash
	resumeProcess();
	
	if (isWindowsXPOrLater())
	{
		typedef BOOL (WINAPI *d_DebugActiveProcessStop)(DWORD);
		typedef BOOL (WINAPI *d_DebugSetProcessKillOnExit)(BOOL);
		
		HMODULE kernel32 = GetModuleHandle("kernel32.dll");
		d_DebugActiveProcessStop f_DebugActiveProcessStop = (d_DebugActiveProcessStop) GetProcAddress(kernel32, "DebugActiveProcessStop");
		d_DebugSetProcessKillOnExit f_DebugSetProcessKillOnExit = (d_DebugSetProcessKillOnExit) GetProcAddress(kernel32, "DebugSetProcessKillOnExit");
		
		if (f_DebugActiveProcessStop && f_DebugSetProcessKillOnExit)
		{
			// Don't kill the process when detaching
			f_DebugSetProcessKillOnExit(false);
		
			// Stop receiving debug events from the target process
			f_DebugActiveProcessStop(getPID());
		}
	}
	
	return NaviErrors::SUCCESS;
}

/**
* Terminates the target process.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::terminateProcess()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	if (isWindowsXPOrLater())
	{
		typedef BOOL (WINAPI *d_DebugActiveProcessStop)(DWORD);
		
		HMODULE kernel32 = GetModuleHandle("kernel32.dll");
		d_DebugActiveProcessStop f_DebugActiveProcessStop = (d_DebugActiveProcessStop) GetProcAddress(kernel32, "DebugActiveProcessStop");
		
		if (f_DebugActiveProcessStop)
		{
			// Stop receiving debug events from the target process
			f_DebugActiveProcessStop(getPID());
		}
	}
	
	return TerminateProcess(hProcess, 0) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_TERMINATE_TARGET_PROCESS;
}

/**
* Stores the original data that is replaced by a given breakpoint.
* 
* @param bp The breakpoint in question.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::storeOriginalData(const BREAKPOINT& bp)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string addressString = cpuAddressToString(bp.addr);
	
	if (originalBytes.find(addressString) != originalBytes.end())
	{
		// Original data already stored
		return NaviErrors::SUCCESS;
	}
	
	char b;
	
	NaviError result = readByte(hProcess, bp.addr, b);
	
	if (result)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't read byte from address %s", addressString.c_str());
		return result;
	}
	
	originalBytes[addressString] = b;
	
	return NaviErrors::SUCCESS;
}

/**
* Sets a breakpoint in the target process.
*
* @param breakpoint The breakpoint to be set.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::setBreakpoint(BREAKPOINT& breakpoint, bool)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	const unsigned char BREAKPOINT_OPCODE = 0xCC;
	
	return writeByte(hProcess, breakpoint.addr, BREAKPOINT_OPCODE);
}

/**
* Removes a breakpoint from the target process.
*
* @param bp The breakpoint to be removed.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::removeBreakpoint(const BREAKPOINT& bp, bool)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string addressString = cpuAddressToString(bp.addr);
	
	if (originalBytes.find(addressString) == originalBytes.end())
	{
		msglog->log(LOG_ALWAYS, "Error: Trying to restore a breakpoint with unknown original byte");
		return NaviErrors::ORIGINAL_DATA_NOT_AVAILABLE;
	}
	
	char b = originalBytes[addressString];
	
	msglog->log(LOG_VERBOSE, "Writing byte %lx to address %s\n", b, addressString.c_str()); 
	
	return writeByte(hProcess, bp.addr, b);
}

/**
* The alreadyStepped set keeps track of all threads that were already single-
* stepped "recently".
*
* This is necessary because of the following scenario:
*
* - Thread A hits breakpoint
* - Thread A is stepped over the breakpoint
* - Before Thread A stepping is complete, Thread B hits breakpoint
* - Thread B is stepped over the breakpoint
* - Before Thread B stepping is complete, Thread A stepping is complete
*
* What has to happen now is that the step-loop of Thread A has to
* be stopped but Thread A does not know this because it was stepped in
* the step-loop of Thread B and the two loops can not communicate with
* each other directly. Therefore alreadyStepped is used.
*
* You can make this scenario arbitrarily difficult by adding more threads.
**/
std::set<unsigned int> alreadyStepped;

/**
* Executes a single instruction.
*
* @param tid The thread ID of the thread that executes the instruction.
* @param address The address of the instruction pointer after the single step.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::doSingleStep(unsigned int& tid, CPUADDRESS& address)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	for (std::vector<Thread>::iterator Iter = tids.begin(); Iter != tids.end(); ++Iter)
	{
		// Before single stepping, we suspend all threads but the one which should step.
		// We do this, so debug events from other threads do not interfere with out stepping thread.
		// After stepping, the other threads are resumed again.

		if (Iter->tid == tid)
		{
			// Set the trap flag in the target thread
			NaviError stepEnter = changeStepMode(Iter->tid, true);
	
			if (stepEnter)
			{
				msglog->log(LOG_ALWAYS, "Error: Couldn't enter single step mode");
				return NaviErrors::COULDNT_ENTER_SINGLE_STEP_MODE;
			}
		}
		else
		{
			NaviError suspendError = suspendThread(Iter->tid);
			if (suspendError)
			{
				msglog->log(LOG_ALWAYS, "Error: Unable to suspend thread (0x%X)\n");
				return suspendError;
			}
		}
	}
	
	// Execute the next instruction
	NaviError resumeResult = resumeProcess();
	
	if (resumeResult)
	{
		msglog->log(LOG_ALWAYS, "Error: Couldn't resume thread");
		return resumeResult;
	}

	DEBUG_EVENT dbg_evt;
	
	do
	{
		// Attempt to catch the next exception (single-step event)
		if (!WaitForDebugEvent(&dbg_evt, INFINITE))
		{
			msglog->log(LOG_ALWAYS, "Error: WaitForDebugEvent failed with error 0x%X", GetLastError());
			return NaviErrors::COULDNT_SINGLE_STEP;
		}
		
		// TODO: Can we single step to somewhere that closes/creates threads, terminates the program, ...?
	
		if (dbg_evt.dwDebugEventCode == EXIT_THREAD_DEBUG_EVENT)
		{
			handleExitThread(dbg_evt);
			
			continue;
		}
		else if (dbg_evt.dwDebugEventCode == CREATE_THREAD_DEBUG_EVENT)
		{
			handleCreateThread(dbg_evt);
			
			continue;
		}
		else if (dbg_evt.dwDebugEventCode == LOAD_DLL_DEBUG_EVENT)
		{
			handleLoadDll(dbg_evt);
			
			continue;
		}
		else if (dbg_evt.dwDebugEventCode == UNLOAD_DLL_DEBUG_EVENT)
		{
			handleUnloadDll(dbg_evt);
			
			return true;
		}
		else if (dbg_evt.dwDebugEventCode != EXCEPTION_DEBUG_EVENT)
		{
			msglog->log(LOG_ALWAYS, "Error: should've gotten EXCEPTION_DEBUG_EVENT, but got %d.", dbg_evt.dwDebugEventCode);
			
			return NaviErrors::COULDNT_SINGLE_STEP;
		}
		else
		{
			if (dbg_evt.u.Exception.ExceptionRecord.ExceptionCode == EXCEPTION_BREAKPOINT)
			{
				// It is possible for another thread to hit a breakpoint here. This needs to be
				// handled transparently.
				
				setActiveThread(dbg_evt.dwThreadId);
				// Tell the base system that a breakpoint was hit
				NaviError hitResult = breakpointHit((CPUADDRESS)dbg_evt.u.Exception.ExceptionRecord.ExceptionAddress, dbg_evt.dwThreadId);
				
				if (hitResult)
				{
					// What could have happened here? Maybe a breakpoint was hit that wasn't
					// set by the debugger. TODO: Care more about error codes.
					
					msglog->log(LOG_ALWAYS, "Error: Breakpoint handler failed");
					
					// Tell the program that we couldn't handle the breakpoint exception
					ContinueDebugEvent(dbg_evt.dwProcessId, dbg_evt.dwThreadId, DBG_EXCEPTION_NOT_HANDLED);
				}
				
				if (alreadyStepped.find(tid) != alreadyStepped.end())
				{
					alreadyStepped.erase(tid);
					break;
				}
			}
			else if (dbg_evt.u.Exception.ExceptionRecord.ExceptionCode != EXCEPTION_SINGLE_STEP)
			{
				msglog->log(LOG_ALWAYS, "Error: should've gotten EXCEPTION_SINGLE_STEP, but got %d.", dbg_evt.u.Exception.ExceptionRecord.ExceptionCode);
				
				return NaviErrors::COULDNT_SINGLE_STEP;
			}
			else if (dbg_evt.u.Exception.ExceptionRecord.ExceptionCode == EXCEPTION_SINGLE_STEP && dbg_evt.dwThreadId != tid)
			{
				// Single-stepped the wrong thread.
				alreadyStepped.insert(tid);
				Win32_resume(dbg_evt.dwThreadId, DBG_CONTINUE);
			}
		}
	} while (dbg_evt.u.Exception.ExceptionRecord.ExceptionCode != EXCEPTION_SINGLE_STEP || dbg_evt.dwThreadId != tid);

	// We need to resume all other threads besides the one which was single stepped.
	for (std::vector<Thread>::const_iterator cit = tids.begin(); cit != tids.end(); ++cit)
	{
		if (cit->tid != tid)
		{
			NaviError resumeError = resumeThread(cit->tid);
			if (resumeError)
			{
				msglog->log(LOG_ALWAYS, "Error: Unable to resume thread (0x%X)\n");
				return resumeError;
			}
			
		}
	}

	// Get the new address of the instruction pointer after the single step
	address = (CPUADDRESS)dbg_evt.u.Exception.ExceptionRecord.ExceptionAddress;
	
	return NaviErrors::SUCCESS;
}

/**
* Resumes the thread with the currently active thread ID.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::resumeProcess()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	// Tricky: We have to ignore tid here because we always
	// have to resume the process with the thread that caused
	// the last debug event.
	
	if (threadContinueState.hasThreadState(getActiveThread()))
	{
		DWORD continueAction = threadContinueState.getThreadState<DWORD>(getActiveThread());
		threadContinueState.removeThreadState(getActiveThread());
		return Win32_resume(getActiveThread(), continueAction) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_RESUME_THREAD;
	}
	else
	{
		return Win32_resume(getActiveThread(), nextContinue) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_RESUME_THREAD;
	}
}

NaviError WindowsSystem::suspendThread(unsigned int tid)
{
	HANDLE handle = OpenThread(THREAD_ALL_ACCESS, FALSE, tid);
	
	if (!handle)
	{
		return NaviErrors::COULDNT_SUSPEND_THREAD;
	}
	
	if (SuspendThread(handle) == -1)
	{
		return NaviErrors::COULDNT_SUSPEND_THREAD;
	}
	
	CloseHandle(handle);
	
	return NaviErrors::SUCCESS;
}

NaviError WindowsSystem::resumeThread(unsigned int tid)
{
	HANDLE handle = OpenThread(THREAD_ALL_ACCESS, FALSE, tid);
	
	if (!handle)
	{
		return NaviErrors::COULDNT_SUSPEND_THREAD;
	}
	
	if (ResumeThread(handle) == -1)
	{
		return NaviErrors::COULDNT_SUSPEND_THREAD;
	}
	
	CloseHandle(handle);
	
	return NaviErrors::SUCCESS;
}
		
/**
* Retrieves the value of the instruction pointer in a given thread.
*
* @param tid The thread ID of the thread.
* @param addr The variable where the value of the instruction pointer is stored.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::getInstructionPointer(unsigned int threadId, CPUADDRESS& addr)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	CONTEXT	threadContext;

	// Open the selected thread
	HANDLE thread = OpenThread(THREAD_ALL_ACCESS, FALSE, threadId);
	
	threadContext.ContextFlags = CONTEXT_FULL;
	
	// Read the values of the registers of the thread
	if (GetThreadContext(thread, &threadContext) == FALSE)
	{
		msglog->log(LOG_ALWAYS, "Error: GetThreadContext failed %s:%d.", __FUNCTION__, __LINE__);
		
		CloseHandle(thread);
		
		return NaviErrors::COULDNT_READ_REGISTERS;
	}
	
	// Select the instruction pointer
	#ifdef CPU_386
		addr = threadContext.Eip;
	#elif CPU_AMD64
		addr = threadContext.Rip;
	#else
		#error Unknown architecture
	#endif
	
	CloseHandle(thread);
		
	return NaviErrors::SUCCESS;
}

/**
* Sets the instruction pointer in the target process to a new value.
*
* @param tid Thread ID of the target thread.
* @param address The new value of the instruction pointer.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::setInstructionPointer(unsigned int tid, CPUADDRESS address)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	CONTEXT	threadContext;

	// Open the selected thread
	HANDLE thread = OpenThread(THREAD_ALL_ACCESS, FALSE, tid);
	
	threadContext.ContextFlags = CONTEXT_FULL;
	
	// Read the values of the registers of the thread
	if (GetThreadContext(thread, &threadContext) == FALSE)
	{
		msglog->log(LOG_ALWAYS, "Error: GetThreadContext failed %s:%d.", __FUNCTION__, __LINE__);
		
		CloseHandle(thread);
		
		return NaviErrors::COULDNT_READ_REGISTERS;
	}
	
	// Set the new instruction pointer address
	#ifdef CPU_386
		threadContext.Eip = address;
	#elif CPU_AMD64
		threadContext.Rip = address;
	#else
		#error Unknown architecture
	#endif
	
	// Set the new register values in the thread
	NaviError result = SetThreadContext(thread, &threadContext) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_WRITE_REGISTERS;
	
	CloseHandle(thread);

	return result;
}

std::vector<char> WindowsSystem::readPointedMemory(CPUADDRESS address)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	unsigned int currentSize = 128;
	
	while (currentSize != 0)
	{
		std::vector<char> memory(currentSize, 0);
		
		if (readMemoryDataInternal(&memory[0], address, memory.size(), true) == NaviErrors::SUCCESS)
		{
			return memory;
		}
	
		currentSize /= 2;
	}

	return std::vector<char>();
}

/**
* Fills a given register container structure with information about the
* current values of the CPU registers in the active thread. 
*
* @param registers The register information structure.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::readRegisters(RegisterContainer& registers)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);

	unsigned int threadId = getActiveThread();
	Thread thread = Thread(threadId, RUNNING);
	HANDLE hThread = OpenThread(THREAD_ALL_ACCESS, FALSE, threadId);

	if (!hThread)
	{
		DWORD lastError = GetLastError();
		if (isThreadDead(threadId))
		{
			// If opening the thread failed and it turns out that the thread didn't exist in our list of known threads
			// we conclude that the read registers message was sent for a thread which does not exist anymore.
			// Note: we perform the isThreadDead check after the API failed in order to prevent running into TOCTOU race conditions.
			return NaviErrors::SUCCESS;
		}
		else
		{
			msglog->log(LOG_ALWAYS, "Opening thread %d failed (Error: 0x%X)", threadId, lastError);
			return NaviErrors::COULDNT_OPEN_THREAD;
		}
	}

	CONTEXT ctx;
	ctx.ContextFlags = CONTEXT_FULL;

	// Read the values of the registers of the thread
	if(GetThreadContext(hThread, &ctx) == FALSE)
	{
		msglog->log(LOG_ALWAYS, "Error: %s: GetThreadContext failed (Code: %d, thread: 0x%X).", __FUNCTION__, GetLastError(), threadId);
		CloseHandle(hThread);

		return NaviErrors::COULDNT_READ_REGISTERS;
	}

	msglog->log(LOG_VERBOSE, "Assigning register values of thread %X", threadId);

#ifdef CPU_386
	thread.registers.push_back(makeRegisterValue("EAX", zylib::zycon::toHexString(ctx.Eax), readPointedMemory(ctx.Eax)));
	thread.registers.push_back(makeRegisterValue("EBX", zylib::zycon::toHexString(ctx.Ebx), readPointedMemory(ctx.Ebx)));
	thread.registers.push_back(makeRegisterValue("ECX", zylib::zycon::toHexString(ctx.Ecx), readPointedMemory(ctx.Ecx)));
	thread.registers.push_back(makeRegisterValue("EDX", zylib::zycon::toHexString(ctx.Edx), readPointedMemory(ctx.Edx)));
	thread.registers.push_back(makeRegisterValue("ESI", zylib::zycon::toHexString(ctx.Esi), readPointedMemory(ctx.Esi)));
	thread.registers.push_back(makeRegisterValue("EDI", zylib::zycon::toHexString(ctx.Edi), readPointedMemory(ctx.Edi)));
	thread.registers.push_back(makeRegisterValue("ESP", zylib::zycon::toHexString(ctx.Esp), readPointedMemory(ctx.Esp), false, true));
	thread.registers.push_back(makeRegisterValue("EBP", zylib::zycon::toHexString(ctx.Ebp), readPointedMemory(ctx.Ebp)));
	thread.registers.push_back(makeRegisterValue("EIP", zylib::zycon::toHexString(ctx.Eip), readPointedMemory(ctx.Eip), true));
	thread.registers.push_back(makeRegisterValue("EFLAGS", zylib::zycon::toHexString(ctx.EFlags)));
	thread.registers.push_back(makeRegisterValue("CF", zylib::zycon::toHexString(ctx.EFlags & 1)));
	thread.registers.push_back(makeRegisterValue("PF", zylib::zycon::toHexString((ctx.EFlags >> 2) & 1)));
	thread.registers.push_back(makeRegisterValue("AF", zylib::zycon::toHexString((ctx.EFlags >> 4) & 1)));
	thread.registers.push_back(makeRegisterValue("ZF", zylib::zycon::toHexString((ctx.EFlags >> 6) & 1)));
	thread.registers.push_back(makeRegisterValue("SF", zylib::zycon::toHexString((ctx.EFlags >> 7) & 1)));
	thread.registers.push_back(makeRegisterValue("OF", zylib::zycon::toHexString((ctx.EFlags >> 11) & 1)));
#elif CPU_AMD64
	thread.registers.push_back(makeRegisterValue("RAX", zylib::zycon::toHexString(ctx.Rax)));
	thread.registers.push_back(makeRegisterValue("RBX", zylib::zycon::toHexString(ctx.Rbx)));
	thread.registers.push_back(makeRegisterValue("RCX", zylib::zycon::toHexString(ctx.Rcx)));
	thread.registers.push_back(makeRegisterValue("RDX", zylib::zycon::toHexString(ctx.Rdx)));
	thread.registers.push_back(makeRegisterValue("RSI", zylib::zycon::toHexString(ctx.Rsi)));
	thread.registers.push_back(makeRegisterValue("RDI", zylib::zycon::toHexString(ctx.Rdi)));
	thread.registers.push_back(makeRegisterValue("RSP", zylib::zycon::toHexString(ctx.Rsp)));
	thread.registers.push_back(makeRegisterValue("RBP", zylib::zycon::toHexString(ctx.Rbp)));
	thread.registers.push_back(makeRegisterValue("RIP", zylib::zycon::toHexString(ctx.Rip)));
	thread.registers.push_back(makeRegisterValue("EFLAGS", zylib::zycon::toHexString(ctx.EFlags)));
#else
#error Unknown architecture
#endif

	CloseHandle(hThread);

	registers.addThread(thread);

	return NaviErrors::SUCCESS;
}

/**
* Updates the value of a given register in a given thread.
*
* @param tid The thread ID of the thread.
* @param index The index of the register.
* @param value The new value of the register.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::setRegister(unsigned int tid, unsigned int index, CPUADDRESS address)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	CONTEXT	threadContext;

	// Open the selected thread
	HANDLE thread = OpenThread(THREAD_ALL_ACCESS, FALSE, tid);
	
	threadContext.ContextFlags = CONTEXT_FULL;
	
	// Read the values of the registers of the thread
	if (GetThreadContext(thread, &threadContext) == FALSE)
	{
		msglog->log(LOG_ALWAYS, "Error: GetThreadContext failed %s:%d.", __FUNCTION__, __LINE__);
		return NaviErrors::COULDNT_READ_REGISTERS;
	}
	
	// Set the register
	#ifdef CPU_386
		switch(index)
		{
			case 0: threadContext.Eax = address; break;
			case 1: threadContext.Ebx = address; break;
			case 2: threadContext.Ecx = address; break;
			case 3: threadContext.Edx = address; break;
			case 4: threadContext.Esi = address; break;
			case 5: threadContext.Edi = address; break;
			case 6: threadContext.Esp = address; break;
			case 7: threadContext.Ebp = address; break;
			case 8: threadContext.Eip = address; break;
			case 10: threadContext.EFlags = (threadContext.EFlags & 0xFFFFFFFE) | (address & 1); break;
			case 11: threadContext.EFlags = (threadContext.EFlags & 0xFFFFFFFB) | ((address & 1) << 2); break;
			case 12: threadContext.EFlags = (threadContext.EFlags & 0xFFFFFFEF) | ((address & 1) << 4); break;
			case 13: threadContext.EFlags = (threadContext.EFlags & 0xFFFFFFBF) | ((address & 1) << 6); break;
			case 14: threadContext.EFlags = (threadContext.EFlags & 0xFFFFFF7F) | ((address & 1) << 7); break;
			case 15: threadContext.EFlags = (threadContext.EFlags & 0xFFFFF7FF) | ((address & 1) << 11); break;
			default: return NaviErrors::INVALID_REGISTER_INDEX;
		}
	#elif CPU_AMD64
		switch(index)
		{
			case 0: threadContext.Rax = address; break;
			case 1: threadContext.Rbx = address; break;
			case 2: threadContext.Rcx = address; break;
			case 3: threadContext.Rdx = address; break;
			case 4: threadContext.Rsi = address; break;
			case 5: threadContext.Rdi = address; break;
			case 6: threadContext.Rsp = address; break;
			case 7: threadContext.Rbp = address; break;
			case 8: threadContext.Rip = address; break;
			default: return NaviErrors::INVALID_REGISTER_INDEX;
		}
	#else
		#error Unknown architecture
	#endif
	
	// Write the new register value back to the thread
    NaviError result = SetThreadContext(thread, &threadContext) ? NaviErrors::SUCCESS : NaviErrors::COULDNT_WRITE_REGISTERS;
    
    CloseHandle(thread);

    return result;
}

/**
* Given a start address, this function returns the first and last offset of the
* memory region the start address belongs to.
*
* @param start The start address.
* @param from The first offset of the memory region.
* @param to The last offset of the memory region.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::getValidMemory(CPUADDRESS start, CPUADDRESS& from, CPUADDRESS& to)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	// TODO: Update page size for Itanium (GetSystemInfo())
	unsigned int PAGE_SIZE = 0x1000;
	
	CPUADDRESS startOffset = start & (~(PAGE_SIZE - 1));
	
	CPUADDRESS current = startOffset;
	
	CPUADDRESS low = (unsigned int)current;
	CPUADDRESS high = (unsigned int)current;
	
	MEMORY_BASIC_INFORMATION mem;
	
	for (;;)
	{
		if (!VirtualQueryEx(hProcess, (void*)current, &mem, sizeof(MEMORY_BASIC_INFORMATION)))
		{
			break;
		}
		
		if (mem.State != MEM_COMMIT)
		{
			break;
		}
		
		current -= PAGE_SIZE;
	}
	
	if (current == startOffset)
	{
		// No valid memory
		return NaviErrors::NO_VALID_MEMORY;
	}
	
	low = current + PAGE_SIZE;
	
	current = startOffset;
	
	for (;;)
	{
		if (!VirtualQueryEx(hProcess, (void*)current, &mem, sizeof(MEMORY_BASIC_INFORMATION)))
		{
			break;
		}
		
		if (mem.State != MEM_COMMIT)
		{
			break;
		}
		
		current += PAGE_SIZE;
	}
	
	high = current;
	
	from = low;
	to = high;
	
	return low != high ? NaviErrors::SUCCESS : NaviErrors::NO_VALID_MEMORY;
}

/**
* Returns a list of all memory regions that are available in the target process.
*
* @param addresses The memory map is written into this list.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::getMemmap(std::vector<CPUADDRESS>& addresses)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	MEMORY_BASIC_INFORMATION mem;
	
	CPUADDRESS offset = 0;
	
	unsigned int consecutiveRegions = 0;
		
	while (VirtualQueryEx(hProcess, (void*)offset, &mem, sizeof(mem)))
	{
		if (mem.State == MEM_COMMIT)
		{
			++consecutiveRegions;
			
			if (consecutiveRegions == 1)
			{
				msglog->log(LOG_ALL, "Found memory section between %X and %X", (CPUADDRESS)mem.BaseAddress, (CPUADDRESS)mem.BaseAddress + mem.RegionSize - 1);
				
				addresses.push_back((CPUADDRESS)mem.BaseAddress);
				addresses.push_back(((CPUADDRESS)mem.BaseAddress + mem.RegionSize - 1));
			}
			else
			{
				msglog->log(LOG_ALL, "Extending memory section to %X", addresses[addresses.size() - 1] + (CPUADDRESS)mem.RegionSize);
				
				addresses[addresses.size() - 1] += (CPUADDRESS)mem.RegionSize;
			}
		}
		else
		{
			consecutiveRegions = 0;
		}

		offset = (unsigned int)mem.BaseAddress + mem.RegionSize;
		
		if (offset == 0)
		{
			// Break if we get the overflow
			break;
		}
	}
	
	return NaviErrors::SUCCESS;
}

NaviError WindowsSystem::readMemoryDataInternal(char* buffer, CPUADDRESS address, CPUADDRESS size, bool silent)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
    SIZE_T        outlen;
    
    msglog->log(LOG_ALL, "Trying to read %d bytes from memory address %X", size, address);
  
	//      Make sure the entire region from addr to addr2 is paged  
	//  VirtualQuery(ntohl(addr->low32bits), & meminf, ntohl(addr2->low32bits) - nothl(addr->low32bits));

	DWORD oldProtection = 0;
	
	unsigned int result = makePageReadable(hProcess, address, oldProtection, silent);

	if (result == PROTECTION_CHANGE_FAILED)
	{
		return NaviErrors::COULDNT_READ_MEMORY;
	}

	if (ReadProcessMemory(hProcess, (void *)address, buffer, size, &outlen) == 0)
	{
		if (!silent)
		{
			msglog->log(LOG_ALWAYS, "Error: ReadProcessMemory failed (Error Code: %d)", GetLastError());
		}
		
		return NaviErrors::COULDNT_READ_MEMORY;
	}
	
	if (result == PROTECTION_CHANGED)
	{
		restoreMemoryProtection(hProcess, address, oldProtection);
	}

	return NaviErrors::SUCCESS;
}

/**
* Fills a buffer with memory data from the current process.
*
* @param buffer The buffer to fill.
* @param address The address from where the memory is read.
* @param size Number of bytes to read.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::readMemoryData(char* buffer, CPUADDRESS address, CPUADDRESS size)
{
	return readMemoryDataInternal(buffer, address, size, false);
}

NaviError WindowsSystem::writeMemory(CPUADDRESS address, const std::vector<char>& data)
{
	return writeBytes(hProcess, address, data);
}

NaviError WindowsSystem::readProcessList(ProcessListContainer& processList)
{
	HANDLE hSnapshot = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
	
	if (hSnapshot == INVALID_HANDLE_VALUE)
	{
		return NaviErrors::COULDNT_GET_PROCESSLIST;
	}
	
	PROCESSENTRY32 pe;
	pe.dwSize = sizeof(PROCESSENTRY32);
	
	BOOL retval = Process32First(hSnapshot, &pe);
	
	while (retval)
	{
		DWORD pid = pe.th32ProcessID;
		std::string name = pe.szExeFile;
		
		ProcessDescription process(pid, name);
		processList.push_back(process);
		
		pe.dwSize = sizeof(PROCESSENTRY32);
		retval = Process32Next(hSnapshot, &pe);
	}
	
	CloseHandle(hSnapshot);
	
	return NaviErrors::SUCCESS;
}

NaviError WindowsSystem::getAttachedProcessInfo(DWORD processId, std::string& exeName, std::string& exePath, CPUADDRESS& imageSize) const
{
	HANDLE hSnapshot = CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, processId);

	if (hSnapshot == INVALID_HANDLE_VALUE)
	{
		return NaviErrors::COULDNT_GET_PROCESSLIST;
	}

	MODULEENTRY32 me;
	me.dwSize = sizeof(MODULEENTRY32);

	BOOL retval = Module32First(hSnapshot, &me);

	while (retval)
	{
		if (processId == me.th32ProcessID)
		{
			exeName =me.szModule;
			exePath = me.szExePath;
			imageSize = me.modBaseSize;
			break;
		}

		me.dwSize = sizeof(MODULEENTRY32);
		retval = Module32Next(hSnapshot, &me);
	}

	CloseHandle(hSnapshot);

	return NaviErrors::SUCCESS;
}

/**
* Finds out whether debug events occurred in the target process.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
NaviError WindowsSystem::readDebugEvents()
{
	//msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);

	return Win32_is_dbg_event_available() ? NaviErrors::SUCCESS : NaviErrors::WAITING_FOR_DEBUG_EVENTS_FAILED;
}

/**
* Stops the debug client for a specified amount of time.
*
* @param millis Time in milliseconds.
**/
void WindowsSystem::sleep(unsigned int millis)
{
	Sleep(millis);
}

/**
 * Returns a list of the names of the registers of the underlying platform.
 *
 * @return A list of register names.
 **/
std::vector<RegisterDescription> WindowsSystem::getRegisterNames() const
{
	std::vector<RegisterDescription> regNames;
	
	#ifdef CPU_386
		RegisterDescription eax("EAX", 4, true);
		RegisterDescription ebx("EBX", 4, true);
		RegisterDescription ecx("ECX", 4, true);
		RegisterDescription edx("EDX", 4, true);
		RegisterDescription esi("ESI", 4, true);
		RegisterDescription edi("EDI", 4, true);
		RegisterDescription ebp("EBP", 4, true);
		RegisterDescription esp("ESP", 4, true);
		RegisterDescription eip("EIP", 4, true);
		RegisterDescription eflags("EFLAGS", 4, false);
		RegisterDescription cf("CF", 0, true);
		RegisterDescription pf("PF", 0, true);
		RegisterDescription af("AF", 0, true);
		RegisterDescription zf("ZF", 0, true);
		RegisterDescription sf("SF", 0, true);
		RegisterDescription of("OF", 0, true);
		
		regNames.push_back(eax);
		regNames.push_back(ebx);
		regNames.push_back(ecx);
		regNames.push_back(edx);
		regNames.push_back(esi);
		regNames.push_back(edi);
		regNames.push_back(esp);
		regNames.push_back(ebp);
		regNames.push_back(eip);
		regNames.push_back(eflags);
		regNames.push_back(cf);
		regNames.push_back(pf);
		regNames.push_back(af);
		regNames.push_back(zf);
		regNames.push_back(sf);
		regNames.push_back(of);
	#elif CPU_AMD64
		RegisterDescription rax("RAX", 8, false);
		RegisterDescription rbx("RBX", 8, false);
		RegisterDescription rcx("RCX", 8, false);
		RegisterDescription rdx("RDX", 8, false);
		RegisterDescription rsi("RSI", 8, false);
		RegisterDescription rdi("RDI", 8, false);
		RegisterDescription rbp("RBP", 8, false);
		RegisterDescription rsp("RSP", 8, false);
		RegisterDescription rip("RIP", 8, false);
		RegisterDescription eflags("EFLAGS", 4, false);
		
		regNames.push_back(rax);
		regNames.push_back(rbx);
		regNames.push_back(rcx);
		regNames.push_back(rdx);
		regNames.push_back(rsi);
		regNames.push_back(rdi);
		regNames.push_back(rsp);
		regNames.push_back(rbp);
		regNames.push_back(rip);
		regNames.push_back(eflags);
	#else
		#error Unknown architecture
	#endif
	
	return regNames;
}

/**
* Returns the maximum size of a memory address of the target machine.
*
* @return The maximum size of a memory address of the target machine.
**/
unsigned int WindowsSystem::getAddressSize() const
{
	#ifdef CPU_386
		return 32;
	#elif CPU_AMD64 || CPU_IA64
		return 64;
	#else
		#error Unknown architecture
	#endif
}

/**
* Returns the debugger options that are supported by the debug client.
*
* @return The debugger options that are supported by the debug client.
**/
DebuggerOptions WindowsSystem::getDebuggerOptions() const
{
	DebuggerOptions empty;
	
	empty.canDetach = isWindowsXPOrLater() == TRUE ? true : false;
	empty.canMultithread = true;
	empty.pageSize = 4096;
	empty.exceptions = getPlatformExceptions();
	empty.canTraceCount = true;

	return empty;
}

namespace
{
	const DebugException ExceptionsArray[] = {
		
		DebugException("Access violation", EXCEPTION_ACCESS_VIOLATION, Halt),
		DebugException("Illegal instruction", EXCEPTION_ILLEGAL_INSTRUCTION, Halt),
		DebugException("Privileged instruction", EXCEPTION_PRIV_INSTRUCTION, Halt),
		DebugException("Integer division by zero", EXCEPTION_INT_DIVIDE_BY_ZERO, Halt),
		DebugException("Integer overflow", EXCEPTION_INT_OVERFLOW, Halt),
		DebugException("Stack overflow", EXCEPTION_STACK_OVERFLOW, Halt),
		DebugException("Guard page", EXCEPTION_GUARD_PAGE, Halt),
		DebugException("Non-continuable exception", EXCEPTION_NONCONTINUABLE_EXCEPTION, Halt),
		DebugException("Floating point division by zero", EXCEPTION_FLT_DIVIDE_BY_ZERO, Halt),
		DebugException("Floating point invalid operation", EXCEPTION_FLT_INVALID_OPERATION, Halt),
		DebugException("Invalid handle closed", EXCEPTION_INVALID_HANDLE, Halt),
		DebugException("Microsoft Visual C++ Exception", 0xE06D7363, Halt)
	};

	DebugExceptionContainer exceptions(ExceptionsArray, ExceptionsArray + sizeof(ExceptionsArray) / sizeof(DebugException));
}

/**
* Returns the list of exceptions that are supported by the debug client
*
* @return The list of exceptions that are supported by the debug client
**/
DebugExceptionContainer WindowsSystem::getPlatformExceptions() const
{
	return exceptions;
}

/**
* Create a module instance for the executable image which was launched by the debug API.
*
* @param dbgInfo The structure which was obtained in the debug event CREATE_PROCESS_DEBUG_EVENT
*
* @return A new Module instance for the executable image.
*
* @throws std::runtime_error exception is the PE header of the executable was not accessible.
**/
Module WindowsSystem::getProcessModule(const CREATE_PROCESS_DEBUG_INFO* dbgInfo) const
{
	if (wasAttached)
	{
		std::string path, name;
		CPUADDRESS imageSize = 0;
		getAttachedProcessInfo(getPID(), name, path, imageSize);
		return Module(name, path, (CPUADDRESS)dbgInfo->lpBaseOfImage, imageSize);
	}
	else
	{
		CPUADDRESS imageBase = (CPUADDRESS)dbgInfo->lpBaseOfImage;
	
		NTHeader ntHeader(getTargetApplicationPath().string());
		CPUADDRESS imageSize = ntHeader.getImageSize();	

		return Module(getTargetApplicationPath().filename().string(), getTargetApplicationPath().string(), imageBase, imageSize);
	}
}

/**
* Returns a list of available file system roots, i.e. the available drives.
**/
NaviError WindowsSystem::getFileSystems(std::vector<boost::filesystem::path>& roots) const
{
	return windowscommon::getFileSystems(roots);
}

/**
* Get the path of the system drive (usually C:/).
**/
NaviError WindowsSystem::getSystemRoot(boost::filesystem::path& root) const
{
	return windowscommon::getSystemRoot(root);
}

/**
* Add module to internal module map and generate debug event.
**/
void WindowsSystem::addModule(const Module& module, unsigned int threadId)
{
	moduleMap.insert(std::make_pair<CPUADDRESS, Module>(module.baseAddress, module));
	moduleLoaded(module, threadId);
}

/**
* Test if the thread represented by the given thread id is not existing in the debuggee.
* Note: this method is inherently vulnerable to a time-of-check-to-time-of-use race condition
* so it must be used with care.
**/
bool WindowsSystem::isThreadDead(unsigned int threadId) const
{
	BOOST_FOREACH(const Thread& t, tids)
	{
		if (t.tid == threadId) return false;
	}

	return true;
}