
// IDA Pro plugin to load function name information from PDB files

// This file was originally by John Robbins
// I added some code to interface it with IDA Pro
// and deleted some code unused by the plugin. Ilfak Guilfanov.

/*----------------------------------------------------------------------
       Bugslayer Column - April '00 MSDN Magazine - John Robbins
----------------------------------------------------------------------*/

/*//////////////////////////////////////////////////////////////////////
                           Necessary Includes
//////////////////////////////////////////////////////////////////////*/

#include "stdafx.h"
#include <imagehlp.h>

// We define this if the dbghelp.h file is not a newer version 
#ifndef SYMF_REGISTER
#define SymDia		7
#endif

// Because some systems could not to have IMAGEHLP.DLL, we are not going
// to link its functions statically.
// Instead, we use LoadLibrary and the following pointers:


typedef
BOOL IMAGEAPI t_SymLoadModule(
    IN  HANDLE          hProcess,
    IN  HANDLE          hFile,
    IN  PSTR            ImageName,
    IN  PSTR            ModuleName,
    IN  DWORD           BaseOfDll,
    IN  DWORD           SizeOfDll
    );

typedef
BOOL IMAGEAPI t_SymEnumerateSymbols(
    IN HANDLE                       hProcess,
    IN DWORD                        BaseOfDll,
    IN PSYM_ENUMSYMBOLS_CALLBACK    EnumSymbolsCallback,
    IN PVOID                        UserContext
    );

typedef 
BOOL IMAGEAPI t_SymGetModuleInfo(
    IN  HANDLE              hProcess,
    IN  DWORD               dwAddr,
    OUT PIMAGEHLP_MODULE    ModuleInfo
    );


typedef
BOOL IMAGEAPI t_SymInitialize(
    IN HANDLE   hProcess,
    IN LPSTR    UserSearchPath,
    IN BOOL     fInvadeProcess
    );


typedef
DWORD IMAGEAPI t_SymSetOptions(
    IN DWORD   SymOptions
    );

t_SymLoadModule       *p_SymLoadModule;
t_SymEnumerateSymbols *p_SymEnumerateSymbols;
t_SymGetModuleInfo    *p_SymGetModuleInfo;
t_SymInitialize       *p_SymInitialize;
t_SymSetOptions       *p_SymSetOptions;


unsigned long delta;            // relocation amount

#include "SymbolEngine.h"
#include "Array.h"

#include "array.cpp"

#include <ida.hpp>
#include <idp.hpp>
#include <name.hpp>
#include <loader.hpp>
#include <diskio.hpp>

/*//////////////////////////////////////////////////////////////////////
                    File Scope Typedefs and Classes
//////////////////////////////////////////////////////////////////////*/

// A function information structure filled in when enumerating the
// public functions.
typedef struct tag_FUNCINFO
{
    // The function address.
    ULONG ulAddr ;
    // The size of the function.
    ULONG ulSymSize ;
    // The name of the function.
    TCHAR szName [ MAX_PATH ] ;
} FUNCINFO , * LPFUNCINFO ;

/*//////////////////////////////////////////////////////////////////////
                   File Scope Prototypes and Inlines
//////////////////////////////////////////////////////////////////////*/
// Enumerates the functions in the symbol engine.
BOOL CALLBACK FunctionEnumeration ( PSTR  szName   ,
                                    ULONG ulAddr   ,
                                    ULONG ulSize   ,
                                    PVOID pContext  ) ;
// Compares function information structures.
int CompareFuncInfos ( LPFUNCINFO p1 , LPFUNCINFO p2 ) ;

// Load IMAGEHLP.DLL
bool SetupPointers(void);

/*//////////////////////////////////////////////////////////////////////
                           File Scope Globals
//////////////////////////////////////////////////////////////////////*/
// The function information structure.
static FASTARRAYINFO g_aFuncArray =
{
    NULL                         ,  // pArray
    0                            ,  // ulTotalSlots
    0                            ,  // ulActiveSlots
    100                          ,  // ulReallocUnits
    sizeof ( FUNCINFO )          ,  // ulStructSize
    (PFNCOMPARE)CompareFuncInfos ,  // pfnCompare
    0                               // ulCurrIterator
} ;


// Handle of IMAGEHLP.DLL
static HINSTANCE imagehlp = NULL;

/*//////////////////////////////////////////////////////////////////////
                  EXTERNALLY VISIBLE CODE STARTS HERE
//////////////////////////////////////////////////////////////////////*/

//----------------------------------------------------------------------
static const char *get_sym_type(int symtype)
{
  switch ( symtype )
  {
    case SymNone:       return "SymNone";
    case SymCoff:       return "SymCoff";
    case SymCv:         return "SymCv";
    case SymPdb:        return "SymPdb";
    case SymExport:     return "SymExport";
    case SymDeferred:   return "SymDeferred";
    case SymSym:        return "SymSym";
	case SymDia:		return "SymDia";
    default:            return "Unknown";
  }
}

/*----------------------------------------------------------------------
FUNCTION        :   plugin_main
DISCUSSION      :
    The entry point for this plugin
PARAMETERS      :
    argc - The number of commmand line arguments
    argv - The array of commmand line arguments
RETURNS         :
    0 - Life is good and everything processed fine.
    1 - There was a general problem.
    2 - The module either did not have symbols or did not have symbols
        that supported line information.
----------------------------------------------------------------------*/
void plugin_main(int)
{
    if ( !SetupPointers() ) return;

    // Calculate the relocation amount (usually 0 since we load
    // the files at their ImageBase)
    {
      delta = 0;
      netnode penode("$ PE header");
      ea_t loaded_base = penode.altval(-2);
      IMAGE_NT_HEADERS *pe = (IMAGE_NT_HEADERS *)penode.value();
      if ( pe != NULL && loaded_base != 0 )
        delta = loaded_base - pe->OptionalHeader.ImageBase;
    }  
    // Get the input file name and try to guess the PDB file locaton
    // If failed, ask the user
    char *input = get_input_file_path();
    if ( !qfileexist(input)
      && (input=askfile_c(false, input, "Please specify the input file")) == NULL )
        return;

    // Now that all the command line stuff is done, it's time to try
    // and load the module into the symbol engine.

    CSymbolEngine g_cSym ;
    // The handle to the module use to load the symbols.
    HANDLE hFile = INVALID_HANDLE_VALUE ;
    try
    {
        p_SymSetOptions(SYMOPT_LOAD_LINES/*SYMOPT_UNDNAME*/);
        // Initialize the symbol engine class.
        show_wait_box("Loading symbols");
        msg("Initializing the symbol engine.\n");
        if ( FALSE == g_cSym.SymInitialize ( (void*)0xBEEFFEED ,
                                             NULL              ,
                                             FALSE              ) )
            throw "The symbol engine failed to initialize.";

        // Open the file so the symbol engine can load it.
        msg("Preparing to open the input file %s\n", input);
        hFile = CreateFile ( input                  ,
                             GENERIC_READ           ,
                             FILE_SHARE_READ        ,
                             NULL                   ,
                             OPEN_EXISTING          ,
                             FILE_ATTRIBUTE_NORMAL  ,
                             NULL                    ) ;
        if ( INVALID_HANDLE_VALUE == hFile )
        {
            warning("Unable to open %s", input);
            throw 0 ;
        }

        // Try and load the symbols
        msg("Loading the symbols from %s\n", input);
        DWORD dwBase = g_cSym.SymLoadModule ( hFile           ,
                                              input           ,
                                              NULL            ,
                                              0               ,
                                              0                ) ;
        if ( 0 == dwBase )
        {
            warning("Unable to load symbols for %s (code %d)", input, GetLastError());
            throw 0;
        }
        msg("Symbols are loaded at image base %08lX\n", dwBase);

        // Double check that the symbols loaded are either C7 (CodeView)
        // or PDB.
        CImageHlp_Module cIM ;

        msg("Getting the module information from the symbol engine.\n");
        if ( FALSE == g_cSym.SymGetModuleInfo ( dwBase , &cIM ) )
        {
            warning("Unable to get the module information for %s "
                    "out of the symbol engine (code %d).\n",
                     input, GetLastError());
            throw 0;
        }
        if ( ( SymCv != cIM.SymType   ) &&
             ( SymPdb != cIM.SymType  ) &&
			 ( SymDia != cIM.SymType  ))
        {
            warning("WARNING: '%s' may not have a valid PDB-style symbol format (symtype is %s [%d]).\n",
                    input, get_sym_type(cIM.SymType), cIM.SymType);

        }

        // The symbols are loaded so I can start the fun.

        msg("Starting to enumerate functions.\n");
        BOOL bEnumRet =
            g_cSym.SymEnumerateSymbols( dwBase              ,
                                        FunctionEnumeration ,
                                        NULL                 ) ;
        if ( !bEnumRet )
          throw "Function enumeration failed.\n";

        // Before getting too carried away, check that there were some
        // functions even there.
        if ( 0 == g_aFuncArray.ulActiveSlots )
            throw "No function information found.\n";

        // Save debug information in the IDA database

        for ( LPFUNCINFO pFInfo =
                    (LPFUNCINFO)GetFirstArrayItem ( &g_aFuncArray )   ;
              NULL != pFInfo                                          ;
              pFInfo = (LPFUNCINFO)GetNextArrayItem ( &g_aFuncArray )  )
        {
            // Relocate the address
            ea_t ea = pFInfo->ulAddr + delta;
            // Tell IDA kernel: rename the function
            set_name(ea, pFInfo->szName, SN_NOWARN);
            // Tell IDA kernel: create the function
            add_func(ea, pFInfo->ulAddr+pFInfo->ulSymSize);
        }

        // That's it for the function information.  I will go ahead and
        // free up the function memory as I no longer need it and
        // there's no sense wasting the working set.
        ClearAllArrayInfo ( &g_aFuncArray ) ;
        msg("Debug information is loaded.\n");

    }
    catch ( const char *errmsg )
    {
        // Yikes!  Some sort of problem cropped up along the way.
        if ( errmsg != NULL )
        {
            // Tell the user.
            warning("%s (code %d)", errmsg, GetLastError());
        }

    }

    hide_wait_box();
    // Cleanup any open and allocated stuff.
    if ( INVALID_HANDLE_VALUE != hFile )
    {
//        g_cSym.SymCleanup();
        CloseHandle ( hFile ) ;
    }
    // Get rid of the func array if need be.
    if ( NULL != g_aFuncArray.pArray )
    {
        ClearAllArrayInfo ( &g_aFuncArray ) ;
    }
}

/*//////////////////////////////////////////////////////////////////////
                      FILE SCOPE CODE STARTS HERE
//////////////////////////////////////////////////////////////////////*/

/*----------------------------------------------------------------------
FUNCTION        :   FunctionEnumeration
DISCUSSION      :
    The function enumeration callback used to enumerate all the
    functions in the loaded module and plop them into the global
    g_cFInfo array.
PARAMETERS      :
    szName   - The function name.
    ulAddr   - The function address.
    ulSize   - The size of the function.
    pContext - The user supplies context.
RETURNS         :
    Always TRUE to continue the enumeration.
----------------------------------------------------------------------*/
BOOL CALLBACK FunctionEnumeration ( PSTR  szName       ,
                                    ULONG ulAddr       ,
                                    ULONG ulSize       ,
                                    PVOID /*pContext*/  )
{
    // All I need to do here is to plop the individual functions into
    // the main vector.
    FUNCINFO stFunc ;
    stFunc.ulAddr = ulAddr ;
    stFunc.ulSymSize = ulSize ;
    _tcscpy ( stFunc.szName , szName ) ;

    AddArrayItem ( &g_aFuncArray , (BYTE*)&stFunc ) ;
    return ( TRUE ) ;
}


/*----------------------------------------------------------------------
FUNCTION        :   CompareFuncInfos
DISCUSSION      :
    Takes care of comparing function info structure.
PARAMETERS      :
    The two func infos to compare.
RETURNS         :
    Same as strcmp.
----------------------------------------------------------------------*/
int CompareFuncInfos ( LPFUNCINFO p1 , LPFUNCINFO p2 )
{
    if ( p1->ulAddr < p2->ulAddr )
    {
        return ( -1 ) ;
    }
    else if ( p1->ulAddr == p2->ulAddr )
    {
        return ( 0 ) ;
    }
    return ( 1 ) ;
}

//----------------------------------------------------------------------
bool SetupPointers(void)
{
  // Since there could be no IMAGEHLP.DLL on the system, we link to
  // the functions at run-time. Usually this is not necessary.
  imagehlp = LoadLibrary("IMAGEHLP.DLL");
  if ( imagehlp == NULL )
  {
    deb(IDA_DEBUG_PLUGIN, "PDB plugin: failed to load IMAGEHLP.DLL");
    return false;         // There is no imagehlp.dll
                          // in this system...
  }
  p_SymLoadModule       = (t_SymLoadModule      *)(GetProcAddress(imagehlp,"SymLoadModule"));
  p_SymEnumerateSymbols = (t_SymEnumerateSymbols*)(GetProcAddress(imagehlp,"SymEnumerateSymbols"));
  p_SymGetModuleInfo    = (t_SymGetModuleInfo   *)(GetProcAddress(imagehlp,"SymGetModuleInfo"));
  p_SymInitialize       = (t_SymInitialize      *)(GetProcAddress(imagehlp,"SymInitialize"));
  p_SymSetOptions       = (t_SymSetOptions      *)(GetProcAddress(imagehlp,"SymSetOptions"));
  if ( p_SymLoadModule       == NULL
    || p_SymEnumerateSymbols == NULL
    || p_SymGetModuleInfo    == NULL
    || p_SymInitialize       == NULL
    || p_SymSetOptions       == NULL )
  {
    deb(IDA_DEBUG_PLUGIN, "PDB plugin: Essential IMAGEHLP.DLL functions are missing\n");
    FreeLibrary(imagehlp);
    return false;
  }
  return true;
}

/*//////////////////////////////////////////////////////////////////////
                      IDA PRO INTERFACE START HERE
//////////////////////////////////////////////////////////////////////*/
//--------------------------------------------------------------------------
//
//      initialize plugin
//
//      IDA will call this function only once.
//      If this function returns PLGUIN_SKIP, IDA will never load it again.
//      If this function returns PLUGIN_OK, IDA will unload the plugin but
//      remember that the plugin agreed to work with the database.
//      The plugin will be loaded again if the user invokes it by
//      pressing the hotkey or selecting it from the menu.
//      After the second load the plugin will stay on memory.
//
//      In this example we check the input file format and make the decision.
//      You may or may not check any other conditions to decide what you do:
//      whether you agree to work with the database or not.
//

int init(void)
{
  if ( inf.filetype != f_PE ) return PLUGIN_SKIP; // only for PE files
  if ( !SetupPointers() ) return PLUGIN_SKIP;
  return PLUGIN_OK;
}

//--------------------------------------------------------------------------
//      terminate
//      usually this callback is empty
//
//      IDA will call this function when the user asks to exit.
//      This function won't be called in the case of emergency exits.

void term(void)
{
  FreeLibrary(imagehlp);
}

char comment[] = "Load debug information from a PDB file";

char help[] =
"PDB file loader\n"
"\n"
"This module allows you to load debug information about function names\n"
"from a PDB file.\n"
"\n"
"The PDB file should be in the same directory as the input file\n";


//--------------------------------------------------------------------------
// This is the preferred name of the plugin module in the menu system
// The preferred name may be overriden in plugins.cfg file

char wanted_name[] = "Load PDB file";


// This is the preferred hotkey for the plugin module
// The preferred hotkey may be overriden in plugins.cfg file
// Note: IDA won't tell you if the hotkey is not correct
//       It will just disable the hotkey.

char wanted_hotkey[] = ""; // Ctrl-F12 is used to draw function call graph now
                           // No hotkey for PDB files, but since it is not
                           // used very often, it is tolerable

//--------------------------------------------------------------------------
//
//      PLUGIN DESCRIPTION BLOCK
//
//--------------------------------------------------------------------------

extern "C" plugin_t PLUGIN =
{
  IDP_INTERFACE_VERSION,
  0,                    // plugin flags
  init,                 // initialize

  term,                 // terminate. this pointer may be NULL.

  plugin_main,          // invoke plugin

  comment,              // long comment about the plugin
                        // it could appear in the status line
                        // or as a hint

  help,                 // multiline help about the plugin

  wanted_name,          // the preferred short name of the plugin
  wanted_hotkey         // the preferred hotkey to run the plugin
};

