/*  sprocryp.h
   (C) Copyright 1986-1994 Rainbow Technologies, Inc.  All rights reserved. */

/*============================ V a r i a b l e s ============================*/

static unsigned char      RNBO_FPTR *CodePtr; /* Ptr to byte to encrypt/decrypt */
static unsigned short int RNBO_FPTR *wordPtr; /* Ptr to a word                  */
static unsigned char           tXseed;  /* Temp for encrypt/decrypt seed     */
static unsigned char           tSwap;   /* Temp byte holder for Swap method  */
static unsigned short int      tCount;  /* Temp for Xcount                   */
/*===========================================================================*/

#define DecryptX CryptX                /* Decrypt/Encrypt for Xor are same   */
#define DecryptS CryptS                /* Decrypt/Encrypt for Swap are same  */


/*****************************************************************************/
/* This code in this Macro is commom to the other Macros.                    */
/* This macro sets up the variable tCount to contain the number of bytes of  */
/* the SuperPro driver that are to be encrypted. This is done as follows:    */
/*   If Xcount is not 0, then tCount is just a copy of Xcount.               */
/*   If Xcount is 0, tCount is the number of bytes between Xstart and the    */
/*   last byte of the driver including both of those bytes.                  */
/*****************************************************************************/
#define DoMacroCore(Xstart, Xcount)                                                   \
  wordPtr = ((unsigned short int RNBO_FPTR *)SUPERPRO) - 2;                         \
                   /* Point 2 words (4 bytes) before the SuperPro entry    */ \
                   /* point. This word is used to decrypt the next word    */ \
                   /* in order to get the length of the SuperPro driver.   */ \
                                                                              \
  tCount = *wordPtr;    /* Use this value to decrypt the SuperPro length.  */ \
  wordPtr++;            /* Advance 1 word (to the SuperPro length).        */ \
  tCount ^= *wordPtr;   /* tCount is now the length of the SuperPro driver */ \
                                                                              \
  if (Xcount == 0)                                                            \
  {                     /* If 0, get the number of bytes from Xstart to    */ \
    tCount -= Xstart;   /* the end of the SuperPro driver (inclusive).     */ \
                        /* Note: we do not need to add 1 to this result    */ \
                        /* because tCount is 1-based and Xstart is 0-based.*/ \
  }                                                                           \
  else                                                                        \
  {                                                                           \
    tCount = Xcount;    /* Not 0, so tCount gets what the user gave us     */ \
  }

/*****************************************************************************/
/* Macro for Decrypting and Encrypting, by Xoring each byte with a seed.     */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define CryptX(Xseed, Xmod, Xstart, Xcount)                                   \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    *CodePtr ^= tXseed; /* Encrypt byte in SuperPro driver.                */ \
    CodePtr++;          /* Point to next byte.                             */ \
    tXseed += Xmod;     /* Modify seed.                                    */ \
    tCount--;           /* Decrement bytes remaining to be encrypted       */ \
  }


/*****************************************************************************/
/* Macro for Encrypting, by adding a seed to each byte.                      */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define CryptA(Xseed, Xmod, Xstart, Xcount)                                   \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    *CodePtr += tXseed; /* Encrypt byte in SuperPro driver.                */ \
    CodePtr++;          /* Point to next byte.                             */ \
    tXseed += Xmod;     /* Modify seed.                                    */ \
    tCount--;           /* Decrement bytes remaining to be encrypted       */ \
  }


/*****************************************************************************/
/* Macro for Encrypting, by Rotating each byte left 1 bit or right 1 bit.    */
/* The direction depends upon the current seed value.                        */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define CryptR(Xseed, Xmod, Xstart, Xcount)                                   \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    if ((tXseed & 1) == 0)                                                    \
    {                   /* If tXseed is EVEN then rotate the bits right    */ \
      *CodePtr = (*CodePtr >> 1) | (*CodePtr << 7);                           \
    }                                                                         \
    else                                                                      \
    {                   /* If tXseed is ODD then rotate the bits left      */ \
      *CodePtr = (*CodePtr << 1) | (*CodePtr >> 7);                           \
    }                                                                         \
                                                                              \
    CodePtr++;          /* Point to next byte.                             */ \
    tXseed += Xmod;     /* Modify seed.                                    */ \
    tCount--;           /* Decrement bytes remaining to be encrypted       */ \
  }


/*****************************************************************************/
/* Macro for Decrypting and Encrypting, by swapping byte pairs. This implies */
/* that the range to encrypt must be an even number of bytes.                */
/* the seed is not currently used in this method (yet).                      */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define CryptS(Xseed, Xmod, Xstart, Xcount)                                   \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  if ((tCount & 1) != 0)         /* If count is odd . . .                  */ \
  {                                                                           \
    tCount -= 1;                 /* make count even                        */ \
  }                                                                           \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    tSwap        = *CodePtr;     /* Copy 1st byte of byte pair             */ \
    *CodePtr     = *(CodePtr+1); /* Move 2nd byte of pair into 1st byte    */ \
    *(CodePtr+1) = tSwap;        /* Move 1st byte of pair into 2nd byte    */ \
    CodePtr     += 2;            /* Point to next byte pair.               */ \
    tXseed      += Xmod;         /* Modify seed.                           */ \
    tCount      -= 2;            /* Decrement bytes left to be encrypted   */ \
  }


/*****************************************************************************/
/* Macro for Decrypting, by Subtracting a seed from each byte.               */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define DecryptA(Xseed, Xmod, Xstart, Xcount)                                 \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    *CodePtr -= tXseed; /* Encrypt byte in SuperPro driver.                */ \
    CodePtr++;          /* Point to next byte.                             */ \
    tXseed += Xmod;     /* Modify seed.                                    */ \
    tCount--;           /* Decrement bytes remaining to be encrypted       */ \
  }


/*****************************************************************************/
/* Macro for Decrypting, by Rotating each byte left 1 bit or right 1 bit.    */
/* The direction depends upon the current seed value.                        */
/* None of the parameters are modified.                                      */
/*****************************************************************************/
#define DecryptR(Xseed, Xmod, Xstart, Xcount)                                 \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  tXseed = Xseed;                                                             \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to encrypt */ \
  while (tCount > 0)                                                          \
  {                                                                           \
    if ((tXseed & 1) == 1)                                                    \
    {                   /* If tXseed is ODD then rotate the bits right     */ \
      *CodePtr = (*CodePtr >> 1) | (*CodePtr << 7);                           \
    }                                                                         \
    else                                                                      \
    {                   /* If tXseed is EVEN then rotate the bits left     */ \
      *CodePtr = (*CodePtr << 1) | (*CodePtr >> 7);                           \
    }                                                                         \
                                                                              \
    CodePtr++;          /* Point to next byte.                             */ \
    tXseed += Xmod;     /* Modify seed.                                    */ \
    tCount--;           /* Decrement bytes remaining to be encrypted       */ \
  }


/*****************************************************************************/
/* Macro to calculate a Long (32-bit) checksum over a given range.           */
/* The variable replaced by Xsum will be modified. Note that it is up to the */
/* programmer to make sure this variable has the desired value before using  */
/* this macro (i.e. this macro does not zero the variable replaced by Xsum   */
/* before using it!)                                                         */
/*****************************************************************************/
#define ChkSum(Xstart, Xcount, Xsum)                                          \
                                                                              \
  DoMacroCore(Xstart, Xcount);                                                \
                                                                              \
  CodePtr = (unsigned char RNBO_FPTR *)(SUPERPRO) + Xstart;                         \
                        /* Point at 1st byte in SuperPro driver to checksum*/ \
  while (tCount > 0)                                                          \
  {                                                                           \
    Xsum   += *CodePtr; /* Add byte of SuperPro driver to checksum value.  */ \
    CodePtr++;          /* Point to next byte.                             */ \
    tCount--;           /* Decrement bytes remaining to be checksummed     */ \
  }
