/*****************************************************************************
W k D k S a m p l e s                         WIBU-KEY Development-Kit Samples
==============================================================================
HackCont/WkDmHC.cpp                Core Part of Application, encrypted by AXAN
==============================================================================

This sample application represents the usage of the WIBU-KEY API in
combination with the AXAN technology to encrypt code and data within the
program by WKCRYPT.EXE and to decrypt this parts during the program execution.

Used WIBU-KEY API Functions
---------------------------
WkbGetFC() ........... retrieve Firm Code from EXE file
WkbGetUC() ........... retrieve User Code from EXE file
WkbGetLastError() .... get last error code returned by a function
WkbGetVersion3() ..... get driver version information
WkbQueryInfo2() ...... get error/version information
WkbApplicationInfo2()  checks the minimum driver version
WkbOpen2() ........... open access to WIBU-BOX
WkbCrypt2() .......... encrypt/decrypt data or program
WkbClose2() .......... close access to WIBU-BOX

Used Structures
---------------
WKBAREA .. structure that describes what kind of data should be encrypted/
           decrypted and how this should be done.
WKBSELECT  structure that describes an indirect encryption selection.


Special Features
----------------
This example demonstrates how to use the new AXAN technologie to encrypt
parts of the executable program. Any part of the program, it doesn't
matter whether program code or data, can be encrypted or decrypted during
runtime.

Important Program Functions
---------------------------
CoreCalc ........ core "know how" routine of the example, encrypted by AXAN
OnCalc .......... core dialog routine, calls CoreCalc
OpenBox ......... open access to WIBU-BOX
CheckBox ........ check if a proper WIBU-BOX is attached
DecryptProgram .. decrypt the CoreCalc() function and some objects
DecryptAreas .... decrypt the protected areas
CloseBox ........ close access to WIBU-BOX

==============================================================================

Copyright  2001 by WIBU-SYSTEMS AG
           Rueppurrer Strasse 52-54, D-76137 Karlsruhe, Germany/Europe
           Website: http://www.wibu.com, Email: support@wibu.com
           Phone: +49-721-93172-14, Fax: +49-721-93172-22
           All rights reserved.

Version .. 3.01 of 2001Mar18
Project .. wibu/wk/dev/samples
Author ... Marcellus Buchheit (mabu)
System ... Windows 95/98/Me, Windows NT 4.00-5.00
Tools .... Microsoft Visual C++ 6.00/7.00

*****************************************************************************/

#define STRICT // for windows.h
#include <windows.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

// project resource definitions
#include "resource.h"

// include WIBU-KEY definitions
#include <wk.h>

// =================================
// general settings (may be adapted)
// =================================

// definitions for the sample
const int nMinDrvVersion = 3;    // major driver version number (required)
const int nMinDrvSubVersion = 1; // minor driver version number (required)

// used algorithm version for WIBU-BOX encryption/decryption
// * WKB_VERSION2 for WIBU-BOX/U, WIBU-BOX/P, WIBU-BOX/M and WIBU-BOX/CI
// * WKB_VERSION3 for WIBU-BOX/RU and WIBU-BOX/RP
// * do NOT use WKB_VERSION1 (minor security)
const UINT uAlgVersion = WKB_VERSION2;

// ==================
// global definitions
// ==================

// structure that holds all important WIBU-KEY informations
typedef struct {
  long lFirmCode;        // Firm Code to be used
  long lUserCode;        // User Code to be used
  ULONG ulSelection;     // Selection Code to be used
  WKBSELECT wkbsel;      // Selection informations
  HWKBENTRY  hwkbe;      // handle of opened WIBU-BOX entry
  HWKBSYSTEM hwkbsys;    // handle of used subsystem
  int nVersion;          // version number as integer
  TCHAR szVersion[10];   // version number as string
  bool fCheckRunning;    // flag "CheckBox is running"
  bool fIsDecrypted;     // flag "functions have been decrypted"
} WIBUKEYDATA; // prefix "wkdata"


// prototypes for the WIBU-BOX access
bool OpenBox(WIBUKEYDATA& wkdata);
bool CheckBox(WIBUKEYDATA& wkdata);
bool DecryptAreas(WIBUKEYDATA& wkdata);
bool CloseBox(WIBUKEYDATA& wkdata);

// window functions
BOOL CALLBACK AboutDlgProc (HWND, UINT, UINT, long);
BOOL CALLBACK MainDlgProc (HWND, UINT, UINT, long);

// global variables
WIBUKEYDATA wkdata;
HINSTANCE hThisInstance;


/*****************************************************************************
WinMain
==============================================================================

This is the main function of the Windows program.

Parameters
  hInstance .... handle to this instance
  hPrevInstance  handle to previous instance
  lpszCmdLine .. command line string
  nCmdShow ..... command how to show at startup

Return
  The function returns the return code of the application which is always 0.

*****************************************************************************/

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
  LPSTR lpszCmdLine, int nCmdShow)
{
  hThisInstance = hInstance;
  DialogBox(hInstance, MAKEINTRESOURCE(IDD_MAIN), NULL, MainDlgProc);
  // verify that all subsystems have been released
  CloseBox(wkdata);
  return 0;
} // WinMain()


/*****************************************************************************
CoreCalc
==============================================================================

This is the kernel routine of the application and it is encrypted by AXAN.

Parameters
  nValue  contains the input value.

Return
  The function returns the calculated value.

Implementation
--------------
This function searchs the next higher prime number after the input value.

*****************************************************************************/

int CoreCalc(int nValue)
{
  // analyse only positive numbers
  nValue = abs(nValue);
  for (;; nValue++) {
    // check if <nValue> is prime number
    if (nValue == 2) {
      // special case: 2 is even and prime
      return nValue;
    } // if
    if (nValue == 1 || nValue % 2 == 0) {
      // 1 or a pair number is never a prime: continue in loop
      continue;
    } // if
    int nDiv = 3;
    bool fStep2 = false;
    for (;;) {
      if (nValue / nDiv < nDiv) {
        // all divisors checked => return, <nValue> is prime
        return nValue;
      } // if
      // check modulo
      if (nValue % nDiv == 0) {
        // <nValue> can be divided by <nDiv> => no prime, exit loop
        break;
      } // if
      // set next value for divisor
      nDiv += fStep2 || nDiv == 3? 2 : 4;
      fStep2 = !fStep2;
    } // for
  } // for
} // CoreCalc()


// the following text is set into IDC_TITLE if the AXAN decryption is possible
// * this text is encrypted by AXAN

TCHAR szTitle[] = "Prime Number Finder";

// the following matrix is used to convert <result2> from <result1>.
// * this table is encrypted by AXAN

ULONG auCalcMatrix[32] = {
  0x7ED8B689, 0x68E1B5FD, 0x560D5D09, 0xDABCFAC3,
  0x1EE3D231, 0xA72BCB55, 0x2EC91EBA, 0xA1B757E8,
  0x5CBEB486, 0x7977630C, 0x36FC425A, 0x5B5FCEE8,
  0x5DB93792, 0xE0FBF9AE, 0x171294AB, 0x4C8861FC,
  0xE4CB8AA4, 0xB6C5B32B, 0x107E1661, 0xAEB3AB7F,
  0x080E760E, 0x11A16AFB, 0x17CF451B, 0x4CB15CA5,
  0xE67D3927, 0xB161178B, 0xC0FB4963, 0xF0763D65,
  0x91783FDC, 0x4E24DA20, 0xC9FFF989, 0x0D7332EC
};


/*****************************************************************************
OnCalc
==============================================================================

This function is called when the "Calc"-button is pressed. For each addition
it will be checked if a WIBU-BOX with the correct Firm- and Usercode is
available.

Parameters
  hDlg  contains the handle of the dialog box

*****************************************************************************/

void OnCalc(HWND hDlg)
{
  // check WIBU-BOX
  HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));
  bool fBoxOk = CheckBox(wkdata);
  SetCursor(hcurSave);
  if (!fBoxOk) {
    WkbQueryInfo2(HWKB_LOCAL, WKB_QI_ERRTEXT | WKB_QI_DIALOG, NULL, 0);
    return;
  } // if
  // get edit box entries
  TCHAR szBuffer[20];
  GetDlgItemText(hDlg, IDC_OP, szBuffer, 20);
  int nOp = atol(szBuffer);
  // calculate result
  int nResult = CoreCalc(nOp);
  // set result edit box
  wsprintf(szBuffer, "%ld", nResult);
  SetDlgItemText(hDlg, IDC_RESULT1, szBuffer);
  // convert value into result 2
  ULONG uResult2 = auCalcMatrix[nResult & 31] ^ nResult;
  wsprintf(szBuffer, "%lu", uResult2);
  SetDlgItemText(hDlg, IDC_RESULT2, szBuffer);
} // OnCalc()


/*****************************************************************************
DecryptProgram
==============================================================================

This function is the main routine to decrypt all encrypted areas of the
executable code.

Parameters
  hDlg  contains the handle of the dialog box

Return
  The function returns <true> if the decryption was successful and <false>
  if not.

*****************************************************************************/

bool DecryptProgram(HWND hDlg)
{
  // check WIBU-BOX
  HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));
  bool fOk = DecryptAreas(wkdata);
  SetCursor(hcurSave);
  if (!fOk) {
    MessageBox(hDlg, "Unable to decrypt program!\n\"Calc\" cannot be used.",
      S(APPNAME), MB_ICONSTOP);
    return false;
  } // if
  return true;
} // DecryptProgram()


/*****************************************************************************
MainDlgProc
==============================================================================

This is the dialog function of the main dialog box.

Parameters
  hDlg .... contains the handle of the dialog box
  uMessage  message id
  uWParam   first parameter
  uLParam   second parameter

*****************************************************************************/

BOOL CALLBACK MainDlgProc(HWND hDlg, UINT uMessage, UINT uWParam,
  long uLParam)
{
  switch (uMessage) {
    case WM_INITDIALOG:
      // set default values
      SetDlgItemText(hDlg, IDC_OP, "100");
      SetDlgItemText(hDlg, IDC_RESULT1, "0");
      SetDlgItemText(hDlg, IDC_RESULT2, "0");
      return TRUE;
    case WM_SYSCOMMAND:
      if (uWParam == SC_CLOSE) {
        EndDialog (hDlg, 0);
        return TRUE;
      } // if
      break;
    case WM_COMMAND:
      switch (uWParam) {
        case IDC_CALC:
          if (!DecryptProgram(hDlg)) {
            // program cannot be decrypted: abort operation
            return FALSE;
          } // if
          // set title text (now decrypted)
          SetDlgItemText(hDlg, IDC_TITLE, szTitle);
          OnCalc(hDlg);
          return TRUE;
        case IDC_ABOUT:
          DialogBox (hThisInstance, MAKEINTRESOURCE(IDD_ABOUT), hDlg,
            AboutDlgProc);
          return TRUE;
        case IDC_CANCEL:
          EndDialog (hDlg, 0);
          return TRUE;
      } // switch
      break;
  } // switch
  return FALSE;
} // MainDlgProc()


/*****************************************************************************
AboutDlgProc
==============================================================================

This is the dialog function of the about dialog box.

Parameters
  hDlg .... contains the handle of the dialog box
  uMessage  message id
  uWParam   first parameter
  uLParam   second parameter

*****************************************************************************/

BOOL CALLBACK AboutDlgProc(HWND hDlg, UINT uMessage, UINT uWParam,
  long uLParam)
{
  switch (uMessage) {
    case WM_INITDIALOG: {
      // set version of kernel driver
      TCHAR szBuffer[100];
      wsprintf(szBuffer, "Access to Kernel Driver, Version %s",
        (LPSTR)wkdata.szVersion);
      SetDlgItemText(hDlg, IDC_ABOUT_VERSION, szBuffer);
      return TRUE;
    } // case
    case WM_COMMAND:
      switch (uWParam) {
        case IDOK:
          EndDialog (hDlg, 0);
          return TRUE;
      } // switch
      break;
  } // switch
  return FALSE;
} // AboutDlgProc()


/*****************************************************************************
PrintWkError
==============================================================================

This function prints an error message in dependence of the last error code.
The error code is read via the WIBU-KEY-driver function WkbGetLastError().

Parameters
  pszFunction  name of function where the error occurred

*****************************************************************************/

void PrintWkError(const TCHAR *pszFunction)
{
  TCHAR szMsg[200];
  wsprintf(szMsg, "Error: Function = %s()", pszFunction);
  // read the error text from the wkwin(32).dll
  TCHAR szError[100];
  int nErr = WkbQueryInfo2 (HWKB_LOCAL, WKB_QI_ERRTEXT, szError, 100);
  wsprintf(szMsg + strlen(szMsg), "\n%s", szError);
  // print the whole text in a message box.
  MessageBox(NULL, szMsg, S(APPNAME), MB_ICONSTOP | MB_OK);
} // PrintWkError()


/*****************************************************************************
ReadPgmCode
==============================================================================

This function reads the expected Firm and User Code of the program and sets
these values into global variables. A message displays the expected values
at screen.

Parameters
  wkdata  refers to the global WIBU-KEY data.

*****************************************************************************/

void ReadPgmCode(WIBUKEYDATA& wkdata)
{
  // read Firm Code and User Code from program code
  int idError = WkbGetFC(&(wkdata.lFirmCode), 0);
  if (idError == WKB_ERROR_NOT_CRYPTIC) {
    MessageBox(NULL, "Program is not encrypted by CRYPT 1.32 or higher.",
     S(APPNAME), MB_ICONSTOP|MB_OK);
    exit(255);
  } // if
  WkbGetUC(&(wkdata.lUserCode), 0);
} // ReadPgmCode()


/*****************************************************************************
OpenBox
==============================================================================

Initialize global structure for the WIBU-KEY.

Parameters
  wkdata  refers to the global WIBU-KEY data.

Return
  This function returns <true> if all actions were completed. If any error
  occurs <false> will be returned.

*****************************************************************************/

bool OpenBox(WIBUKEYDATA& wkdata)
{
  // initialize the whole structure with 0.
  memset(&wkdata, 0, sizeof(wkdata));

  // NOTE  Every application should check the version of the driver, which is
  // ----  required as minimum to avoid any unexpected results when older
  //       drivers are used.
  // * determine the driver version for information purposes only in a string
  wkdata.nVersion = WkbGetVersion3(HWKB_LOCAL);
  int nVersion = wkdata.nVersion / 256;
  int nSubVersion = wkdata.nVersion % 256;
  wsprintf((LPSTR)wkdata.szVersion, "%d.%02d", nVersion, nSubVersion);
  // now check the minimum driver via WkbApplicationInfo2()
  WKBAPPINFO wkbai;
  memset(&wkbai, 0, sizeof(wkbai));
  wkbai.flCtrl = WKB_AIS_SEL_VERSION;    // check only version
  wkbai.bVersionLow = nMinDrvSubVersion; // set mimimum major version
  wkbai.bVersionHigh = nMinDrvVersion;   // set minimum minor version
  wkbai.fsAccessMask = WKB_ACC_LOCAL;    // only local check
  // check required driver version, display message if not proper
  BOOL fOK = WkbApplicationInfo2(HWKB_LOCAL,
    WKB_AI_SETDATA | WKB_AI_DIALOG | WKB_STDCTRL,
    &wkbai, sizeof(wkbai));
  if (!fOK) {
    // wrong driver verison specified (user was notified by dialog box)
    // * abort operation
    return false;
  } // if


  // read Firm Code and User Code
  ReadPgmCode(wkdata);

  // search & open WIBU-BOX entry of accessed WIBU-KEY subsystem
  // NOTE  If you use WKB_VERSION2 it will not be possible to access
  // ====  WIBU-BOX/RU or WIBU-BOX/RP.
  UINT flCtrl = WKB_OPEN_FIND | WKB_STDCTRL | uAlgVersion;
  wkdata.hwkbe = WkbOpen2(
    HWKB_LOCAL,          // handle to local system
    flCtrl,              // control flags (use version 2, find entry)
    NULL,                // search all available ports for WIBU-BOXes
    wkdata.lFirmCode,    // Firm Code
    wkdata.lUserCode,    // User Code
    NULL                 // no optional data
  );
  if (!wkdata.hwkbe) {
    PrintWkError("WkbOpen2");
    return FALSE;
  } // if
  return true;
} // OpenBox()


/*****************************************************************************
GetRandomChar
==============================================================================

This function returns a random character between 'A' and 'Z'.

Return
  The created character ist returned.

*****************************************************************************/

TCHAR GetRandomChar()
{
  ULONG ul = rand() * 26;
  ul = ul / RAND_MAX;
  TCHAR ch = (TCHAR)ul + 65;
  return ch;
} // GetRandomChar()


/*****************************************************************************
GenerateRandomText
==============================================================================

This function fills the given array of chars with <cch> random characters
ranging from 'A' to 'Z'. The text is terminated with 0 character.

Parameters
  pszText  points to the text which is created
  cchText  contains the length of the text in characters.

*****************************************************************************/

void GenerateRandomText(TCHAR *pszText, UINT cchText)
{
  // initialize random generator
  srand((UINT)time(NULL));
  // fill string with random characters ('A' ... 'Z')
  for (UINT i = 0; i < cchText; i++) {
    pszText[i] = GetRandomChar();
  } // for
  pszText[cchText] = 0;
} // GenerateRandomText()



/*****************************************************************************
CheckBox
==============================================================================

This function checks, if a suitable WIBU-BOX is connected at the local system.

This function checks if a correct WIBU-BOX is attached to the local system.
In the most simple way a string is encrypted, then decryped and the
original data is being compared to the decrypted data. If the two string
are identical TRUE is returned. If the comparison of the two string fails
or any error occurs in the communication with the WIBU-KEY the function
will return <false>.

Parameters
  wkdata  refers to the global WIBU-KEY data.

Return
  This function returns <true> if all actions were completed. If any error
  occurs <false> will be returned.

*****************************************************************************/

bool CheckBox(WIBUKEYDATA& wkdata)
{
  if (wkdata.fCheckRunning) {
    // were already running a check for the WIBU-BOX
    return true;
  } // if
  const UINT cbMaxData = 100;
  TCHAR szTest[cbMaxData];
  memset(szTest, 0, cbMaxData);
  TCHAR szCrypted[cbMaxData];
  memset(szCrypted, 0, cbMaxData);
  TCHAR szDecrypted[cbMaxData];
  memset(szDecrypted, 0, cbMaxData);

  // * normally the WIBU-KEY should be already opened by OpenBox(),
  //   but when an error has occured it may have beend closed, therefore
  //   we check this
  if (wkdata.hwkbe == NULL) {
    // search & open WIBU-BOX entry of accessed WIBU-KEY subsystem
    // NOTE  If you use WKB_VERSION2 it will not be possible to access
    // ====  WIBU-BOX/RU or WIBU-BOX/RP.
    ULONG flCtrl = WKB_OPEN_FIND | WKB_STDCTRL | uAlgVersion;
    wkdata.hwkbe = WkbOpen2(
      HWKB_LOCAL,          // handle to local system
      flCtrl,              // control flags (use version 2, find entry)
      NULL,                // search all available ports for WIBU-BOXes
      wkdata.lFirmCode,    // Firm Code
      wkdata.lUserCode,    // User Code
      NULL                 // no optional data
    );
    if (!wkdata.hwkbe) {
      PrintWkError("WkbOpen2");
      return false;
    } // if
  } // if

  // generate random Selection Code from system time
  // * use the seconds since January 1, 1970 and clear
  //   the 4 topmost bits because bit 28 to 31
  //   have special meanings.
  ULONG ulSC = ((ULONG)time(NULL)) & WKB_SC_COREMASK;

  // generate random string to encrypt
  GenerateRandomText(szTest, 20);
  UINT cbData = strlen(szTest);

  wkdata.fCheckRunning = TRUE;

  // fill WKBAREA structure
  // * This structure describes what data should be encrypted and with
  //   which paramters the encryption should be performed.
  WKBAREA wkbar;
  memset(&wkbar, 0, sizeof(wkbar));
  // the driver should perform the Select/UnSelect operation
  // * activate CRC calculation for original data
  wkbar.flCtrl = WKB_AREA_SELECT | WKB_AREA_CALCCRC;
  // use direct encryption
  wkbar.flSelCtrl = WKB_SEL_DIRECT;
  // the random Selection Code
  wkbar.ulSelectCode = ulSC;
  // we use direct encryption
  wkbar.pvCtrl = NULL;
  // encrypt raw data
  memcpy(szCrypted, szTest, cbMaxData);
  UINT cbCrypt;
  BOOL fValid = WkbCrypt2(
    wkdata.hwkbe,                 // handle to opened entry
    WKB_CRYPT_CTRL | WKB_STDCTRL, // flags for crypt & copy
    szCrypted,                    // source AND destination buffer
    &wkbar,                       // control structure
    cbData,                       // bytes to encrypt
    &cbCrypt                      // bytes encrypted by WkCrypt2
  );
  if (!fValid) {
    PrintWkError("WkbCrypt2");
    WkbClose2(wkdata.hwkbe);
    wkdata.hwkbe = NULL;
    wkdata.fCheckRunning = FALSE;
    return FALSE;
  } // if
  // decrypt encrypted data
  // * check now CRC value of original data
  wkbar.flCtrl &= ~WKB_AREA_CALCCRC;
  wkbar.flCtrl |= WKB_AREA_CHKCRC;
  memcpy(szDecrypted, szCrypted, cbMaxData);
  fValid = WkbCrypt2(
    wkdata.hwkbe,                 // handle to opened entry
    WKB_CRYPT_CTRL | WKB_STDCTRL, // flags for crypt & copy
    szDecrypted,                  // destination buffer
    &wkbar,                       // source AND destination buffer
    cbData,                       // bytes to encrypt
    &cbCrypt                      // bytes encrypted by WkCrypt2
  );
  if (!fValid) {
    PrintWkError("WkbCrypt2");
    WkbClose2(wkdata.hwkbe);
    wkdata.hwkbe = NULL;
    wkdata.fCheckRunning = false;
    return false;
  } // if

  wkdata.fCheckRunning = false;

  // * compare original data <szText> to decrypted data <szDecrypted>
  //   to assure that the encryption was correct and by this that
  //   the WIBU-KEY is connected.
  for (UINT i = 0; i < cbData; i++) {
    if (szTest[i] != szDecrypted[i]) {
      return false;
    } // if
  } // for
  // everythings correct
  return true;
} // CheckBox()


/*****************************************************************************

---- Area Descriptions for Encryption/Decryption of Code and Data ------------

*****************************************************************************/

// NOTE  See the Developer's Guide manual V3.01 for a description of the
// ----  principles behind AXAN and the special Online document
//       "AXAN Introduction.Pdf" of the WIBU-KEY Development Kit.
//     . The detailed explanation of the WKBAREA structure and of all the
//       flags can be found in the WIBU-KEY API online help file.

// =====================================
// area 1: first 100 bytes of CoreCalc()
// =====================================

#define CB_AREA_CORECALC 100

struct {
  WKBAREA wkbar;
  WKBSELECT wkbsel;
  BYTE abGapMap[WKB_GET_MAP_SIZE(CB_AREA_CORECALC)];
} wkbarCoreCalc = {
  {// WKBAREA structure (prepared for AXAN)
    // (WKB_AREA_CHKCRC set by WKCRYPT)
    WKB_AREA_NONE | WKB_AREA_CODE | WKB_AREA_SELECT,
    WKB_SEL_DEC_FEAL,
    0, // Selection Code filled by WKCRYPT
    0, // CRC filled by WKCRYPT
    0, // reserved
    NULL, // wkbarCoreCalc.abGapMap,
    &wkbarCoreCalc.wkbsel // link to indirect encryption
  },
  {// WKBSELECT structure
    WKB_SEL_FEAL32 | WKB_SEL_CBC_CRYPT, // WKB_SEL_CHKCRC set by WKCRYPT
    0, // ulCacheId
    0, // ulCrc filled by WKCRYPT
    8, // cbInit
    { 0xDC, 0x75, 0x5B, 0x84, 0xB8, 0x26, 0x65, 0xEB }
  }
};


// =============================================
// area 2: complete encryption of <szTitle> text
// =============================================

#define CB_AREA_TITLE sizeof(szTitle)

struct {
  WKBAREA wkbar;
} wkbarTitle = {
  {// WKBAREA structure (prepared for AXAN)
    // (WKB_AREA_CHKCRC set by WKCRYPT)
    WKB_AREA_NONE | WKB_AREA_DATA | WKB_AREA_SELECT,
    WKB_SEL_DIRECT,
    // * Selection Code, core part filled by WKCRYPT
    //   but set Limit Counter Reduction and Expiration Date Check
    WKB_SC_BOXLIMIT | WKB_SC_EXPDATE | 0,
    0, // CRC filled by WKCRYPT
    0, // reserved
    NULL, // no gap map required
    NULL // none because direct encryption
  }
};

// ===================================================
// area 3: complete encryption of <auCalcMatrix> table
// ===================================================

#define CB_AREA_CALCMATRIX sizeof(auCalcMatrix)

struct {
  WKBAREA wkbar;
  WKBSELECT wkbsel;
} wkbarCalcMatrix = {
  {// WKBAREA structure (prepared for AXAN)
    // (WKB_AREA_CHKCRC set by WKCRYPT)
    WKB_AREA_NONE | WKB_AREA_DATA | WKB_AREA_SELECT,
    WKB_SEL_DEC_FEAL,
    // * Selection Code, core part filled by WKCRYPT
    //   but set Expiration Date Check
    WKB_SC_EXPDATE | 0,
    0, // CRC filled by WKCRYPT
    0, // reserved
    NULL, // no gap map required
    &wkbarCalcMatrix.wkbsel // link to indirect encryption
  },
  {// WKBSELECT structure
    WKB_SEL_FEAL32 | WKB_SEL_CBC_CRYPT, // WKB_SEL_CHKCRC set by WKCRYPT
    0, // ulCacheId
    0, // ulCrc filled by WKCRYPT
    8, // cbInit
    { 0x80, 0x12, 0x44, 0x79, 0x18, 0xFC, 0x76, 0x53 }
  }
};

// disable C++ error warning for AXAN table structure
#pragma warning(disable:4200)

// =========================================
// AXAN support macros define all AXAN areas
// =========================================

WKB_BEGIN_AXAN_CODE_TABLE(Func)
  WKB_AXAN_CODE(CoreCalc,
    CB_AREA_CORECALC, &wkbarCoreCalc.wkbar, NULL, WKBAXAN_SC_FCUC)
WKB_END_AXAN_CODE_TABLE()

WKB_BEGIN_AXAN_CODE_TABLE(Data)
  WKB_AXAN_CODE(szTitle,
    CB_AREA_TITLE, &wkbarTitle.wkbar, NULL, WKBAXAN_SC_FCUC)
  WKB_AXAN_CODE(auCalcMatrix,
    CB_AREA_CALCMATRIX, &wkbarCalcMatrix.wkbar, NULL, WKBAXAN_SC_RANDOM)
WKB_END_AXAN_CODE_TABLE()

// * "Main" is the name of the WKBAREAHEAD. You need call WKCRYPT.EXE
//   with the option /CH:wkbaxhMain to use this table. The prefix "wkbaxh"
//   is added by the WKB_BEGIN_AXAN_HEAD macro. This is done to get a unique
//   variable name.
WKB_BEGIN_AXAN_HEAD(Main, "WKAXANH", 0)
  WKB_AXAN_CODE_TABLE(Func)
  WKB_AXAN_CODE_TABLE(Data)
WKB_END_AXAN_HEAD()


/*****************************************************************************
DecryptAreas
==============================================================================

This function decrypts all encrypted parts of the executable. It is important
to call the function before any attempt to call the CoreCalc() function or to
read the <szTitle> or <auCalcMatrix> objects is made. Otherwise the program
will crash when Windows tries to execute the encrypted code or displays wrong
data.

Parameters
  wkdata  refers to the global WIBU-KEY data.

Return
  This function returns <true> if all actions were completed. If any error
  occurs, <false> will be returned.

*****************************************************************************/

bool DecryptAreas(WIBUKEYDATA& wkdata)
{
  // we can decrypt only once
  if (!wkdata.fIsDecrypted) {
    // initialize <wkdata>-structure, check version, etc.
    if (!OpenBox(wkdata)) {
      return false;
    } // if
    // decrypt CoreCalc function
    // * select entry, decrypt area, unselect entry
    BOOL fValid = WkbCrypt2(
      wkdata.hwkbe,          // handle to opened entry
      WKB_STDCTRL,           // standard flags
      MAKECODEPTR(CoreCalc), // pointer to the function (start of area)
      &wkbarCoreCalc.wkbar,  // control structure (filled by WKCRYPT)
      CB_AREA_CORECALC,      // number of bytes to decrypt
      NULL                   // we do not need the return value
    );
    if (!fValid) {
      // any error
      PrintWkError("WkbCrypt2");
      return false;
    } // if
    // decrypt <szTitle> text
    // * select entry, decrypt area, unselect entry
    fValid = WkbCrypt2(
      wkdata.hwkbe,         // handle to opened entry
      WKB_STDCTRL,          // standard flags
      MAKECODEPTR(szTitle), // pointer to function (start of decryption area)
      &wkbarTitle.wkbar,    // control structure (filled by WKCRYPT)
      CB_AREA_TITLE,        // number of bytes to decrypt
      NULL                  // we do not need the return value
    );
    if (!fValid) {
      // any error
      PrintWkError("WkbCrypt2");
      return false;
    } // if
    // decrypt <auCalcMatrix> table
    // * select entry, decrypt area, unselect entry
    fValid = WkbCrypt2(
      wkdata.hwkbe,              // handle to opened entry
      WKB_STDCTRL,               // standard flags
      MAKECODEPTR(auCalcMatrix), // pointer to the data area
      &wkbarCalcMatrix.wkbar,    // control structure (filled by WKCRYPT)
      CB_AREA_CALCMATRIX,        // number of bytes to decrypt
      NULL                       // we do not need the return value
    );
    if (!fValid) {
      // any error
      PrintWkError("WkbCrypt2");
      return false;
    } // if
    // we can decrypt only once, set flag to remember
    wkdata.fIsDecrypted = true;
  } // if

  // decryption successfully finished
  return true;
} // DecryptAreas()


/*****************************************************************************
CloseBox
==============================================================================

This function closes the  access to the opened WIBU-BOX entry.

Parameters
  wkdata  refers to the global WIBU-KEY data.

Return
  This function returns <true> if all actions were completed. If any error
  occurs, <false> will be returned.

*****************************************************************************/

bool CloseBox(WIBUKEYDATA& wkdata)
{
  if (wkdata.hwkbe != NULL) {
    // close opened WIBU-BOX entry
    if (!WkbClose2(wkdata.hwkbe)) {
      PrintWkError("WkbClose2");
    } // if
  } // if
  return true;
} // CloseBox()


// ===============================================
// wibu/wk/dev/samples: end of HackCont/WkDmHC.cpp
// ===============================================
