/*
 *  AX25CONF.C - Configuration utility for the ETHRAX25 packet driver.
 *  Copyright (C) 1994  Gary L. Grebus
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This utility sets the AX.25 related parameters of the loaded driver,
 * and the parameters in the TNC (i.e. those set via KISS commands).
 * The protocol stack won't configure these things because it is expecting
 * to talk to an Ethernet device.
 *
 * Original Version:
 * 	7 April 1994 - Gary L. Grebus, K8LT, glg@k8lt.ampr.org
 *
 * Modifications:
 *      18 Aug 1996 - glg@k8lt.ampr.org
 *      Reworked mechanism for passing TNC port etc to use a different
 *      receive mode.  The old mechanism totally broke the driver.
 *      Also add -promisc|nopromisc switches for promiscuous mode.
 *
 *	21 July 1996 - glg@k8lt.ampr.org
 *	Added -port switch for multi-port TNC's.
 *	Create new method for passing parameters to the driver by
 *	  overloading the SET_ADDRESS function.
 *
 * Usage:
 *    ax25conf <intno> -mycall <ax25 addr> -txdelay n -slottime n ...
 *
 */

#include <stdio.h>
#include <dos.h>
#include <string.h>
#include <stdlib.h>
#include "ax25util.h"
#include "pktdrvr.h"

/* The most common collection of KISS params */
#define	PARAM_DATA	0
#define	PARAM_TXDELAY	1
#define	PARAM_PERSIST	2
#define	PARAM_SLOTTIME	3
#define	PARAM_TXTAIL	4
#define	PARAM_FULLDUP	5

struct statistics {
    unsigned long packets_in;
    unsigned long packets_out;
    unsigned long bytes_in;
    unsigned long bytes_out;
    unsigned long errors_in;
    unsigned long errors_out;
    unsigned long packets_dropped;
};

struct axargs {
    unsigned char version;
    unsigned char type;
    union {
      unsigned char kiss_port;
    } m;
};

enum axargs_type { AX_KISS_PORT,
		   AX_LAST };

enum rcv_modes { RX_RSRVD0, RX_RXOFF, RX_ADDRONLY, RX_BROADCAST,
	         RX_MYMULTI, RX_ALLMULTI, RX_PROMISC, RX_RAW, RX_COMMAND };
#define AX_ARG_VERSION 0

extern int Derr;                /* Error variable for packet driver routines */
int intno;                      /* Packet driver interrupt number */

char access_err_msg[] = "ax25conf: unable to access driver: %s\n";
char mode_err_msg[] = "ax25conf: unable to switch modes: %s\n";

void usage(void);
int do_kissparm(int argc, char **argv, int type);
int do_kissport(int argc, char **argv);
int do_mycall(int argc, char **argv);
int do_promiscuous(int argc, char **argv);
int do_stats(int argc, char **argv);
int get_statistics( int intno, int handle, struct statistics *s );
int set_address(int intno, unsigned char *addr, unsigned length);
int set_ax_args(int intno, struct axargs *addr, unsigned length);
char *pkt_error(void);

int
main(int argc, char **argv)
{

    int args_used;

    if( argc < 2 ) {
        usage();
        return(1);
    }

	argc--; argv++;

    /* First parameter must be the interrupt number */
    intno = (int)strtol(*argv, NULL, 0);
    if( intno == 0) {
        printf("ax25conf: Invalid interrupt number: %s\n", *argv);
	    usage();
        return(1);
        }

    if( test_for_pd(intno) == 0 ) {
        printf("ax25conf: No packet driver on interrupt 0x%x\n", intno);
        return(1);
        }

    argv++; argc--;
    args_used = 1;
    while( argc > 0 && args_used > 0 ) {

	if ( strcmp("-mycall", *argv) == 0 )
	    args_used = do_mycall(argc, argv);

        else if (strcmp("-txdelay", *argv) == 0 )
            args_used = do_kissparm(argc, argv, PARAM_TXDELAY);

        else if (strcmp("-persist", *argv) == 0 )
            args_used = do_kissparm(argc, argv, PARAM_PERSIST);

        else if (strcmp("-slottime", *argv) == 0 )
            args_used = do_kissparm(argc, argv, PARAM_SLOTTIME);

        else if (strcmp("-txtail", *argv) == 0 )
            args_used = do_kissparm(argc, argv, PARAM_TXTAIL);

	else if (strcmp("-stats", *argv) == 0 )
            args_used = do_stats(argc, argv);

        else if (strcmp("-port", *argv) == 0 )
            args_used = do_kissport(argc, argv);

        else if (strcmp("-promisc", *argv) == 0 ||
		 strcmp("-nopromisc", *argv) == 0)
            args_used = do_promiscuous(argc, argv);

        else {
		printf("ax25conf: Unrecognized parameter %s\n", *argv);
		usage();
		return( 1 );
	}
        argc-=args_used;
        argv+=args_used;
    }

    /* Display the current configuration */
    /* NYI */
	return( 0 );
}
/*
 * Set the AX.25 address associated with the driver.
 */
int
do_mycall(int argc, char **argv)
{
    unsigned char pseudo_adr[PSEUDOLEN];
    unsigned char ax25_adr[AXALEN];
    int status;

    if( argc > 1 ) {
	    StrToAX25(*(argv+1), ax25_adr);
	    AX25ToEther(ax25_adr, pseudo_adr);
	    if( set_address(intno, pseudo_adr, sizeof(pseudo_adr)) != 0 ) {
		    printf("ax25conf: unable to set AX.25 address: %s\n",
		            pkt_error() );
		    status = 0;
	    } else 
	        status = 2;			/* Two argv's used */
    } else {
	    printf("ax25conf: missing parameter to %s\n", argv);
	    status = 0;
    }
    return( status );
}
/*
 * Send a raw KISS frame to set one of the TNC KISS parameters.
 */
#pragma argsused
int
do_kissparm(int argc, char **argv, int type)
{
    int protocol=1234;
    int handle;
    int interrupt dummy_rx();

    int status;
    int old_mode;
    char kiss_msg[2];
    int i;

    if( (handle = access_type(intno, CL_ETHERNET, 0, 0, 
                             (char *)&protocol, 4, dummy_rx)) < 0) {
        printf(access_err_msg, pkt_error());
        return( 0 );
        }

    /* Switch the interface to raw mode to send the KISS frame */
    status = 0;                         /* Assume failure */
    if( (old_mode = set_mode(intno, handle, RX_RAW)) < 0  ) {
	    printf(mode_err_msg, pkt_error());
    } else {
	    kiss_msg[0] = type;	 /* Port number is added by the driver */
	    i = atoi(argv[1]);
	    kiss_msg[1] = i < 255 ? i : 255;
	    if( send_pkt(intno, kiss_msg, 2) < 0 ) {
		    printf("ax25conf: unable to send kiss msg: %s\n",
		           pkt_error());
	    } else {
		if( set_mode(intno, handle, old_mode) < 0 ) {
		    printf(mode_err_msg, pkt_error());
		    printf("ax25conf: further operation may be impossible\n");
		} else 
		    status = 2;
	    }
    }
    release_type(intno, handle);
    return( status );
}

/*
 * Bind the driver to a KISS port on a multi-port TNC.
 * This switch has to come first, before setting the other parameters.
 */
#pragma argsused
int
do_kissport(int argc, char **argv)
{
    struct axargs a;
    int status;

    if( argc > 1 ) {
      	    a.version = AX_ARG_VERSION;
	    a.type = AX_KISS_PORT;
	    a.m.kiss_port = atoi(argv[1]);

	    if( set_ax_args(intno, &a, sizeof(a)) != 0 ) {
		    printf("ax25conf: unable to select KISS port\n",
		            pkt_error() );
		    status = 0;
	    } else 
	        status = 2;			/* Two argv's used */
    } else {
	    printf("ax25conf: missing parameter to %s\n", argv);
	    status = 0;
    }
    return( status );
}
/*
 * Place the driver into promiscuous receive mode i.e. all incoming
 * packets will be sent up the stack, regardless of link layer address.
 */
#pragma argsused
int
do_promiscuous(int argc, char **argv)
{
    int rx_mode;
    int protocol=1234;
    int handle;
    int interrupt dummy_rx();

    int status;

    if( strcmp(argv[0], "-nopromisc") == 0 )
	rx_mode = RX_BROADCAST;		/* Default mode */
    else
	rx_mode = RX_PROMISC;

    if( (handle = access_type(intno, CL_ETHERNET, 0, 0, 
                             (char *)&protocol, 4, dummy_rx)) < 0) {
        printf(access_err_msg, pkt_error());
        return( 0 );
        }

    /* Switch the interface to requested recevie mode. */
    status = 0;                         /* Assume failure */
    if( set_mode(intno, handle, rx_mode) < 0  ) {
	printf(mode_err_msg, pkt_error());
    } else
	status = 1;			/* 1 argv used */

    release_type(intno, handle);

    return( status );
}
#pragma argsused
int do_stats(int argc, char **argv)
{

  int status;
  int protocol=1234;
  int handle;
  struct statistics s;
  int interrupt dummy_rx();

  if( (handle = access_type(intno, CL_ETHERNET, 0, 0, 
			    (char *)&protocol, 4, dummy_rx)) < 0) {
    printf(access_err_msg, pkt_error());
    return( 0 );
  }

  if( get_statistics( intno, handle, &s ) < 0 ) {
      printf("ax25conf: unable to get statistics: %s\n", pkt_error());
      status = 0;

  } else {
      printf("Packets In: %6ld   Packets Out: %6ld\n",
	   s.packets_in, s.packets_out);
      printf("Bytes   In: %6ld   Bytes   Out: %6ld\n",
	   s.bytes_in,   s.bytes_out);
      printf("Errors  In: %6ld   Errors  Out: %6ld\n",
	   s.errors_in,  s.errors_out);
      printf("Packets Dropped: %6ld\n", s.packets_dropped);
      status = 1;
  }

  release_type(intno, handle);

  return (status );
}
/*
 * Packet driver utility routine to get interface statistics.
 */
int
get_statistics( int intno, int handle, struct statistics *s )
{
    union REGS regs;
    struct SREGS sregs;

    segread(&sregs);
    regs.x.bx = handle;
    regs.h.ah = GET_STATISTICS;
    int86x(intno,&regs,&regs,&sregs);
    if(regs.x.cflag){
	Derr = regs.h.dh;
	return -1;
    } else {
      movedata(sregs.ds, regs.x.si, FP_SEG(s), FP_OFF(s),
	       sizeof(struct statistics));
      return 0;
  }
}
/*
 * Return the error message associated with the last error status 
 * detected by the packet driver utility routines.
 */
char *pkt_error(void)
{
    char *message[] = {
	    "No error",
	    "Invalid handle number",
	    "No interface of specified class",
	    "No interface of specified type",
	    "No interface of specified number",
	    "Bad packet type specified",
	    "Interface does not support multicast",
	    "Driver cannot terminate",
	    "Invalid receiver mode",
	    "Insufficient space",
	    "Packet type already in use",
	    "Bad or unimplemented command",
	    "Cannot send packet",
	    "Cannot set address when driver is in use",
	    "Bad hardware address format",
	    "Cannot reset when driver is in use"
    };

    if( Derr < 0 || Derr > CANT_RESET )
	return("Unknown error");
    else
        return( message[Derr] );
}
/* 
 * Packet driver utility routine to set an interface address.
 */
int
set_address(int intno, unsigned char *buffer, unsigned length)
{
	union REGS regs;
	struct SREGS sregs;

	segread(&sregs);
	sregs.es = FP_SEG(buffer);
	regs.x.di = FP_OFF(buffer);
	regs.x.cx = length;
	regs.h.ah = SET_ADDRESS;
	int86x(intno,&regs,&regs,&sregs);
	if(regs.x.cflag){
		Derr = regs.h.dh;
		return -1;
	} else
		return 0;
}
/* 
 * Packet driver utility routine to set special AX.25 parameters.
 * Place the packet driver into a special receive mode in which
 * transmitted data is treated as driver commands.
 */
int
set_ax_args(int intno, struct axargs *args, unsigned length)
{
    int protocol=1234;
    int handle;
    int interrupt dummy_rx();

    int status;
    int old_mode;

    if( (handle = access_type(intno, CL_ETHERNET, 0, 0, 
                             (char *)&protocol, 4, dummy_rx)) < 0) {
        printf(access_err_msg, pkt_error());
        return( 0 );
        }

    /* Switch the interface to command mode and send the arguments
     * as if they were outgoing data.
     */
    status = -1;                         /* Assume failure */
    if( (old_mode = set_mode(intno, handle, RX_COMMAND)) < 0  ) {
	    printf(mode_err_msg, pkt_error());
    } else {
	if( send_pkt(intno, (char *)args, length) < 0 ) {
	    printf("ax25conf: unable to send command msg: %s\n",
		   pkt_error());
	}
	if( set_mode(intno, handle, old_mode) < 0 ) {
	    printf(mode_err_msg, pkt_error());
	    printf("ax25conf: further operation may be impossible\n");
	} else 
	    status = 0;			/* Success */
    }
    release_type(intno, handle);
    return( status );

}

void
usage(void)
{

    printf("Usage:\n");
    printf("ax25conf <intno> [-port <n>] [-mycall <ax25 addr>]\n");
    printf("         [-stats] [-promisc|nopromisc] [-<kissparm> <val>]\n");
    printf("\n");
    printf("    <intno>      - Packet driver interrupt number\n");
    printf("    <n>          - Bind the driver to this port number on a multi-port \n");
    printf("                   TNC.  This switch must appear first if used\n");
    printf("    <ax25 addr>  - AX.25 layer address (callsign with optional SSID)\n");
    printf("    -stats       - Display packet driver statistics\n");
    printf("    -[no]promisc - Receive all link layer addresses\n");
    printf("    -<kissparam> - one of:\n");
    printf("                   -txdelay n: txdelay (in 10 msec units)\n");
    printf("                   -persist n: p-persistence\n");
    printf("                   -slottime n: slottime (in 10 msec units)\n");
    printf("                   -txtail n: txtail (in 10 msec units)\n");

}

/*
 * Dummy upcall function for whenever we need a handle.
 */
#pragma argsused
int interrupt dummy_rx(unsigned bp, unsigned di, unsigned si,
		       unsigned ds, unsigned es, unsigned dx,
		       unsigned cx, unsigned bx, unsigned as)
{
      es = 0;			/* Discard the packet */
      di = 0;
      return 0;
}
