/*
**  AX25UTIL.C - utility routines for Ether to AX.25 packet driver
**  Copyright (C) 1994  Gary L. Grebus
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; version 2.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
**
** Utilities for handling AX.25 addresses encoded into a form that 
** resembles an Ethernet address.
**
** An AX.25 address consists of 7 octets in the "shifted" form described
** in the AX.25 spec.
**
** The psuedo-Ethernet form consists six bytes:
**	The low order byte contains the first octet of the AX.25
**	  address.  The octet is right shifted by 1 (out of normal
**	  AX.25 form) to insure that the MSB is zero.  A non-zero
**	  MSB would imply an Ethernet multicast address.
**	The next byte contains the 2nd octet of the AX.25 address
**	  in normal (left-shifted) form.
** 	The remaining 4 bytes contain the AX.25 address in normal
**	  (left-shifted) form. The LSB of each byte is or'ed with
**	  one bit of the 4 bit SSID value, starting with the
**	  LSB of the SSID.
**
** The AX.25 address QST-0 is interpreted as the broadcast address and
** is converted to the pseudo-ethernet address FF-FF-FF-FF-FF-FF, and
** vice versa.
**
** Original version:
**	7 April 1994 - Gary L. Grebus, K8LT, glg@k8lt.ampr.org
**
**      28 Aug 1996 - Gary L. Grebus, glg@k8lt.ampr.org
**	Use a new, simpler pseudo-ethernet address encoding.
*/

#ifndef DRIVER
#include <ctype.h>
#include <stdlib.h>
#endif

#define ARP_REQUEST 0x0100                /* ARP request opcode */
#define ARP_REPLY 0x0200                  /* ARP reply opcode */

#include "ax25util.h"
#define SSID 0x1e                        /* Mask for SSID bits in AX.25 addr */

typedef struct _AX25Arp {
        int hardware;                       /* Hardware type */
	int protocol;                       /* Protocol type */
	char hwalen;                        /* Hardware addr length in bytes */
	char pralen;                        /* Protocol addr length in bytes */
	int  opcode;                        /* ARP opcode */
	unsigned char shwaddr[AXALEN];      /* Sender hardware address */
	long sprotoaddr;                    /* Sender protocol address */
        unsigned char thwaddr[AXALEN];      /* Target hardware address */
        long tprotoaddr;                    /* Target protocol address */
} AX25Arp;

typedef struct _EtherArp {
	int hardware;                       /* Hardware type */
	int protocol;                       /* Protocol type */
	char hwalen;                        /* Hardware addr length in bytes */
	char pralen;                        /* Protocol addr length in bytes */
	int  opcode;                        /* ARP opcode */
	char shwaddr[PSEUDOLEN];            /* Sender hardware address */
	long sprotoaddr;                    /* Sender protocol address */
    char thwaddr[PSEUDOLEN];            /* Target hardware address */
    long tprotoaddr;                    /* Target protocol address */
} EtherArp;

char qst[] = { 'Q'<<1, 'S'<<1, 'T'<<1, ' '<<1, ' '<<1, ' '<<1, '\0'<<1 };

/* Routine to convert an ARP packet containing AX.25 addresses to the
 * equivalent packet containing pseudo-Ethernet addresses.
 * Since the packet will shrink, do the conversion such that it could be
 * happening in-place.
 */
void
AArpToEArp(AX25Arp far *a_arp, EtherArp far *e_arp)
{
    int i;

    e_arp->hardware = 0x0100;
    e_arp->protocol = 0x0008;
    e_arp->hwalen = 6;
    e_arp->pralen = 4;
    e_arp->opcode = a_arp->opcode;
    a_arp->shwaddr[6] &= SSID;        /* Turn off any AX.25 control bits */
    AX25ToEther(&(a_arp->shwaddr[0]), (ether_t *)&(e_arp->shwaddr[0]));
    e_arp->sprotoaddr = a_arp->sprotoaddr;
    if( a_arp->opcode == ARP_REPLY ) {
	a_arp->thwaddr[6] &= SSID;        /* Turn off any AX.25 control bits */
        AX25ToEther(&(a_arp->thwaddr[0]), (ether_t *)&(e_arp->thwaddr[0]));
    } else {
	for( i = 0; i < PSEUDOLEN; i++)
	    e_arp->thwaddr[i] = '\0';
    }
    e_arp->tprotoaddr = a_arp->tprotoaddr;
}

/* Convert an AX.25 address in network form into a pseudo-ethernet address */
void
AX25ToEther(unsigned char far *ax25, ether_t *ether)
{
    register int i;
    register int ssid;

    for(i = 0; i < AXALEN; i++) {
	    if( ax25[i] != qst[i] )
	        break;
    }
    if (i == AXALEN) {
	    for( i = 0; i < PSEUDOLEN; i++)
	        *ether++ = (unsigned char)0xff;
	return;
    }

    ssid = (ax25[6] & SSID);		/* LSB will be zero */
    ether[0] = (ax25[0] >> 1) & 0x7f;

    for(i = 1; i < 6 ; i++) {
	ether[i] = ax25[i] | (ssid & 1);
	ssid = ssid >> 1;
    }
}

/* Routine to convert an ARP packet containing pseudo-Ethernet addresses
 * to the equivalent packet containing AX.25 addresses.
 *
 * This routine assumes that the constant fields of the AX.25 ARP packet
 * have been statically initialized.
 */
void
EArpToAArp(EtherArp far *e_arp, AX25Arp *a_arp)
{
	int i;

	a_arp->opcode = e_arp->opcode;

 	EtherToAX25((ether_t *)&(e_arp->shwaddr[0]), &(a_arp->shwaddr[0]));
 	a_arp->sprotoaddr = e_arp->sprotoaddr;

	if( e_arp->opcode == ARP_REPLY )
		EtherToAX25((ether_t *)&(e_arp->thwaddr[0]), &(a_arp->thwaddr[0]));
	else {
		for( i = 0; i < AXALEN; i++)
	        a_arp->thwaddr[i] = '\0';
	}
	a_arp->tprotoaddr = e_arp->tprotoaddr;
}

/* Convert a pseudo-Ethernet address to an AX.25 address in network form */
void
EtherToAX25(ether_t *ether, unsigned char *ax25)
{
    register int i;
    register int ssid;

    for (i = 0; i < PSEUDOLEN; i++) {
	    if( ether[i] != (unsigned char)0xff )
	        break;
    }

    if( i == PSEUDOLEN ) {
      for( i = 0; i < AXALEN; i++)
	*ax25++ = qst[i];
      return;
    }
    
    ssid = 0;

    for(i = 5; i > 0; i--) {
	ssid = (ssid << 1) | (ether[i] & 1);
	ax25[i] = ether[i] & 0xfe;
    }
    ax25[0] = ether[0] << 1;
    ax25[6] = ssid;
}

#ifndef DRIVER

/* Convert an AX.25 address in null terminated string into network form */

void
StrToAX25(char *s, unsigned char *ax25)
{

    unsigned char *axp = &ax25[0];

    while(*s != '\0' && *s != '-') {
	    *axp++ = toupper(*s) << 1;
        s++;
    }

    while( axp < &ax25[6] )
        *axp++ = ' ' << 1;
		    
    if( *s++ == '-')
        *axp = atoi(s) << 1;
	else
        *axp = '\0';

}

/* Convert an AX.25 address in network form into a null terminated string */
void
AX25ToStr(unsigned char *ax25, char *s)
{
    int i;

    for( i = 0; i < 6; i++) {
        if( (*ax25 >> 1) != ' ')
            *s++ = *ax25 >> 1;
        ax25++;
    }
    if( *ax25 != '\0' ) {
        *s++ = '-';
        (void)itoa(*ax25 >> 1, s, 10);
    } else
        *s = '\0';
}
#endif


/*
 * A test driver for the AX.25 utility routines.
 */
#undef TEST
#ifdef TEST
#include <stdio.h>

main(int argc, char **argv)
{
    char c;
    unsigned char ax25[7];
    unsigned char ether[6];
    char callsign[8];
    int i;

    while( 1 ) {
	printf("[A]X.25 or [E]thernet?: ");
	scanf(" %c", &c);

	if( c == 'A' ) {

	    /* Pervert the callsign into AX.25 form including shift */
	    scanf(" %s", callsign);
            StrToAX25(callsign, ax25);

	    /* Translate, and translate back */
	    AX25ToEther(ax25, ether);
	    for( i = 0; i < 5; i++)
		printf("%02x-", ether[i]);
	    printf("%02x\n", ether[5]);

	    EtherToAX25(ether, ax25);
            AX25ToStr(ax25, callsign);
            printf("%s\n", callsign);

	} else {

	    scanf(" %x-%x-%x-%x-%x-%x", &ether[0], &ether[1], &ether[2],
		  &ether[3], &ether[4], &ether[5]);

	    /* Translate, and translate back */
	    EtherToAX25(ether, ax25);
	    for(i = 0; i < 6; i++)
		printf("%c", ax25[i]>>1);
	    printf("-%d\n", ax25[6]>>1);

	    AX25ToEther(ax25, ether);
	    for( i = 0; i < 5; i++)
		printf("%02x-", ether[i]);
	    printf("%02x\n", ether[5]);

	}
    }
}
#endif
