// $Id: PacketCapture.java,v 1.9 2001/06/18 05:05:30 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.capture;


/**
 * This class is the core of packet capture in jpcap. It provides a 
 * high-level interface for capturing network packets by encapsulating 
 * libpcap.
 * <p>
 * If you want to capture network packets, implement PacketListener
 * and register with an instance of this class. When packets arrive, 
 * the object will call you back via packetArrived().
 * <p>
 * Examples can be found in net.sourceforge.jpcap.tutorial.
 * <p>
 * For more documentation on this class's methods, see PacketCaptureCapable;
 * Javadoc is 'inherited' from this interface.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.9 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/06/18 05:05:30 $
 */
public class PacketCapture extends PacketCaptureBase
  implements PacketCaptureCapable
{
  /**
   * Name of the java-enabled wrapper around libpcap.
   */
  protected static String LIB_PCAP_WRAPPER = "jpcap";


  // methods for controlling a packet capture session

  public void open(String device, boolean promiscuous)
   throws CaptureDeviceOpenException {
    open(device, DEFAULT_SNAPLEN, promiscuous, DEFAULT_TIMEOUT);
  }

  public native void open(String device, int snaplen, 
                          boolean promiscuous, int timeout) 
    throws CaptureDeviceOpenException;

  public native void setFilter(String filterExpression, boolean optimize)
    throws InvalidFilterException;

  public native void capture(int count)
    throws CapturePacketException;

  public CaptureStatistics getStatistics() {
    setupStatistics();

    // transfer the values setup by the native method into
    // the container and return to the caller
    return new CaptureStatistics(receivedCount, droppedCount);
  }

  public native void close();


  // the following methods could be static, but aren't so that they 
  // can be included in the PacketCaptureCapable interface.

  public native String findDevice() 
    throws CaptureDeviceNotFoundException;

  public native int getNetwork(String device)
    throws CaptureConfigurationException;

  public native int getNetmask(String device)
    throws CaptureConfigurationException;

  public native int getLinkLayerType()
    throws CaptureConfigurationException;


  /**
   * The packet capture library sets up the statistic counter members
   * when this method is invoked internally.
   */
  private native void setupStatistics();


  // static initialization

  static {
    System.err.print("PacketCapture: loading native library jpcap.. ");
    System.loadLibrary(LIB_PCAP_WRAPPER);
    System.err.println("ok");
  }


  private String _rcsid = 
    "$Id: PacketCapture.java,v 1.9 2001/06/18 05:05:30 pcharles Exp $";
}
