// $Id: jpcap.c,v 1.11 2001/07/25 21:34:06 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/

// A libpcap wrapper with java native hooks.
//
// @author Jonas Lehmann and Patrick Charles
// @version $Revision: 1.11 $
// @lastModifiedBy $Author: pcharles $
// @lastModifiedAt $Date: 2001/07/25 21:34:06 $
//
#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <jni.h>
// native function headers generated by javah
#include "net_sourceforge_jpcap_capture_PacketCapture.h"


/*****************************************************************************
/* global variables 
 */
static pcap_t *PD = NULL; // packet capture device.
jobject javaObject; // reference to java object hooked into this wrapper lib.
JNIEnv *javaEnvironment; // java vm containing running java object.
const int VERBOSE = 0; // for debugging

const char *DEVICE_NOT_FOUND_EXCEPTION =
  "net/sourceforge/jpcap/capture/CaptureDeviceNotFoundException";
const char *DEVICE_OPEN_EXCEPTION =
  "net/sourceforge/jpcap/capture/CaptureDeviceOpenException";
const char *CAPTURE_EXCEPTION = 
  "net/sourceforge/jpcap/capture/CapturePacketException";
const char *CONFIGURATION_EXCEPTION = 
  "net/sourceforge/jpcap/capture/CaptureConfigurationException";
const char *INVALID_FILTER_EXCEPTION = 
  "net/sourceforge/jpcap/capture/InvalidFilterException";
const char *CLASS_EXCEPTION = 
  "java/lang/ClassNotFoundException";


/*****************************************************************************
 * utility functions
 */

/*
 * Write captured packet data to stderr.
 */
void printPacket(u_char *user, u_char *cp, u_int caplen, u_int length, 
                 u_int seconds, u_int useconds)
{
  register u_int i, s;
  register int nshorts;
  
  nshorts = (u_int) caplen / sizeof(u_short);

  fprintf(stderr, "Packet: u = %s, l = %d of %d, t = %u.%06u, d = ", 
          user, caplen, length, seconds, useconds);

  i = 0;
  while (--nshorts >= 0) {
    if ((i++ % 8) == 0)
      fprintf(stderr, "\n\t\t\t");
    s = *cp++;
    fprintf(stderr, " %02x%02x", s, *cp++);
  }

  if (caplen & 1) {
    if ((i % 8) == 0)
      (void)fprintf(stderr, "\n\t\t\t");
    (void)fprintf(stderr, " %02x", *cp);
  }

  fprintf(stderr, "\t\n");
}

/*
 * Create and throw an exception into the Java VM using this wrapper.
 * If the specified exception class can't be found, the VM will throw
 * a class not found exception.
 */
void throwException(JNIEnv *env, const char *excClassName, char *message) {
  // create an instance of the specified exception
  jclass exception = (*env)->FindClass(env, excClassName);

  if(exception != NULL) 
    // throw the new exception back to the java wrapper
    (*env)->ThrowNew(env, exception, message);

  // free the local reference. if exception is still null, delete is a noop.
  (*env)->DeleteLocalRef(env, exception);
}


/*****************************************************************************
 * implementation of pcap wrapper functions with java hooks
 */

/*
 * Callback called by pcap_loop() for processing packet data.
 * This method is called whenever a packet is captured by the pcap_loop()
 * in _capture(). It decomposes the captured data and transfers it 
 * to the PacketCapture object in the attached java VM.
 */
void processData(u_char *user, struct pcap_pkthdr *h, u_char *sp) {
  jmethodID mid;
  jclass cls;
  jbyteArray jba;
  int seconds;
  int useconds;
  register const struct timeval *tvp;

  /* decompose the timestamp contained in the packet header */
  tvp = &h->ts;
  seconds = tvp->tv_sec;
  useconds = tvp->tv_usec;

  if(VERBOSE) {
    (void)printPacket(user, sp, h->caplen, h->len, seconds, useconds);
  }

  /* Check to see if any unhandled exceptions are pending in the VM before 
     calling back the packet handler. If an exception is pending, return 
     control immediately so that the VM can handle it. 
  */
  if((*javaEnvironment)->ExceptionOccurred(javaEnvironment))
    return;

  cls = (*javaEnvironment)->GetObjectClass(javaEnvironment, javaObject);

  mid = (*javaEnvironment)->
    GetMethodID(javaEnvironment, cls, "handlePacket", "(IIII[B)V");

  if(mid == 0) {
    fprintf(stderr, "jpcap: method lookup failure\n");
    fflush(stdout); fflush(stderr);
    return;
  }

  if(VERBOSE)
    fprintf(stderr, "jpcap: invoking java callback..\t\n");

  jba = (*javaEnvironment)->NewByteArray(javaEnvironment, h->caplen);

  (*javaEnvironment)->SetByteArrayRegion
    (javaEnvironment, jba, 0, h->caplen, sp); 

  // dispatch captured data to the handle method in the java capture object..
  (*javaEnvironment)->CallVoidMethod(javaEnvironment, javaObject, mid, 
                                     h->len, h->caplen, seconds, useconds,
                                     jba);
  
  if(VERBOSE) {
    fprintf(stderr, "jpcap: regained control from java\t\n");
    fflush(stdout); fflush(stderr);
  }
}

/*
 * Look up a network device.
 * Throws a CaptureDeviceNotFoundException if no suitable device is found.
 *
 * JNI PacketCapture.findDevice()
 */
JNIEXPORT jstring JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_findDevice
(JNIEnv *env, jobject obj) {
  char ebuf[PCAP_ERRBUF_SIZE];
  char *device = pcap_lookupdev(ebuf);

  if(device == NULL)
    // if no device can be found, throw an exception back to the java wrapper
    throwException(env, DEVICE_NOT_FOUND_EXCEPTION, ebuf);
  else 
    return (*env)->NewStringUTF(env, device);
}

/*
 * Open a network device for data capture.
 * Throws an OpenException if the device cannot be opened.
 * If the open is successful, sets the global device pointer.
 *
 * JNI PacketCapture.open()
 */
JNIEXPORT void JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_open
  (JNIEnv *env, jobject object, jstring jdevice, jint snaplen, 
   jboolean promiscuous, jint timeout)
{
  char ebuf[PCAP_ERRBUF_SIZE];
  int linkType;
  jfieldID fid;
  jclass cls;
  const char *device;

  if(jdevice == NULL) {
    // device can't be null; throw an exception back to java wrapper
    throwException(env, DEVICE_OPEN_EXCEPTION, "null device specified");
    return;
  }

  device = (*env)->GetStringUTFChars(env, jdevice, 0);

  PD = pcap_open_live((char*)device, snaplen, promiscuous, timeout, ebuf);
  if(PD == NULL) {
    // if the open fails, throw an exception back to the java wrapper
    throwException(env, DEVICE_OPEN_EXCEPTION, ebuf);
    return;
  }

  // set the link type in the java wrapper encapsulating the capture system
  linkType = pcap_datalink(PD);
  cls = (*env)->GetObjectClass(env, object);
  fid = (*env)->GetFieldID(env, cls, "linkType", "I");
  if (fid == 0) {
    // catch native/java field inconsistencies
    throwException(env, CLASS_EXCEPTION, 
                   "couldn't find member PacketCapture.linkType!");
    return;
  }
  (*env)->SetIntField(env, object, fid, linkType);
}

/*
 * Capture packets.
 * Throws a CapturePacketException if a problem occurs during the capture.
 *
 * JNI PacketCapture.capture()
 */
JNIEXPORT void JNICALL Java_net_sourceforge_jpcap_capture_PacketCapture_capture
(JNIEnv *env, jobject obj, jint count) {
  // assign global vm environment and object to for callback entry point
  javaEnvironment = env;
  javaObject = obj;

  // make sure a device is open before allowing the capture session to start
  if(PD == NULL) {
    throwException(env, CAPTURE_EXCEPTION, 
                   "a device must be open before capturing packets");
    return;
  }

  if (pcap_loop(PD, count, (pcap_handler)processData, 0) < 0)
    // if the capture failed, throw an exception back to the java wrapper
    throwException(env, CAPTURE_EXCEPTION, pcap_geterr(PD));
}

/*
 * Fetch the network associated with the device.
 * Throws a CaptureConfigurationException if the device is messed up.
 * 
 * JNI PacketCapture.getNetwork()
 */
JNIEXPORT jint JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_getNetwork
(JNIEnv *env, jobject obj, jstring jdevice) {
  char ebuf[PCAP_ERRBUF_SIZE];
  int localnet = 0;
  int netmask = 0;

  const char *device = (*env)->GetStringUTFChars(env, jdevice, 0);

  if(pcap_lookupnet((char*)device, &localnet, &netmask, ebuf) < 0)
    // if the lookup failed, throw an exception back to the java wrapper
    throwException(env, CONFIGURATION_EXCEPTION, ebuf);
  else
    return localnet;
}

/*
 * Fetch the netmask associated with the device.
 * Throws a CaptureConfigurationException if the device is messed up.
 *
 * JNI PacketCapture.getNetmask()
 */
JNIEXPORT jint JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_getNetmask
(JNIEnv *env, jobject obj, jstring jdevice) {
  char ebuf[PCAP_ERRBUF_SIZE];
  const char *device = (*env)->GetStringUTFChars(env, jdevice, 0);
  int localnet = 0;
  int netmask = 0;

  if(pcap_lookupnet((char*)device, &localnet, &netmask, ebuf) < 0)
    // if the lookup failed, throw an exception back to the java wrapper
    throwException(env, CONFIGURATION_EXCEPTION, ebuf);
  else
    return netmask;
}

/*
 * Fetch the link layer type associated with this device.
 *
 * JNI PacketCapture.getLinkLayerType()
 */
JNIEXPORT jint JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_getLinkLayerType
(JNIEnv *env, jobject obj) {
  return pcap_datalink(PD);
}

/*
 * Create, compile and activate a filter from a filter expression.
 *
 * JNI PacketCapture.setFilter()
 */
JNIEXPORT void JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_setFilter
(JNIEnv *env, jobject object, jstring jfilter, jboolean optimize) {
  struct bpf_program bpp;
  const char *filter = (*env)->GetStringUTFChars(env, jfilter, 0);

  // the device must be open in order to set the filter.
  if(PD == NULL) {
    throwException(env, INVALID_FILTER_EXCEPTION, 
                   "A device must be open before setting the filter.");
    return;
  }

  // compile bpf program
  if(pcap_compile(PD, &bpp, (char*)filter, optimize, 0) == -1) {
    // if the filter wouldn't compile, throw an exception back to java
    throwException(env, INVALID_FILTER_EXCEPTION, pcap_geterr(PD));
    return;
  }

  // activate program
  if(pcap_setfilter(PD, &bpp) == -1)
    // if the filter couldn't be activated, throw an exception back to java
    throwException(env, INVALID_FILTER_EXCEPTION, pcap_geterr(PD));
}

/*
 * Close the capture device.
 *
 * JNI PacketCapture.close()
 */
JNIEXPORT void JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_close
(JNIEnv *env, jobject object) {
  if(PD != NULL)
    pcap_close(PD);
}

/*
 * Fetch and set statistics in the java vm.
 * 
 * JNI PacketCapture.setupStatistics()
 */
JNIEXPORT void JNICALL 
Java_net_sourceforge_jpcap_capture_PacketCapture_setupStatistics
(JNIEnv *env, jobject object) {
  struct pcap_stat stat;

  if (PD != NULL && pcap_file(PD) == NULL) {
    if (pcap_stats(PD, &stat) < 0) {
      /*
       * todo: throw the following as an exception..
       fprintf(stderr, "jpcap: pcap_stats(%s) %s\n", 
         g_device, pcap_geterr(PD));
      */
    }
    else {
      jfieldID fid;
      jclass cls = (*env)->GetObjectClass(env, object);
      // received count
      fid = (*env)->GetFieldID(env, cls, "receivedCount", "I");
      if (fid == 0)
        return;
      (*env)->SetIntField(env, object, fid, stat.ps_recv);
      // dropped count
      fid = (*env)->GetFieldID(env, cls, "droppedCount", "I");
      if (fid == 0)
        return;
      (*env)->SetIntField(env, object, fid, stat.ps_drop);
    }
  }
}

/*
 * This function is currently unused. might be used later by capture
 * to potentially adjust the snapshot length in the event that the 
 * capture client got it wrong.
 */
int gx_snaplen;
int adjustSnaplen(int verbose) {
  int i = -1;
  i = pcap_snapshot(PD);
  if(i == -1)
    return(0);
  if(i != gx_snaplen) {
    if(verbose) {
      printf("jpcap: snaplen was adjusted from %d to %d.\n", gx_snaplen, i);
      gx_snaplen = i;
    }
  } else {
    if(verbose)
      printf("jpcap: snaplen confirmed at %d.\n", gx_snaplen);
  }
  return(1);
}


const char *rcsid = 
  "$Id: jpcap.c,v 1.11 2001/07/25 21:34:06 pcharles Exp $";
