// $Id: EthernetPacket.java,v 1.13 2001/07/01 21:35:34 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * An ethernet packet.
 * <p>
 * Contains link-level header and data payload encapsulated by an ethernet
 * packet.
 * <p>
 * There are currently two subclasses. IP and ARP protocols are supported.
 * IPPacket extends with ip header and data information.
 * ARPPacket extends with hardware and protocol addresses.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.13 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/01 21:35:34 $
 */
public class EthernetPacket extends Packet implements EthernetFields
{
  /**
   * Construct a new ethernet packet.
   * <p>
   * For the purpose of jpcap, when the type of ethernet packet is 
   * recognized as a protocol for which a class exists network library, 
   * then a more specific class like IPPacket or ARPPacket is instantiated.
   * The subclass can always be cast into a more generic form.
   */
  public EthernetPacket(int lLen, byte [] bytes) {
    this.header = PacketEncoding.extractHeader(0, lLen, bytes);
    this.data = PacketEncoding.extractData(0, lLen, bytes);
  }

  /**
   * Fetch the ethernet header, excluding data payload.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the ethernet header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Fetch the hardware address from which the packet originated.
   */
  public String getSourceHwAddress() {
    return MACAddress.extract(ETH_SRC_POS, header);
  }

  /**
   * Fetch the hardware address of the target host.
   */
  public String getDestinationHwAddress() {
    return MACAddress.extract(ETH_DST_POS, header);
  }

  /**
   * Fetch the code stored in the ethernet type field which indicates the 
   * type of datagram embedded in this packet. @see EthernetProtocols.
   */
  public int getProtocol() {
    return ArrayHelper.extractInteger(header, ETH_CODE_POS, ETH_CODE_LEN);
  }

  /**
   * Convert this ethernet packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this ethernet packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("EthernetPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(getSourceHwAddress() + " -> " + 
                  getDestinationHwAddress());
    buffer.append(" proto=0x" + Integer.toHexString(getProtocol()));
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.DARK_GRAY;
  }


  /**
   * Ethernet header.
   */
  byte [] header;

  /**
   * Ethernet data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: EthernetPacket.java,v 1.13 2001/07/01 21:35:34 pcharles Exp $";
}
