// $Id: ICMPPacket.java,v 1.13 2001/07/30 00:01:49 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * An ICMP packet.
 * <p>
 * Extends an IP packet, adding an ICMP header and ICMP data payload.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.13 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/30 00:01:49 $
 */
public class ICMPPacket extends IPPacket implements ICMPFields
{
  public ICMPPacket(int lLen, byte [] bytes) {
    super(lLen, bytes);

    // offset to icmp header start is length of link-layer plus IP header
    int offset = lLen + IPPacket.getHeaderLength(lLen, bytes);

    this.header = PacketEncoding.extractHeader(offset, ICMP_HEADER_LEN, bytes);
    this.data = PacketEncoding.extractData(offset, ICMP_HEADER_LEN, bytes);
  }

  /**
   * Fetch the icmp header, excluding icmp data payload.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the icmp header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Fetch the ICMP message type, including subcode. Return value can be 
   * used with ICMPMessage.getDescription().
   * @return a 2-byte value containing the message type in the high byte
   * and the message type subcode in the low byte.
   */
  public int getMessageCode() {
    return ArrayHelper.extractInteger(header, 
                                      ICMP_CODE_POS, ICMP_CODE_LEN * 2);
  }

  /**
   * Fetch the ICMP message type code. Formerly .getMessageType().
   */
  public int getMessageMajorCode() {
    return ArrayHelper.extractInteger(header, ICMP_CODE_POS, ICMP_CODE_LEN);
  }

  /**
   * For backward compatibility. @deprecated use getMessageMajorCode().
   */
  public int getMessageType() {
    return getMessageMajorCode();
  }

  /**
   * Fetch the ICMP message subcode.
   */
  public int getMessageMinorCode() {
    return ArrayHelper.extractInteger(header, ICMP_CODE_POS + 1, 
                                      ICMP_CODE_LEN);
  }

  /**
   * Fetch the ICMP header checksum.
   */
  public int getChecksum() {
    return ArrayHelper.extractInteger(header, ICMP_CSUM_POS, ICMP_CSUM_LEN);
  }


  /**
   * Convert this ICMP packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this ICMP packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("ICMPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(ICMPMessage.getDescription(getMessageCode()));
    buffer.append(", ");
    buffer.append(getSourceAddress() + " -> " + 
                  getDestinationAddress());
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.LIGHT_BLUE;
  }


  /**
   * ICMP header.
   */
  byte [] header;

  /**
   * ICMP data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: ICMPPacket.java,v 1.13 2001/07/30 00:01:49 pcharles Exp $";
}
