// $Id: IGMPPacket.java,v 1.1 2001/07/30 00:00:02 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * An IGMP packet.
 * <p>
 * Extends an IP packet, adding an IGMP header and IGMP data payload.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.1 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/30 00:00:02 $
 */
public class IGMPPacket extends IPPacket implements IGMPFields
{
  public IGMPPacket(int lLen, byte [] bytes) {
    super(lLen, bytes);

    // offset to igmp header start is length of link-layer plus IP header
    int offset = lLen + IPPacket.getHeaderLength(lLen, bytes);

    this.header = PacketEncoding.extractHeader(offset, IGMP_HEADER_LEN, bytes);
    this.data = PacketEncoding.extractData(offset, IGMP_HEADER_LEN, bytes);
  }

  /**
   * Fetch the igmp header, excluding igmp data payload.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the igmp header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Fetch the IGMP message type, including subcode. Return value can be 
   * used with IGMPMessage.getDescription().
   * @return a 2-byte value containing the message type in the high byte
   * and the message type subcode in the low byte.
   */
  public int getMessageType() {
    return ArrayHelper.extractInteger(header, IGMP_CODE_POS, IGMP_CODE_LEN);
  }

  /**
   * Fetch the IGMP max response time.
   */
  public int getMaxResponseTime() {
    return ArrayHelper.extractInteger(header, IGMP_MRSP_POS, IGMP_MRSP_LEN);
  }

  /**
   * Fetch the IGMP header checksum.
   */
  public int getChecksum() {
    return ArrayHelper.extractInteger(header, IGMP_CSUM_POS, IGMP_CSUM_LEN);
  }

  /**
   * Fetch the IGMP group address.
   */
  public String getGroupAddress() {
    return IPAddress.extract(IGMP_GADDR_POS, header);
  }


  /**
   * Convert this IGMP packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this IGMP packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("IGMPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(IGMPMessage.getDescription(getMessageType()));
    buffer.append(", ");
    buffer.append(getGroupAddress() + ": ");
    buffer.append(getSourceAddress() + " -> " + 
                  getDestinationAddress());
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.BROWN;
  }


  /**
   * IGMP header.
   */
  byte [] header;

  /**
   * IGMP data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: IGMPPacket.java,v 1.1 2001/07/30 00:00:02 pcharles Exp $";
}
