// $Id: IPPacket.java,v 1.15 2001/07/30 00:02:41 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * An IP protocol packet.
 * <p>
 * Extends an ethernet packet, adding IP header information and an IP 
 * data payload. 
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.15 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/30 00:02:41 $
 */
public class IPPacket extends EthernetPacket implements IPFields
{
  /**
   * Create a new IP packet.
   */
  public IPPacket(int lLen, byte [] bytes) {
    super(lLen, bytes);

    this.header = PacketEncoding.extractHeader(lLen, IP_HEADER_LEN, bytes);

    // fetch the actual header length from the header data
    int realHdrLength = getIpHeaderLength();

    if(IP_HEADER_LEN != realHdrLength)
      // if options are present, resize the header to include them
      this.header = PacketEncoding.extractHeader(lLen, realHdrLength, bytes);

    this.data = PacketEncoding.extractData(lLen, realHdrLength, bytes);
  }

  /**
   * Get the IP version code. i.e. 0x4 is IPv4.
   */
  public int getVersion() {
    // the version is stored in the high nibble of the length/version field
    return 
      (ArrayHelper.extractInteger(header, IP_VER_POS, IP_VER_LEN) >> 4) & 0xf;
  }

  /**
   * Fetch the IP header length in bytes.
   * @param lLen the offset in bytes past the link-layer header.
   * @param bytes an ethernet header containing an IP datagram.
   */
  protected static int getHeaderLength(int lLen, byte [] bytes) {
    byte [] hdr = PacketEncoding.extractHeader(lLen, IP_HEADER_LEN, bytes);
    // fetch the actual header length from the header data
    return getHeaderLength(hdr);
  }

  /**
   * Fetch the IP header length in bytes.
   * @param hdr a truncated IP header.
   */
  protected static int getHeaderLength(byte [] hdr) {
    // should return the same value as header.length.
    // the length is stored in the low nibble of the length/version field.
    // It is stored as the number of 32-bit words.
    return 
      (ArrayHelper.extractInteger(hdr, IP_VER_POS, IP_VER_LEN) & 0xf) * 4;
  }

  /**
   * Fetch the IP header length in bytes.
   */
  protected int getIpHeaderLength() {
    return IPPacket.getHeaderLength(header);
  }

  /**
   * Fetch the IP header length in bytes.
   */
  public int getHeaderLength() {
    return getIpHeaderLength();
  }

  /**
   * Fetch the type of service. @see TypesOfService.
   */
  public int getTypeOfService() {
    return ArrayHelper.extractInteger(header, IP_TOS_POS, IP_TOS_LEN);
  }

  /**
   * Fetch the total length of the IP packet, including header and
   * data payload, in bytes.
   */
  public int getLength() {
    // should produce the same value as header.length + data.length
    return ArrayHelper.extractInteger(header, IP_LEN_POS, IP_LEN_LEN);
  }

  /**
   * Fetch the unique ID of this IP datagram. The ID normally 
   * increments by one each time a datagram is sent by a host.
   */
  public int getId() {
    return ArrayHelper.extractInteger(header, IP_ID_POS, IP_ID_LEN);
  }

  /**
   * Fetch fragmentation flags.
   */
  public int getFragmentFlags() {
    // fragment flags are the high 3 bits
    return (ArrayHelper.extractInteger(header, IP_FRAG_POS, IP_FRAG_LEN) 
            >> 13) & 0x7;
  }

  /**
   * Fetch the fragment offset.
   */
  public int getFragmentOffset() {
    // offset is the low 13 bits
    return ArrayHelper.extractInteger(header, IP_FRAG_POS, IP_FRAG_LEN) 
      & 0x1fff;
  }

  /**
   * Fetch the time to live. TTL sets the upper limit on the number of routers
   * through which this IP datagram is allowed to pass.
   */
  public int getTimeToLive() {
    return ArrayHelper.extractInteger(header, IP_TTL_POS, IP_TTL_LEN);
  }

  /**
   * Fetch the code indicating the type of protocol embedded in the IP
   * datagram. @see IPProtocols.
   */
  public int getProtocol() {
    return ArrayHelper.extractInteger(header, IP_CODE_POS, IP_CODE_LEN);
  }

  /**
   * Fetch the header checksum.
   */
  public int getChecksum() {
    return ArrayHelper.extractInteger(header, IP_CSUM_POS, IP_CSUM_LEN);
  }

  /**
   * Fetch the IP address of the host where the packet originated from.
   */
  public String getSourceAddress() {
    return IPAddress.extract(IP_SRC_POS, header);
  }

  /**
   * Fetch the IP address of the host where the packet is destined for.
   */
  public String getDestinationAddress() {
    return IPAddress.extract(IP_DST_POS, header);
  }

  /**
   * Fetch the IP header, excluding ip data.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the IP header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Convert this IP packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this IP packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("IPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(getSourceAddress() + " -> " + 
                  getDestinationAddress());
    buffer.append(" proto=" + getProtocol());
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Convert this IP packet to a more verbose string.
   */
  public String toColoredVerboseString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("IPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append("version=" + getVersion() + ", ");
    buffer.append("hlen=" + getHeaderLength() + ", ");
    buffer.append("tos=" + getTypeOfService() + ", ");
    buffer.append("length=" + getLength() + ", ");
    buffer.append("id=" + getId() + ", ");
    buffer.append("flags=0x" + Integer.toHexString(getFragmentFlags()) + ", ");
    buffer.append("offset=" + getFragmentOffset() + ", ");
    buffer.append("ttl=" + getTimeToLive() + ", ");
    buffer.append("proto=" + getProtocol() + ", ");
    buffer.append("sum=0x" + Integer.toHexString(getChecksum()) + ", ");
    buffer.append("src=" + getSourceAddress() + ", ");
    buffer.append("dest=" + getDestinationAddress());
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.WHITE;
  }


  /**
   * IP header.
   */
  byte [] header;

  /**
   * IP data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: IPPacket.java,v 1.15 2001/07/30 00:02:41 pcharles Exp $";
}
