// $Id: TCPPacket.java,v 1.16 2001/07/30 00:02:41 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * A TCP packet.
 * <p>
 * Extends an IP packet, adding a TCP header and TCP data payload.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.16 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/30 00:02:41 $
 */
public class TCPPacket extends IPPacket implements TCPFields
{
  /**
   * Create a new TCP packet.
   */
  public TCPPacket(int lLen, byte [] bytes) {
    super(lLen, bytes);

    // offset to tcp header start is length of link-layer plus IP header
    int offset = lLen + IPPacket.getHeaderLength(lLen, bytes);

    // initially assume default header length in order to extract from header
    this.header = PacketEncoding.extractHeader(offset, TCP_HEADER_LEN, bytes);

    // fetch the actual header length from the header data.
    int realHdrLength = getTcpHeaderLength();

    if(TCP_HEADER_LEN != realHdrLength)
      // if options are present, resize the header to include them
      this.header = PacketEncoding.extractHeader(offset, realHdrLength, bytes);

    this.data = PacketEncoding.extractData(offset, realHdrLength, bytes);
  }

  /**
   * Fetch the port number on the source host.
   */
  public int getSourcePort() {
    return ArrayHelper.extractInteger(header, TCP_SP_POS, TCP_PORT_LEN);
  }

  /**
   * Fetch the port number on the target host.
   */
  public int getDestinationPort() {
    return ArrayHelper.extractInteger(header, TCP_DP_POS, TCP_PORT_LEN);
  }

  /**
   * Fetch the packet sequence number.
   */
  public int getSequenceNumber() {
    return ArrayHelper.extractInteger(header, TCP_SEQ_POS, TCP_SEQ_LEN);
  }

  /**
   * Fetch the packet acknowledgment number.
   */
  public int getAcknowledgmentNumber() {
    return ArrayHelper.extractInteger(header, TCP_ACK_POS, TCP_ACK_LEN);
  }

  /**
   * Fetch the packet header length.
   */
  public int getHeaderLength() {
    // should return the same value as header.length.
    // the length is stored in the high nibble of the flag block.
    // It is stored as the number of 32-bit words.
    return ((ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
             >> 12) & 0xf) * 4;
  }

  /**
   * Fetch the packet header length.
   */
  protected int getTcpHeaderLength() {
    return getHeaderLength();
  }

  /**
   * Fetch the window size.
   */
  public int getWindowSize() {
    return ArrayHelper.extractInteger(header, TCP_WIN_POS, TCP_WIN_LEN);
  }

  /**
   * Fetch the header checksum.
   */
  public int getChecksum() {
    return ArrayHelper.extractInteger(header, TCP_CSUM_POS, TCP_CSUM_LEN);
  }

  /**
   * Fetch the urgent pointer.
   */
  public int getUrgentPointer() {
    return ArrayHelper.extractInteger(header, TCP_URG_POS, TCP_URG_LEN);
  }

  /**
   * Check the URG flag.
   * The flag indicates if the urgent pointer is valid.
   */
  public boolean isUrg() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_URG_MASK) != 0;
  }

  /**
   * Check the ACK flag.
   * The flag indicates whether the acknowledgment number is valid.
   */
  public boolean isAck() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_ACK_MASK) != 0;
  }

  /**
   * Check the PSH flag.
   * The flag indicates that the receiver should pass the data to the 
   * application as soon as possible.
   */
  public boolean isPsh() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_PSH_MASK) != 0;
  }

  /**
   * Check the RST flag.
   * The flag indicates that the session should be reset between
   * the sender and the receiver.
   */
  public boolean isRst() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_RST_MASK) != 0;
  }

  /**
   * Check the SYN flag.
   * The flag indicates that the sequence numbers should be synchronized
   * between the sender and receiver to initiate a connection.
   */
  public boolean isSyn() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_SYN_MASK) != 0;
  }

  /**
   * Check the FIN flag.
   * The flag indicates that the sender is finished sending data.
   */
  public boolean isFin() {
    return (ArrayHelper.extractInteger(header, TCP_FLAG_POS, TCP_FLAG_LEN) 
            & TCP_FIN_MASK) != 0;
  }

  /**
   * Fetch the tcp header, excluding tcp data payload.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the tcp header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Convert this TCP packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this TCP packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("TCPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(getSourceAddress());
    buffer.append('.');
    buffer.append(IPPort.getName(getSourcePort()));
    buffer.append(" -> ");
    buffer.append(getDestinationAddress());
    buffer.append('.');
    buffer.append(IPPort.getName(getDestinationPort()));
    if(isUrg()) 
      buffer.append(" urg[0x" + Integer.toHexString(getUrgentPointer()) + "]");
    if(isAck()) 
      buffer.append(" ack[0x" + 
                    Integer.toHexString(getAcknowledgmentNumber()) + "]");
    if(isPsh()) buffer.append(" psh");
    if(isRst()) buffer.append(" rst");
    if(isSyn()) 
      buffer.append(" syn[0x" + 
                    Integer.toHexString(getSequenceNumber()) + "]");
    if(isFin()) buffer.append(" fin");
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Convert this TCP packet to a verbose.
   */
  public String toColoredVerboseString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("TCPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append("sport=" + getSourcePort() + ", ");
    buffer.append("dport=" + getDestinationPort() + ", ");
    buffer.append("seqn=0x" + Integer.toHexString(getSequenceNumber()) + ", ");
    buffer.append("ackn=0x" + 
                  Integer.toHexString(getAcknowledgmentNumber()) + ", ");
    buffer.append("hlen=" + getHeaderLength() + ", ");
    buffer.append("urg=" + isUrg() + ", ");
    buffer.append("ack=" + isAck() + ", ");
    buffer.append("psh=" + isPsh() + ", ");
    buffer.append("rst=" + isRst() + ", ");
    buffer.append("syn=" + isSyn() + ", ");
    buffer.append("fin=" + isFin() + ", ");
    buffer.append("wsize=" + getWindowSize() + ", ");
    buffer.append("sum=0x" + Integer.toHexString(getChecksum()) + ", ");
    buffer.append("uptr=0x" + Integer.toHexString(getUrgentPointer()));
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.YELLOW;
  }


  /**
   * TCP header.
   */
  byte [] header;

  /**
   * TCP data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: TCPPacket.java,v 1.16 2001/07/30 00:02:41 pcharles Exp $";
}
