// $Id: UDPPacket.java,v 1.11 2001/07/30 00:02:41 pcharles Exp $

/***************************************************************************
 * Copyright (C) 2001, Patrick Charles and Jonas Lehmann                   *
 * Distributed under the Mozilla Public License                            *
 *   http://www.mozilla.org/NPL/MPL-1.1.txt                                *
 ***************************************************************************/
package net.sourceforge.jpcap.net;

import net.sourceforge.jpcap.util.AnsiEscapeSequences;
import net.sourceforge.jpcap.util.ArrayHelper;


/**
 * A UDP packet.
 * <p>
 * Extends an IP packet, adding a UDP header and UDP data payload.
 *
 * @author Patrick Charles and Jonas Lehmann
 * @version $Revision: 1.11 $
 * @lastModifiedBy $Author: pcharles $
 * @lastModifiedAt $Date: 2001/07/30 00:02:41 $
 */
public class UDPPacket extends IPPacket implements UDPFields
{
  /**
   * Create a new UDP packet.
   */
  public UDPPacket(int lLen, byte [] bytes) {
    super(lLen, bytes);

    // offset to udp header start is length of link-layer plus IP header
    int offset = lLen + IPPacket.getHeaderLength(lLen, bytes);

    this.header = PacketEncoding.extractHeader(offset, UDP_HEADER_LEN, bytes);
    this.data = PacketEncoding.extractData(offset, UDP_HEADER_LEN, bytes);
  }

  /**
   * Fetch the port number on the source host.
   */
  public int getSourcePort() {
    return ArrayHelper.extractInteger(header, UDP_SP_POS, UDP_PORT_LEN);
  }

  /**
   * Fetch the port number on the target host.
   */
  public int getDestinationPort() {
    return ArrayHelper.extractInteger(header, UDP_DP_POS, UDP_PORT_LEN);
  }

  /**
   * Fetch the total length of the UDP packet, including header and
   * data payload, in bytes.
   */
  public int getLength() {
    // should produce the same value as header.length + data.length
    return ArrayHelper.extractInteger(header, UDP_LEN_POS, UDP_LEN_LEN);
  }

  /**
   * Fetch the header checksum.
   */
  public int getChecksum() {
    return ArrayHelper.extractInteger(header, UDP_CSUM_POS, UDP_CSUM_LEN);
  }

  /**
   * Fetch the udp header, excluding udp data payload.
   */
  public byte [] getHeader() {
    return header;
  }

  /**
   * Fetch data portion of the udp header.
   */
  public byte [] getData() {
    return data;
  }

  /**
   * Convert this UDP packet to a readable string.
   */
  public String toString() {
    return toColoredString(false);
  }

  /**
   * Generate string with contents describing this UDP packet.
   * @param colored whether or not the string should contain ansi
   * color escape sequences.
   */
  public String toColoredString(boolean colored) {
    StringBuffer buffer = new StringBuffer();
    buffer.append('[');
    if(colored) buffer.append(getColor());
    buffer.append("UDPPacket");
    if(colored) buffer.append(AnsiEscapeSequences.RESET);
    buffer.append(": ");
    buffer.append(getSourceAddress());
    buffer.append('.');
    buffer.append(IPPort.getName(getSourcePort()));
    buffer.append(" -> ");
    buffer.append(getDestinationAddress());
    buffer.append('.');
    buffer.append(IPPort.getName(getDestinationPort()));
    buffer.append(" l=" + header.length + "," + data.length);
    buffer.append(']');

    return buffer.toString();
  }

  /**
   * Fetch ascii escape sequence of the color associated with this packet type.
   */
  public String getColor() {
    return AnsiEscapeSequences.LIGHT_GREEN;
  }


  /**
   * UDP header.
   */
  byte [] header;

  /**
   * UDP data.
   */
  byte [] data;

  private String _rcsid = 
    "$Id: UDPPacket.java,v 1.11 2001/07/30 00:02:41 pcharles Exp $";
}
