/*
Mognet, 802.11b Frame Analyzer
Copyright (C) 2001 Sean Whalen <lucidity@gmx.net>
http://chocobospore.org/mognet/

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import net.sourceforge.jpcap.net.*;
import net.sourceforge.jpcap.capture.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class Mognet implements RawPacketListener
{
	public void rawPacketArrived(RawPacket packet)
	{
		if(!captureEnabled)
		{
			return;
		}

		addFrame(packet);
	}

	public static void addFrame(RawPacket packet)
	{
		byte[] frame		=packet.getData();
		int frameMainType	=(frame[0] & 0x0c) >> 2;

		int frameSubType	=(frame[0] & 0xf0) >> 4;
		int frameType		=(frameMainType << 4) | frameSubType;
		int toDS		=frame[1] & P80211.FLAG_TO_DS;
		int fromDS		=frame[1] & P80211.FLAG_FROM_DS;

		Object[] data		=new Object[4];
		byte[] destMac		=new byte[6];
		byte[] srcMac		=new byte[6];
		data[0]			=(String)P80211.subtypeHash.get(new Integer(frameType));

		switch(frameType)
		{
			case P80211.DATA:
				if(toDS==0 && fromDS==0)
				{
					System.arraycopy(frame, 4, destMac, 0, 6);
					System.arraycopy(frame, 10, srcMac, 0, 6);
				}
				else if(toDS==0 && fromDS==1)
				{
					System.arraycopy(frame, 4, destMac, 0, 6);
					System.arraycopy(frame, 16, srcMac, 0, 6);
				}
				else if(toDS==1 && fromDS==0)
				{
					System.arraycopy(frame, 16, destMac, 0, 6);
					System.arraycopy(frame, 10, srcMac, 0, 6);
				}
				else if(toDS==1 && fromDS==1)
				{
					System.arraycopy(frame, 16, destMac, 0, 6);
					System.arraycopy(frame, 24, srcMac, 0, 6);
				}

				data[1]=MognetUtils.getHex(srcMac);
				data[2]=MognetUtils.getHex(destMac);
				data[3]=new String();

				break;

			case P80211.MGT_BEACON:
				System.arraycopy(frame, 4, destMac, 0, 6);
				System.arraycopy(frame, 10, srcMac, 0, 6);
				int length=frame[37] & 0xff;
				byte[] ssid=new byte[length];

				if(38+length<frame.length)
				{
					System.arraycopy(frame, 38, ssid, 0, length);
				}

				data[1]=MognetUtils.getHex(srcMac);
				data[2]=MognetUtils.getHex(destMac);
				data[3]=new String(ssid);
				break;

			default:
				data[1]=new String();
				data[2]=data[1];
				data[3]=data[1];
				break;
		}


		captureTableModel.addRow(data);
		captureScrollBar.setValue(captureScrollBar.getMaximum());		

		if(simpleMode)
		{
			frameVector.add(frameCount, null);
		}
		else
		{
			frameVector.add(frameCount, frame);
		}

		++frameCount;
	}

	private void layoutGui()
	{
		frame=new JFrame("Mognet");
		frame.addWindowListener(new WindowAdapter()
			{
				public void windowClosing(WindowEvent ev)
				{
					System.exit(0);
				}
			});

		contents=frame.getContentPane();
		contents.setLayout(new BorderLayout());

		status=new JLabel("Click Capture/Start to begin");

		JMenuBar menuBar=new JMenuBar();
		frame.setJMenuBar(menuBar);

		JMenu fileMenu=new JMenu("File");
		fileMenu.setMnemonic(KeyEvent.VK_F);

		fileMenuLoad=new JMenu("Load Frames from File");
		fileMenuLoad.addItemListener(new ItemListener()
			{
				public void itemStateChanged(ItemEvent ev)
				{
					fileMenuLoad.removeAll();
					File[] logfiles=new File(".").listFiles();
					int entries=0;

					for(int i=0; i<logfiles.length; i++)
					{
						String filename=logfiles[i].toString();
						
						if(filename.indexOf("mognet-")!=-1 && filename.endsWith(".log"))
						{
							JMenuItem fileMenuLoadEntry=new JMenuItem(filename);
							fileMenuLoadEntry.addActionListener(new LoadListener(filename, status));
							fileMenuLoad.add(fileMenuLoadEntry);
							entries++;
						}
					}
							
					if(entries==0)
					{
						JMenuItem fileMenuLoadEntry=new JMenuItem("No logs found");
						fileMenuLoad.add(fileMenuLoadEntry);
					}
				}
			});

		JMenuItem fileMenuSave=new JMenuItem("Save Frames to File");
		fileMenuSave.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					try
					{
						new LogWriter(frameVector);
						status.setText("Saved "+frameCount+" frames");
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
						status.setText("Error saving to file");
					}
				}
			});

		JMenuItem fileMenuExit=new JMenuItem("Exit");
		fileMenuExit.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					System.exit(0);
				}
			});

		fileMenu.add(fileMenuLoad);
		fileMenu.add(fileMenuSave);
		fileMenu.add(fileMenuExit);		

		JMenu captureMenu=new JMenu("Capture");
		captureMenu.setMnemonic(KeyEvent.VK_C);

		captureMenuStart=new JCheckBoxMenuItem("Start", captureEnabled);
		captureMenuStart.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					captureEnabled=!captureEnabled;
					captureMenuStop.setState(false);
					status.setText("Capture started");
				}
			});

		captureMenuStop=new JCheckBoxMenuItem("Stop", !captureEnabled);
		captureMenuStop.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					captureEnabled=!captureEnabled;
					captureMenuStart.setState(false);
					status.setText("Capture stopped");
				}
			});

		JMenuItem captureMenuClear=new JMenuItem("Clear frames");
		captureMenuClear.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					MognetUtils.clearTable(captureTableModel);
					detailArea.setText("");
					frameCount=0;
				}
			});

		captureMenu.add(captureMenuStart);
		captureMenu.add(captureMenuStop);
		captureMenu.add(captureMenuClear);

		JMenu modeMenu=new JMenu("Mode");
		modeMenu.setMnemonic(KeyEvent.VK_M);

		modeMenuSimple=new JCheckBoxMenuItem("Simple", simpleMode);
		modeMenuSimple.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					simpleMode=!simpleMode;
					modeMenuAdvanced.setState(false);
					splitPane.setBottomComponent(null);
				}
			});

		modeMenuAdvanced=new JCheckBoxMenuItem("Advanced", !simpleMode);
		modeMenuAdvanced.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent ev)
				{
					simpleMode=!simpleMode;
					modeMenuSimple.setState(false);
					splitPane.setBottomComponent(tabbedPane);
					splitPane.setDividerLocation(0.5);
				}
			});

		modeMenu.add(modeMenuSimple);
		modeMenu.add(modeMenuAdvanced);

		menuBar.add(fileMenu);
		menuBar.add(captureMenu);
		menuBar.add(modeMenu);
	
		captureTableModel=new DefaultTableModel(captureCols, 0);
		captureTable=new JTable(captureTableModel);
		captureTable.getSelectionModel().addListSelectionListener(new ListSelectionListener()
			{
				public void valueChanged(ListSelectionEvent ev)
				{
					if(!ev.getValueIsAdjusting())
					{
						ListSelectionModel m=(ListSelectionModel)ev.getSource();
						int row=m.getMinSelectionIndex();
						byte[] frame=(byte[])frameVector.elementAt(row);

						if(frame!=null)
						{
							status.setText("Frame number: "+row+", Frame size: "+frame.length+" bytes");
							detailArea.setText(P80211.dissect(frame));
							MognetUtils.showHex(frame, hexTableModel, hexRowSize);
							MognetUtils.showAscii(frame, asciiTableModel, asciiRowSize);
						}
					}
				}
			});

		captureScrollPane=new JScrollPane(captureTable);
		captureScrollBar=captureScrollPane.getVerticalScrollBar();

		detailArea=new JTextArea();
		detailScrollPane=new JScrollPane(detailArea);
		detailScrollBar=detailScrollPane.getVerticalScrollBar();

		Object[] hexCols=new Object[hexRowSize];
		for(int i=0; i<hexRowSize; i++)
		{
			hexCols[i]=new Integer(i);
		}

		hexTableModel=new DefaultTableModel(0, hexRowSize);
		hexTableModel.setColumnIdentifiers(hexCols);
		hexTable=new JTable(hexTableModel);
		hexScrollPane=new JScrollPane(hexTable);

		Object[] asciiCols=new Object[asciiRowSize];
		for(int i=0; i<asciiRowSize; i++)
		{
			asciiCols[i]=new Integer(i);
		}

		asciiTableModel=new DefaultTableModel(0, asciiRowSize);
		asciiTableModel.setColumnIdentifiers(asciiCols);
		asciiTable=new JTable(asciiTableModel);
		asciiScrollPane=new JScrollPane(asciiTable);

		tabbedPane=new JTabbedPane(1);
		tabbedPane.insertTab("Detail", null, detailScrollPane, null, 0);
		tabbedPane.insertTab("Hex Dump", null, hexScrollPane, null, 1);
		tabbedPane.insertTab("ASCII Dump", null, asciiScrollPane, null, 2);

		splitPane=new JSplitPane(JSplitPane.VERTICAL_SPLIT, false);
		splitPane.setTopComponent(captureScrollPane);

		if(simpleMode)
		{
			splitPane.setBottomComponent(null);
		}
		else
		{
			splitPane.setBottomComponent(tabbedPane);
		}

		contents.add(splitPane, BorderLayout.CENTER);
		contents.add(status, BorderLayout.SOUTH);

		frame.setSize(400,300);
		frame.show();
		
		splitPane.setDividerLocation(0.5);
	}

	public Mognet(String device)
	{
		try
		{
			if(System.getProperty("os.arch").indexOf("arm")!=-1)
			{
				simpleMode=true;
			}

			P80211.initHash();
			PacketCapture pc=new PacketCapture();
			frameVector=new Vector(vectorSize, vectorSize);
			layoutGui();

			pc.addRawPacketListener(this);
			pc.open(device, true);
			pc.capture(-1);
		}
		catch(CaptureDeviceOpenException cdoEx)
		{
			System.out.println("Invalid device name passed to Mognet.");
			System.out.println("Check device name and status.");
			System.exit(1);
		}
		catch(CapturePacketException cpEx)
		{
			System.out.println("Device not available.");
			System.out.println("Try 'ifconfig "+device+" up' and re-run Mognet.");
			System.exit(1);
		}
	}

	public static void main(String[] args)
	{
		String device=args[0];

		if(device==null)
		{
			System.out.println("Specify a device on the command line, ex: java Mognet eth1");
			System.exit(1);
		}

		new Mognet(device);
	}

	//capture view
	private static final Object[] captureCols		={"Type", "Source", "Dest", "SSID"};
	private static JTable captureTable			=null;
	private static DefaultTableModel captureTableModel	=null;
	private static JScrollPane captureScrollPane		=null;
	private static JScrollBar captureScrollBar		=null;

	//detail tab
	private static JTextArea detailArea			=null;
	private static JScrollPane detailScrollPane		=null;
	private static JScrollBar detailScrollBar		=null;

	//hex tab
	private static JTable hexTable				=null;
	private static DefaultTableModel hexTableModel		=null;
	private static JScrollPane hexScrollPane		=null;
	private static final int hexRowSize			=8;

	//ascii tab
	private static JTable asciiTable			=null;
	private static DefaultTableModel asciiTableModel	=null;
	private static JScrollPane asciiScrollPane		=null;
	private static final int asciiRowSize			=8;

	//menus
	private static JMenuItem fileMenuLoad			=null;
	private static JCheckBoxMenuItem captureMenuStart	=null;
	private static JCheckBoxMenuItem captureMenuStop	=null;
	private static JCheckBoxMenuItem modeMenuSimple		=null;
	private static JCheckBoxMenuItem modeMenuAdvanced	=null;
	private static JCheckBoxMenuItem filterMenuAll		=null;
	private static JCheckBoxMenuItem filterMenuData		=null;
	private static JCheckBoxMenuItem filterMenuControl	=null;
	private static JCheckBoxMenuItem filterMenuManagement	=null;

	//general globals
	private static Container contents		=null;
	private static JFrame frame			=null;
	private static JSplitPane splitPane		=null;
	private static JTabbedPane tabbedPane		=null;
	private static JLabel status			=null;
	
	private static Vector frameVector		=null;
	private static int vectorSize			=100;
	private static int frameCount			=0;
	private static boolean captureEnabled		=false;
	private static boolean simpleMode		=false;
}
