#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#define _GNU_SOURCE
#include <getopt.h>
#include <string>
#include <stdlib.h>
#include <signal.h>

#include "config.h"
#include "packet.h"
#include "prism2source.h"
#include "pcapsource.h"
#include "wtapfilesource.h"
#include "genericsource.h"
#include "dumpfile.h"
#include "wtapdump.h"
#include "wtaplocaldump.h"
#include "gpsd.h"
#include "gpsdump.h"
#include "frontend.h"
#include "packetracker.h"
#include "ncursefront.h"
#include "configfile.h"
#include "kismet.h"

#define MAJOR 1
#define MINOR 4

const char *config_base = "kismet.conf";

// Some globals for command line options
char *configfile = NULL;
int no_log = 0;
int noise_log = 0;
int data_log = 0, net_log = 0, crypt_log = 0, cisco_log = 0, gps_log = 0, gps_enable = 1;
string logname, dumplogfile, netlogfile, cryptlogfile, ciscologfile, gpslogfile;
FILE *net_file = NULL, *crypt_file = NULL, *cisco_file = NULL;
DumpFile *dumpfile;
PacketSource *packsource;
int packnum = 0;
Frontend *gui = NULL;
Packetracker tracker;
GPSD gps;
GPSDump gpsdump;

// This/these have to be globals, unfortunately, so that the interrupt
// handler can shut down logging and write out to disk the network list.


// Expand a logfile into a full filename
string ExpandLogPath(string path, string logname, string type) {
    string logtemplate;
    int inc = 0;

    logtemplate = path;

    for (unsigned int nl = logtemplate.find("%"); nl < logtemplate.length();
         nl = logtemplate.find("%", nl+1))
    {
        char op = logtemplate[nl+1];
        logtemplate.erase(nl, 2);
        if (op == 'n')
            logtemplate.insert(nl, logname);
        else if (op == 'd') {
            time_t tnow;
            struct tm *now;

            tnow = time(0);
            now = localtime(&tnow);

            char datestr[24];
            strftime(datestr, 24, "%b-%d-%Y", now);

            logtemplate.insert(nl, datestr);
        }
        else if (op == 'l')
            logtemplate.insert(nl, type.c_str());
        else if (op == 'i')
            inc = nl;
    }

    // If we've got an incremental, go back and find it and start testing
    if (inc) {
        int found = 0;
        for (int num = 1; num < 100; num++) {
            string copied = logtemplate;
            struct stat fstat;

            // This is annoying
            char numstr[5];
            snprintf(numstr, 5, "%d", num);

            copied.insert(inc, numstr);

            if (stat(copied.c_str(), &fstat) == -1) {
                found = 1;
                logtemplate = copied;
                break;
            }
        }

        if (!found) {
            fprintf(stderr, "ERROR:  Unable to find a logging file within 100 hits.  If you really are\n"
                    "        logging this many times in 1 day, change log names or edit the \n"
                    "        source.\n");
            exit(1);
        }
    }

    return logtemplate;
}

// Catch our interrupt
void CatchShutdown(int sig) {
    if (gui != NULL)
        gui->EndDisplay();

    packsource->CloseSource();

    if (net_log) {
        if (tracker.FetchNumNetworks() != 0) {
            tracker.WriteNetworks(net_file);
            fclose(net_file);
        } else {
            fprintf(stderr, "NOTICE: Didn't detect any networks, unlinking network dump.\n");
            fclose(net_file);
            unlink(netlogfile.c_str());
        }
    }

    if (cisco_log) {
        if (tracker.FetchNumCisco() != 0) {
            tracker.WriteCisco(cisco_file);
            fclose(cisco_file);
        } else {
            fprintf(stderr, "NOTICE: Didn't detect any Cisco Discovery Packets, unlinking cisco dump\n");
            unlink(ciscologfile.c_str());
        }
    }

    if (data_log) {
        dumpfile->CloseDump();

        if (dumpfile->FetchDumped() == 0) {
            fprintf(stderr, "NOTICE: Didn't capture any packets, unlinking dump file\n");
            unlink(dumpfile->FetchFilename());
        }
    }

    if (crypt_log) {
        fclose(crypt_file);

        if (tracker.FetchNumInteresting() == 0) {
            fprintf(stderr, "NOTICE: Didn't see any weak encryption packets, unlinking weak file\n");
            unlink(cryptlogfile.c_str());
        }
    }

    if (gps_log) {
        if (gpsdump.CloseDump(1) < 0)
            fprintf(stderr, "NOTICE:  Didn't log any GPS coordinates, unlinking gps file\n");
    }

    exit(1);
}

int Usage(char *argv) {
    printf("Usage: %s [OPTION]\n", argv);
    printf("Most (or all) of these options can (and should) be configured via the\n"
           "kismet.conf global config file, but can be overridden here.\n");
    printf("  -t, --log-title <title>      Custom log file title\n"
           "  -n, --no-logging             No logging (only process packets)\n"
           "  -f, --config-file <file>     Use alternate config file\n"
           "  -g, --gui <type>             Change type of GUI\n"
           "  -c, --capture-type <type>    Type of packet capture device (prism2, pcap, etc)\n"
           "  -i, --capture-interface <if> Packet capture interface (eth0, eth1, etc)\n"
           "  -l, --log-types <types>      Comma seperated list of types to log,\n"
           "                                (ie, dump,cisco,weak,network,gps)\n"
           "  -d, --dump-type <type>       Dumpfile type (wiretap)\n"
           "  -m, --max-packets <num>      Maximum number of packets before starting new dump\n"
           "  -q, --quiet                  Don't play sounds\n"
           "  -p, --gps                    GPS server (host:port or off)\n"
           "  -v, --version                Kismet version\n"
           "  -h, --help                   What do you think you're reading?\n");
    exit(1);
}

// Fork and run a system call to play a sound
void PlaySound(string player, string sound, map<string, string> soundmap) {
    // Why doesn't the above work?  It seems to bugger up wtap dumping.
    // If we don't have this sound event defined
    if (soundmap.find(sound) == soundmap.end())
        return;

    char snd_call[1024];
    snprintf(snd_call, 1024, "%s %s &", player.c_str(),
             soundmap[sound].c_str());
    system(snd_call);

}

int main(int argc,char *argv[]) {
    int sleepu = 0;
    time_t last_draw = time(0);
    int log_packnum = 0;
    int limit_logs = 0;
    char status[STATUS_MAX];

    const char *sndplay = NULL;
    int sound = -1;

    map<string, string> wav_map;

    const char *captype = NULL, *capif = NULL, *logtypes = NULL, *dumptype = NULL,
        *guitype = NULL;

    char gpshost[1024];
    int gpsport = -1;

    static struct option long_options[] = {   /* options table */
        { "log-title", required_argument, 0, 't' },
        { "no-logging", no_argument, 0, 'n' },
        { "config-file", required_argument, 0, 'f' },
        { "gui", required_argument, 0, 'g' },
        { "capture-type", required_argument, 0, 'c' },
        { "capture-interface", required_argument, 0, 'i' },
        { "log-types", required_argument, 0, 'l' },
        { "dump-type", required_argument, 0, 'd' },
        { "max-packets", required_argument, 0, 'm' },
        { "quiet", no_argument, 0, 'q' },
        { "gps", required_argument, 0, 'p' },
        { "help", no_argument, 0, 'h' },
        { "version", no_argument, 0, 'v' },
        // No this isn't documented, and no, you shouldn't be screwing with it
        { "microsleep", required_argument, 0, 'M' },
        { 0, 0, 0, 0 }
    };
    int option_index;
    int decay = 5;

    // Catch the interrupt handler to shut down
    signal(SIGINT, CatchShutdown);

    while(1) {
        int r = getopt_long(argc, argv, "d:M:t:nf:c:i:l:m:g:p:qhv",
                            long_options, &option_index);
        if (r < 0) break;
        switch(r) {
        case 'M':
            // Microsleep
            if (sscanf(optarg, "%d", &sleepu) != 1) {
                fprintf(stderr, "Invalid microsleep\n");
                Usage(argv[0]);
            }
            break;
        case 't':
            // Logname
            logname = optarg;
            fprintf(stderr, "Using logname: %s\n", logname.c_str());
            break;
        case 'n':
            // No logging
            no_log = 1;
            fprintf(stderr, "Not logging any data\n");
            break;
        case 'f':
            // Config path
            configfile = optarg;
            fprintf(stderr, "Using alternate config file: %s\n", configfile);
            break;
        case 'c':
            // Capture type
            captype = optarg;
            break;
        case 'g':
            // GUI
            guitype = optarg;
            break;
        case 'i':
            // Capture interface
            capif = optarg;
            break;
        case 'l':
            // Log types
            logtypes = optarg;
            break;
        case 'd':
            // dump type
            dumptype = optarg;
            break;
        case 'm':
            // Maximum log
            if (sscanf(optarg, "%d", &limit_logs) != 1) {
                fprintf(stderr, "Invalid maximum packet number.\n");
                Usage(argv[0]);
            }
            break;
        case 'p':
            // GPS
            if (strcmp(optarg, "off") == 0) {
                gps_enable = 0;
            } else if (sscanf(optarg, "%[^:]:%d", gpshost, &gpsport) < 2) {
                fprintf(stderr, "Invalid GPS host '%s' (host:port or off required)\n",
                       optarg);
                gps_enable = 1;
                Usage(argv[0]);
            }
            break;
        case 'q':
            // Quiet
            sound = 0;
            break;
        case 'v':
            // version
            fprintf(stderr, "Kismet %d.%d\n", MAJOR, MINOR);
            exit(0);
            break;
        default:
            Usage(argv[0]);
            break;
        }
    }

    // If we haven't gotten a command line config option...
    if (configfile == NULL) {
        configfile = (char *) malloc(1024*sizeof(char));
        snprintf(configfile, 1024, "%s/%s", SYSCONF_LOC, config_base);
    }

    // Parse the config and load all the values from it and/or our command
    // line options.  This is a little soupy but it does the trick.
    if (ParseConfig(configfile) < 0) {
        exit(1);
    }

    if (guitype == NULL && FetchOpt("gui") != "")
        guitype = FetchOpt("gui").c_str();

    if (guitype != NULL ) {

        if (!strcasecmp(guitype, "ncurses")) {
#ifdef HAVE_LIBNCURSES
            gui = new NCurseFront();
#else
            fprintf(stderr, "FATAL:  libncurses support not found, cannot use ncurses GUI.\n");
            exit(1);
#endif
        } else {
            fprintf(stderr, "FATAL:  Unknown GUI type, '%s'\n", guitype);
            exit(1);
        }

        gui->AddTracker(&tracker);
    } else {
        fprintf(stderr, "FATAL:  No GUI specified.\n");
        exit(1);
    }


    if (logname == "") {
        if (FetchOpt("logdefault") == "") {
            fprintf(stderr, "FATAL:  No default log name in config and no log name provided on the command line.\n");
            exit(1);
        }
        logname = FetchOpt("logdefault").c_str();
    }

    if (logtypes == NULL) {
        if (FetchOpt("logtypes") == "") {
            fprintf(stderr, "FATAL:  No log types in config and none provided on the command line.\n");
            exit(1);
        }
        logtypes = FetchOpt("logtypes").c_str();
    }

    if (FetchOpt("noiselog") == "true")
        noise_log = 1;

    if ((strstr(logtypes, "dump")) && !no_log) {
        data_log = 1;

        if (FetchOpt("logtemplate") == "") {
            fprintf(stderr, "FATAL:  Logging (network dump) enabled but no logtemplate given in config.\n");
            exit(1);
        }

        dumplogfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "dump");

        fprintf(stderr, "Logging data to %s\n", dumplogfile.c_str());

        if (FetchOpt("dumplimit") != "" || limit_logs != 0) {
            if (limit_logs == 0)
                if (sscanf(FetchOpt("dumplimit").c_str(), "%d", &limit_logs) != 1) {
                    fprintf(stderr, "FATAL:  Illegal config file value for dumplimit.\n");
                    exit(1);
                }

            if (limit_logs != 0)
                fprintf(stderr, "Limiting dumpfile to %d packets each.\n",
                        limit_logs);
        }

        if (FetchOpt("dumptype") == "" && dumptype == NULL) {
            fprintf(stderr, "FATAL: Dump file logging requested but no dump type given.\n");
            exit(1);
        }

        if (FetchOpt("dumptype") != "" && dumptype == NULL)
            dumptype = FetchOpt("dumptype").c_str();

        if (!strcasecmp(dumptype, "wiretap")) {
            dumpfile = new WtapDumpFile;
        }
        /*
         else if (!strcasecmp(dumptype, "pcap")) {
#ifdef HAVE_LIBPCAP
            dumpfile = new PcapDumpFile;
#else
            fprintf(stderr, "FATAL: Pcap support was not compiled in.\n");
            exit(1);
#endif
}
*/
        else {
            fprintf(stderr, "FATAL:  Unknown dump file type '%s'\n", dumptype);
            exit(1);
        }

    }

    if ((strstr(logtypes, "network")) && !no_log) {
        net_log = 1;

        if (FetchOpt("logtemplate") == "") {
            fprintf(stderr, "FATAL:  Logging (network list) enabled but no logtemplate given in config.\n");
            exit(1);
        }

        netlogfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "network");
        fprintf(stderr, "Logging networks to %s\n", netlogfile.c_str());
    }

    if ((strstr(logtypes, "weak")) && !no_log) {
        crypt_log = 1;

        if (FetchOpt("logtemplate") == "") {
            fprintf(stderr, "FATAL:  Logging (weak packets) enabled but no logtemplate given in config.\n");
            exit(1);
        }

        cryptlogfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "weak");
        fprintf(stderr, "Logging cryptographically weak packets to %s\n", cryptlogfile.c_str());
    }

    if ((strstr(logtypes, "cisco")) && !no_log) {
        cisco_log = 1;

        if (FetchOpt("logtemplate") == "") {
            fprintf(stderr, "FATAL: Logging (cisco packets) enabled but no logtemplate given in config.\n");
            exit(1);
        }

        ciscologfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "cisco");
        fprintf(stderr, "Logging cisco product information to %s\n", ciscologfile.c_str());
    }

    if ((strstr(logtypes, "gps")) && !no_log) {
        gps_log = 1;

        if (FetchOpt("logtemplate") == "") {
            fprintf(stderr, "FATAL:  Logging (gps coordinates) enabled but no logtemplate given in config.\n");
            exit(1);
        }

        gpslogfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "gps");
        fprintf(stderr, "Logging gps coordinates to %s\n", gpslogfile.c_str());
    }

    if (FetchOpt("decay") != "") {
        if (sscanf(FetchOpt("decay").c_str(), "%d", &decay) != 1) {
            fprintf(stderr, "FATAL:  Illegal config file value for decay.\n");
            exit(1);
        }
    }

    if (sleepu == 0) {
        if (FetchOpt("microsleep") == "") {
            sleepu = 100;
        } else if (sscanf(FetchOpt("microsleep").c_str(), "%d", &sleepu) != 1) {
            fprintf(stderr, "FATAL:  Illegal config file value for microsleep.\n");
            exit(1);
        }
    }

    // Process sound stuff
    if (FetchOpt("sound") == "true" && sound == -1) {
        if (FetchOpt("soundplay") != "") {
            sndplay = FetchOpt("soundplay").c_str();
            sound = 1;

            if (FetchOpt("sound_new") != "")
                wav_map["new"] = FetchOpt("sound_new");
            if (FetchOpt("sound_traffic") != "")
                wav_map["traffic"] = FetchOpt("sound_traffic");

        } else {
            fprintf(stderr, "ERROR:  Sound alerts enabled but no sound playing binary specified.\n");
        }
    } else if (sound == -1)
        sound = 0;

    // Open the captype
    if (captype == NULL) {
        if (FetchOpt("captype") == "") {
            fprintf(stderr, "FATAL:  No capture type specified.");
            exit(1);
        }
        captype = FetchOpt("captype").c_str();
    }

    // Create a capture source
    if (!strcasecmp(captype, "prism2")) {
        fprintf(stderr, "Using prism2 to capture packets.\n");

        packsource = new Prism2Source;
    } else if (!strcasecmp(captype, "pcap")) {
#ifdef HAVE_LIBPCAP
        if (capif == NULL) {
            if (FetchOpt("capinterface") == "") {
                fprintf(stderr, "FATAL:  No capture device specified.\n");
                exit(1);
            }
            capif = FetchOpt("capinterface").c_str();
        }

        fprintf(stderr, "Using pcap to capture packets from %s\n", capif);

        packsource = new PcapSource;
#else
        fprintf(stderr, "FATAL: Pcap support was not compiled in.\n");
        exit(1);
#endif
    } else if (!strcasecmp(captype, "generic")) {
#ifdef HAVE_LINUX_WIRELESS
        if (capif == NULL) {
            if (FetchOpt("capinterface") == "") {
                fprintf(stderr, "FATAL:  No capture device specified.\n");
                exit(1);
            }
            capif = FetchOpt("capinterface").c_str();
        }

        fprintf(stderr, "Using generic kernel extentions to capture SSIDs from %s\n", capif);

        fprintf(stderr, "Generic capture does not support cisco, weak, or dump logs.\n");
        cisco_log = crypt_log = data_log = 0;

        fprintf(stderr, "Generic kernel capture will ONLY work as root.  Kismet must be run as\n"
                "root, not suid, for this to function.\n");

        packsource = new GenericSource;
#else
        fprintf(stderr, "FATAL: Kernel wireless (wavelan/generic) support was not compiled in.\n");
        exit(1);
#endif
    } else if (!strcasecmp(captype, "wtapfile")) {
        if (capif == NULL) {
            if (FetchOpt("capinterface") == "") {
                fprintf(stderr, "FATAL:  No capture file specified.\n");
                exit(1);
            }
            capif = FetchOpt("capinterface").c_str();
        }
#ifdef HAVE_LIBWIRETAP
        fprintf(stderr, "Loading packets from dump file %s\n", capif);

        // Drop root privs NOW, because we don't want them reading any
        // files in the system they shouldn't be.
        setuid(getuid());

        packsource = new WtapFileSource;
#else
        fprintf(stderr, "FATAL: Wtap support was not compiled in.\n");
        exit(1);
#endif

    } else {
        fprintf(stderr, "FATAL: Unknown capture type '%s'\n", captype);
        exit(1);
    }

    // Open the packet source
    if (packsource->OpenSource(capif) < 0) {
        fprintf(stderr, "%s\n", packsource->FetchError());
        exit(1);
    }

    // Once the packet source is opened, we shouldn't need special privileges anymore
    // so lets drop to a normal user.  We also don't want to open our logfiles as root
    // if we can avoid it.
    setuid(getuid());

    // Now lets open the GPS host if specified
    if (gpsport == -1 && gps_enable) {
        if (FetchOpt("gps") == "true") {
            if (sscanf(FetchOpt("gpshost").c_str(), "%[^:]:%d", gpshost, &gpsport) != 2) {
                fprintf(stderr, "Invalid GPS host in config (host:port required)\n");
                exit(1);
            }

            gps_enable = 1;
        } else {
            gps_enable = 0;
            gps_log = 0;
        }
    }

    if (gps_enable == 1) {
        // Open the GPS
            if (gps.OpenGPSD(gpshost, gpsport) < 0) {
                fprintf(stderr, "%s\n", gps.FetchError());
                gps_enable = 0;
                if (gps_log)
                    fprintf(stderr, "Disabling GPS logging.\n");
                gps_log = 0;
            } else {
                fprintf(stderr, "Opened GPS connection to %s port %d\n",
                        gpshost, gpsport);

                gui->AddGPS(&gps);
                tracker.AddGPS(&gps);
                gpsdump.AddGPS(&gps);

                if (gps_log) {
                    if (gpsdump.OpenDump(gpslogfile.c_str()) < 0) {
                        fprintf(stderr, "FATAL: %s\n", gpsdump.FetchError());
                        exit(1);
                    }
                }
            }
    } else {
        if (gps_log)
            fprintf(stderr, "Disabling GPS logging.\n");
        gps_log = 0;
    }

    if (data_log) {
        if (dumpfile->OpenDump(dumplogfile.c_str()) < 0) {
            fprintf(stderr, "FATAL: %s\n", dumpfile->FetchError());
            exit(1);
        }

        fprintf(stderr, "Dump file format: %s\n", dumpfile->FetchType());
    }

    if (net_log) {
        if ((net_file = fopen(netlogfile.c_str(), "w")) == NULL) {
            perror("Unable to open net file");
            exit(1);
        }
    }

    if (crypt_log) {
        if ((crypt_file = fopen(cryptlogfile.c_str(), "w")) == NULL) {
            perror("Unable to open weak log file");
            exit(1);
        }
    }

    if (cisco_log) {
        if ((cisco_file = fopen(ciscologfile.c_str(), "w")) == NULL) {
            perror("Unable to open cisco file");
            exit(1);
        }
    }

    gui->InitDisplay(decay);

    snprintf(status, 80, "Kismet %d.%d", MAJOR, MINOR);
    gui->WriteStatus(status);

    snprintf(status, 80, "Capturing packets from %s",
             packsource->FetchType());
    gui->WriteStatus(status);

    if (data_log || net_log || crypt_log) {
        snprintf(status, 80, "Logging%s%s%s%s%s",
                 data_log ? " data" : "" ,
                 net_log ? " networks" : "" ,
                 crypt_log ? " weak" : "",
                 cisco_log ? " cisco" : "",
                 gps_log ? " gps" : "");
        gui->WriteStatus(status);
    } else if (no_log) {
        snprintf(status, 80, "Not logging any data.");
        gui->WriteStatus(status);
    }

    gui->DrawDisplay();

    time_t last_click = 0;
    while (1) {
        // Update the GPS
        if (gps_enable) {
            int gpsret;
            gpsret = gps.Scan();
//            fprintf(stderr, "ret: %d\n", gpsret);
            if (gpsret < 0) {
                gui->WriteStatus(gps.FetchError());
                gps_enable = 0;
            }
        }

        pkthdr header;
        u_char data[MAX_PACKET_LEN];

        int len;

        // Capture the packet from whatever device
        len = packsource->FetchPacket(&header, data);

        // Handle a packet
        if (len > 0) {
            packnum++;
            log_packnum++;

            // Play sounds
            if (time(0) - last_click > decay && sound == 1) {
                PlaySound(sndplay, "traffic", wav_map);
                last_click = time(0);
            }


            if (limit_logs && log_packnum > limit_logs) {
                dumpfile->CloseDump();

                dumplogfile = ExpandLogPath(FetchOpt("logtemplate"), logname, "dump");

                if (dumpfile->OpenDump(dumplogfile.c_str()) < 0) {
                    perror("Unable to open new dump file");
                    CatchShutdown(1);
                }

                snprintf(status, 80, "Opened new packet log file %s",
                         dumplogfile.c_str());
                gui->WriteStatus(status);

                log_packnum = 0;
            }

            packet_info info;

            info = GetPacketInfo(&header, data);

            int process_ret;

            if (gps_log && info.type != packet_noise) {
                process_ret = gpsdump.DumpPacket(&info);
                if (process_ret < 0) {
                    snprintf(status, 80, "%s", gpsdump.FetchError());
                    gui->WriteStatus(status);
                }
            }

            process_ret = tracker.ProcessPacket(info, status);
            if (process_ret > 0)
                gui->WriteStatus(status);

            if (process_ret == 1 && sound == 1)
                PlaySound(sndplay, "new", wav_map);

            if (data_log) {
                if (!(info.type == packet_noise && noise_log == 1)) {
                    dumpfile->DumpPacket(&header, data);
                }
            }

            if (crypt_log && info.interesting) {
                // Airsnort crypt files are the packet data from the WEP
                // header until the CRC check -- so, offset 24, length
                // -4
                int len = header.len - 28;

                if (fwrite(&len, sizeof(int), 1, crypt_file) < 1) {
                    snprintf(status, 80, "ERROR:  Couldn't write packet length to weak file.");
                    gui->WriteStatus(status);
                }

                if (fwrite(data+24, len, 1, crypt_file) < 1) {
                    snprintf(status, 80, "ERROR:  Couldn't write weak packet to file.");
                    gui->WriteStatus(status);
                }
            }

        } else if (len < 0) {
            // Fail on error
            gui->WriteStatus(packsource->FetchError());
            gui->WriteStatus("Terminating.");
            gui->DrawDisplay();
            sleep(5);
            CatchShutdown(1);
        }

        // Draw if it's time
        if (time(0) != last_draw) {
            // Log GPS once per draw
            if (gps_log) {
                gpsdump.DumpPacket(NULL);
            }

            last_draw = time(0);
            gui->DrawDisplay();
        }

        if (sleepu > 0)
            usleep(sleepu);
    }

    gui->EndDisplay();

}


