/* am930cs.c: PCMCIA cs elements for PRISM/AM79C930 card from Harris
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*
*	This file contains the entry points/cleanup routines for modules and
*	PCMCIA drivers.  Additionally, all "driver global" variables are
*	defined here.
*
*	MSM 02/13/97
*/

#if defined(__LINUX_WLAN__)
/* PCMCIA headers generated during PCMCIA package installation */
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

/* Module related headers, non-module drivers should not include */
#include <linux/module.h>
#include <linux/version.h>

/* Standard driver includes */
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/kernel_stat.h>
#include <linux/delay.h>

#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>

#include <linux/errno.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>

/* Card and Driver services includes */
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/mem_op.h>

#endif

/*================================================================*/
/* Local Includes */
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include "am930di.h"

/*================================================================*/
/* Local macros */

/*================================================================*/
/* Local statics */

static char *version = "am930cs.c 0.1.2 1998/08/07";

/* Linked list of device "instances" */
static dev_link_t *am930DI_list_head = NULL;


/*================================================================*/
/* "Driver global" definitions */

int		am930_use_mmio=0;		/* use memory mapped i/o or not */
MODULE_PARM( am930_use_mmio, "i");

int		am930_iodelay=0;		/* udelay between memory or i/o reads */
MODULE_PARM( am930_iodelay, "i");

int		am930_isap=0;			/* are we an AP? */
MODULE_PARM( am930_isap, "i");

int		wlan_debug=0;			/* Debug output level, declared in */
MODULE_PARM( wlan_debug, "i");	/* wlan_compat.h */

/*================================================================*/
/* Local functions */
int		init_module(void);
void	cleanup_module(void);

dev_link_t *am930_drvr_attach(void);
void        am930_drvr_detach(dev_link_t *);



/*----------------------------------------------------------------
*	init_module
*
*	Module initialization routine. For a PCMCIA device, this function
*	tests the version of the running card services with the version
*	number of the headers used to compile the module. In case of 
*	mismatch, an error is reported.
*	If the version is OK, then the driver is registered with the PCMCIA
*	support system, identifying it's name, attach and detach functions.
*
*	returns: zero on success, non-zero on failure.
----------------------------------------------------------------*/
int init_module(void)
{
	servinfo_t serv;

	DBFENTER;

	printk( KERN_DEBUG"init_module:%s\n", version);

	/* Check to make sure compile and run-time versions match */
	CardServices(GetCardServicesInfo, &serv);
	if (serv.Revision != CS_RELEASE_CODE)
	{
		printk(KERN_NOTICE "am79c930: Card Services release does not match!\n");
		return -1;
	}

	printk(KERN_INFO "Linux WLAN " WLAN_RELEASE "\n");

	/* Notify driver services that we are loaded and are a pcmcia driver */
	register_pcmcia_driver( &am930_driver_name,
							&am930_drvr_attach, 
							&am930_drvr_detach);

	DBFEXIT;
	return 0;
}


/*----------------------------------------------------------------
*	cleanup_module
*
*	This is the destructor for the am930_drvr object.
*
*	Standard module unload callback function. For PCMCIA, that
*	means that the driver must be unregistered AND any outstanding
*	resources must be released.
*
*	returns: nothing
----------------------------------------------------------------*/
void cleanup_module(void)
{
	DBFENTER;


	unregister_pcmcia_driver(&am930_driver_name);

	while (am930DI_list_head != NULL)
	{
		if ( am930DI_list_head->state & DEV_CONFIG )
		{
			am930DI_destruct(am930DI_list_head);
		}
		am930_drvr_detach(am930DI_list_head);
	}

	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930_drvr_attach
*
*	Callback function for linux pcmcia Driver Services. Called 
*	when Driver Services has detected a card in a socket that
*	needs to be bound to a driver.
*
*	The function calls the "device instance" object constructor
*	which allocates and initializes the "device instance" member
*	variables. The address of the device instance member variables
*	is returned from the constructor.
*	
*	The "device instance" member variables are stored in the pcmcia 
*	defined "dev_link_t" structure type. 
*
*	If the instance was created successfully, then the instance is
*	inserted into the list of instances maintained by this object.
*
*	returns: the dev_link_t structure that represents this 
*            instance or NULL on failure
----------------------------------------------------------------*/

dev_link_t *am930_drvr_attach(void)
{
    dev_link_t		*instance = NULL;

	DBFENTER;

	/* call the device instance constructor */
	instance = am930DI_construct();

	/* if successful, then add it to the list */
	if ( instance != NULL )
	{
		instance->next = am930DI_list_head;
		am930DI_list_head = instance;
	}

	DBFEXIT;
    return instance;
} 


/*----------------------------------------------------------------
*	am930_drvr_detach
*
*	Callback function for linux pcmcia Driver Services. Called
*	when driver services receives a DS_UNBIND ioctl. This usually
*	occurs when socket services detects that a card has been 
*	removed.
*
*	Kindof a wierd function, it has to determine if it's OK to do
*	its job. If not then it sets some flags and expects to be
*	called later.
*
*	NOTE: THIS FUNCTION FREES THE INSTANCE STRUCTURE!!!
*
*	returns: nothing
----------------------------------------------------------------*/
void am930_drvr_detach(dev_link_t *instance)
{
    dev_link_t **curr;

	DBFENTER;

    /* Locate device structure in the linked list */
    for (curr = &am930DI_list_head; *curr != NULL; curr = &(*curr)->next)
    {
		if (*curr == instance)
			break;
	}

    if (*curr != NULL)  /* if instance is good, start work, otherwise bail */
    {

		if (instance->state & DEV_CONFIG)
    	{
    		WLAN_LOG_DEBUG0(2, "detached postponed until release\n");
			am930DI_destruct(instance);
		}

		
		if ( instance->state & DEV_STALE_CONFIG )
		{
			instance->state |= DEV_STALE_LINK;
		}
		else
		{
	
		    /* Break the link with Card Services */
		    if (instance->handle)
				CardServices(DeregisterClient, instance->handle);
	    
		    /* Unlink the instance from the list */
		    *curr = instance->next;

			/* free the memory */
		    kfree_s(instance, sizeof(struct dev_link_t));
	   }
    }

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930_drvr_cs_error
*
*	This function simply provides a convenient way of reporting 
*	Card Services errors.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930_drvr_cs_error(client_handle_t handle, int func, int ret)
{
	error_info_t err = {func, ret};
    CardServices(ReportError, handle, &err);
}


