/* am930_di.c : Handles the PCMCIA "device instance" functions
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*
*	PCMCIA "device Instance" 
*
*	The functions in this file are responsible for dealing with the
*	life of a PCMCIA card. This involves dealing with the PCMCIA elements
*	of the card itself and dealing with socket, driver, and card services.
*
*/

#if defined(__LINUX_WLAN__)
/* The following prevents "kernel_version" from being set in this file. */
#define __NO_VERSION__

/* PCMCIA headers generated during PCMCIA package installation */
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

/* Module related headers, non-module drivers should not include */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

/* Standard driver includes */
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <asm/bitops.h>
#include <linux/errno.h>

/* Card and Driver services includes */
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>

#endif

/* Local Includes */
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include "am930di.h"
#include "am930llc.h"
#include "am930mac.h"
#include "am930hw.h"

#if (LINUX_VERSION_CODE < VERSION(2,2,0))
#				define ioremap_nocache ioremap
#endif

/*================================================================*/
/* Local macros */

/*================================================================*/
/* Local prototypes */

static void am930DI_interrupt (int irq, void * dev_id, struct pt_regs *regs );

static int am930DI_config(dev_link_t *instance, UINT32 *pMemBase);
static int am930DI_deconfig(dev_link_t *instance);

/*================================================================*/
/* Local statics (lifetime not scope) */

/* Driver name, must match the name in the PCMCIA database */
dev_info_t	am930_driver_name = "am930_cs";

/*----------------------------------------------------------------
*	am930DI_construct
*
*	Device constructor, contains alot of the code commonly found 
*	in other pcmcia driver's "attach" function.
*
*	The function first allocates an "instance" structure, then 
*	zeros it. Then the RegisterClient Card Services call is 
*	set up and done. Note that we register for ALL of the 
*	Card Services events at this point. Therefore, the event
*	handler must be coded to deal with ALL of them.
*
*	returns: nothing
----------------------------------------------------------------*/
dev_link_t* am930DI_construct(void)
{
    dev_link_t		*instance;
    client_reg_t	client_reg;
    int 			ret = 0;
   
	DBFENTER;

    /* Allocate the instance */
    instance = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);

	if ( instance != NULL )
	{
		/* zero the instance members */
	    memset(instance, 0, sizeof(struct dev_link_t));

	    /* remaining initialization of instance will be done */
	    /*  in the config function */
	
		/*---------- Register with Card Services ---------------------*/
		/* Set up the client registration function */
	    client_reg.dev_info = &am930_driver_name;
	    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
	    client_reg.EventMask =
	    	/* client services */
				CS_EVENT_REGISTRATION_COMPLETE | 
			/* host battery */
				CS_EVENT_BATTERY_LOW |
				CS_EVENT_BATTERY_DEAD |
	    	/* insertion/removal */
				CS_EVENT_CARD_INSERTION | 
				CS_EVENT_CARD_REMOVAL |
			/* reset */
				CS_EVENT_RESET_REQUEST | 
				CS_EVENT_RESET_PHYSICAL |
				CS_EVENT_CARD_RESET |
				CS_EVENT_RESET_COMPLETE |
			/* power management */
				CS_EVENT_PM_SUSPEND | 
				CS_EVENT_PM_RESUME ;
	    client_reg.event_handler = &am930DI_event;
	    client_reg.Version = 0x0210;
	    client_reg.event_callback_args.client_data = instance;

		/* allocate the device node */
	    instance->dev = kmalloc( sizeof(dev_node_t), GFP_KERNEL );

	    if ( instance->dev == NULL )
	    {
	    	WLAN_LOG_ERROR0("dev_node_t allocation failed!\n");
	    	am930DI_destruct( instance );
	    	instance = NULL;
	    }
	    else
	    {
	    	memset( instance->dev, 0, sizeof(dev_node_t) );

		    ret = CardServices(RegisterClient, &instance->handle, &client_reg);
	
		    if (ret != 0) 
		    {
				am930_drvr_cs_error(instance->handle, RegisterClient, ret);
				kfree_s( instance->dev, sizeof(struct dev_node_t));
				kfree_s( instance, sizeof(struct dev_link_t));
				instance = NULL;
		    }
		}
	}
	DBFEXIT;
    return instance;
} 




/*----------------------------------------------------------------
*	am930DI_destruct
*
*	Destructor for the device instance object. This function 
*	roughly corresponds to the "xxx_release" function described for
*	pcmcia drivers.
*
*	This function first check to see if the Linux device corresponding
*	to this pcmcia device instance is still open,
*	if so, then we mark the instance as having a STALE configuration
*	and bail, hoping that this function will be called again later.
*	If the device is not open (closed? ;-) ), 1st we call the
*	call the destructors of the subobjects, then we call the 
*	deconfig method to release all of the resources allocated when
*	this instance was configured. Then, we check to see if the
*	driver object's detach function wants to be called (indicated
*	by STALE_LINK) and if so, call it. After the detach function is
*	called, the structure representing this instance no longer exists
*	and should not be referenced.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930DI_destruct(dev_link_t *instance)
{
    
	DBFENTER;

	if ( instance->open )
	{
		WLAN_LOG_DEBUG1(2, "inst 0x%x still open, destruct (release) deferred.\n", 
					(UINT)instance);

		instance->state |= DEV_STALE_CONFIG;
	}
	else
	{
		am930mac_t	*mac;

		/* Destroy the subobjects */
		mac = (am930mac_t*)instance->priv;
		if ( mac != NULL )
		{
			am930mac_destruct(mac);
		}

		am930DI_deconfig(instance);

		/* check to see if drvr_detach has been called previously and
			expects to be called again.
		*/
		if ( instance->state & DEV_STALE_LINK )
		{
			am930_drvr_detach(instance);
		}

		if (instance->dev != NULL )
		{
			kfree_s( instance->dev, sizeof(dev_node_t));
			instance->dev = NULL;
		}

		/* instance memory is kfree'd in am930cs_detach */
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930DI_event
*
*	Responsible for dispatching and/or scheduling the handling of
*	PCMCIA events. 
*	Events and the responses:
*		CS_EVENT_REGISTRATION_COMPLETE
*			in debug, log the event, otherwise nothing.
*		CS_EVENT_BATTERY_LOW
*			currently unsupported.
*		CS_EVENT_BATTERY_DEAD
*			currently unsupported.
*		CS_EVENT_CARD_INSERTION
*			call the DI_config method to configure the interface
*			to the device and get the ball rolling on the 
*			OS interface.
*		CS_EVENT_CARD_REMOVAL
*			schedule a delayed run of the destructor, delayed
*			because the event handler can't wait for the destructor
*			to complete, we must return immediately.
*		CS_EVENT_RESET_REQUEST 
*		CS_EVENT_RESET_PHYSICAL
*		CS_EVENT_CARD_RESET
*		CS_EVENT_RESET_COMPLETE
*		CS_EVENT_PM_SUSPEND 
*		CS_EVENT_PM_RESUME
*			all unsupported, but logged in debug
*
*	returns: zero
----------------------------------------------------------------*/
int am930DI_event( event_t event, int priority,
                        	event_callback_args_t *args)
{
    dev_link_t		*instance = args->client_data;
    

	DBFENTER;
    
    switch (event)
    {
		case CS_EVENT_REGISTRATION_COMPLETE:
			WLAN_LOG_DEBUG1(2, 
				"CS_EVENT_REGISTRATION_COMPLETE event, instance %x\n", 
				(UINT)instance);
			break;

		case CS_EVENT_CARD_INSERTION:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_CARD_INSERTION event, instance %x\n", 
				(UINT)instance);

			instance->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
			{
				UINT32 membase = 0;  /* temp to hold membase from config */

				if ( am930DI_config(instance, &membase) == AM930DI_SUCCESS)
				{
					/* Build the mac object */
					/* NOTE: am930mac_construct modifies instance->priv */
					am930mac_construct(instance, instance->io.BasePort1, 
										membase, instance->irq.AssignedIRQ);
					if ( instance->priv == NULL )
					{
						WLAN_LOG_ERROR0("am930hw_construct failed!\n");
						am930DI_deconfig(instance);
						instance->state &= ~DEV_CONFIG_PENDING;						
					}
					
				}
				else
				{
					WLAN_LOG_ERROR0("am930DI_config failed!\n");
					instance->state &= ~DEV_CONFIG_PENDING;
				}
			}
			break;

		case CS_EVENT_CARD_REMOVAL:
			WLAN_LOG_DEBUG1( 2, 
				"CS_EVENT_CARD_REMOVAL event, instance %x\n", (UINT)instance);

			instance->state &= ~DEV_PRESENT;
			if (instance->state & DEV_CONFIG)
			{
				WLAN_LOG_DEBUG0(2,"Setting up timer destructor call.\n"); 
				instance->release.expires = RUN_AT(HZ/20);
				add_timer(&instance->release);
			}
			break;

		case CS_EVENT_BATTERY_LOW:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_BATTERY_LOW event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_BATTERY_DEAD:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_BATTERY_DEAD event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_RESET_REQUEST: 
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_RESET_REQUEST event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_RESET_PHYSICAL:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_RESET_PHYSICAL event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_CARD_RESET:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_CARD_RESET event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_RESET_COMPLETE:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_RESET_COMPLETE event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_PM_SUSPEND: 
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_PM_SUSPEND event, instance %x\n", (UINT)instance);
			break;
		case CS_EVENT_PM_RESUME:
			WLAN_LOG_DEBUG1(2,
				"CS_EVENT_PM_RESUME event, instance %x\n", (UINT)instance);
			break;

		default:
			WLAN_LOG_DEBUG1(2,
				"unknown event, instance %x\n", (UINT)instance);
			break;

    }

	DBFEXIT
    return 0;
} 

/*----------------------------------------------------------------
*	am930DI_config
*
*	This function configures the PCMCIA socket, and allocates the
*	system resources for the card. Once this function is complete
*	we can treat the card like any other ISA and ignore the fact
*	that it's a PCMCIA.
*
*	returns: nothing
----------------------------------------------------------------*/
#define CS_CHECK( fn, args... ) while( ( last_ret = CardServices( last_fn = ( fn ), args ) ) != 0 ) goto cs_failed
#define CFG_CHECK( fn, args... ) if( CardServices( fn, args ) != 0 ) goto next_entry

int am930DI_config(dev_link_t *instance, UINT32 *pMemBase)
{
	client_handle_t	handle = instance->handle;
	tuple_t			tuple;
	cisparse_t		parse;
	int				last_fn;
	int				last_ret;
	cisdata_t		buf[AM930DI_TPL_BUFSIZE];
	win_req_t 		req;
	dev_node_t		*pnode;
	cistpl_cftable_entry_t dflt = { 0 };
	cistpl_cftable_entry_t *cfg = NULL;

	DBFENTER;

	/*--Allocate a node_t for the instance to point to, apparently --*/
	/*-- driver services requires this ------*/
	pnode = kmalloc(sizeof(dev_node_t), GFP_KERNEL);
	memset( pnode, 0, sizeof(dev_node_t));
	instance->dev = pnode;

	/*-- set up the deconfig (release) function --*/
	instance->release.function = 
		(void (*)(u_long))&am930DI_destruct;
	instance->release.data = (u_long)instance;

	/* clear the membase */
	*pMemBase = 0;

	/* Pick up the CONFIG tuple items */
	tuple.DesiredTuple = CISTPL_CONFIG;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof( buf );
	tuple.TupleOffset = 0;
	CS_CHECK( GetFirstTuple, handle, &tuple );
	CS_CHECK( GetTupleData, handle, &tuple );
	CS_CHECK( ParseTuple, handle, &tuple, &parse );
	instance->conf.ConfigBase = parse.config.base;
	instance->conf.Present = parse.config.rmask[0];
	instance->conf.IntType = INT_MEMORY_AND_IO;

	instance->state |= DEV_CONFIG;

	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	CS_CHECK( GetFirstTuple, handle, &tuple );

	while( 1 ) 
	{
		CFG_CHECK( GetTupleData, handle, &tuple );
		CFG_CHECK( ParseTuple, handle, &tuple, &parse );
		cfg = &(parse.cftable_entry);

		printk(KERN_DEBUG"Inspecting CFTABLE[%d]\n", cfg->index);

		if( cfg->flags & CISTPL_CFTABLE_DEFAULT ) { dflt = *cfg; }

		if( cfg->index == 0 || ((am930_use_mmio == 0) && (cfg->mem.nwin > 0)) )
		{ 
			goto next_entry;
		}

		instance->conf.ConfigIndex = cfg->index;

		if( cfg->vcc.present & ( 1<<CISTPL_POWER_VNOM ) )
			instance->conf.Vcc = cfg->vcc.param[CISTPL_POWER_VNOM] / 10000;
		else if( dflt.vcc.present & ( 1<<CISTPL_POWER_VNOM ) )
			instance->conf.Vcc = dflt.vcc.param[CISTPL_POWER_VNOM] / 10000;

		if( cfg->vpp1.present & ( 1<<CISTPL_POWER_VNOM ) )
			instance->conf.Vpp1 = instance->conf.Vpp2 =
				cfg->vpp1.param[CISTPL_POWER_VNOM] / 10000;
		else if( dflt.vpp1.present & ( 1<<CISTPL_POWER_VNOM ) )
			instance->conf.Vpp1 = instance->conf.Vpp2 =
				dflt.vpp1.param[CISTPL_POWER_VNOM] / 10000;

		if( cfg->irq.IRQInfo1 || dflt.irq.IRQInfo1 )
		{
			instance->conf.Attributes |= CONF_ENABLE_IRQ;
			instance->irq.Handler = am930DI_interrupt;
			instance->irq.Instance = instance;
			instance->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
			instance->irq.IRQInfo1 = cfg->irq.IRQInfo1 ? 
				cfg->irq.IRQInfo1 : dflt.irq.IRQInfo1;
			if( cfg->irq.IRQInfo2 || dflt.irq.IRQInfo2 )
			{
				instance->irq.IRQInfo1 |= IRQ_INFO2_VALID;
				instance->irq.IRQInfo2 = cfg->irq.IRQInfo2 ? 
					cfg->irq.IRQInfo2 : dflt.irq.IRQInfo2;
			}
		}

		instance->io.NumPorts1 = instance->io.NumPorts2 = 0;
		if( ( cfg->io.nwin > 0 ) || ( dflt.io.nwin > 0 ) ) 
		{
			cistpl_io_t *io = ( cfg->io.nwin ) ? &cfg->io : &dflt.io;
			instance->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
			if( !( io->flags & CISTPL_IO_8BIT ) ) 
				instance->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
			if( !( io->flags & CISTPL_IO_16BIT ) ) 
				instance->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
			instance->io.BasePort1 = io->win[0].base;
			instance->io.NumPorts1 = io->win[0].len;
		}

		CFG_CHECK( RequestIO, instance->handle, &instance->io );
			

		if( ( cfg->mem.nwin > 0 ) || ( dflt.mem.nwin > 0 ) ) 
		{
			cistpl_mem_t *mem = (cfg->mem.nwin) ? &cfg->mem : &dflt.mem;
			req.Attributes = WIN_DATA_WIDTH_16 | WIN_MEMORY_TYPE_CM;
			req.Base = mem->win[0].host_addr;
			req.Size = mem->win[0].len;
			req.AccessSpeed = 0;
			instance->win = (window_handle_t)instance->handle;
			CFG_CHECK( RequestWindow, &instance->win, &req );

			/*-- Save the mem base ----*/
			*pMemBase = (UINT32)ioremap_nocache( req.Base, 0x8000 );
		}
		break;

next_entry:
		CS_CHECK( GetNextTuple, handle, &tuple );
	}

	if( instance->conf.Attributes & CONF_ENABLE_IRQ )
	{
		CS_CHECK( RequestIRQ, instance->handle, &instance->irq );
	}

	CS_CHECK( RequestConfiguration, instance->handle, &instance->conf );

	IRQ_MAP(instance->irq.AssignedIRQ, instance);

	printk(KERN_DEBUG"CFTABLE index 0x%02x: Vcc %d.%d irq %d\n",
		instance->conf.ConfigIndex,
		instance->conf.Vcc / 10, 
		instance->conf.Vcc % 10,
		instance->irq.AssignedIRQ );
	printk(KERN_DEBUG"  io 0x%04x-0x%04x\n", 
		instance->io.BasePort1,
		instance->io.BasePort1 + instance->io.NumPorts1 - 1 );
	if( instance->win )
		printk(KERN_DEBUG"  mem 0x%06lx-0x%06lx\n", 
			req.Base, 
			req.Base + req.Size - 1 );

	instance->state &= ~DEV_CONFIG_PENDING;

	DBFEXIT;
	return AM930DI_SUCCESS;

 cs_failed:
	am930_drvr_cs_error( handle, last_fn, last_ret );
	am930DI_deconfig( instance );
	return AM930DI_FAILURE;
}


/*----------------------------------------------------------------
*	am930DI_deconfig
*
*	This function releases all of the resources established for
*	the pcmcia card instance. It then marks the instance as NOT
*	configured.
*
*	returns: always returns success
----------------------------------------------------------------*/
int am930DI_deconfig(dev_link_t *instance)
{
	int nResult = AM930DI_SUCCESS;

	DBFENTER;

	/*-------Clear the irq2dev_map entry -----*/
	IRQ_MAP( instance->irq.AssignedIRQ, NULL );

    /* Don't bother checking to see if these succeed or not */

	if( instance->priv != NULL ) {
		am930mac_t * mac = (am930mac_t *) instance->priv;
		if( mac->hw != NULL ) {
			am930hw_t * hw = (am930hw_t *) mac->hw;
			if( hw->membase != 0 ) {
				iounmap( (void *) hw->membase );
			}
		}
	}
	
    CardServices(ReleaseWindow, instance->win); 
    CardServices(ReleaseConfiguration, instance->handle);
    CardServices(ReleaseIO, instance->handle, &instance->io);
    CardServices(ReleaseIRQ, instance->handle, &instance->irq);
    instance->state &= ~DEV_CONFIG;

	DBFEXIT;
	return nResult;
}

/*----------------------------------------------------------------
*	am930DI_interrupt
*
*	Wrapper interrupt handle. This function receives interrupts
*	logically from the PCMCIA hardware since it's the one mapping
*	card interrupts to a given host interrupt line.
*
*	This function checks to see if the DI object is referenced in
*	the irq entry in the irq2dev_map. If so, then we check to see
*	if an am930hw object is present in the given DI object. If so
*	we call the am930hw object's ISR method.
*
*	TODO: check that it's OK for a dev_link_t to be in irq2dev_map
*
*	returns: nothing
----------------------------------------------------------------*/
static void am930DI_interrupt IRQ(int irq, void * dev_id,	struct pt_regs *regs ) 
{
	dev_link_t*	instance = NULL;

	DBFENTER;
	if ( (instance = (dev_link_t*)DEV_ID) != NULL )
	{
		if ( instance->priv != NULL )
		{
			am930mac_ISR( (am930mac_t*)instance->priv );
		}
	}
	else
	{
		WLAN_LOG_DEBUG0(2, "*** am930di_interrupt with unknown receiver!\n");
	}

	DBFEXIT;
	return;
}

