/* am930_hw.c: Handles the specifics of the AM79C930/PRISM card
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*
*	The functions here implement all of the stuff that's specific
*	to this card.  I've tried to keep all of the AM79C930 hardware and
*	firmware stuff here.  Some specifics have bled into the mac and mgr
*	sections though.
------------------------------------------------------------------------*/


#include <linux/config.h>
#include <wlan/wlan_compat.h>

#if (WLAN_OS == WLAN_LINUX_KERNEL)

/* The following prevents "kernel_version" from being set in this file. */
#define __NO_VERSION__

/* PCMCIA headers generated during PCMCIA package installation */
#ifdef WLAN_PCMCIA
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#endif


/* Module related headers, non-module drivers should not include */
#include <linux/version.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

/* Standard driver includes */
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>

/* Card and Driver services includes */
#ifdef WLAN_PCMCIA
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#endif

/* Local Includes */
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_ioctl.h>

#ifdef WLAN_PCMCIA
#include "am930di.h"
#endif

#include "am930llc.h"
#include "am930mac.h"
#include "am930hw.h"
#include "am930mgr.h"

#elif (WLAN_OS == WLAN_LWOS)

#include <stddef.h>
#include <stdlibsys.h>
#include <clock.h>
#include <direct.h>
#include <hal.h>
#include <kernel.h>
#include <spi.h>

#include <wlan/wlan_compat.h>
#include <wlan/version.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_stable.h>
#include <lw-wlan.h>
#include <wlan/am930mac.h>
#include <wlan/am930hw.h>
#include <wlan/am930mgr.h>

#else
	#error "No WLAN_OS match!"
#endif

/*================================================================*/
/* Local Macros  */

/* Am79c930, io port address methods (16 port mode) */
#define GCR(hw)		((hw)->iobase)
#define BSS(hw)		(((hw)->iobase) + 0x01)
#define LMAL(hw)	(((hw)->iobase) + 0x02)
#define LMAU(hw)	(((hw)->iobase) + 0x03)
#define DPLL(hw)	(((hw)->iobase) + 0x04)
#define DPLM(hw)	(((hw)->iobase) + 0x05)
#define DPUM(hw)	(((hw)->iobase) + 0x06)
#define DPUU(hw)	(((hw)->iobase) + 0x07)

#define TIR(index,base)	 /* TODO: write an inline for accessing the TIR's */

/* General */
#define ABS(N) ((N) < 0 ? -(N) : (N))
#define AM930HW_FWNOTREADY		(0x01)
#define AM930HW_FWCMDTIMEOUT	(0x02)

#define AM930HW_CMD_SUCCESS			0
#define AM930HW_CMD_GETMIBFAILED	1
#define AM930HW_CMD_INITTXFAILED	2
#define AM930HW_CMD_INITRXFAILED	3
#define AM930HW_CMD_SCANFAILED		4
#define AM930HW_CMD_SETMIBFAILED	5
#define AM930HW_CMD_SYNCFAILED		6

/*--- getset function action values ------------------*/
#define SU_GET_INTMASK1		0x01
#define SU_SET_INTMASK1		0x02
#define SU_GET_INTMASK2		0x04
#define SU_SET_INTMASK2		0x08
#define SU_GET_INTSTATUS1	0x10
#define SU_SET_INTSTATUS1	0x20
#define SU_GET_INTSTATUS2	0x10
#define SU_SET_INTSTATUS2	0x20

#define SU_GETSET_INT_FAILED	(BIT7) 

/* Queue setup values */
#define AM930_NTXDESC		5
#define AM930_MAX_TXDATA	1550   /* NEED TO CHANGE THIS TO 802.11 MAX */
#define AM930_TXSLOT_LEN	(sizeof(am930tx_desc_t) + AM930_MAX_TXDATA)

/*================================================================*/
/* Local Types  */

/* return type for the f/w command send method */
__WLAN_PRAGMA_PACK1__
typedef struct am930hw_cmd_result
{
	UINT8	fw_status		__WLAN_ATTRIB_PACK__;		/* f/w returned status  */
	UINT8	fw_err_off		__WLAN_ATTRIB_PACK__;		/* f/w error offset */
	UINT8	drvr_status		__WLAN_ATTRIB_PACK__;		/* driver code errors */
	UINT8	rsvd			__WLAN_ATTRIB_PACK__;			
} am930hw_cmd_result_t;
__WLAN_PRAGMA_PACKDFLT__


/* types for setting up the tx queues in card memory */
__WLAN_PRAGMA_PACK1__
typedef struct am930tx_dataslot
{
	am930tx_desc_t	desc						__WLAN_ATTRIB_PACK__;
	UINT8			data[AM930_MAX_TXDATA]		__WLAN_ATTRIB_PACK__;
} am930tx_dataslot_t;
__WLAN_PRAGMA_PACKDFLT__


/*================================================================*/
/* Local Functions  */

/* private: */
/* Hardware/firmware helper methods, (private) */
static int	am930hw_lockint( am930hw_t *hw);
static void	am930hw_unlockint( am930hw_t *hw);
static UINT32 am930hw_read_rxnext(am930hw_t *hw);

static am930hw_cmd_result_t 
am930hw_fw_cmd( am930hw_t *hw, UINT8 cmd, void* cmdparms, INT32 parmsize);

static UINT16 am930hw_getset_int( am930hw_t *hw, int action, UINT8 mask, UINT8 status );

static int am930hw_init_tx_queues( am930hw_t *hw);

UINT32 am930hw_sync( am930hw_t *hw, UINT32 ch, UINT32 startBSS, 
					wlan_bss_ts_t ts, UINT32 ref_time );

/* firmware command helpers */
static UINT32 am930hw_tx_enable(am930hw_t *hw);
static UINT32 am930hw_rx_enable(am930hw_t *hw);

/* ISR helpers */
static void am930hw_onint_scancomplete( am930hw_t *hw);
static void am930hw_onint_rx( am930hw_t *hw );
static void am930hw_onint_tx( am930hw_t *hw );
static void am930hw_onint_cmdcomplete( am930hw_t *hw );

/* Card tx buffer memory mgmt */
void am930mem_init( am930mem_t *mem, UINT32 start, UINT32 len);
UINT32 am930mem_alloc( am930mem_t *mem, UINT32 size );
UINT32 am930mem_free( am930mem_t *mem, UINT32 p, UINT32 len);

/* Card memcpy functions */
static void readcard( am930hw_t *hw, UINT32 cardaddr, void *buf, UINT32 len);
static void writecard( am930hw_t *hw, UINT32 cardaddr, void *buf, UINT32 len);
static void memsetcard( am930hw_t *hw, UINT32 cardaddr, UINT8 val, UINT32 len);
static UINT8  read8( am930hw_t *hw, UINT32 cardaddr );
static UINT16 read16( am930hw_t *hw, UINT32 cardaddr );
static UINT32 read32( am930hw_t *hw, UINT32 cardaddr );
static UINT32 reread32( am930hw_t *hw, UINT32 cardaddr );
static void   write8( am930hw_t *hw, UINT32 cardaddr, UINT8 val );
static void   write16( am930hw_t *hw, UINT32 cardaddr, UINT16 val );
static void   write32( am930hw_t *hw, UINT32 cardaddr, UINT32 val );

/* Debug Helper Methods*/
void am930hw_dbprintregs( am930hw_t *hw);
void am930hw_dbprintCS_blk( am930hw_t *hw);
void am930hw_dbprintCMD_blk( am930hw_t *hw);
void am930hw_dbprintRXdesc( am930rx_desc_t *d);
void am930hw_dbprintTXdesc( am930tx_desc_t *d);
void am930hw_dbprintallMIBs(am930hw_t *hw);
void am930hw_dbprintMIBlocal(am930hw_t *hw);
void am930hw_dbprintMIBmac(am930hw_t *hw);
void am930hw_dbprintMIBmac_addr_stat_grp(am930hw_t *hw);
void am930hw_dbprintMIBmac_statistics(am930hw_t *hw);
void am930hw_dbprintMIBmac_mgmt( am930hw_t *hw);
void am930hw_dbprintMIBphy(am930hw_t *hw);

void am930db_prrxdesc( am930rx_desc_t *desc);

/*================================================================*/
/* Static variables  */

#if (WLAN_OS == WLAN_LWOS)

static am930hw_t	ghw;

#endif


/*================================================================*/
/* defined here because of inline */

/*----------------------------------------------------------------
*	am930hw_(un)lockint
*
*	These two functions handle locking and unlocking driver access
*	to the control/status fields int_status and int_mask. 
*	This is handled via the lockout_host and lockout_fw fields of
*	the cs block.
*
*	PS: if the firmware isn't locked out already, it doesn't hurt
*       anything to call unlockint
*
*	returns: 0 on success
*			 non-zero on failure
----------------------------------------------------------------*/
static __WLAN_INLINE__ void am930hw_unlockint( am930hw_t *hw)
{
	DBFENTER;
	write8(hw, hw->cs + CS_OFF_LOCKOUT_FW, 0);
	DBFEXIT;
}

static __WLAN_INLINE__ int am930hw_lockint( am930hw_t *hw)
{
	int				result = 0;
	int				ntries;
	int				count;
	wlan_flags_t	flags;

	DBFENTER;

	WLAN_INT_DISABLE(flags);

	/* Attempt the access procedure at most three times */
	ntries = 3;
	do
	{
		/*--- check access ---*/
		count = 10; /* waitloop */
		while( count-- > 0 && read8(hw, hw->cs + CS_OFF_LOCKOUT_HOST))
		{
			udelay(5);
		}

		if ( count > 0)
		{
			/*--- attempt lockout ---*/
			write8(hw, hw->cs + CS_OFF_LOCKOUT_FW, 0xff); /* lockout the f/w */
	
			/*--- check access again ---*/
			if ( read8(hw, hw->cs + CS_OFF_LOCKOUT_HOST) != 0 )
			{
				write8( hw, hw->cs + CS_OFF_LOCKOUT_FW, 0);
			}
		}
	} while ( read8(hw, hw->cs + CS_OFF_LOCKOUT_FW) == 0 && ntries--);

	WLAN_INT_ENABLE(flags);

	if ( ntries <= 0 )
	{
		WLAN_LOG_DEBUG0(2, "Attempt to lockout f/w from cs block failed!\n");
		result = 1;
	}
	DBFEXIT;
	return result;
}



/*----------------------------------------------------------------
*	am930hw_construct
*
*	Called _after_ someone has determined that we do in fact have
*	an am930 device present in the system _and_ we know where it 
*	is.  Compared to some drivers, this might be considered to be
*	"probe part 2".
*
*	This function will:
*		allocate the hw struct
*		fill in constant (possibly calculated) fields
*		initialize the ioports on the card,
*		perform a reset of the card (includes the f/w NOP test)
*		read the f/w banner to determine the f/w version
*		test the f/w version for a match with this driver
*
*
*	When complete, we know that the card has been successfully reset
*	and the firmware is up and running.  Additionally, the
*	firmware has been identified
*
*	Subsequent code will either prepare the card for tx/rx
*	or prepare the card for flash writes.
*
*	Arguments:
*		irq		- IRQ being used by the card
*		iobase	- base addr for io window
*		membase	- base addr for memory window, if zero then there
*					is no memory window, use the io method to access
*					on card SRAM
*		
*	returns: addr. of the hw object if successful, NULL otherwise
*
----------------------------------------------------------------*/
am930hw_t* am930hw_construct( UINT32 irq, UINT32 iobase, UINT32 membase,
							  am930mac_t *mac)
{
	am930hw_t				*hw = NULL;
	char					banner[SU_LEN_BANNER];
	char					*endp;
	UINT8					reg;
	su_mib_local_t			mib;
	UINT32					rx_buffer_top; 
	UINT8					one = 1;
	su_mib_mac_statistics_t	mib_stats;

	DBFENTER;

	/* allocate and zero the object */
#if (WLAN_OS == WLAN_LINUX_KERNEL)
	hw = kmalloc( sizeof(am930hw_t), GFP_KERNEL);
#elif (WLAN_OS == WLAN_LWOS)
	hw = &ghw;
#else
	#error "No WLAN_OS match!"
#endif

	if (hw != NULL )
	{
		memset( hw, 0, sizeof(am930hw_t));

		hw->state |= AM930HW_CONFIG_PENDING;

		/* Initialize some of the members of the object */
		hw->mac = mac;
		hw->membase = membase;
		hw->usemem = (membase != 0);
		hw->iobase = iobase;
		hw->irq = irq;

		if (hw->membase) 
		{
			WLAN_LOG_NOTICE3("am930: io=0x%lx irq=%ld\n membase=0x%lx\n", 
				hw->iobase, hw->irq, hw->membase);
		}
		else
		{
			WLAN_LOG_NOTICE2("am930: io=0x%lx irq=%ld\n", 
				hw->iobase, hw->irq);
		}

		hw->banner =	SU_OFF_BANNER;
		hw->cs =		SU_OFF_CNTL_STATUS_BLK;
		hw->cmd =		SU_OFF_CMD_BLK;
		hw->txcmplt =	SU_OFF_LAST_TXDESC;
		hw->vbm =		SU_OFF_VBM;

		/* Card is in an unknown state, perform a reset */
		hw->mac->hw = hw;   /* hook up with mac so interrupts are handled */
		if ( am930hw_reset(hw) != 0 )
		{
			am930hw_destruct( hw );
			hw = NULL;
		}
		else  
		{
			/* card should be in a post reset state and 
			   f/w NOP was successful, we can now issue f/w cmds and
			   access oncard SRAM 
			*/
#if (WLAN_OS == WLAN_LWOS )
//			LED_Blink( WLAN_HLED_POWER, LED_ON);
#endif

			/* first get and check the firmware banner for correct version */
			readcard( hw, hw->banner, banner, SU_LEN_BANNER);
			WLAN_LOG_NOTICE1("am930: Firmware Banner: %s \n", banner);
#if (WLAN_OS == WLAN_LINUX_KERNEL )
			hw->fw_major = simple_strtoul(banner+13, &endp, 10);
			hw->fw_minor = simple_strtoul(endp+1, &endp, 10);
			hw->fw_date  = simple_strtoul(endp+1, &endp, 10);
			hw->fw_api   = simple_strtoul(endp+4, &endp, 10);
#elif (WLAN_OS == WLAN_LWOS )
			hw->fw_major = 2;
			hw->fw_minor = 1;
			hw->fw_date  = 0;
			hw->fw_api   = 0;
#else
	#error "No WLAN_OS match!"
#endif

			WLAN_LOG_NOTICE4( "f/w version:%lu.%02lu  date:%lu api:%lu\n", 
				hw->fw_major, hw->fw_minor, hw->fw_date, hw->fw_api);

			if ( !( (hw->fw_major == 2 && hw->fw_minor == 0) ||  
			        (hw->fw_major == 2 && hw->fw_minor == 1) ||  
					(hw->fw_major == 1 && hw->fw_minor == 50) )  )
			{
				WLAN_LOG_ERROR0("Unsupported firmware version!\n");
				am930hw_destruct( hw );
				hw = NULL;
			}
			else
			{
				/*--------------------------------------------*/
				/* Set the control area variables */
				/* Lets set usr_return to zero */
				write8(hw, hw->cs + CS_OFF_USER_RTN, 0);

				/* Set the pwr_dn to one */
				write8(hw, hw->cs + CS_OFF_DIS_PWR_DN, 1);

				/* set the state bit for the object */
				hw->state &= ~AM930HW_CONFIG_PENDING;
				hw->state |= AM930HW_CONFIG;
			
				WLAN_LOG_INFO0("Reset and f/w TestInterface are successful\n");

				/*--------------------------------------------*/
				/* Set some MIB items */

				/* disable TX fragmentation, because of host-based WEP */
				am930hw_mibsetitem( 
					hw, LOC_FRAGDISABLE, &one, MI_SZ(LOC_FRAGDISABLE));

				/* disable RX fragment reassembly, too much SRAM load */
				am930hw_mibsetitem( 
					hw, LOC_REASSEMBLY_DISABLE, &one, MI_SZ(LOC_REASSEMBLY_DISABLE));

				/* clear the MIB counter items */
				memset( &mib_stats, 0, sizeof(mib_stats));
				am930hw_mibset( hw, SUMIB_STAT, 0, 
							sizeof(mib_stats), &mib_stats);

				/* increase the tx buffer size */
				am930hw_mibget( hw, SUMIB_LOCAL, 0, 
							sizeof(su_mib_local_t), &mib);

				WLAN_LOG_DEBUG2(1, "default f/w tx.off=0x%04lx tx.len=0x%04lx\n", 
					amd2host32(mib.tx_buffer_offset), amd2host32(mib.tx_buffer_size));
				WLAN_LOG_DEBUG2(1, "default f/w rx.off=0x%04lx rx.len=0x%04lx\n", 
					amd2host32(mib.rx_buffer_offset), amd2host32(mib.rx_buffer_size));

				mib.tx_buffer_size = 
					host2amd32(AM930_NTXDESC * AM930_TXSLOT_LEN);

				rx_buffer_top = 
					amd2host32(mib.rx_buffer_offset) +
					amd2host32(mib.rx_buffer_size);

				mib.rx_buffer_offset = 
					host2amd32(
						amd2host32(mib.tx_buffer_offset) + 
						amd2host32(mib.tx_buffer_size) );

				mib.rx_buffer_size = 
					host2amd32(
						rx_buffer_top - 
						amd2host32(mib.rx_buffer_offset));

				WLAN_LOG_DEBUG2(1, "f/w tx.off=0x%04lx tx.len=0x%04lx\n", 
					amd2host32(mib.tx_buffer_offset), amd2host32(mib.tx_buffer_size));
				WLAN_LOG_DEBUG2(1, "f/w rx.off=0x%04lx rx.len=0x%04lx\n", 
					amd2host32(mib.rx_buffer_offset), amd2host32(mib.rx_buffer_size));

				am930hw_mibset( hw, SUMIB_LOCAL, 0, 
							sizeof(su_mib_local_t), &mib);

				/* If we're an AP, set some extra items */
				if (am930_isap)  /* can't do this, it isn't set yet hw->mac->mode == AM930_MACMODE_ESS_AP) */
				{
					/* set the AP mib */
					am930hw_mibsetitem( hw, LOC_NETWORK_MODE, &one, MI_SZ(LOC_NETWORK_MODE));
					/* set the AP mib */
					am930hw_mibsetitem( hw, LOC_ACTING_AS_AP, &one, MI_SZ(LOC_ACTING_AS_AP));

					/* Set the driver state */
					if (hw->mac->mode == AM930_MACMODE_ESS_AP)
					{
						am930hw_stateset( hw, SUCS_DS_AP_NO_ASSOC);
#if (WLAN_OS == WLAN_LWOS)
//						LED_Blink( WLAN_HLED_LINK, LED_OFF);
#endif
					}
				}

				/* clear the VBM area */
				memsetcard( hw, hw->vbm, 0x00, 0x100);

				/* Get the rx buffer loc and size for sanity checking in */
				/*  onint_rx */
				am930hw_mibget( hw, SUMIB_LOCAL, 0, 
						sizeof(su_mib_local_t), &mib);

				hw->rx_base = amd2host32(mib.rx_buffer_offset);
				hw->rx_len  = amd2host32(mib.rx_buffer_size);

				/* Save our MAC address */
				am930hw_mibgetitem( hw, ADDR_MAC_ADDR, 
					hw->addr, MI_SZ(ADDR_MAC_ADDR) );

				/* Set the f/w interrupt mask and status */
				am930hw_getset_int( hw, SU_SET_INTMASK1 | SU_SET_INTSTATUS1, 
					SUCS_INT_WAKEUP |
					SUCS_INT_CFPEND |
					SUCS_INT_DTIMTX |
					SUCS_INT_CFPSTART |
					SUCS_INT_MASK_CMD,
					0x00 );
				am930hw_getset_int( hw, SU_SET_INTMASK2 | SU_SET_INTSTATUS2, 
					0x00, 0x00 );
				
				/* Enable Core Generated Interrupts and clear any pending */
				reg = inb_p( GCR(hw) );
				reg |= BIT1 | BIT3;
				outb_p( reg, GCR(hw) );
			}
		}
	}

	DBFEXIT;
    return hw;
} 


/*----------------------------------------------------------------
*	am930hw_destruct
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_destruct(am930hw_t *hw)
{
	DBFENTER;

#if (WLAN_OS == WLAN_LINUX_KERNEL)
	kfree_s( hw, sizeof(am930hw_t));
#elif (WLAN_OS == WLAN_LWOS)
#else
	#error "No WLAN_OS match!"
#endif

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930hw_fw_cmd
*
*	This is a helper function used to send commands to the
*	firmware. It copies its arguments to the command block
*	and then sets the command. After setting the command, we then
*	wait for the command to complete. The wait includes a timeout
*	and if the command times out, that status is returned.
*	Otherwise we return the firmware indicated status. If the
*	firmware status is OK and if needed, the command parms are
*	copied to the buffer space provided by the caller.
*
*	Argument notes:
*		the ABS of the parmsize argument provides the number of 
*		bytes to copy to the command parm area, and if the parm
*		is < 0 then after the command completes, we copy that
*		many bytes back from the command parm area as well.
*
*	returns: an am930cmd_result_t structure, which should be 
*			four bytes long and can be played with like an int.
----------------------------------------------------------------*/
am930hw_cmd_result_t 
am930hw_fw_cmd( am930hw_t *hw, UINT8 cmd, void* cmdparms, INT32 parmsize)
{
	am930hw_cmd_result_t	result;
	UINT32					timeout;

	DBFENTER;

	/* clear the result struct */
	memset( &result, 0, sizeof(result));

	/* wait for the cmd block to be available */
#if (WLAN_OS == WLAN_LINUX_KERNEL)
	timeout = jiffies + wlan_ms_to_ticks(100);
	while(  read8(hw, hw->cmd + CMD_OFF_CODE) != 0 &&
			read8(hw, hw->cmd + CMD_OFF_STATUS) == 0 &&
			jiffies < timeout );
#elif (WLAN_OS == WLAN_LWOS)
	timeout = 1000;
	while(  read8(hw, hw->cmd + CMD_OFF_CODE) != 0 &&
			read8(hw, hw->cmd + CMD_OFF_STATUS) == 0 &&
			timeout )
	{
		udelay(10);
		timeout--;
	}
#else
	#error "No WLAN_OS match!"
#endif

#if (WLAN_OS == WLAN_LINUX_KERNEL)
	if ( jiffies >= timeout )
#elif (WLAN_OS == WLAN_LWOS)
	if ( timeout == 0 )
#else
	#error "No WLAN_OS match!"
#endif
	{
		WLAN_LOG_WARNING0("Timed out waiting for f/w cmd byte available\n");
		result.drvr_status = AM930HW_FWNOTREADY;
	}
	else
	{
		/* Make sure the status and code are set properly */
		write8(hw, hw->cmd + CMD_OFF_CODE, 0);
		write8(hw, hw->cmd + CMD_OFF_STATUS, 0);
		
		/* copy the parms to the command block */
		if ( parmsize != 0 )
		{
			writecard(hw, hw->cmd + CMD_OFF_PARMS, cmdparms, ABS(parmsize));
		}

		/* Tell the hw object that this cmd is the most recent */
		hw->last_cmd = cmd;

		/* set the command */
		write8(hw, hw->cmd + CMD_OFF_CODE, cmd);

		/* Now, wait for the result */
#if (WLAN_OS == WLAN_LINUX_KERNEL)
		timeout = jiffies + wlan_ms_to_ticks(1500);
		while ( read8(hw, hw->cmd + CMD_OFF_STATUS) == SUCMD_STAT_IDLE && 
				jiffies < timeout );
#elif (WLAN_OS == WLAN_LWOS)
		timeout = 1500;
		while ( read8(hw, hw->cmd + CMD_OFF_STATUS) == SUCMD_STAT_IDLE && 
				timeout )
		{
			udelay(10);
			timeout--;
		}
#else
	#error "No WLAN_OS match!"
#endif

		/* Did we time out or did the fw finish with something? */
#if (WLAN_OS == WLAN_LINUX_KERNEL)
		if ( jiffies >= timeout )
#elif (WLAN_OS == WLAN_LWOS)
		if ( timeout == 0 )
#else
	#error "No WLAN_OS match!"
#endif
		{
			/* If we timed out, simply set the drvr status and return */
			result.fw_status = read8(hw, hw->cmd + CMD_OFF_STATUS);
			result.drvr_status = AM930HW_FWCMDTIMEOUT;
			WLAN_LOG_WARNING1("We timed out on command: 0x%02x\n", hw->last_cmd);
		}
		else
		{
			/* The fw finished with something, what? */
			if ( read8(hw, hw->cmd + CMD_OFF_STATUS) != SUCMD_STAT_CMPLT )
			{
				/* some error occurred, save the result and move on */
				result.fw_status = read8(hw, hw->cmd + CMD_OFF_STATUS);
				result.fw_err_off = read8(hw, hw->cmd + CMD_OFF_ERR_OFF);
				WLAN_LOG_DEBUG1(1,"f/w cmd error, code=0x%x\n", result.fw_status);
			}
			else
			{
				/* Everything appears to be OK, now collect any results */
				result.drvr_status = 0;
				result.fw_status = read8(hw, hw->cmd + CMD_OFF_STATUS);

				if ( parmsize < 0 )
				{
					readcard( hw, hw->cmd + CMD_OFF_PARMS, cmdparms, ABS(parmsize));
				}
			}

			/* clear the cmd and status */
			write8(hw, hw->cmd + CMD_OFF_CODE, 0);
			write8(hw, hw->cmd + CMD_OFF_STATUS, 0);
		}
	}

	DBFEXIT;
	return result;
} 



/*----------------------------------------------------------------
*	am930hw_getset_int
*
*	Attempts to access the sutro f/w control structure int_mask or
*	int_status field of the given control/status block. It uses the
*	lockout_fw and lockout_host fields to block the firmware from
*	accessing the mask and status fields while we get or update
*	them.
*
*	action values:
*		SU_GET_INTMASK1		get the interrupt mask
*		SU_SET_INTMASK1		set the interrupt mask
*		SU_GET_INTMASK2		get the interrupt mask
*		SU_SET_INTMASK2		set the interrupt mask
*		SU_GET_INTSTATUS1	get the interrupt status
*		SU_GET_INTSTATUS1	get the interrupt status
*		SU_SET_INTSTATUS2	set the interrupt status
*		SU_SET_INTSTATUS2	set the interrupt status
*
*	The value argument is only used by the SET actions.
*
*	returns: -1 on failure, 
*			 0 or GET result on success
----------------------------------------------------------------*/
static UINT16 am930hw_getset_int( am930hw_t *hw, 
								int action,
								UINT8 mask,
								UINT8 status )
{
	UINT16 result = 0;
	DBFENTER;

	if ( am930hw_lockint(hw) == 0 )
	{
		if (action & SU_GET_INTMASK1 )
			result = read8(hw, hw->cs + CS_OFF_INT_MASK);

		if ( action & SU_SET_INTMASK1 )
			write8(hw, hw->cs + CS_OFF_INT_MASK, mask);

		if (action & SU_GET_INTMASK2 )
			result = read8(hw, hw->cs + CS_OFF_INT_MASK2);

		if ( action & SU_SET_INTMASK2 )
			write8(hw, hw->cs + CS_OFF_INT_MASK2, mask);

		if ( action & SU_GET_INTSTATUS1 )
			result = read8(hw, hw->cs + CS_OFF_INT_STATUS);

		if ( action & SU_SET_INTSTATUS1 )
			write8(hw, hw->cs + CS_OFF_INT_STATUS, status);

		if ( action & SU_GET_INTSTATUS2 )
			result = read8(hw, hw->cs + CS_OFF_INT_STATUS2);

		if ( action & SU_SET_INTSTATUS2 )
			write8(hw, hw->cs + CS_OFF_INT_STATUS2, status);

		am930hw_unlockint(hw);
	}
	else
	{
		result = -1;
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_init_rx_tx
*
*	Performs the second part of the object initialization. We have
*	to do this here so that all of the links are in place in the
*	other objects. Particularly so that interrupts get routed to
*	this object.
*
*	returns: zero on success, non-zero on failure
----------------------------------------------------------------*/
int am930hw_init_rx_tx( am930hw_t *hw )
{
	int		result = 0;

	DBFENTER;


	/* Configure the tx queues */
	if ( am930hw_init_tx_queues(hw) != 0 )
	{
		result = 1;
	}
	else
	{
		/* Enable tx */
		if ( am930hw_tx_enable(hw) != AM930HW_CMD_SUCCESS )
		{
			result = 2;
		}
		else
		{
			udelay(100);

			/* Enable rx */
			if ( am930hw_rx_enable(hw) != AM930HW_CMD_SUCCESS )
			{
				result = 3;
			}
			else
			{
				udelay(100);
			}
		}
	}
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_init_tx_queue
*
*	Initialize the tx queue related data structures and members of
*	the hw object.
*
*	NOTE: this function takes care of am930 related queues, NOT
*			any of the upper level protocol packet queues.
*
*	returns: nothing
----------------------------------------------------------------*/
int am930hw_init_tx_queues( am930hw_t *hw)
{
	int					result = 0;
	int					i;
	UINT32				curr_off;
	am930tx_dataslot_t	ds;

	DBFENTER;

	/* From the MIB, find out where the TX queue is supposed to go */
	if ( am930hw_mibgetitem( hw, LOC_TX_BUFFER_OFFSET, 
			&(hw->tx_tail), sizeof(hw->tx_tail)) != AM930HW_CMD_SUCCESS )
	{
		WLAN_LOG_ERROR0("mibgetitem(LOC_TX_BUFFER_OFFSET) Failed!\n");
		result = 1;
	}
	else
	{
		/* We've got the MIB, now init. the main queue */
		hw->tx_tail = amd2host32(hw->tx_tail);
		hw->tx_base = hw->tx_tail;
		am930hw_mibgetitem( hw, LOC_TX_BUFFER_SIZE, 
			&(hw->tx_len), sizeof(hw->tx_len));
		hw->tx_len = amd2host32(hw->tx_len);

		curr_off = hw->tx_base;

		hw->ntxbuf = hw->tx_len / sizeof(am930tx_dataslot_t);
		hw->used_txbuf = 0;

		WLAN_LOG_INFO1("Allocated %d tx slots\n", hw->ntxbuf);

		memset( &ds.desc, 0, sizeof(am930tx_desc_t));
		for ( i = 0; i < hw->ntxbuf; i++)
		{
			/* first set the address where the next desc will go */
			ds.desc.next = (i < hw->ntxbuf - 1) ?
				host2amd32(curr_off + sizeof(ds)) :
				host2amd32(hw->tx_base);

			/* set the prev ptr */
			ds.desc.prev = (i > 0) ?
				host2amd32(curr_off - sizeof(ds)) :
				host2amd32(hw->tx_base + (sizeof(ds) * (hw->ntxbuf - 1)));

			/* write the descriptor */
			writecard( hw, curr_off, &ds.desc, sizeof(ds.desc));
			/* update the offset */
			curr_off += sizeof(ds);
		}
	}
	
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_ISR
*
*	HW object interrupt service routine. Called when the pcmcia
*	circuitry has generated a host interrupt in response to an
*	interrupt from supposedly our card.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_ISR( am930hw_t *hw )
{
	UINT8	int_status1;
	UINT8	int_status2;
	UINT8	int_mask1;
	UINT8	int_mask2;
	UINT8	gcr;
	UINT8	handled = 0;
	DBFENTER;
	/* Make sure the hw object is ready */
	/*  - additional checks make sure the H/W is _really_ in the slot */
	/*  - sometimes an int reaches here after a card is removed and before */
	/*    card services has been notified. (Thanks David) */
	if ( (hw != NULL) && 
		 (hw->state & AM930HW_CONFIG)
#if (WLAN_OS == WLAN_LINUX_KERNEL)
#ifdef WLAN_PCMCIA
		 && (((dev_link_t*)(hw->mac->di))->state & DEV_PRESENT)
		 && read8(hw, hw->banner) == 'P' && read8(hw, hw->banner+1) == 'C'
#endif
		)
#elif (WLAN_OS == WLAN_LWOS)
		)
#else
	#error "No WLAN_OS match!"
#endif
	{
		/* Make sure we have not overlapped ISR calls */
		if ( hw->state & AM930HW_INTOCURRED )
		{
			WLAN_LOG_WARNING0("Overlapping Interrupt detected!\n");
		}
		else
		{
			hw->state |= AM930HW_INTOCURRED;

			/* Lock the int fields */
			if ( am930hw_lockint(hw) != 0 )
			{
				WLAN_LOG_DEBUG0(2,"Unable to lockout f/w, skip interrupt\n");

				/* Clear the ECWAIT bit of GCR (by setting...wierd) */
				gcr = inb_p( GCR(hw));
				gcr |= BIT3;
				outb_p( gcr, GCR(hw));

				/* Clear the int bit of the hw->state */
				hw->state &= ~AM930HW_INTOCURRED;
			}
			else
			{
				/* save the int_mask for future restoration */
				int_mask1 = read8( hw, hw->cs + CS_OFF_INT_MASK);
				int_mask2 = read8( hw, hw->cs + CS_OFF_INT_MASK2);
				write8(hw, hw->cs + CS_OFF_INT_MASK, 0xff);
				write8(hw, hw->cs + CS_OFF_INT_MASK2, 0xff);

				int_status1 = read8(hw, hw->cs + CS_OFF_INT_STATUS);
				int_status2 = read8(hw, hw->cs + CS_OFF_INT_STATUS2);
				write8(hw, hw->cs + CS_OFF_INT_STATUS, 0);
				write8(hw, hw->cs + CS_OFF_INT_STATUS2, 0);

				am930hw_unlockint(hw);

				WLAN_LOG_DEBUG2(4,"Int mask=0x%02x, status=0x%02x\n", int_mask1, int_status1);

				/* Handle various int sources */
				if ( int_status1 == 0 )
				{
					WLAN_LOG_DEBUG0(1,"in ISR, int_status1 == 0\n");
				}

				do {
					handled++;
					if ( SUCS_INT_IS_RX(int_status1) )
					{
						#if (WLAN_OS == WLAN_LWOS)
							LED_Blink( WIRELESS_DATA_LED, 1);
						#endif
						am930hw_onint_rx(hw);
					}
		
					if ( SUCS_INT_IS_TX(int_status1) )
					{
						#if (WLAN_OS == WLAN_LWOS)
							LED_Blink( WIRELESS_DATA_LED, 1);
						#endif
						am930hw_onint_tx(hw);
					}

					if ( SUCS_INT_IS_SCANCMPLT(int_status1) )
					{
						am930hw_onint_scancomplete(hw);
					}
		
					if ( SUCS_INT_IS_CMD(int_status1) )
					{
						am930hw_onint_cmdcomplete(hw);
					}

					/* Check the int status again, to see if more int events */
					/*   have occurred */		
					if ( am930hw_lockint(hw) != 0 )
					{
						WLAN_LOG_DEBUG0(2,"Unable to lockout f/w, 2nd time\n");
						int_status1 = 0;
					}
					else
					{
						int_status1 = read8(hw, hw->cs + CS_OFF_INT_STATUS);
						int_status2 = read8(hw, hw->cs + CS_OFF_INT_STATUS2);
						if ( int_status1 != 0 ) /* are we handling again? */
						{
							write8(hw, hw->cs + CS_OFF_INT_STATUS, 0);
							write8(hw, hw->cs + CS_OFF_INT_STATUS2, 0);
						}
						else
						{
							/* reset the int_mask to the setting it had on ISR entry */
							write8(hw, hw->cs + CS_OFF_INT_MASK, int_mask1);
							write8(hw, hw->cs + CS_OFF_INT_MASK2, int_mask2);
						}
						am930hw_unlockint(hw);
					}
				}
				while ( int_status1 != 0 );

				if ( handled > 1 )
				{
					WLAN_LOG_DEBUG1(3, "hw_ISR: handled=%d\n", handled);
				}

				/* Clear the ECWAIT bit of GCR (by setting...wierd) */
				gcr = inb_p( GCR(hw));
				gcr |= BIT3;
				outb_p( gcr, GCR(hw));

				/* Clear the int bit of the hw->state */
				hw->state &= ~AM930HW_INTOCURRED;


			} /* endif lockint */
		} /* endif !overlapped */
	} /* hw OK, card in slot */
	else
	{
		WLAN_LOG_WARNING0("ISR called when hw is NULL, not CONFIG or "
					"no card in slot, future ints may be lost!\n");
	}
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930hw_joinbss
*
*	I/F function used by mac and mgr to join a specific bss. The
*	only nod here to the sutro implementation is the sutro_ref_time
*	argument used by the sutro to update it's internal timers.
*
*	returns: AM930HW_SUCCESS or
*			 AM930HW_FAILURE
----------------------------------------------------------------*/
UINT32 am930hw_joinbss( am930hw_t *hw, UINT32 ch, UINT32 newBSS, 
						UINT8 *bssid, wlan_ie_ssid_t *ssid, UINT32 bcn_int,
						wlan_bss_ts_t ts, UINT32 sutro_ref_time )
{
	UINT32 				result = AM930HW_SUCCESS;
	su_mib_mac_mgmt_t	mm_mib;

	DBFENTER;

	/* retrieve the mib */
	if ( am930hw_mibget(hw, SUMIB_MGMT, 0, 
				sizeof(mm_mib), &mm_mib) != AM930HW_SUCCESS )
	{
		result = AM930HW_FAILURE;
	}
	else
	{
		/* set the bssid */
		memcpy( mm_mib.current_bssid, bssid, WLAN_ADDR_LEN );

		/* set the ssid */
		memcpy( mm_mib.current_essid, ssid, ssid->len + 2);

		/* set the beaconrate */
		mm_mib.beacon_period = host2amd16(bcn_int);

		/* save the mib */
		if ( am930hw_mibset(hw, SUMIB_MGMT, 0, 
					sizeof(mm_mib), &mm_mib) != AM930HW_SUCCESS )
		{
			result = AM930HW_FAILURE;
		}
		else
		{
			/* send the sync command */
			if ( am930hw_sync( hw, 
								ch, 
								newBSS, 
								(newBSS == 0) ? ts : NULL, 
								sutro_ref_time ) != AM930HW_CMD_SUCCESS )
			{
				result = AM930HW_FAILURE;
			}
			else
			{
				/* Set the driver state */
				if (hw->mac->mode == AM930_MACMODE_ESS_AP)
				{
					write8( hw, hw->cs + CS_OFF_DRIVER_STATE, SUCS_DS_AP_NO_ASSOC);
				}
			}
		}
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_onint_cmdcomplete
*
*	ISR helper for f/w generated command complete interrupts
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_onint_cmdcomplete( am930hw_t *hw )
{
	DBFENTER;

	WLAN_LOG_DEBUG1(3,"cmd complete int, last_cmd= 0x%x\n", hw->last_cmd);

	hw->last_cmd = 0;
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930hw_onint_rx
*
*	ISR helper for f/w generated rx complete interrupts
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_onint_rx( am930hw_t *hw )
{
	UINT8 			old_state;
	UINT32			old_head;
	UINT32 			next;
	UINT32			len;
	UINT32			data_off;
	UINT8 			state;
	am930rxstats_t	*stats;
	wlan_pb_t		*pb;
	UINT			nhandled = 0;
	UINT			frm_rxd = 0;
	UINT			payloadlen;

	DBFENTER;

	/* read the state and next fields from the old head */
	state = read8(hw, hw->rx_head + RXD_OFF_STATE);
	next = am930hw_read_rxnext(hw);

	while ( !SURXD_ST_IS_FWOWN(state) &&
			SURXD_ST_IS_CONSUMED(state) &&
			!SURXD_ISLAST(next) &&
			((next&0x0000ffffL) >= hw->rx_base && 
			 (next&0x0000ffffL) < (hw->rx_base+hw->rx_len)) )
	{
		/* we've got a valid next ptr, it's time to move on */	
		old_head = hw->rx_head;
		old_state = state;
		nhandled++;

		hw->rx_head = next & 0x0000ffffL;
		state = read8( hw, hw->rx_head + RXD_OFF_STATE);

		/* make sure we own the new descriptor */
		if ( SURXD_ST_IS_FWOWN(state) )
		{
			/* we've been handed a desc that's still owned by the f/w */
			/*  set the head back to the old one, and stop the loop */
			hw->rx_head = old_head;
			break;
		}
		else
		{
			/* we have a new, valid descriptor...process it */
			/*  first, get rid of the old one */
			old_state |= SURXD_ST_FWOWN;
			write8( hw, old_head + RXD_OFF_STATE, old_state);

			if ( SURXD_ST_IS_CONSUMED(state) )
			{
				/* it's a dummy, do nothing */
				WLAN_LOG_DEBUG0(1, "We got a dummy!\n");
			}

			if ( SURXD_ST_IS_RXERR(state) )
			{

				if ( SURXD_ST_IS_CRC_ERR(state) )
				{
					WLAN_LOG_WARNING0( "RX int, desc err=CRC\n");
					hw->stats.rx_crcerr++;
				}
				if ( SURXD_ST_IS_BUF_OFLOW(state) )
				{
					WLAN_LOG_WARNING0( "RX int, desc err=BUF_OFLOW\n");
					hw->stats.rx_buf_oflow++;
				}

				state |= SURXD_ST_CONSUMED;
				write8(hw, hw->rx_head+RXD_OFF_STATE, state);

				am930mac_rxframe_err(hw->mac);
			}
			else if ( !SURXD_ST_IS_CONSUMED(state) )
			{
				/* hey! we've actually got something! */
				/*  copy the frame to host memory and pass it up */
				len = read16(hw, hw->rx_head + RXD_OFF_LEN);

				/* Make a packet buffer */
				pb = am930shim_pballoc();

				if ( pb == NULL )
				{
					WLAN_LOG_DEBUG0(2,"pballoc failed.\n");
					state |= SURXD_ST_CONSUMED;
					write8(hw, hw->rx_head+RXD_OFF_STATE, state);
				}
				else
				{
					am930shim_pballoc_p80211(pb, len);

					if ( pb->p80211hostbuf == NULL )
					{
						WLAN_LOG_DEBUG0(2,"pballoc_p80211 failed!\n");
						state |= SURXD_ST_CONSUMED;
						write8(hw, hw->rx_head+RXD_OFF_STATE, state);
					}
					else
					{

						data_off = read32(hw, hw->rx_head + RXD_OFF_START_FRAME);
						data_off &= 0x0000ffff;
						payloadlen = len;

						readcard(hw, data_off, pb->p80211buf, WLAN_HDR_A3_LEN);
						data_off += WLAN_HDR_A3_LEN;
						payloadlen -= (WLAN_HDR_A3_LEN + WLAN_CRC_LEN);

						if ( WLAN_GET_FC_ISWEP(ieee2host16(pb->p80211_hdr->a3.fc)))
						{
							/* read the encrypted frame into the pb fields */
							payloadlen -= (WLAN_WEP_IV_LEN + WLAN_WEP_ICV_LEN);

							/* wep iv */
							readcard(hw, 
								data_off, 
								&(pb->wep_iv), 
								WLAN_WEP_IV_LEN); 
							data_off += WLAN_WEP_IV_LEN;

							/* wep icv */
							readcard(hw, 
								data_off + payloadlen,
								&(pb->wep_icv), 
								WLAN_WEP_ICV_LEN);

							/* payload */
							readcard(hw, 
								data_off,
								pb->p80211buf + WLAN_HDR_A3_LEN, 
								payloadlen);

							/* wlan crc */
							readcard(hw, 
								data_off + payloadlen + WLAN_WEP_ICV_LEN,
								pb->p80211buf + WLAN_HDR_A3_LEN + payloadlen,
								WLAN_CRC_LEN );

							/* adjust the pb len fields and mark as encrypted */
							pb->p80211frmlen = 
								WLAN_HDR_A3_LEN + payloadlen + WLAN_CRC_LEN;
							pb->p80211_payloadlen = payloadlen;
							pb->wep_iscrypt = 1;
						}
						else
						{
							readcard( hw, 
								data_off,
								pb->p80211buf + WLAN_HDR_A3_LEN, 
								len - WLAN_HDR_A3_LEN);
						}

						stats = (am930rxstats_t*)kmalloc(sizeof(am930rxstats_t), GFP_ATOMIC);
						if ( stats == NULL )
						{
							WLAN_LOG_DEBUG0(2,"rxstats alloc_p80211 failed!\n");
							state |= SURXD_ST_CONSUMED;
							write8(hw, hw->rx_head+RXD_OFF_STATE, state);
						}
						else
						{
							stats->rssi = read8(hw, hw->rx_head + RXD_OFF_RSSI);
							stats->ch = read8(hw, hw->rx_head + RXD_OFF_INDEX_OR_CH);
							stats->local_time = read32(hw, hw->rx_head + RXD_OFF_LOCAL_TIME);
							stats->rate = read8(hw, hw->rx_head + RXD_OFF_RATE);

							state |= SURXD_ST_CONSUMED;
							write8(hw, hw->rx_head+RXD_OFF_STATE, state);

							WLAN_HEX_DUMP(3,"hwrxf",pb->p80211buf,pb->p80211frmlen);

							am930mac_rxenqueue( hw->mac, pb, stats);

							frm_rxd = 1;

							#ifdef WLAN_INCLUDE_SNIF
							if ( hw->mac->snifflags & SNIFFLAG_SNIFRX )
							{
								am930mac_snifframe(hw->mac, pb, stats);
							}
							#endif
						}
					}
				}
			}
		}

		/* read the state and next fields from the head */
		state = read8(hw, hw->rx_head + RXD_OFF_STATE);
		next = am930hw_read_rxnext(hw);
	}

	if ( nhandled > 1 )
	{
		WLAN_LOG_DEBUG1(2, "onint_rx handled %d\n", nhandled);
	}

	#if (WLAN_OS == WLAN_LINUX_KERNEL)
	if ( frm_rxd )
	{
		am930mac_rxframe(hw->mac);
/*		am930hw_txkick(hw); */
	}
	#endif

	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930hw_read_rxnext
*
*	Repeated read of the rxnext field to make _sure_ we have real
*	data.  Very paranoid.
*
*	returns: nothing
----------------------------------------------------------------*/
static UINT32 am930hw_read_rxnext(am930hw_t *hw)
{
	UINT32 result = 0;
	UINT32 old = 0;

	result = read32( hw, hw->rx_head + RXD_OFF_NEXT);
	while ( result != old )
	{
		old = result;
		result = read32( hw, hw->rx_head + RXD_OFF_NEXT);
		if ( old != result )
		{
			WLAN_LOG_DEBUG1(1, "reread of rxnext:0x%04lx didn't match!\n",
				hw->rx_head + RXD_OFF_NEXT);
		}
	}
	return result;
}


/*----------------------------------------------------------------
*	am930hw_onint_tx
*
*	ISR helper for f/w generated tx complete interrupts
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_onint_tx( am930hw_t *hw )
{
	UINT32	last;
	UINT32	back;
	UINT8	state;
	UINT32	host;
	UINT32	txresult = 0;
	UINT32	nhandled = 0;

	DBFENTER;

	last = reread32(hw, hw->txcmplt + TXCMPLT_OFF_DATA);
	state = read8(hw, last + TXD_OFF_STATE);
	host = read32(hw, last + TXD_OFF_HOST);

	/* backwalk the queue to find last unserviced desc */
	back = last;
	while ( !(state & SUTXD_ST_FWOWN) && host != 0 )
	{
		last = back;
		write32(hw, back + TXD_OFF_HOST, 0);
		back = reread32(hw, back + TXD_OFF_PREV);
		state = read8(hw, back + TXD_OFF_STATE);
		host = read32(hw, back + TXD_OFF_HOST);
	}

	state = read8(hw, last + TXD_OFF_STATE);
	host = read32(hw, last + TXD_OFF_HOST);

	while ( !(state & SUTXD_ST_FWOWN) && 
			(state & SUTXD_ST_DONE) ) 
	{
		write8(hw, last + TXD_OFF_STATE, 0);
		write32(hw, last + TXD_OFF_HOST, 0);

		if ( state & SUTXD_ST_TXERR )
		{
			WLAN_LOG_DEBUG0(2, "TX err:\n");

			if ( state & SUTXD_ST_REJECTED )
			{
				WLAN_LOG_DEBUG0(2, "TX err=frame rejected by f/w.\n");
				hw->stats.tx_rejected++;
			}
			if ( state & SUTXD_ST_MSDU_TMOUT )
			{
				WLAN_LOG_DEBUG0(2, "TX err=msdu timeout.\n");
				hw->stats.tx_tmout++;
			}
			if ( state & SUTXD_ST_ABRT )
			{
				WLAN_LOG_DEBUG0(2, "TX err=frame aborted, after >1 tx attempt.\n");
			}
			if ( state & SUTXD_ST_ABRT_NOTX )
			{
				WLAN_LOG_DEBUG0(2, "TX err=frame aborted, no tx attempts.\n");
			}
			if ( state & SUTXD_ST_MAX_RETRY )
			{
				WLAN_LOG_DEBUG0(2, "TX err=too many RTS or MPDU retries.\n");
				hw->stats.tx_retry++;
			}
		}
	
		txresult = state & (SUTXD_ST_TXERR | SUTXD_ST_ABRT) ? 1 : 0;

		hw->used_txbuf--;
		hw->used_txbuf = wlan_max( 0, hw->used_txbuf);
		nhandled++;

		last = reread32(hw, last + TXD_OFF_NEXT);
		state = read8(hw, last + TXD_OFF_STATE);
		host = read32(hw, last + TXD_OFF_HOST);
	}

	if ( nhandled > 1 )
	{
		WLAN_LOG_DEBUG0(2, "onint_tx handled >1 txdesc\n");
	}

	#if (WLAN_OS == WLAN_LINUX_KERNEL)
		am930mac_ontxcomplete(hw->mac, txresult);
	#elif (WLAN_OS == WLAN_LWOS)
		am930lw_readerkick();
	#else
		#error "No WLAN_OS match!"
	#endif

	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930hw_onint_scancomplete
*
*	ISR helper for f/w generated scan complete interrupts
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_onint_scancomplete( am930hw_t *hw)
{
	DBFENTER;
	hw->state &= ~AM930HW_SCANNING;

	if ( hw->scan_callback != NULL )
	{
		/* alert the callback function */
		(*hw->scan_callback)(hw->scan_callbackarg);

		/* check to see if the callback put us back in scan mode */
		if ( (hw->state & AM930HW_SCANNING) == 0 )
		{
			/* if not, clear the callback and arg fields */
			hw->scan_callback = NULL;
			hw->scan_callbackarg = NULL;
		}
	}
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930hw_reset
*
*	This function performs a reset of the am930 processor, waits
*	for the firmware to "reboot" and then sends the firmware a 
*	test command.
*
*	returns: 0 on success, non-zero on failure
----------------------------------------------------------------*/
UINT32	am930hw_reset( am930hw_t *hw )
{
	UINT32 	result = 0;
	UINT32 	init_retry = 0;
	UINT8	reg;
	UINT32	count;
	UINT32	i;
	UINT8	selftest;
	am930hw_cmd_result_t	cmd_result;

	DBFENTER;

	do
	{
		/* Lets go through a reset cycle */
		/* Set the CORESET bit of GCR and wait 10 us */
		outb( BIT6, GCR(hw) );
		udelay(10);
		/* Set the SWRESET bit of GCR and wait 10 us */
		outb( BIT7, GCR(hw) );
		udelay(10);
		/* Clear the GCR and wait 10 us */
		outb( 0, GCR(hw) );
		udelay(10);
		/* Set the CORESET bit of GCR and wait 10 us */
		outb( BIT6, GCR(hw) );
		udelay(10);

		/* Clear the ctl memory  regions (f/w should do this, but doesn't) */
		for ( i = SU_OFF_LAST_TXDESC; i < SU_OFF_BUFFER; i++)
		{
			write8(hw, i, 0);
		}
	
		/* Clear the GCR and wait 10 us */
		outb( 0, GCR(hw) );
		udelay(10);

		/* Before doing the ALE test, do an initial read of BSS */
		reg = inb( BSS(hw) );
		count = 0;
		do 
		{
			udelay(50);
			
			/* Write zero to BSS and wait 10us */
			outb( 0, BSS(hw) );
			udelay(50);
			
			/* Read BSS and check the ALE Test Read bit */
			reg = inb( BSS(hw) );
		} while ( reg != 0x80 && ++count < 20 );
	
		/* Now wait for 2 seconds */
#if (WLAN_OS == WLAN_LINUX_KERNEL)
		schedule_timeout( 2*HZ );

		/* Now, check the selftest status */
		count = jiffies + (1 * HZ);
		while ( 
			((selftest = read8(hw, hw->cs + CS_OFF_SELF_TEST)) < 0xf0) && 
			(jiffies < count) )
		{
		}
#elif (WLAN_OS == WLAN_LWOS)
		count = 200;
		while( count-- )
		{
			udelay(1000);
		}

		/* Now, check the selftest status */
		count = 500;
		while ( 
			((selftest = read8(hw, hw->cs + CS_OFF_SELF_TEST)) < 0xf0) && 
			(count--) )
		{
			udelay(10)
		}
#else
	#error "No WLAN_OS match!"
#endif

		if ( selftest != 0xFF )
		{
		   WLAN_LOG_WARNING1("Card self-test failed, code=0x%02x\n", selftest);
		   result = 1;
		}

		/* Issue the test interface command */
		cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_TEST_NOP, NULL, 0); 
	
		if ( cmd_result.fw_status != SUCMD_STAT_CMPLT ||
		 	 cmd_result.drvr_status != 0 )
		{
			WLAN_LOG_WARNING1( "Firmware Test Interface failed, status=0x%x\n",
				cmd_result.fw_status);
			result = 1;
		}
	
		init_retry++;
	
	} while ( init_retry < 10 &&
			  ( cmd_result.fw_status != SUCMD_STAT_CMPLT ||
			    cmd_result.drvr_status != 0 ) );

	/* if the reset failed, log it */
	if ( init_retry >= 10 &&
	 		( cmd_result.fw_status != SUCMD_STAT_CMPLT ||
		      cmd_result.drvr_status != 0 ) )
	{
		WLAN_LOG_ERROR1( 
			"All retries of reset and Test Interface failed status=0x%x\n",
			cmd_result.fw_status);
		result = 1;
	}
	return result;
}
	
/*----------------------------------------------------------------
*	am930hw_rx_enable
*
*	Helper function to send the sutro ENABLE_RX command. 
*
*	returns: AM930HW_CMD_SUCCESS or
*			 AM930HW_CMD_INITTXFAILED
----------------------------------------------------------------*/
UINT32 am930hw_rx_enable(am930hw_t *hw)
{
	UINT32 					result = AM930HW_CMD_SUCCESS;
	sucmd_en_rcvr_t			cmd;
	am930hw_cmd_result_t	cmd_result;

	DBFENTER;

	cmd.data_desc = host2amd32(0);

/* TODO: handle the setting up of the pspoll descriptor */

	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_EN_RCVR, &cmd, -sizeof(cmd));
	
	if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || 
		cmd_result.drvr_status != 0 )
	{
		WLAN_LOG_ERROR0("init_rx failed!\n");
		result = AM930HW_CMD_INITRXFAILED;
	}
	else
	{
		hw->rx_head = amd2host32(cmd.data_desc);
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_resumefromscan
*
*	Public method used to tell the MAC to stop scanning. 
*
*	returns: AM930HW_CMD_SUCCESS
----------------------------------------------------------------*/
UINT32 am930hw_resumefromscan( am930hw_t *hw)
{
	UINT32		result = AM930HW_SUCCESS;
	am930hw_cmd_result_t	cmd_result;

	DBFENTER;
	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_RESUME, NULL, 0);
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_scan
*
*	Public method used to tell the MAC to scan. In this case it 
*   is the hardware element of the MAC that's being told what to
*   do.
*
*	The cntl argument has two purposes, it specifies
*	what kind of BSS to scan for (BSSType)and what kind of scan to
*	perform (active or passive).
*	The values of the bit fields are:
*	AM930HWSCAN_ACTIVE
*	AM930HWSCAN_PASSIVE
*	AM930HWSCAN_IBSS
*	AM930HWSCAN_ESS
*	note 1: that IBSS and ESS can be used together.
*	note 2: the units for the duration field are Kus.
*
*	returns: AM930HW_CMD_SUCCESS or
*			 AM930HW_CMD_INITTXFAILED
----------------------------------------------------------------*/
UINT32 am930hw_scan( am930hw_t *hw, UINT32 cntl, UINT8 *bssid, 
					UINT32 ch, UINT32 duration,
					wlan_ie_ssid_t *ssid,
					am930hw_scan_callback_t cb, void *callback_arg)
{
	UINT32 					result = AM930HW_SUCCESS;
	sucmd_scan_t			cmd;
	am930hw_cmd_result_t	cmd_result;
	su_mib_mac_mgmt_t		mib;
	DBFENTER;

	/* Get the mac_mgmt mib */
	if ( am930hw_mibget( hw, SUMIB_MGMT, 0, sizeof(mib), &mib) 
			!= AM930HW_CMD_SUCCESS )
	{
		WLAN_LOG_WARNING0("MIB_MGMT read failed!\n");
		result = AM930HW_FAILURE;
	}
	else
	{
		/* set the scan mode and duration (if passive) */
		if ( cntl & AM930HWSCAN_PASSIVE )
		{
			mib.scan_mode = SUMIB_SCANMODE_PASSIVE;
			mib.passive_scan_duration = host2amd16(duration);
		}
		else /* ACTIVE */
		{
			mib.scan_mode = SUMIB_SCANMODE_ACTIVE;
		}
	
		if ( am930hw_mibset( hw, SUMIB_MGMT, 0, sizeof(mib), &mib) 
				!= AM930HW_CMD_SUCCESS )
		{
			WLAN_LOG_WARNING0("MIB write failed!\n");
			result = AM930HW_FAILURE;
		}
		else
	{
			/* Set the desired ssid */
			am930hw_mibsetitem( hw, MAC_DESIRED_ESSID, 
								ssid, 
								MI_SZ(MAC_DESIRED_ESSID));

			/* set the callback */
			hw->scan_callback = cb;
			hw->scan_callbackarg = callback_arg;

			/* send the scan command */
			memset( &cmd, 0, sizeof(cmd));
			cmd.duration = host2amd16(duration);
			cmd.set_or_ch = ch;
			hw->state |= AM930HW_SCANNING;
			hw->scan_ch = ch;
			cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_SCAN, &cmd, sizeof(cmd));
			
			if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || cmd_result.drvr_status != 0 )
			{
				hw->state &= ~AM930HW_SCANNING;
				hw->scan_ch = 0;
				WLAN_LOG_WARNING1("scan command failed! ch=%ld\n", ch);
				result = AM930HW_FAILURE;
			}
		}
	}
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_stateset
*
*	Sets the driver state field of the f/w control/status block.
*
*	returns: Prior value of the state field
----------------------------------------------------------------*/
UINT8 am930hw_stateset( am930hw_t *hw, UINT8 state)
{
	UINT8 result;

	DBFENTER;

	result = read8( hw, hw->cs + CS_OFF_DRIVER_STATE);
	write8( hw, hw->cs + CS_OFF_DRIVER_STATE, state);

	DBFEXIT;
	return result;
}

/*----------------------------------------------------------------
*	am930hw_stateget
*
*	Gets the driver state field of the f/w control/status block.
*
*	returns: Prior value of the state field
----------------------------------------------------------------*/
UINT8 am930hw_stateget( am930hw_t *hw)
{
	UINT8 result;

	DBFENTER;

	result = read8( hw, hw->cs + CS_OFF_DRIVER_STATE);

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_mibget
*
*	I/F function used by mac and mgr to retrieve elements of or an
*	entire mib section. Also used by the mibgetitem function.
*
*	returns: AM930HW_SUCCESS or
*			 AM930HW_FAILURE
----------------------------------------------------------------*/
UINT32 am930hw_mibget( am930hw_t *hw, UINT32 mibsec, UINT32 offset,
						UINT32 len, void *buf)
{
	UINT32					result = AM930HW_SUCCESS;
	sucmd_get_mib_var_t		cmd;
	am930hw_cmd_result_t	cmd_result;

	DBFENTER;

	memset( &cmd, 0, sizeof(cmd));
	cmd.type = mibsec;
	cmd.size = len;
	cmd.index = offset;

	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_GET_MIB_VAR, &cmd, -sizeof(cmd));
	
	if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || 
		 cmd_result.drvr_status != 0 )
	{
		WLAN_LOG_DEBUG3(1, 
			"GetMIB failed! mibsec=0x%02lx, offset=%lu, len=%lu\n", 
			mibsec, offset, len);
		result = AM930HW_FAILURE;
	}
	else
	{
		/* TODO: keep an eye on the where the f/w copies the result to */
		memcpy( buf, cmd.data, len);
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_mibgetitem
*
*	I/F function used by mac and mgr to retrieve the values of 
*	particular MIB elements.
*
*	returns: AM930HW_SUCCESS or
*			 AM930HW_FAILURE
----------------------------------------------------------------*/
UINT32 am930hw_mibgetitem( am930hw_t *hw, UINT32 itemid, void *buf, UINT32 len)
{
	UINT32	result = AM930HW_SUCCESS;

	DBFENTER;

	result = am930hw_mibget( hw, MI_SEC(itemid), MI_OFF(itemid), 
								MI_SZ(itemid), buf);

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_mibset
*
*	I/F function used by mac and mgr to set the values of MIB
*	elements.
*
*	TODO: for now this function operates on entire sutro MIB structures.
*	TODO: need to make the function work on individual elements
*
*	returns: AM930HW_SUCCESS or
*			 AM930HW_FAILURE
----------------------------------------------------------------*/
UINT32 am930hw_mibset( am930hw_t *hw, UINT32 mibsec, UINT32 offset,
						UINT32 len, void *buf)
{
	UINT32					result = AM930HW_SUCCESS;
	sucmd_set_mib_var_t		cmd;
	am930hw_cmd_result_t	cmd_result;

	DBFENTER;

	memset( &cmd, 0, sizeof(cmd));
	cmd.type = mibsec;
	cmd.size = len;
	cmd.index = offset;
	memcpy( cmd.data, buf, len);

	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_SET_MIB_VAR, &cmd, sizeof(cmd));
	
	if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || 
		cmd_result.drvr_status != 0 )
	{
		WLAN_LOG_WARNING0("SetMIB failed!\n");
		result = AM930HW_FAILURE;
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_mibsetitem
*
*	I/F function used by mac and mgr to set the values of MIB
*	items.
*
*	returns: AM930HW_SUCCESS or
*			 AM930HW_FAILURE
----------------------------------------------------------------*/
UINT32 am930hw_mibsetitem( am930hw_t *hw, UINT32 itemid, void *buf, UINT32 len)
{
	UINT32	result = AM930HW_SUCCESS;

	DBFENTER;

	result = am930hw_mibset( hw, MI_SEC(itemid), MI_OFF(itemid), 
								MI_SZ(itemid), buf);

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_sync
*
*	Helper function to send the sutro SYNC command. 
*
*	returns: AM930HW_CMD_SUCCESS or
*			 AM930HW_CMD_SYNCFAILED
----------------------------------------------------------------*/
UINT32 am930hw_sync( am930hw_t *hw, UINT32 ch, UINT32 startBSS, 
					wlan_bss_ts_t ts, UINT32 ref_time )
{
	UINT32 					result = AM930HW_CMD_SUCCESS;
	sucmd_sync_t			cmd;
	am930hw_cmd_result_t	cmd_result;
	DBFENTER;

	memset( &cmd, 0, sizeof(cmd));

	/* values from arguments */
	cmd.set_or_ch = ch;
	cmd.start_BSS = startBSS;
	cmd.ref_time = host2amd32(ref_time);
	if ( ts != NULL )
	{
		memcpy( &cmd.bss_timestamp, ts, WLAN_BSS_TS_LEN);
	}
	else
	{
		memset( &cmd.bss_timestamp, 0, WLAN_BSS_TS_LEN);
	}


/*
	cmd.pattern = 5;
	cmd.index = 1;
	cmd.dwell_interval = 0x3c;
*/

	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_SYNC, &cmd, sizeof(cmd));
	
	if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || 
		cmd_result.drvr_status != 0 )
	{
		WLAN_LOG_ERROR0("sync command failed!\n");
		result = AM930HW_CMD_SYNCFAILED;
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_tx_enable
*
*	Helper function to send the sutro INIT_TX command. 
*	ASSUMPTION: The queue heads have already been set in the
*				members of the object.
*
*	returns: AM930HW_CMD_SUCCESS or
*			 AM930HW_CMD_INITTXFAILED
----------------------------------------------------------------*/
UINT32 am930hw_tx_enable(am930hw_t *hw)
{
	UINT32 					result = AM930HW_CMD_SUCCESS;
	sucmd_init_tx_t			cmd;
	am930hw_cmd_result_t	cmd_result;
	int						i;
	DBFENTER;

	/* Lets zero the last complete block */
	for ( i = 0; i < 5; i++)
	{
		write32(hw, hw->txcmplt + (i * sizeof(UINT32)), 0);
	}

	/* Now send the init_tx command */
	memset( &cmd, 0, sizeof(cmd));
	cmd.data_desc = host2amd32(hw->tx_tail);

	cmd_result = am930hw_fw_cmd( hw, SUCMD_CMD_INIT_TX, &cmd, sizeof(cmd));
	
	if ( cmd_result.fw_status != SUCMD_STAT_CMPLT || 
		cmd_result.drvr_status != 0 )
	{
		WLAN_LOG_ERROR0("init_tx failed!\n");
		result = AM930HW_CMD_INITTXFAILED;
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_txdata
*
*	I/F function to send a packet via the sutro data queue.
*	rate is in 100Kbps.
*
*	returns: 0 - pkt was handed to f/w and more tx buffers avail
*			>0 - pkt was handed to f/w and no more tx buffers avail
*			<0 - pkt was not sent due to no tx buffers avail
----------------------------------------------------------------*/
UINT32 am930hw_txdata( am930hw_t* hw, wlan_pb_t *pb, UINT32 rate)
{
	UINT32				result = 0;
	UINT32				slot;
	UINT8				state;
	wlan_flags_t		flags;
	UINT				writeaddr;
	UINT				totallen = 0;

	DBFENTER;
	
	if ( pb == NULL )
	{
		result = 0; /* bogus success */
	}
	else
	{
		WLAN_INT_DISABLE(flags);

		/* Test if there is room in the queue */
		state = read8(hw, hw->tx_tail + TXD_OFF_STATE);

		WLAN_INT_ENABLE(flags);

		if ( !SUTXD_ST_IS_HOSTOWN(state) ) 
		{
			WLAN_LOG_DEBUG0(1, "f/w tx queue full, frame not tx'd.\n");
			result = -1;	/* no tx, no buffers */
		} 
		else
		{
			/* if so, update the tail pointer */
			slot = hw->tx_tail;
			hw->tx_tail = read32(hw, hw->tx_tail + TXD_OFF_NEXT);
			writeaddr = slot + sizeof(am930tx_desc_t);

			if ( pb->ethhostbuf != NULL )
			{
				/* It's a converted frame with pieces in multiple bufs */
				
				/* write the 802.11 header */
				writecard( hw, writeaddr, pb->p80211buf, WLAN_HDR_A3_LEN);
				totallen = WLAN_HDR_A3_LEN;
				writeaddr += WLAN_HDR_A3_LEN;
				WLAN_HEX_DUMP(3, "txf1.1", pb->p80211buf, WLAN_HDR_A3_LEN);

				/* is it encrypted ? */
				if ( pb->wep_iscrypt )
				{
					/* write the IV */
					writecard( hw, writeaddr, (UINT8*)&(pb->wep_iv), sizeof(pb->wep_iv));
					totallen += sizeof(pb->wep_iv);
					writeaddr += sizeof(pb->wep_iv);
					WLAN_HEX_DUMP(3, "txf1.2", &(pb->wep_iv), sizeof(pb->wep_iv));
				}
				
				/* Now write any remaining in p80211buf */
				if ( pb->p80211buflen > WLAN_HDR_A3_LEN)
				{
					writecard( hw, 
						writeaddr, 
						pb->p80211buf + WLAN_HDR_A3_LEN,
						pb->p80211buflen - WLAN_HDR_A3_LEN);

					totallen += (pb->p80211buflen - WLAN_HDR_A3_LEN);
					writeaddr += (pb->p80211buflen - WLAN_HDR_A3_LEN);
					WLAN_HEX_DUMP(3, "txf1.3",
						pb->p80211buf + WLAN_HDR_A3_LEN,
						pb->p80211buflen - WLAN_HDR_A3_LEN);
				}

				/* It's a converted frame, write the payload from p80211 */
				writecard( hw, 
					writeaddr, 
					pb->p80211_payload, 
					pb->p80211_payloadlen);
				totallen += pb->p80211_payloadlen;
				writeaddr += pb->p80211_payloadlen;
				WLAN_HEX_DUMP(3, "txf1.4", 
					pb->p80211_payload, 
					pb->p80211_payloadlen);

				/* is it encrypted ? */
				if ( pb->wep_iscrypt )
				{
					/* write the ICV */
					writecard( hw, writeaddr, (UINT8*)&(pb->wep_icv), sizeof(pb->wep_icv));
					totallen += sizeof(pb->wep_icv);
					writeaddr += sizeof(pb->wep_icv);
					WLAN_HEX_DUMP(3, "txf1.5", 
						(UINT8*)&(pb->wep_icv), 
						sizeof(pb->wep_icv)); 
				}

			}
			else
			{
				/* write the 802.11 header */
				writecard( hw, writeaddr, pb->p80211buf, WLAN_HDR_A3_LEN);
				totallen = WLAN_HDR_A3_LEN;
				writeaddr += WLAN_HDR_A3_LEN;
				WLAN_HEX_DUMP(3, "txf2.1", pb->p80211buf, WLAN_HDR_A3_LEN);

				/* is it encrypted ? */
				if ( pb->wep_iscrypt )
				{
					/* write the IV */
					writecard( hw, writeaddr, (UINT8*)&(pb->wep_iv), sizeof(pb->wep_iv));
					totallen += sizeof(pb->wep_iv);
					writeaddr += sizeof(pb->wep_iv);
					WLAN_HEX_DUMP(3, "txf2.2", 
						&(pb->wep_iv), 
						sizeof(pb->wep_iv));
				}

				/* write the body */
				writecard( hw, 
					writeaddr, 
					pb->p80211_payload, 
					pb->p80211_payloadlen);
				totallen += pb->p80211_payloadlen;
				writeaddr += pb->p80211_payloadlen;
				WLAN_HEX_DUMP(3, "txf2.3", 
					pb->p80211_payload, 
					pb->p80211_payloadlen);

				/* is it encrypted ? */
				if ( pb->wep_iscrypt )
				{
					/* write the ICV */
					writecard( hw, writeaddr, (UINT8*)&(pb->wep_icv), sizeof(pb->wep_icv));
					totallen += sizeof(pb->wep_icv);
					writeaddr += sizeof(pb->wep_icv);
					WLAN_HEX_DUMP(3, "txf2.4", 
						(UINT8*)&(pb->wep_icv), 
						sizeof(pb->wep_icv)); 
				}
			}

			/* set the offset and len */
			WLAN_LOG_DEBUG1(3, "totallen=%d.\n", totallen);
			write32(hw, slot + TXD_OFF_START_FRAME, slot + sizeof(am930tx_desc_t));

			#ifdef DB_TX_READBACK
			{
				UINT8 buf[2000];
				readcard(hw, slot+sizeof(am930tx_desc_t), buf, totallen);
				WLAN_HEX_DUMP(2,"tx rdback:", buf, totallen);
			}
			#endif

			write16(hw, slot + TXD_OFF_LEN, totallen);

			write8(hw, slot + TXD_OFF_SIFS_ATTEMPTS, 0);
			write8(hw, slot + TXD_OFF_SIFS_FAILURES, 0);
			write8(hw, slot + TXD_OFF_DIFS_ATTEMPTS, 0);
			write8(hw, slot + TXD_OFF_DIFS_FAILURES, 0);
			write8(hw, slot + TXD_OFF_RTS_ATTEMPTS, 0);
			write8(hw, slot + TXD_OFF_DATA_ATTEMPTS, 0);
			write8(hw, slot + TXD_OFF_TX_CNTL, 0);
			write8(hw, slot + TXD_OFF_RATE, rate);
			write32(hw, slot + TXD_OFF_HOST, 1);

			/* give sutro ownership of the descriptor */
			write8(hw, slot + TXD_OFF_STATE, SUTXD_ST_FWOWN);

			hw->used_txbuf++;

			result = (hw->used_txbuf >= hw->ntxbuf) ? 1 : 0;

			#ifdef WLAN_INCLUDE_SNIF
			if ( hw->mac->snifflags & SNIFFLAG_SNIFTX )
			{
				am930mac_snifframe(hw->mac, pb, NULL);
			}
			#endif
		}

		/* WLAN_INT_ENABLE(flags); */
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930hw_txkick
*
*	If there is space in the f/w queue, generate a 'kicker' 
*	ontxcomplete for the mac.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930hw_txkick( am930hw_t* hw )
{
	wlan_flags_t	flags;

	WLAN_INT_DISABLE(flags);

	if ( hw->used_txbuf < hw->ntxbuf )
	{
		am930mac_ontxcomplete(hw->mac, 0xffffffffUL);
	}

	WLAN_INT_ENABLE(flags);
	return;
}


/*----------------------------------------------------------------
*	am930mem_init
*
*	Initializes a structure to handle _very_ simple memory 
*	management.  The structure contains a bitarray that's used
*	to indicate the alloc/free state of each fragment of a block
*	of memory.  The region of memory is identified by the start and
*	len members.  This implementation is used to manage the
*	tx buffer memory on the PRISM card.
*
*	Note: If len is not evenly divisible by sizeof(mem->bits)*8, then
*			there will be a small chunk of memory at the end of the
*			managed region that will never be allocated.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mem_init( am930mem_t *mem, UINT32 start, UINT32 len)
{
	memset( mem, 0, sizeof(*mem));
	mem->start = start;
	mem->len = len;
	mem->fragsize = len / (sizeof(mem->bits) * 8);
}


/*----------------------------------------------------------------
*	am930mem_alloc
*
*	Tests the structure lock.  If set, returns NULL
*	Searches the bitarray for the appropriate sized chunk,
*	If found
*		sets the bits
*		unlocks the structure
*		returns the offset
*	else
*		unlocks the structure
*		returns NULL
*
*	Note: LSb in mem->bits corresponds to fragment at address mem->start
*	Note: mem->start can't be zero
*	Note: len of alloc'd region isn't saved, caller _must_ save it!
*
*	returns: 0 on failure, non-zero on success
----------------------------------------------------------------*/
UINT32 am930mem_alloc( am930mem_t *mem, UINT32 size )
{
	UINT32			amask;
	UINT32			nbits;
	INT				i;
	UINT32			allocaddr = 0;
	wlan_flags_t	flags;

	WLAN_INT_DISABLE(flags);

	if ( mem->lock == 0)
	{
		mem->lock = 1;

		nbits = size / mem->fragsize;
		nbits += ((size % mem->fragsize) ? 1 : 0);
		amask = ((UINT32)0xffffffff) >> ((sizeof(mem->bits) * 8) - nbits);

		allocaddr = 0;
		i = 0;
		while (allocaddr == 0 && i < ((sizeof(mem->bits) * 8) - nbits) )
		{
			if ( (amask & mem->bits) == 0 )
			{
				mem->bits |= amask;
				allocaddr = mem->start + (mem->fragsize * i);
			}
			amask = amask << 1;
			i++;
		}
		WLAN_INT_ENABLE(flags);
		mem->lock = 0;
	}
	else
	{
		WLAN_LOG_DEBUG0(1, "failed! mem mgr locked.\n");
	}

	WLAN_INT_ENABLE(flags);

	return allocaddr;
}


/*----------------------------------------------------------------
*	am930mem_free
*
*	Tests the structure lock.  If set, returns the value of p
*	clears the bits associated with the fragments in p,
*	returns NULL
*
*	returns: 0 on failure, non-zero on success
----------------------------------------------------------------*/
UINT32 am930mem_free( am930mem_t *mem, UINT32 p, UINT32 size)
{
	UINT32			amask;
	UINT32			nbits;
	UINT32			lshift;
	wlan_flags_t	flags;

	WLAN_INT_DISABLE(flags);

	if ( mem->lock == 0)
	{
		mem->lock = 1;

		nbits = size / mem->fragsize;
		nbits += ((size % mem->fragsize) ? 1 : 0);
		amask = ((UINT32)0xffffffff) >> ((sizeof(mem->bits)*8) - nbits);

		lshift = p / mem->fragsize;
		amask = amask << lshift;

		mem->bits &= ~amask;

		WLAN_INT_ENABLE(flags);
	}
	else
	{
		WLAN_LOG_DEBUG0(1, "failed! mem mgr locked.\n");
		p = 0UL;
	}

	WLAN_INT_ENABLE(flags);

	return p;
}


/*----------------------------------------------------------------
*	readcard, writecard, 
*   read8, read16, read32, 
*   write8, write16, write32
*
*	Functions to read/write to the 'shared' memory on the card.
*	These functions are abstracted here so we can choose between
*	either the io register or memory mapped access to the card 
*   memory.
*
*	The 8, 16, and 32 functions are for reading/writing bytes, 
*	halfwords and words respectively.
*
----------------------------------------------------------------*/
void readcard( am930hw_t *hw, UINT32 cardaddr, void *buf, UINT32 len)
{
	if ( hw->usemem ) {
		#if (WLAN_OS == WLAN_LINUX_KERNEL)
			UINT8 * dp = (UINT8 *) buf;
			volatile UINT8 *sp = ((volatile UINT8*) hw->membase) + cardaddr;
			while( len-- != 0 ) {
				*dp++ = readb(sp++);
				udelay(am930_iodelay);
			} 
		#else
			memcpy(buf, ((volatile UINT8 *) hw->membase) + cardaddr, len);
		#endif
	}
	else {
		UINT8			byte;
		INT				i;
		wlan_flags_t	flags;
		UINT8			lmal;
		UINT8			lmau;
		UINT16			scardaddr;

		/* set bank select and flash deselect */
		byte = inb_p(BSS(hw));
		byte &= ~0x38;  
		outb_p(byte, BSS(hw));

		/* save the address regs and set the new address */
		WLAN_INT_DISABLE(flags);

		lmal = inb( LMAL(hw));
		lmau = inb( LMAU(hw));

		/* if necessary, mess with the byte order */
		scardaddr = cardaddr;
		scardaddr = host2amd32(scardaddr);

		/* write the address */
		outb( (cardaddr) & 0x00ff, LMAL(hw));
		udelay(am930_iodelay);
		outb( ((cardaddr) & 0x7f00) >> 8, LMAU(hw));
		udelay(am930_iodelay);

		WLAN_INT_ENABLE(flags);

		/* now, copy the data */
		for ( i = 0; i < len; i++)
		{
			/* retrieve the byte */
			((UINT8*)buf)[i] = inb( DPLL(hw) );
			udelay(am930_iodelay);
		}

		/* restore the address regs */
		WLAN_INT_DISABLE(flags);

		outb( lmal, LMAL(hw));
		udelay(am930_iodelay);
		outb( lmau, LMAU(hw));
		udelay(am930_iodelay);

		WLAN_INT_ENABLE(flags);
	}
}

void writecard( am930hw_t *hw, UINT32 cardaddr, void *buf, UINT32 len)
{
	if ( hw->usemem ) {
		#if (WLAN_OS == WLAN_LINUX_KERNEL)
			UINT8 * sp = (UINT8 *) buf;
			volatile UINT8 * dp = ((volatile UINT8 *) hw->membase) + cardaddr;
			while( len-- != 0 ) {
				writeb( *sp++, dp++ );
				udelay(am930_iodelay);
			}
		#else
			memcpy(((volatile UINT8 *) hw->membase) + cardaddr, buf, len);
		#endif
	}
	else {
		UINT8			byte;
		INT				i;
		wlan_flags_t	flags;
		UINT8			lmal;
		UINT8			lmau;
		UINT16  		scardaddr;

		/* set bank select and flash deselect */
		byte = inb_p(BSS(hw));
		byte &= ~0x38;  
		outb_p(byte, BSS(hw));

		/* save the address regs and set the new address */
		WLAN_INT_DISABLE(flags);

		lmal = inb( LMAL(hw));
		lmau = inb( LMAU(hw));

		/* if necessary, mess with the byte order */
		scardaddr = cardaddr;
		scardaddr = host2amd32(scardaddr);

		/* write the address */
		outb( (cardaddr) & 0x00ff, LMAL(hw));
		udelay(am930_iodelay);
		outb( ((cardaddr) & 0x7f00) >> 8, LMAU(hw));
		udelay(am930_iodelay);

/*		WLAN_INT_ENABLE(flags); */

		/* now, copy the data */
		for ( i = 0; i < len; i++)
		{
			/* set the byte */
			outb( ((UINT8*)buf)[i], DPLL(hw) );
			udelay(am930_iodelay);
		}

		/* restore the address regs */
/*		WLAN_INT_DISABLE(flags); */

		outb( lmal, LMAL(hw));
		udelay(am930_iodelay);
		outb( lmau, LMAU(hw));
		udelay(am930_iodelay);

		WLAN_INT_ENABLE(flags);
	}
}

void memsetcard( am930hw_t *hw, UINT32 cardaddr, UINT8 val, UINT32 len)
{
	if ( hw->usemem ) {
		#if (WLAN_OS == WLAN_LINUX_KERNEL)
		volatile UINT8 * dp = ((volatile UINT8 *) hw->membase) + cardaddr;
		while( len-- != 0 ) {
			writeb( val, dp++ );
			udelay(am930_iodelay);
		}
		#else
		memset(((volatile UINT8*)hw->membase) + cardaddr, val, len);
		#endif
	}
	else {
		UINT8			byte;
		INT				i;
		wlan_flags_t	flags;
		UINT8			lmal;
		UINT8			lmau;
		UINT16  		scardaddr;

		/* set bank select and flash deselect */
		byte = inb_p(BSS(hw));
		byte &= ~0x38;  
		outb_p(byte, BSS(hw));

		/* save the address regs and set the new address */
		WLAN_INT_DISABLE(flags);

		lmal = inb( LMAL(hw));
		lmau = inb( LMAU(hw));

		/* if necessary, mess with the byte order */
		scardaddr = cardaddr;
		scardaddr = host2amd32(scardaddr);

		/* write the address */
		outb( (cardaddr) & 0x00ff, LMAL(hw));
		udelay(am930_iodelay);
		outb( ((cardaddr) & 0x7f00) >> 8, LMAU(hw));
		udelay(am930_iodelay);

/*		WLAN_INT_ENABLE(flags); */

		/* now, copy the data */
		for ( i = 0; i < len; i++)
		{
			/* set the byte */
			outb( val, DPLL(hw) );
			udelay(am930_iodelay);
		}

		/* restore the address regs */
/*		WLAN_INT_DISABLE(flags); */

		outb( lmal, LMAL(hw));
		udelay(am930_iodelay);
		outb( lmau, LMAU(hw));
		udelay(am930_iodelay);

		WLAN_INT_ENABLE(flags);
	}
}

UINT8  read8( am930hw_t *hw, UINT32 cardaddr ) 
{
	UINT8 result;
	readcard( hw, cardaddr, &result, 1);
	return result;
}

UINT16 read16( am930hw_t *hw, UINT32 cardaddr ) 
{
	UINT16 result;
	readcard( hw, cardaddr, &result, 2);
	return amd2host16(result);
}

UINT32 read32( am930hw_t *hw, UINT32 cardaddr ) 
{
	UINT32 result;
	readcard( hw, cardaddr, &result, 4);
	return amd2host32(result);
}

UINT32 reread32( am930hw_t *hw, UINT32 cardaddr )
{
	UINT32 result = 0;
	UINT32 old = 0;

	result = read32( hw, cardaddr);
	while ( result != old )
	{
		old = result;
		result = read32( hw, cardaddr);
		if ( old != result )
		{
			WLAN_LOG_DEBUG1(1, "reread32(0x%04lx) didn't match!\n", cardaddr);
		}
	}
	return result;
}

void   write8( am930hw_t *hw, UINT32 cardaddr, UINT8 val )
{
	writecard( hw, cardaddr, &val, 1);
}

void   write16( am930hw_t *hw, UINT32 cardaddr, UINT16 val )
{
	val = host2amd16(val);
	writecard( hw, cardaddr, &val, 2);
}

void   write32( am930hw_t *hw, UINT32 cardaddr, UINT32 val )
{
	val = host2amd32(val);
	writecard( hw, cardaddr, &val, 4);
}

void am930db_prrxdesc( am930rx_desc_t *desc)
{
/*
	printk(KERN_DEBUG"rxd:");
	printk("nxt:%lx ",	host2amd32(desc->next));
	printk("st:%02x ",	desc->state);
	printk("ss:%02x ",	desc->rssi);
	printk("ch:%d ",	desc->index_or_ch);
	printk("lt:%ld ",	amd2host32(desc->local_time));
	printk("dp:%lx ",	amd2host32(desc->rx_start_frame));
	printk("l:%d ",		amd2host16(desc->rx_len));
	printk("r:%x\n",	desc->rate);
*/
}

/*
#ifdef DBMSG_ENABLED
	#include "am930hw_db.c"
#endif
*/
