/* am930hw.c: Handles the AM79C930 specifics
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*
*	--------------------------------------------------------------------
*/

#ifndef _LINUX_AM930HW_H
#define _LINUX_AM930HW_H

#include <wlan/p80211hdr.h>

/*===== Control/Status and Command block Related Constants =*/

/*--- Fixed memory offsets --------------------------*/
#define SU_OFF_LAST_TXDESC		0x3ec
#define SU_OFF_RSVD1			0x400
#define SU_OFF_BANNER			0x480
#define SU_OFF_CMD_BLK			0x4a0
#define SU_OFF_CNTL_STATUS_BLK	0x4f0
#define SU_OFF_VBM				0x500
#define SU_OFF_BUFFER			0x600

/*--- Global Sizes ----------------------------------*/
#define SU_LEN_BANNER			32


/*===== Descriptor Related Constants & Macros =========*/

/*--- next testing macro (applies to Rx and Tx) -------*/
#define SUTXD_ISLAST(x)			((x) & BIT31)
#define SURXD_ISLAST(x)			((x) & BIT31)


/*=============================================================*/
/*------ Types and their related constants --------------------*/
/*=============================================================*/
/*--- Last Completed Tx Descriptor Block ---------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930txcmplt_blk
{
	volatile UINT32	last_bcast		__WLAN_ATTRIB_PACK__;
	volatile UINT32	last_mgmt		__WLAN_ATTRIB_PACK__;
	volatile UINT32	last_data		__WLAN_ATTRIB_PACK__;
	volatile UINT32	last_pspoll		__WLAN_ATTRIB_PACK__;
	volatile UINT32	last_cfpoll		__WLAN_ATTRIB_PACK__;
} am930txcmplt_blk_t;
__WLAN_PRAGMA_PACKDFLT__

#define TXCMPLT_OFF_BCAST	0
#define	TXCMPLT_OFF_MGMT	4
#define TXCMPLT_OFF_DATA	8
#define TXCMPLT_OFF_PSPOLL	12
#define TXCMPLT_OFF_CFPOLL	16

/*=============================================================*/
/*--- Command Block for sending commands to f/w ---------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930cmd_blk
{
	volatile UINT8	code			__WLAN_ATTRIB_PACK__;
	volatile UINT8	status			__WLAN_ATTRIB_PACK__;
	volatile UINT8	error_offset	__WLAN_ATTRIB_PACK__;
	volatile UINT8	reserved		__WLAN_ATTRIB_PACK__;
	volatile UINT8	cmd_parms[76]	__WLAN_ATTRIB_PACK__;
} am930cmd_blk_t;
__WLAN_PRAGMA_PACKDFLT__

#define CMD_OFF_CODE	0
#define CMD_OFF_STATUS	1
#define CMD_OFF_ERR_OFF	2
#define CMD_OFF_PARMS	4

/*--- Command Block - Command values -----------------*/
#define	SUCMD_CMD_TEST_NOP		0x01
#define	SUCMD_CMD_SET_MIB_VAR	0x02
#define SUCMD_CMD_INIT_TX		0x03
#define SUCMD_CMD_FLUSH_TX		0x04
#define SUCMD_CMD_EN_RCVR		0x05
#define SUCMD_CMD_DIS_RCVR		0x06
#define SUCMD_CMD_SLEEP			0x07
#define SUCMD_CMD_WAKE			0x08
#define	SUCMD_CMD_GET_MIB_VAR	0x09
#define	SUCMD_CMD_SCAN			0x0a
#define	SUCMD_CMD_SYNC			0x0b
#define SUCMD_CMD_RESUME		0x0c

/*--- Command Block - Status values ------------------*/
#define SUCMD_STAT_IDLE			0x00
#define SUCMD_STAT_CMPLT			0x01
#define	SUCMD_STAT_REJECT_UNK	0x02
#define SUCMD_STAT_REJECT_ARG	0x03
#define SUCMD_STAT_REJECT_NOS	0x04
#define SUCMD_STAT_REJECT_ARG2	0x05
#define SUCMD_STAT_REJECT_MODE	0x06
/* other error codes undocumented */

/*=============================================================*/
/*--- Control/Status Block ------------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930cs_blk
{
	volatile UINT8		self_test_status	__WLAN_ATTRIB_PACK__;
	volatile UINT8		sta_state			__WLAN_ATTRIB_PACK__;
	volatile UINT8		user_rtn			__WLAN_ATTRIB_PACK__;
	volatile UINT8		int_status			__WLAN_ATTRIB_PACK__;
	volatile UINT8		int_mask			__WLAN_ATTRIB_PACK__;
	volatile UINT8		lockout_fw			__WLAN_ATTRIB_PACK__;
	volatile UINT8		lockout_host		__WLAN_ATTRIB_PACK__;
	volatile UINT8		int_status2			__WLAN_ATTRIB_PACK__;
	volatile UINT32		rsvd				__WLAN_ATTRIB_PACK__;
	volatile UINT8		sw_dis_pwrdn		__WLAN_ATTRIB_PACK__;
	volatile UINT8		int_mask2			__WLAN_ATTRIB_PACK__;
	volatile UINT8		driver_state		__WLAN_ATTRIB_PACK__;
	volatile UINT8		rsvd2				__WLAN_ATTRIB_PACK__;
} am930cs_blk_t;
__WLAN_PRAGMA_PACKDFLT__

#define CS_OFF_SELF_TEST			0
#define CS_OFF_STA_STATE			1
#define CS_OFF_USER_RTN				2
#define CS_OFF_INT_STATUS			3
#define CS_OFF_INT_MASK				4
#define CS_OFF_LOCKOUT_FW			5
#define CS_OFF_LOCKOUT_HOST			6
#define CS_OFF_INT_STATUS2			7
#define CS_OFF_RSVD					8
#define CS_OFF_DIS_PWR_DN			12
#define CS_OFF_INT_MASK2			13
#define CS_OFF_DRIVER_STATE			14
#define CS_OFF_RSVD2				15

/*--- Control/Status Block - self_test_status values -*/
#define SUCS_POST_INIT				0x00
#define SUCS_POST_FIRMWARE			0x01
#define SUCS_POST_HW				0x02
#define SUCS_POST_INIT_MIB			0x03
#define SUCS_POST_MIB_CHKSUM_FAIL	0xfa
#define SUCS_POST_RADIO_FAIL		0xfb
#define SUCS_POST_SUTRO_FAIL		0xfc
#define SUCS_POST_PROM_CKSUM_FAIL	0xfd
#define SUCS_POST_RAM_FAIL			0xfe
#define SUCS_POST_SUCCESS			0xff

/*--- Control/Status Block - state testing macros ----*/
#define SUCS_ST_IS_RSVD1(x)			((x) & BIT7)
#define SUCS_ST_IS_RSVD2(x)			((x) & BIT6)
#define SUCS_ST_IS_AP_ACTIVE(x)		((x) & BIT5)
#define SUCS_ST_IS_PSP_DISABLED(x)	((x) & BIT4)
#define SUCS_ST_IS_DOZE(x)			((x) & BIT3)
#define SUCS_ST_IS_PSPMODE(x)		((x) & BIT2)
#define SUCS_ST_IS_RX_ENABLED(x)	((x) & BIT1)
#define SUCS_ST_IS_TX_ENABLED(x)	((x) & BIT0)

/*--- Control/Status Block - int_status testing macros --*/
#define SUCS_INT_IS_WAKEUP(x)		((x) & BIT7)
#define SUCS_INT_IS_CFPEND(x)		((x) & BIT6)
#define SUCS_INT_IS_DTIMTX(x)		((x) & BIT5)
#define SUCS_INT_IS_CFPSTART(x)		((x) & BIT4)
#define SUCS_INT_IS_SCANCMPLT(x)	((x) & BIT3)
#define SUCS_INT_IS_TX(x)			((x) & BIT2)
#define SUCS_INT_IS_RX(x)			((x) & BIT1)
#define SUCS_INT_IS_CMD(x)			((x) & BIT0)

/*--- Control/Status Block - int_mask (and status) values --*/
#define SUCS_INT_WAKEUP				(BIT7)
#define SUCS_INT_CFPEND				(BIT6)
#define SUCS_INT_DTIMTX				(BIT5)
#define SUCS_INT_CFPSTART			(BIT4)
#define SUCS_INT_SCANCMPLT			(BIT3)
#define SUCS_INT_TX					(BIT2)
#define SUCS_INT_RX					(BIT1)
#define SUCS_INT_MASK_CMD			(BIT0)

/*--- Control/Status Block - int_status2 testing macros --*/
#define SUCS_INT2_IS_RX_MGMT_PSPOLL(x)	((x) & (BIT7))
#define SUCS_INT2_IS_RX_DATA(x)		((x) & (BIT6))
#define SUCS_INT2_IS_RSVD(x)		((x) & (BIT5))
#define SUCS_INT2_IS_TX_MGMT(x)		((x) & (BIT4))
#define SUCS_INT2_IS_TX_CFPOLL(x)	((x) & (BIT3))
#define SUCS_INT2_IS_TX_PSPOLL(x)	((x) & (BIT2))
#define SUCS_INT2_IS_TX_BCAST(x)	((x) & (BIT1))
#define SUCS_INT2_IS_TX_DATA(x)		((x) & (BIT0)

/*--- Control/Status Block - int_mask2 (and status) values --*/
#define SUCS_INT2_RX_MGMT_PSPOLL	(BIT7)
#define SUCS_INT2_RX_DATA			(BIT6)
#define SUCS_INT2_RSVD				(BIT5)
#define SUCS_INT2_TX_MGMT			(BIT4)
#define SUCS_INT2_TX_CFPOLL			(BIT3)
#define SUCS_INT2_TX_PSPOLL			(BIT2)
#define SUCS_INT2_TX_BCAST			(BIT1)
#define SUCS_INT2_TX_DATA			(BIT0)

/*--- Control/Status Block - driver state values --*/
/*-- bits 0-3  --------*/
#define SUCS_DS_RESET			(0x00)
#define SUCS_DS_ESS_SYNC		(0x01)
#define SUCS_DS_IBSS_SCAN		(0x02)
#define SUCS_DS_IBSS_SYNC		(0x03)
#define SUCS_DS_ESS_SCAN		(0x04)
#define SUCS_DS_ESS_AUTH		(0x05)
#define SUCS_DS_ESS_ASSOC		(0x06)
#define SUCS_DS_ESS_HANDOFF		(0x07)
#define SUCS_DS_AP_NO_ASSOC		(0x08)
#define SUCS_DS_AP_ASSOC_NONE	(0x0c)
#define SUCS_DS_AP_ASSOC_LIGHT	(0x0d)
#define SUCS_DS_AP_ASSOC_MOD	(0x0e)
#define SUCS_DS_AP_ASSOC_HVY	(0x0f)

/*-- enable f/w control of the LEDS ---*/
#define SUCS_DS_AUTO_RXLED_EN	(BIT4)
#define SUCS_DS_AUTO_TXLED_EN	(BIT5)

/*-- if f/w control of LEDs is off, then driver controls them with these bits*/
#define SUCS_DS_RXLED_ON		(BIT6)
#define SUCS_DS_TXLED_ON		(BIT7)


/*=============================================================*/
/*--- Virtual Bitmap Block ------------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930vbm_blk
{
	volatile UINT8		offset		__WLAN_ATTRIB_PACK__;
	volatile UINT8		length		__WLAN_ATTRIB_PACK__;
	volatile UINT8		bm[251]		__WLAN_ATTRIB_PACK__;
} am930vbm_blk_t;
__WLAN_PRAGMA_PACK1__

#define VBM_OFF_OFFSET	0
#define VBM_OFF_LENGTH	1
#define VBM_OFF_BM		2


/*=============================================================*/
/*--- Rx Descriptor Block  ------------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930rx_desc
{
	volatile UINT32	rsvd1				__WLAN_ATTRIB_PACK__;
	volatile UINT32	next				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rsvd2				__WLAN_ATTRIB_PACK__;
	volatile UINT8	state				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rssi				__WLAN_ATTRIB_PACK__;
	volatile UINT8	index_or_ch			__WLAN_ATTRIB_PACK__;
	volatile UINT32	local_time			__WLAN_ATTRIB_PACK__;
	volatile UINT32	rx_start_frame		__WLAN_ATTRIB_PACK__;
	volatile UINT16	rx_len				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rate				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rsvd3				__WLAN_ATTRIB_PACK__;
} am930rx_desc_t;
__WLAN_PRAGMA_PACKDFLT__

#define RXD_OFF_RSVD1				0
#define RXD_OFF_NEXT				4
#define RXD_OFF_RSVD2				8
#define RXD_OFF_STATE				9
#define RXD_OFF_RSSI				10
#define RXD_OFF_INDEX_OR_CH			11
#define RXD_OFF_LOCAL_TIME			12
#define RXD_OFF_START_FRAME			16
#define RXD_OFF_LEN					20
#define RXD_OFF_RATE				22
#define RXD_OFF_RSVD3				23

/*--- RX Descriptor - state values and testing macros -*/
#define SURXD_ST_FWOWN				(BIT7)
#define SURXD_ST_CONSUMED			(BIT6)
#define SURXD_ST_LAST_FRAG			(BIT5)
#define SURXD_ST_RSVD1				(BIT4)
#define SURXD_ST_CRCERR				(BIT3)
#define SURXD_ST_RSVD2				(BIT2)
#define SURXD_ST_BUF_OFLOW			(BIT1)
#define SURXD_ST_RXERR				(BIT0)

#define SURXD_ST_IS_FWOWN(x)		((x) & SURXD_ST_FWOWN)
#define SURXD_ST_IS_HOSTOWN(x)		(~(x) & SURXD_ST_FWOWN)
#define SURXD_ST_IS_CONSUMED(x)		((x) & SURXD_ST_CONSUMED)
#define SURXD_ST_IS_LAST_FRAG(x)	((x) & SURXD_ST_LAST_FRAG)
#define SURXD_ST_IS_CRC_ERR(x)		((x) & SURXD_ST_CRCERR)
#define SURXD_ST_IS_BUF_OFLOW(x)	((x) & SURXD_ST_BUF_OFLOW)
#define SURXD_ST_IS_RXERR(x)		((x) & SURXD_ST_RXERR)

/*=============================================================*/
/*--- Tx Descriptor Block  ------------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct am930tx_desc
{
	volatile UINT32	tx_start_frame		__WLAN_ATTRIB_PACK__;
	volatile UINT32	next				__WLAN_ATTRIB_PACK__;
	volatile UINT16	tx_len				__WLAN_ATTRIB_PACK__;
	volatile UINT8	state				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rate				__WLAN_ATTRIB_PACK__;
	volatile UINT8	difs_attempts		__WLAN_ATTRIB_PACK__;
	volatile UINT8	difs_failures		__WLAN_ATTRIB_PACK__;
	volatile UINT8	sifs_attempts		__WLAN_ATTRIB_PACK__;
	volatile UINT8	sifs_failures		__WLAN_ATTRIB_PACK__;
	volatile UINT32	rsvd1				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rts_attempts		__WLAN_ATTRIB_PACK__;
	volatile UINT8	data_attempts		__WLAN_ATTRIB_PACK__;
	volatile UINT8	tx_cntl				__WLAN_ATTRIB_PACK__;
	volatile UINT8	rsvd2				__WLAN_ATTRIB_PACK__;
	volatile UINT32	prev				__WLAN_ATTRIB_PACK__;
	volatile UINT32	host				__WLAN_ATTRIB_PACK__;
} am930tx_desc_t;
__WLAN_PRAGMA_PACKDFLT__

#define TXD_OFF_START_FRAME		0
#define TXD_OFF_NEXT			4
#define TXD_OFF_LEN				8
#define TXD_OFF_STATE			10
#define TXD_OFF_RATE			11
#define TXD_OFF_DIFS_ATTEMPTS	12
#define TXD_OFF_DIFS_FAILURES	13
#define TXD_OFF_SIFS_ATTEMPTS	14
#define TXD_OFF_SIFS_FAILURES	15
#define TXD_OFF_RSVD1			16
#define TXD_OFF_RTS_ATTEMPTS	20
#define TXD_OFF_DATA_ATTEMPTS	21
#define TXD_OFF_TX_CNTL			22
#define TXD_OFF_RSVD2			23
#define TXD_OFF_PREV			24
#define TXD_OFF_HOST			28

/*--- TX Descriptor - state values and testing macros -*/
#define SUTXD_ST_FWOWN				(BIT7)
#define SUTXD_ST_DONE				(BIT6)
#define SUTXD_ST_REJECTED			(BIT5)
#define SUTXD_ST_MSDU_TMOUT			(BIT4)
#define SUTXD_ST_ABRT				(BIT3)
#define SUTXD_ST_ABRT_NOTX			(BIT2)
#define SUTXD_ST_MAX_RETRY			(BIT1)
#define SUTXD_ST_TXERR				(BIT0)

#define SUTXD_ST_IS_FWOWN(x)		((x) & SUTXD_ST_FWOWN)
#define SUTXD_ST_IS_HOSTOWN(x)		(~(x) & SUTXD_ST_FWOWN)
#define SUTXD_ST_IS_DONE(x)			((x) & SUTXD_ST_DONE)
#define SUTXD_ST_IS_REJECTED(x)		((x) & SUTXD_ST_REJECTED)
#define SUTXD_ST_IS_MSDU_TMOUT(x)	((x) & SUTXD_ST_MSDU_TMOUT)
#define SUTXD_ST_IS_ABRT(x)			((x) & SUTXD_ST_ABRT)
#define SUTXD_ST_IS_ABRT_NOTX(x)	((x) & SUTXD_ST_ABRT_NOTX)
#define SUTXD_ST_IS_MAX_RETRY(x)	((x) & SUTXD_ST_MAX_RETRY)
#define SUTXD_ST_IS_TXERR(x)		((x) & SUTXD_ST_TXERR)

/*--- TX Descriptor - tx_cntl values ----*/
#define SUTXD_CTL_PRESERVE_SEQ		(BIT7)
#define SUTXD_CTL_RSVD1				(BIT6)
#define SUTXD_CTL_RSVD2				(BIT5)
#define SUTXD_CTL_RSVD3				(BIT4)
#define SUTXD_CTL_RSVD4				(BIT3)
#define SUTXD_CTL_RSVD5				(BIT2)
#define SUTXD_CTL_BURST				(BIT1)
#define SUTXD_CTL_FRAGS				(BIT0)

/*===== Variables  ==================================*/
extern int			am930_iodelay;		/* udelay between i/o or mem reads */

/*=============================================================*/
/*--- Command Argument Types  ---------------------------------*/
/*=============================================================*/

/*--- Test Interface/NOP -SUCMD_TEST_NOP ----------------------*/
/*
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_test_nop
{
} sucmd_test_nop_t;
__WLAN_PRAGMA_PACKDFLT__
*/

/*--- Set MIB Variables - SUCMD_SET_MIB_VAR -------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_set_mib_var
{
	UINT8	type		__WLAN_ATTRIB_PACK__;
	UINT8	size		__WLAN_ATTRIB_PACK__;
	UINT8	index		__WLAN_ATTRIB_PACK__;
	UINT8	rsvd		__WLAN_ATTRIB_PACK__;
	UINT8	data[72]	__WLAN_ATTRIB_PACK__;
} sucmd_set_mib_var_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Init Tx Routines --- SUCMD_INIT_TX ----------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_init_tx
{
	UINT32	data_desc		__WLAN_ATTRIB_PACK__;
	UINT32	mgmt_desc		__WLAN_ATTRIB_PACK__;
	UINT32	bcast_desc		__WLAN_ATTRIB_PACK__;
	UINT32	pspoll_desc		__WLAN_ATTRIB_PACK__;
	UINT32	cfpoll_desc		__WLAN_ATTRIB_PACK__;
} sucmd_init_tx_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Flush Tx ----------- SUCMD_FLUSH_TX ---------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_flush_tx
{
	UINT8	data			__WLAN_ATTRIB_PACK__;
	UINT8	mgmt			__WLAN_ATTRIB_PACK__;
	UINT8	bcast			__WLAN_ATTRIB_PACK__;
	UINT8	pspoll			__WLAN_ATTRIB_PACK__;
	UINT8	cfpoll			__WLAN_ATTRIB_PACK__;
} sucmd_flush_tx_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Enable Receiver ---- SUCMD_EN_RCVR ----------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_en_rcvr
{
	UINT32	data_desc		__WLAN_ATTRIB_PACK__;
	UINT32	pspoll_desc		__WLAN_ATTRIB_PACK__;
} sucmd_en_rcvr_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Disable Receiver --- SUCMD_DIS_RCVR ---------------------*/
/*
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_dis_rcvr
{
} sucmd_dis_rcvr_t;
__WLAN_PRAGMA_PACKDFLT__
*/

/*--- Sleep -------------- SUCMD_SLEEP ------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_sleep
{
	UINT64	wake_up_time;
} sucmd_sleep_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Wake --------------- SUCMD_WAKE -------------------------*/
/*
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_wake
{
} sucmd_wake_t;
__WLAN_PRAGMA_PACKDFLT__
*/

/*--- Get MIB variables -- SUCMD_GET_MIB_VAR ------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_get_mib_var
{
	UINT8	type		__WLAN_ATTRIB_PACK__;
	UINT8	size		__WLAN_ATTRIB_PACK__;
	UINT8	index		__WLAN_ATTRIB_PACK__;
	UINT8	rsvd		__WLAN_ATTRIB_PACK__;
	UINT8	data[72]	__WLAN_ATTRIB_PACK__;
} sucmd_get_mib_var_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Scan --------------- SUCMD_SCAN -------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_scan
{
	UINT16	duration	__WLAN_ATTRIB_PACK__;
	UINT8	set_or_ch	__WLAN_ATTRIB_PACK__;
	UINT8	pattern		__WLAN_ATTRIB_PACK__;
	UINT8	index		__WLAN_ATTRIB_PACK__;
	UINT8	suspend		__WLAN_ATTRIB_PACK__;
} sucmd_scan_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Sync --------------- SUCMD_SYNC -------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_sync
{
	UINT8	set_or_ch							__WLAN_ATTRIB_PACK__;
	UINT8	pattern								__WLAN_ATTRIB_PACK__;
	UINT8	index								__WLAN_ATTRIB_PACK__;
	UINT8	start_BSS							__WLAN_ATTRIB_PACK__;
	UINT16	dwell_interval						__WLAN_ATTRIB_PACK__;
	UINT16	rsvd								__WLAN_ATTRIB_PACK__;
	UINT8	bss_timestamp[WLAN_BSS_TS_LEN]		__WLAN_ATTRIB_PACK__;
	UINT32	ref_time							__WLAN_ATTRIB_PACK__;
} sucmd_sync_t;
__WLAN_PRAGMA_PACKDFLT__

/*--- Resume ------------- SUCMD_RESUME -----------------------*/
/*
__WLAN_PRAGMA_PACK1__
typedef struct sucmd_resume
{
} sucmd_resume_t;
__WLAN_PRAGMA_PACKDFLT__
*/

/*=============================================================*/
/*--- AM930 Hardware interface types        -------------------*/
/*=============================================================*/

typedef void (*am930hw_scan_callback_t)(void *);

typedef struct am930mem		/* used for tx buffer mem mgmt */
{
	UINT32		lock;
	UINT32		bits;
	UINT32		start;
	UINT32		len;
	UINT32		fragsize;
} am930mem_t;

typedef struct am930hw_stats
{
	UINT32	tx_tmout;
	UINT32	tx_fifo_uflow;
	UINT32	tx_retry;
	UINT32	tx_rejected;
	UINT32	rx_crcerr;
	UINT32	rx_buf_oflow;
} am930hw_stats_t;

typedef struct am930hw
{
	/* object status indicator */
	volatile UINT32	state;

	/* driver support */
	struct am930mac	*mac;

	/* resource assignments */
	UINT32			membase;
	UINT32			usemem;		/* use mem mapped io?? */
	UINT32			iobase;
	UINT32			irq;

	/* firmware revision */
	UINT32			fw_major;
	UINT32			fw_minor;
	UINT32			fw_api;
	UINT32			fw_date;

	/* firmware interface, note: the addrs are in card space */
	UINT32			banner;
	UINT32			cs;
	UINT32			cmd;
	UINT32			txcmplt;
	UINT32			vbm;

	/* tx queue location */
	UINT32			tx_base;
	UINT32			tx_len;
	UINT			ntxbuf;
	UINT			used_txbuf;
	UINT32			mgmt_tx_base;

	/* rx queue location */
	UINT32			rx_base;
	UINT32			rx_len;

	/* tx queue management */
	am930mem_t		*mem;
	UINT32			tx_tail;
	UINT32			mgmt_tx_tail;

	UINT8			*txfrbuf;

	/* rx queue current head ptr */
	UINT32			rx_head;

	/* MAC address */
	UINT8			addr[WLAN_ADDR_LEN];

	/* F/W command support elements */
	volatile UINT8	last_cmd;
	volatile UINT8	scan_ch;		/* when scannning,  this is the channel */

	am930hw_stats_t	stats;

	/* callback ptr(s) */
	am930hw_scan_callback_t		scan_callback;
	void						*scan_callbackarg;
	void						(*timerfunc)(struct am930hw *hw);

} am930hw_t;

/* bit flags for state information */
#define AM930HW_CONFIG_PENDING	(BIT0)
#define AM930HW_CONFIG			(BIT1)
#define AM930HW_INTOCURRED		(BIT2)
#define AM930HW_SCANNING		(BIT3)
#define AM930HW_RXENABLED		(BIT4)
#define AM930HW_TXENABLED		(BIT5)

/* Method return codes */
#define AM930HW_SUCCESS			0
#define AM930HW_FAILURE			1
#define AM930HW_NOTSUPPORTED	2

/* Scan method bit flags */
#define	AM930HWSCAN_ACTIVE		(BIT0)
#define	AM930HWSCAN_PASSIVE		(BIT1)
#define	AM930HWSCAN_IBSS		(BIT2)
#define	AM930HWSCAN_ESS			(BIT3)

/*=============================================================*/
/*--- Methods -------------------------------------------------*/
/*=============================================================*/
/* public: */
am930hw_t*	am930hw_construct(UINT32 irq, UINT32 iobase,
								UINT32 membase, am930mac_t *mac);

void 		am930hw_destruct(am930hw_t *hw);

int		am930hw_init_rx_tx( am930hw_t *hw );
void 	am930hw_ISR( am930hw_t *hw );

UINT32 am930hw_joinbss( am930hw_t *hw, UINT32 ch, UINT32 newBSS, 
						UINT8 *bssid, wlan_ie_ssid_t *ssid, UINT32 bcn_int,
						wlan_bss_ts_t ts, UINT32 sutro_ref_time );

UINT32 am930hw_resumefromscan( am930hw_t *hw);
UINT32	am930hw_scan( am930hw_t *hw, UINT32 cntl, UINT8 *bssid, 
		      UINT32 ch, UINT32 duration,
		      wlan_ie_ssid_t *ssid,
		      am930hw_scan_callback_t cb, void *callback_arg );

UINT32	am930hw_reset( am930hw_t *hw );

UINT8 am930hw_stateset( am930hw_t *hw, UINT8 state);
UINT8 am930hw_stateget( am930hw_t *hw);

UINT32 am930hw_txdata( am930hw_t* hw, wlan_pb_t *pb, UINT32 rate);
UINT32 am930hw_txmgmt( am930hw_t* hw, UINT8 *buf, UINT32 len);

void am930hw_txkick( am930hw_t* hw);

UINT32 am930hw_mibget( am930hw_t *hw, UINT32 mibsec, UINT32 offset,
						UINT32 len, void *buf);
UINT32 am930hw_mibgetitem( am930hw_t *hw, UINT32 itemid, void *buf, UINT32 len);

UINT32 am930hw_mibset( am930hw_t *hw, UINT32 mibsec, UINT32 offset,
						UINT32 len, void *buf);
UINT32 am930hw_mibsetitem( am930hw_t *hw, UINT32 itemid, void *buf, UINT32 len);


/*=============================================================*/
/*--- AM79C930 byte order swapping macros ---------------------*/
/*=============================================================*/

#if (WLAN_CPU_FAMILY == WLAN_Ix86)
	#define amd2host16(n)	(n)
	#define amd2host32(n)	(n)
	#define host2amd16(n)	(n)
	#define host2amd32(n)	(n)
#elif (WLAN_CPU_FAMILY == WLAN_PPC )
	#define amd2host16(n)	(__swab16((n)))
	#define amd2host32(n)	(__swab32((n)))
	#define host2amd16(n)	(__swab16((n)))
	#define host2amd32(n)	(__swab32((n)))
#else
	#error "No WLAN_CPU_FAMILY match!"
#endif

/*=============================================================*/
/*--- Functions -----------------------------------------------*/
/*=============================================================*/

#endif /* _LINUX_AM930HW_H */
