/* am930llc.c: Handles the Linux network interface
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*
*	Seperate file isolating the functions that handle the Linux network
*	interface. LLC may not be the appropriate name, but this code _does_
*	sit on top of the MAC...
*
*/

#include <linux/config.h>
#include <wlan/wlan_compat.h>

/* The following prevents "kernel_version" from being set in this file. */
#define __NO_VERSION__

/* PCMCIA headers generated during PCMCIA package installation */
#ifdef WLAN_PCMCIA
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#endif

/* Module related headers, non-module drivers should not include */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>
#include <net/sock.h>
#include <linux/netlink.h>

/* Standard driver includes */
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>

/* Card and Driver services includes */
#ifdef WLAN_PCMCIA
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#endif

/* Local Includes */
#include <wlan/version.h>
#include <wlan/wlan_compat.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_ioctl.h>

#ifdef WLAN_PCMCIA
#include "am930di.h"
#endif

#include "am930llc.h"
#include "am930mac.h"
#include "am930hw.h"
#include "am930mgr.h"

/*================================================================*/
/* Local Macros  */
#define WLAN_DEVNAME_LEN	9

/*================================================================*/
/* Local Types  */

#if LINUX_VERSION_CODE <= 0x20139
typedef struct enet_statistics	enet_stats_t;
#else
typedef struct net_device_stats	enet_stats_t;
#endif

typedef struct devpriv
{
	enet_stats_t	stats;
	am930llc_t*		llc;

} devpriv_t;

typedef int (*do_ioctl_t)(device_t *dev, struct ifreq *ifr, int cmd);

typedef void (*timerfunc_t)(unsigned long);

/*================================================================*/
/* Local Functions  */

static int am930llc_devinit(device_t *dev);
static int am930llc_devopen(device_t *dev);
static int am930llc_devstop(device_t *dev);
static int am930llc_devhard_start_xmit( struct sk_buff *skb, device_t *dev);
static enet_stats_t* am930llc_devgetstats(device_t *dev);
static void am930llc_devset_multicast_list(device_t *dev);
static int am930llc_devdo_ioctl(device_t *dev, wlan_req_t *req, int cmd);

void am930llc_timerfunc(am930llc_t *llc);
void am930llc_cmdtimerfunc(am930llc_t *llc);
void am930llc_nlfunc(struct sock *sk, int len);

/*================================================================*/
/* Static variables  */

UINT8	oui_rfc1042[] = {0x00, 0x00, 0x00};
UINT8	oui_8021h[] = {0x00, 0x00, 0xf8};


/*================================================================*/

/*----------------------------------------------------------------
*	am930llc_construct
*
*	returns: addr. of the lnd object if successful, NULL otherwise
----------------------------------------------------------------*/
am930llc_t *am930llc_construct(am930mac_t *mac, am930mgr_t *mgr)
{
	am930llc_t *llc;

	DBFENTER;

	llc = kmalloc( sizeof(am930llc_t), GFP_KERNEL);
	if ( llc != NULL )
	{
		memset( llc, 0, sizeof(am930llc_t));

		llc->mac = mac;
		llc->mgr = mgr;

		llc->ethconv = WLAN_ETHCONV_ENCAP;	/* set by user later */

		/* lets begin the initialization of the linux device */
		llc->dev = kmalloc( sizeof(device_t), GFP_KERNEL);
		if ( llc->dev == NULL )
		{
			kfree_s(llc, sizeof(am930llc_t));
			return NULL;
		}
		memset( llc->dev, 0, sizeof(device_t));

		llc->dev->priv = kmalloc( sizeof(devpriv_t), GFP_KERNEL);
		if ( llc->dev->priv == NULL )
		{
		 	kfree_s(llc->dev, sizeof(device_t));
			kfree_s(llc, sizeof(am930llc_t));
			return NULL;
		}
		memset( llc->dev->priv, 0, sizeof(devpriv_t));

		/* the dev name field will point to the same memory used by the 
			node_t field of the pcmcia di object */
		#ifdef WLAN_PCMCIA
		llc->dev->name = ((dev_link_t*)(llc->mac->di))->dev->dev_name;
		#else
		llc->dev->name = kmalloc(WLAN_DEVNAME_LEN, GFP_KERNEL);
		if ( llc->dev->name == NULL )
		{
			WLAN_LOG_ERROR0("Failed to alloc device name space.\n");
			kfree_s( llc->dev->priv, sizeof(devpriv_t));
			kfree_s( llc->dev, sizeof(device_t));
			kfree_s( llc, sizeof(am930llc_t));
			return NULL;
		}
		llc->dev->name[0] = '\0';
		#endif

		/* set the private data to point back at the llc */
		V2P(llc->dev->priv)->llc = llc;

		/* set the hardware address */
		am930mgr_mibgetitem( llc->mgr, ADDR_MAC_ADDR, llc->dev->dev_addr, 6);

		/* set the method pointers */
		llc->dev->init					= am930llc_devinit;
		llc->dev->open					= am930llc_devopen;
		llc->dev->stop					= am930llc_devstop;
		llc->dev->hard_start_xmit 		= am930llc_devhard_start_xmit;
		llc->dev->get_stats				= am930llc_devgetstats;
		llc->dev->set_multicast_list	= am930llc_devset_multicast_list;
		llc->dev->do_ioctl				= (do_ioctl_t)am930llc_devdo_ioctl;

		ether_setup( llc->dev );

		if ( register_netdev(llc->dev) != 0 )
		{
			WLAN_LOG_ERROR0("register_netdev failed!\n");
			kfree( llc->dev );
			llc->dev = NULL;
			am930llc_destruct( llc );
			llc = NULL;
		}
		WLAN_LOG_INFO1("Device %s registered\n", llc->dev->name);

#if 0
		/* now set up the kernel timer */
		init_timer(&llc->timer);
		llc->timer.data = (unsigned long)llc;
		llc->timer.function = (timerfunc_t)am930llc_timerfunc;
		llc->timer.expires = jiffies + (HZ / 10);
		add_timer(&llc->timer);
#endif

		/* set up the netlink i/f for a potential sniffer */
		llc->nlsk = netlink_kernel_create( NETLINK_USERSOCK, am930llc_nlfunc);
		if ( llc->nlsk == NULL )
		{
			WLAN_LOG_WARNING0("Failed to create netlink i/f.\n");
		}
	}

	DBFEXIT

    return llc;
} 


/*----------------------------------------------------------------
*	am930llc_cmdtimerfunc
*	One shot timer func called when an ioctl command has timed out.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_cmdtimerfunc(am930llc_t *llc)
{
	DBFENTER;
	/* clear the expires field as a flag that we've timed out */
	llc->timer.expires = 0;

	/* Assume the timer was set up to watch us while an ioctl's in */
	/*  the wait queue.  Check for NULL and if not, wake it */
	if ( llc->cmdwq != NULL ){
		wake_up_interruptible(&(llc->cmdwq));
	}

	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930llc_timerfunc
*	Called 10 times per second.  Calls any timer functions
*	established by the other components.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_timerfunc(am930llc_t *llc)
{
	DBFENTER;
	if ( llc->mac != NULL && llc->mac->timerfunc != NULL )
	{
		(*llc->mac->timerfunc)(llc->mac);

		if ( llc->mac->hw != NULL && llc->mac->hw->timerfunc != NULL )
		{
			(*llc->mac->hw->timerfunc)(llc->mac->hw);
		}
	}
	if ( llc->mgr != NULL && llc->mgr->timerfunc != NULL )
	{
		(*llc->mgr->timerfunc)(llc->mgr);
	}

	init_timer(&llc->timer);
	llc->timer.data = (unsigned long)llc;
	llc->timer.function = (timerfunc_t)am930llc_timerfunc;
	llc->timer.expires = jiffies + (HZ / 10);
	add_timer(&llc->timer);
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930llc_destruct
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_destruct( am930llc_t *llc)
{
	DBFENTER;

	if ( llc->dev != NULL )
	{
		del_timer(&llc->timer);
		llc->dev->start = 0;
		unregister_netdev( llc->dev );
		kfree(llc->dev);
		llc->dev = NULL;
	}

	kfree_s( llc, sizeof(am930llc_t));

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930llc_devinit
*	init method for the linux net device. Called by 
*	register_netdevice. For this driver, it doesn't have to do
*	anything.
*
*	returns: zero
----------------------------------------------------------------*/
int am930llc_devinit(device_t *dev)
{
	int result = 0;
	DBFENTER;

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930llc_cmdcomplete
*	Called by mgr whenever one of our asynch commands 
*	(scan, auth,assoc,etc.)	finishes.  Our primary responsibility
*	here is to wake up any waiting process (blocked in ioctl).
*	See also: cmdtimerfunc
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_cmdcomplete(am930llc_t *llc, UINT status)
{
	DBFENTER;
	
	if ( llc->cmdwq != NULL ){
		llc->cmdstatus = status;
		wake_up_interruptible(&(llc->cmdwq));
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930llc_devdo_ioctl
*	do_ioctl method for the linux net device. Private ioctl handler
*	for the net device. Used here to pass commands to the mgr
*	object.
*	TODO: Define the interface between the mgr object and user
*			mode utilities. 
*
*	returns: zero
----------------------------------------------------------------*/
int am930llc_devdo_ioctl(device_t *dev, wlan_req_t *req, int cmd)
{
	int 		result = 0;
	am930llc_t	*llc = V2P(dev->priv)->llc;
	DBFENTER;

	/* According to Alan, we're supposed to check the req ptr and data area */
	/* here, but aside from a NULL check, I'm not quite sure how. */

	/* Now that we have the blocks and timers, etc. We should sync access */

	if ( req == NULL ) {
		result = -EFAULT;
	} else {

		if ( test_and_set_bit( 0, (void*)&(llc->cmdbusy))) {
			return -EBUSY;
		}
		llc->currcmd = cmd;
		switch( cmd )
		{
			case WLAN_TEST:
				req->result = 0xf0f0;
				break;
			case WLAN_SCAN:
			{
				wlan_scan_t	cmd;
				UINT32		timeout = HZ;	/* default 1 second */
				UINT32		scanticks = 0;
				
				if ( copy_from_user( &cmd, req->data, sizeof(cmd)) ) {
					result = -EFAULT;
					break;
				}
				
				/* Call mgr to begin the scan */
				req->result = am930mgr_scanbegin( llc->mgr,
										cmd.scantype,	/*active/passive*/
										cmd.bsstype,	/*ind./infra./both*/
										cmd.bssid,		/*bcast/specific*/
										cmd.startch,	
										cmd.endch,
										cmd.timech,	/*time per ch.*/
										(wlan_ie_ssid_t*)cmd.ssid); 

				/* Set a timer for (timech * nchannels + slop) to make */
				/*  sure we  return */
				scanticks = wlan_tu2ticks(((cmd.endch-cmd.startch)*cmd.timech));
				scanticks += (HZ/5);  /* a little extra for slop */
				if ( scanticks > timeout ) {
					timeout = scanticks;
				}
				init_timer(&llc->cmdtimer);
				llc->cmdtimer.data = (unsigned long)llc;
				llc->cmdtimer.function = (timerfunc_t)am930llc_cmdtimerfunc;
				llc->cmdtimer.expires = jiffies + timeout;
				add_timer(&llc->cmdtimer);

				/* Now, do an interruptible_sleep, we'll be awakened by: */
				/*  a signal the process, cmdtimerfunc (timeout), or llc_scancomplete */
				interruptible_sleep_on(&(llc->cmdwq));
				llc->cmdwq = NULL;

				/* If sleep return is via signal */
				if ( signal_pending(current) ) {
					WLAN_LOG_DEBUG0(2,"Scan unblocked via signal\n");
					/* stop scan, clean up scan and timer */
					am930mgr_scanstop(llc->mgr);
					del_timer(&llc->cmdtimer);
					result = -EINTR;
				} else {
					/* If timeout */
					if ( llc->cmdtimer.expires == 0 )
					{
						WLAN_LOG_DEBUG0(2,"Scan unblocked via timeout\n");
						/* stop scan, clean up scan */
						am930mgr_scanstop(llc->mgr);
						result = -ETIME;
					} else {
						/* Success */
						/* clean up timer */
						del_timer(&llc->cmdtimer);
						llc->cmdtimer.expires=0;
						req->result = llc->cmdstatus;
						/* return success */
						result = 0;
					}
				}
				break;
			}
			case WLAN_NETLIST_LEN:
			{	
				wlan_netlist_len_t cmd;
				knownbss_t		*curr;
				int				i;
				am930mgr_t*		mgr = llc->mgr;
				i = 0;
				for( curr = mgr->bsslist; curr != NULL; curr = curr->next) {
					i++;
				}

				cmd.nitems = i;
				if ( copy_to_user( req->data, &cmd, sizeof(cmd)) ) {
					result = -EFAULT;
				}
				req->result = 0;
				break;
			}
			case WLAN_NETLIST:
			{
				wlan_netlist_t 	*cmd;
				wlan_netlist_t 	tmp;
				knownbss_t		*curr;
				wlan_ie_ssid_t	*ie;
				int				i;
				am930mgr_t*		mgr = llc->mgr;

				if ( copy_from_user( &tmp, req->data, sizeof(tmp)) ) {
					result = -EFAULT;
					break;
				}

				cmd = kmalloc(sizeof(wlan_netlist_t) + 
						(tmp.nitems * sizeof(netitem_t)), GFP_ATOMIC);
				if ( cmd == NULL ) {
					result = -ENOMEM;
					break;
				}
				cmd->nitems = tmp.nitems;
				i = 0;
				for( curr = mgr->bsslist; curr != NULL; curr = curr->next)
				{
					cmd->netlist[i].channel = curr->channel;
					cmd->netlist[i].bcn_int = curr->bcn_int;
					cmd->netlist[i].cap_info = curr->cap_info;
					memcpy(cmd->netlist[i].bssid, curr->bssid, 6);
					ie = (wlan_ie_ssid_t*)curr->ssid;
					memcpy(cmd->netlist[i].ssid, ie, ie->len + 2);
					i++;
				}

				if ( copy_to_user( req->data, cmd, 
						sizeof(wlan_netlist_t) + 
						(cmd->nitems * sizeof(netitem_t))) ) {
					result = -EFAULT;
				}
				kfree_s(cmd, sizeof(wlan_netlist_t)+(tmp.nitems*sizeof(netitem_t)));
				break;
			}
			case WLAN_BSSCREATE:
			{
				wlan_bsscreate_t cmd;
				if (copy_from_user( &cmd, req->data, sizeof(cmd))) {
					result = -EFAULT;
					break;
				}
				req->result = am930mgr_createbss( llc->mgr, 
									cmd.channel, 
									(wlan_ie_ssid_t*)cmd.ssid,
									cmd.beacon_int, 
									cmd.atim_win);
				break;
			}
			case WLAN_BSSJOIN:
			{
				wlan_bssjoin_t	cmd;
				if (copy_from_user( &cmd, req->data, sizeof(cmd))) {
					result = -EFAULT;
					break;
				} 
				req->result = am930mgr_joinbss( llc->mgr, cmd.bssid );
				break;
			}
			case WLAN_AUTHENTICATE:
			{
				UINT32		timeout = 2 * HZ;	/* default 2 second */
				
				/* Call mgr to begin the authentication */
				am930mgr_authen_begin_sta( llc->mgr, WLAN_AUTH_ALG_OPENSYSTEM);

				/* Set a timer for (mgmt_timeout) to make sure we  return */
				init_timer(&llc->cmdtimer);
				llc->cmdtimer.data = (unsigned long)llc;
				llc->cmdtimer.function = (timerfunc_t)am930llc_cmdtimerfunc;
				llc->cmdtimer.expires = jiffies + timeout;
				add_timer(&llc->cmdtimer);

				/* Now, do an interruptible_sleep, we'll be awakened by: */
				/*  a signal the process, cmdtimerfunc (timeout), or llc_scancomplete */
				interruptible_sleep_on(&(llc->cmdwq));
				llc->cmdwq = NULL;

				/* If sleep return is via signal */
				if ( signal_pending(current) ) {
					/* stop authen and clean up */
					am930mgr_authen_stop(llc->mgr);
					del_timer(&llc->cmdtimer);
					result = -EINTR;
				} else {
					/* If timeout */
					if ( llc->cmdtimer.expires == 0 )
					{
						/* stop authen and clean up */
						am930mgr_authen_stop(llc->mgr);
						result = -ETIME;
					} else {
						/* Success */
						/* clean up timer */
						del_timer(&llc->cmdtimer);
						llc->cmdtimer.expires=0;
						req->result = llc->cmdstatus;
						/* return success */
						result = 0;
					}
				}
				break;
			}
			case WLAN_ASSOCIATE:
			{
				UINT32		timeout = 2 * HZ;	/* default 2 second */
				
				/* Call mgr to begin the association */
				am930mgr_assoc_begin_sta(llc->mgr);

				/* Set a timer for (mgmt_timeout) to make sure we  return */
				init_timer(&llc->cmdtimer);
				llc->cmdtimer.data = (unsigned long)llc;
				llc->cmdtimer.function = (timerfunc_t)am930llc_cmdtimerfunc;
				llc->cmdtimer.expires = jiffies + timeout;
				add_timer(&llc->cmdtimer);

				/* Now, do an interruptible_sleep, we'll be awakened by: */
				/*  a signal the process, cmdtimerfunc (timeout), or llc_scancomplete */
				interruptible_sleep_on(&(llc->cmdwq));
				llc->cmdwq = NULL;

				/* If sleep return is via signal */
				if ( signal_pending(current) ) {
					/* stop assoc and clean up */
					am930mgr_assoc_stop(llc->mgr);
					del_timer(&llc->cmdtimer);
					result = -EINTR;
				} else {
					/* If timeout */
					if ( llc->cmdtimer.expires == 0 )
					{
						/* stop assoc and clean up */
						am930mgr_assoc_stop(llc->mgr);
						result = -ETIME;
					} else {
						/* Success */
						/* clean up timer */
						del_timer(&llc->cmdtimer);
						llc->cmdtimer.expires=0;
						req->result = llc->cmdstatus;
						/* return success */
						result = 0;
					}
				}
				break;
			}
			case WLAN_GETMIB:
			{
				UINT8*	p = NULL;
				UINT32	size = 0;
				UINT32	mibcode;
				
				p = kmalloc(req->len, GFP_KERNEL);
				if (p == NULL) {
					result = -ENOMEM;
					break;
				}

				if ( copy_from_user( &mibcode, req->data, 4 ) ) {
					kfree_s(p, req->len);
					result = -EFAULT;
					break;
				}

				switch(mibcode) {
					case SUMIB_LOCAL:
						size = sizeof(su_mib_local_t);
						break;
					case SUMIB_ADDR:
						size = sizeof(su_mib_mac_addr_stat_grp_t);
						break;
					case SUMIB_MAC:
						size = sizeof(su_mib_mac_t); 
						break;
					case SUMIB_STAT:
						size = sizeof(su_mib_mac_statistics_t); 
						break;
					case SUMIB_MGMT:
						size = sizeof(su_mib_mac_mgmt_t); 
						break;
					case SUMIB_DRVR:
						/* size = sizeof(su_mib_local_t); */
						size = 0;
						break;
					case SUMIB_PHY:
						size = sizeof(su_mib_phy_t); 
						break;
				}
				am930mgr_mibget( llc->mgr, mibcode, size, p);

				if ( copy_to_user( ((UINT8*)(req->data)) + 4, p, size)) {
					result = -EFAULT;
				}
				kfree_s(p, req->len);
				break;
			}
			case WLAN_SETMIBITEM:
			case WLAN_GETMIBITEM:
				result = -ENOSYS;
				break;
			case WLAN_PRIVACY:
			{
				wlan_privacy_t	cmd;
				int				i;

				if ( copy_from_user( &cmd, req->data, sizeof(cmd)) ) {
					result = -EFAULT;
					break;
				}
				llc->mac->wep_defkeyid = cmd.defkey;
				llc->mac->exclude_unencrypted = cmd.exclude_unencrypted;
				for ( i = 0; i < WLAN_WEP_NKEYS; i++)
				{
					memcpy( llc->mac->wep_key[i], cmd.keys[i], WLAN_WEP_KEYLEN);
				}
				llc->mac->privacy_invoked = 1;

				WLAN_LOG_DEBUG2(2, "Wep set: defkey=%lu, exclude=%lu\n", 
					(UINT32)llc->mac->wep_defkeyid,
					(UINT32)llc->mac->exclude_unencrypted);
				WLAN_LOG_DEBUG5(2, "Wep key0: %02x:%02x:%02x:%02x:%02x\n",
					llc->mac->wep_key[0][0],
					llc->mac->wep_key[0][1],
					llc->mac->wep_key[0][2],
					llc->mac->wep_key[0][3],
					llc->mac->wep_key[0][4] );
				WLAN_LOG_DEBUG5(2, "Wep key1: %02x:%02x:%02x:%02x:%02x\n",
					llc->mac->wep_key[1][0],
					llc->mac->wep_key[1][1],
					llc->mac->wep_key[1][2],
					llc->mac->wep_key[1][3],
					llc->mac->wep_key[1][4] );
				WLAN_LOG_DEBUG5(2, "Wep key2: %02x:%02x:%02x:%02x:%02x\n",
					llc->mac->wep_key[2][0],
					llc->mac->wep_key[2][1],
					llc->mac->wep_key[2][2],
					llc->mac->wep_key[2][3],
					llc->mac->wep_key[2][4] );
				WLAN_LOG_DEBUG5(2, "Wep key3: %02x:%02x:%02x:%02x:%02x\n",
					llc->mac->wep_key[3][0],
					llc->mac->wep_key[3][1],
					llc->mac->wep_key[3][2],
					llc->mac->wep_key[3][3],
					llc->mac->wep_key[3][4] );
				break;
			}
			case WLAN_ETHCONV:
			{
				wlan_ethconv_t	cmd;

				if ( copy_from_user( &cmd, req->data, sizeof(cmd)) ) {
					result = -EFAULT;
					break;
				}

				llc->ethconv = cmd.ethconv_type;
				switch( llc->ethconv )
				{
					case WLAN_ETHCONV_ENCAP:
						WLAN_LOG_NOTICE0("ethconv=encapsulation\n");
						break;
					case WLAN_ETHCONV_RFC1042:
						WLAN_LOG_NOTICE0("ethconv=rfc1042\n");
						break;
					case WLAN_ETHCONV_8021h:
						WLAN_LOG_NOTICE0("ethconv=802.1h\n");
						break;
					default:
						WLAN_LOG_NOTICE0("ethconv=UNKNOWN...this is bad.\n");
						break;
				}
				break;
			}
			case WLAN_SNIFFERCMD:
			{
				#ifdef WLAN_INCLUDE_SNIF
				wlan_sniffercmd_t	cmd;
	
				if ( copy_from_user( &cmd, req->data, sizeof(cmd)) ) {
					result = -EFAULT;
					break;
				}
				cmd.result = 0;
				if ( cmd.cmd == SNIFFERCMD_REG ) {
					WLAN_LOG_DEBUG0(2, "Doing sniffer cmd reg.\n");
					if ( llc->nlsk == NULL || llc->snifferpid != 0 ) {
						WLAN_LOG_DEBUG1(1, "llc->nlsk=%lx\n", (UINT32)llc->nlsk);
						WLAN_LOG_DEBUG1(1, "llc->snifferpid=%d\n", llc->snifferpid);
						WLAN_LOG_DEBUG0(1, "Failed to reg sniffer: no netlink or sniffer already reg.\n");
						cmd.result = -1;
					} else {
						llc->snifferpid = cmd.pid;
						llc->mac->snifflags = SNIFFLAG_SNIFRX | cmd.flags;
					}
				} else if ( cmd.cmd == SNIFFERCMD_UNREG ) {
					WLAN_LOG_DEBUG0(2, "Doing sniffer cmd unreg.\n");
					llc->snifferpid = 0;
					llc->mac->snifflags = 0;
				}
				if ( copy_to_user( req->data, &cmd, sizeof(cmd)) ) {
					result = -EFAULT;
				}
				#else
				result = -ENOSYS;
				#endif
				break;
			}
			default:
				printk(KERN_DEBUG "am930_cs: Unknown or unsupported wlan ioctl!\n");
				break;
		}
		llc->currcmd = 0;
	}

	clear_bit( 0, (void*)&(llc->cmdbusy));
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930llc_devgetstats
*	getstats method for the linux net device. Called as a result
*	of a number of user mode utilities used to check the rx/tx
*	statistics of an interface.
*
*	returns: zero
----------------------------------------------------------------*/
enet_stats_t* am930llc_devgetstats(device_t *dev)
{
	DBFENTER;
	DBFEXIT;
	return &(V2P(dev->priv)->stats);
}


/*----------------------------------------------------------------
*	am930llc_devhard_start_xmit
*	hard_start_xmint method for the linux net device. Called by 
*	the higher level protocol code when it has a packet to xmit.
*
*	returns: zero
----------------------------------------------------------------*/
int am930llc_devhard_start_xmit( struct sk_buff *skb, device_t *dev)
{
	int			result = 0;
	int			txresult = -1;
	wlan_pb_t	*pb;
	am930llc_t	*llc = V2P(dev->priv)->llc;

	DBFENTER;

	if ( dev->start == 1 )
	{
		/* If some higher layer thinks we've missed a tx-done, we are passed
			NULL. Caution: dev_tint handles the cli/sti ..
		*/
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,25))
		if ( skb == NULL )
		{
			dev->tbusy = 0;
			dev_tint(dev);
			return 0;
		}
#endif

		if ( test_and_set_bit(0, (void*)&(dev->tbusy)) != 0 )
		{
			/* TODO: add a f/w reset capability here. see skeleton.c */
			WLAN_LOG_DEBUG2(1, "called when tbusy set,"
			                   "txllc.len=%d txmac.len=%d\n",
							   llc->mac->llcq->len,
							   llc->mac->macq->len);
			result = 1;
		}
		else
		{
			dev->trans_start = jiffies;

			/* OK, now we setup the ether to 802.11 conversion */
			pb = am930llc_pballoc();

			if ( pb != NULL )
			{
				pb->ethhostbuf = skb;
				pb->ethfree = am930llc_pbfreeskb;
				pb->ethbuf = skb->data;
				pb->ethbuflen = skb->len;
				pb->ethfrmlen = skb->len;
				pb->eth_hdr = (wlan_ethhdr_t*)pb->ethbuf;

				if ( am930llc_pb_ether_to_p80211( llc, pb) != 0 )
				{
					/* convert failed */
					result = 1;
					am930llc_pbfree(pb);
				}
				else
				{
					txresult = 
						am930mac_txllc( llc->mac, pb->eth_hdr->daddr,
									 pb->eth_hdr->saddr, pb);
		
					if ( txresult == 0)  /* success and more buf avail, re: hw_txdata */
					{
						dev->tbusy = 0;
						result = 0;
					}
					else if ( txresult == 1 ) /* success, no more avail */
					{
						result = 0;
					}
					else if ( txresult == 2 ) /* alloc failure, drop frame */
					{
						result = 0;
						am930llc_pbfree(pb);
					}
					else /* buffer full or queue busy */
					{
						result = 1;
						pb->ethfree = NULL;
					}
				}
			}
		}
	}

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930llc_devopen
*	open method for the linux net device. Called when ifconfig
*	is used to set up the interface.
*
*	returns: zero
----------------------------------------------------------------*/
int am930llc_devopen(device_t *dev)
{
	int result = 0;
	DBFENTER;

	/* set the flags in the device object */
	dev->tbusy = 0;
	dev->interrupt = 0;
	dev->start = 1;

	#ifdef WLAN_PCMCIA
	MOD_INC_USE_COUNT;
	#endif

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930llc_devstop
*	stop method for the linux net device. Called  when ifconfig
*	is used to shut down an interface.
*
*	returns: zero
----------------------------------------------------------------*/
int am930llc_devstop(device_t *dev)
{
	int result = 0;

	DBFENTER;

	/* set the flags in the device object */
	dev->start = 0;
	dev->tbusy = 1;

	#ifdef WLAN_PCMCIA
	MOD_DEC_USE_COUNT;
	#endif

	DBFEXIT;
	return result;
}



/*----------------------------------------------------------------
*	am930llc_nlfunc
*	Called by netlink when an skb is ready on our netlink i/f.
*	Currently does nothing since we don't expect anything to be
*	coming in.
*	Arguments:
*		sk		- sock representing our netlink i/f
*		len		- buf len
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_nlfunc(struct sock *sk, int len)
{
	return;
}


/*----------------------------------------------------------------
*	am930llc_pb_ether_to_p80211
*	Uses the contents of the ether frame and the etherconv setting
*   to build the elements of the 802.11 frame.  
*
*	We don't actually set 
*	up the frame header here.  That's the MAC's job.  We're only handling
*	conversion of DIXII or 802.3+LLC frames to something that works
*	with 802.11.
*
*	Assume the following are already set:
*		pb->ethfree
*		pb->ethhostbuf
*		pb->ethbuf;
*		pb->ethbuflen
*		pb->eth_hdr
*	returns: zero on success, non-zero on failure
----------------------------------------------------------------*/
int am930llc_pb_ether_to_p80211( am930llc_t *llc, wlan_pb_t *pb)
{
	UINT16	proto;

	if ( llc->ethconv == WLAN_ETHCONV_ENCAP ) /* simplest case */
	{
		/* here, we don't care what kind of ether frm. Just stick it */
		/*  in the 80211 payload */
		pb->p80211hostbuf = kmalloc( WLAN_HDR_A3_LEN, GFP_ATOMIC);
		if ( pb->p80211hostbuf == NULL ) return 1;
		pb->p80211buflen = WLAN_HDR_A3_LEN;
		pb->p80211free = am930llc_pbkfree_s;
		pb->p80211buf = (UINT8*)(pb->p80211hostbuf);
		pb->p80211_hdr = (p80211_hdr_t*)pb->p80211buf;
		pb->p80211_payload = pb->ethbuf;
		pb->p80211_payloadlen = pb->ethbuflen;
	}
	else
	{
		/* step 1: classify ether frame, DIX or 802.3? */
		proto = ntohs(pb->eth_hdr->type);
		if ( proto <= 0x05DC )  /* type|len <= 1500 ? */
		{
			/* it's 802.3, pass ether payload unchanged,  */
			/*   leave off any PAD octets.  */
			pb->p80211hostbuf = kmalloc( WLAN_HDR_A3_LEN, GFP_ATOMIC);
			if ( pb->p80211hostbuf == NULL ) return 1;
			pb->p80211buflen = WLAN_HDR_A3_LEN;
			pb->p80211free = am930llc_pbkfree_s;
			pb->p80211buf = (UINT8*)(pb->p80211hostbuf);
			pb->p80211_hdr = (p80211_hdr_t*)pb->p80211buf;

			/* setup the payload ptrs */
			pb->p80211_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
			pb->p80211_payloadlen = ntohs(pb->eth_hdr->type);
		}
		else
		{
			/* it's DIXII, time for some conversion */
			pb->p80211hostbuf = kmalloc( 
						WLAN_HDR_A3_LEN + 
						sizeof(wlan_llc_t) +
						sizeof(wlan_snap_t), GFP_ATOMIC);
			if ( pb->p80211hostbuf == NULL ) return 1;
			pb->p80211buflen = 
						WLAN_HDR_A3_LEN + 
						sizeof(wlan_llc_t) +
						sizeof(wlan_snap_t);
			pb->p80211free = am930llc_pbkfree_s;
			pb->p80211buf = (UINT8*)pb->p80211hostbuf;
			pb->p80211_hdr = (p80211_hdr_t*)pb->p80211buf;
			pb->p80211_llc = (wlan_llc_t*)(pb->p80211buf + WLAN_HDR_A3_LEN);
			pb->p80211_snap = (wlan_snap_t*)(((UINT8*)pb->p80211_llc) + sizeof(wlan_llc_t));

			/* setup the LLC header */
			pb->p80211_llc->dsap = 0xAA;	/* SNAP, see IEEE 802 */
			pb->p80211_llc->ssap = 0xAA;
			pb->p80211_llc->ctl = 0x03;
		
			/* setup the SNAP header */
			pb->p80211_snap->type = htons(proto);
			if ( llc->ethconv == WLAN_ETHCONV_8021h && 
				 am930llc_stt_findproto(llc, proto) )
			{
				memcpy( pb->p80211_snap->oui, oui_8021h, WLAN_IEEE_OUI_LEN);
			}
			else
			{
				memcpy( pb->p80211_snap->oui, oui_rfc1042, WLAN_IEEE_OUI_LEN);
			}

			/* setup the payload ptrs */
			pb->p80211_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
			pb->p80211_payloadlen = pb->ethbuflen - sizeof(wlan_ethhdr_t);

		}
	}
	return 0;
}


/*----------------------------------------------------------------
*	am930llc_p80211_to_ether
*	Uses the contents of a received 802.11 frame and the etherconv 
*	setting to build an ether frame.
*
*	This function extracts the src and dest address from the 802.11
*	frame to use in the construction of the eth frame.
*
*	This function _will_ set the ethfree member.  If a caller wants
*	to allow some other component (a higher layer?) take responsiblity
*	for freeing the ethhostbuf, the caller should set ethfree to NULL.
*
*	Assume the following are already set:
*		pb->p80211free
*		pb->p80211hostbuf
*		pb->p80211buf
*		pb->p80211buflen
*		pb->p80211_hdr
*		pb->p80211_payload
*		pb->p80211_payloadlen
*	returns: zero on success, non-zero on failure
----------------------------------------------------------------*/
int am930llc_pb_p80211_to_ether( am930llc_t *llc, wlan_pb_t *pb)
{
	UINT8			*daddr = NULL;
	UINT8			*saddr = NULL;
	wlan_ethhdr_t	*ethhdr;
	UINT			llclen;		/* 802 LLC+data length */
	UINT			dixlen;		/* DIX data length */
	UINT			buflen;		/* full frame length, including PAD */
	UINT16			fc;

	/* setup some vars for convenience */
	fc = ieee2host16(pb->p80211_hdr->a3.fc);
	if ( (WLAN_GET_FC_TODS(fc) == 0) && (WLAN_GET_FC_FROMDS(fc) == 0) )
	{
		daddr = pb->p80211_hdr->a3.a1;
		saddr = pb->p80211_hdr->a3.a2;
	}
	else if( (WLAN_GET_FC_TODS(fc) == 0) && (WLAN_GET_FC_FROMDS(fc) == 1) ) 
	{
		daddr = pb->p80211_hdr->a3.a1;
		saddr = pb->p80211_hdr->a3.a3;
	}
	else if( (WLAN_GET_FC_TODS(fc) == 1) && (WLAN_GET_FC_FROMDS(fc) == 0) ) 
	{
		daddr = pb->p80211_hdr->a3.a3;
		saddr = pb->p80211_hdr->a3.a2;
	}
	else
	{
		WLAN_LOG_ERROR0("HDR_A4 detected! A4 currently not supported.\n");
		/* set some bogus pointers so at least we won't crash */
		daddr = pb->p80211_hdr->a3.a1;
		saddr = pb->p80211_hdr->a3.a2;
	}

	ethhdr = (wlan_ethhdr_t*)(pb->p80211_payload);
	pb->p80211_llc = (wlan_llc_t*)(((UINT8*)pb->p80211_hdr) + sizeof(p80211_hdr_a3_t));
	pb->p80211_snap = (wlan_snap_t*)(((UINT8*)pb->p80211_llc) + sizeof(wlan_llc_t));

	/* Test for the various encodings */
	if ( memcmp( daddr, ethhdr->daddr, WLAN_ETHADDR_LEN) == 0 && 
		memcmp(  saddr, ethhdr->saddr, WLAN_ETHADDR_LEN) == 0 )
	{
		/* ENCAP */
		/* Test for an overlength frame */
		if ( pb->p80211frmlen > WLAN_HDR_A3_LEN+WLAN_CRC_LEN+WLAN_MAX_ETHFRM_LEN)
		{
			/* A bogus length ethfrm has been encap'd. */
			/* Is someone trying an oflow attack? */
			return 1;
		}

		/* allocate space and setup host buffer */
		buflen = llclen = pb->p80211frmlen - WLAN_HDR_A3_LEN - WLAN_CRC_LEN;
		pb->ethhostbuf = dev_alloc_skb(buflen + 2); /* +2 is attempt to align IP header */
		if ( pb->ethhostbuf == NULL ) return 1;
		pb->ethfree = am930llc_pbfreeskb;
		skb_reserve( pb->ethhostbuf, 2);
		skb_put( ((struct sk_buff*)pb->ethhostbuf), buflen);	/* make room */
		pb->ethbuf = ((struct sk_buff*)pb->ethhostbuf)->data;
		pb->ethbuflen = buflen;

		/* setup the pointers */
		pb->eth_hdr = (wlan_ethhdr_t*)pb->ethbuf;
		pb->eth_llc = (wlan_llc_t*)(pb->ethbuf + sizeof(wlan_ethhdr_t));
		pb->eth_snap = (wlan_snap_t*)
			(pb->ethbuf + sizeof(wlan_ethhdr_t) + sizeof(wlan_llc_t));
		pb->eth_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
		pb->eth_payloadlen = buflen - sizeof(wlan_ethhdr_t);

		/* now copy the data from the 80211 frame */
		memcpy( pb->ethbuf, pb->p80211_payload, buflen);	/* copy the data */
	}
	else if (	pb->p80211_llc->dsap == 0xaa &&
				pb->p80211_llc->ssap == 0xaa &&
				pb->p80211_llc->ctl == 0x03 &&
				memcmp( pb->p80211_snap->oui, oui_rfc1042, WLAN_IEEE_OUI_LEN) == 0 &&
				llc->ethconv == WLAN_ETHCONV_8021h && 
				am930llc_stt_findproto(llc, pb->p80211_snap->type) )
	{
		/* it's a SNAP + RFC1042 frame && protocol is in STT */
		/* build 802.3 + RFC1042 */
		
		/* Test for an overlength frame */
		if ( pb->p80211_payloadlen > WLAN_MAX_ETHFRM_LEN - WLAN_ETHHDR_LEN )
		{
			/* A bogus length ethfrm has been sent. */
			/* Is someone trying an oflow attack? */
			return 1;
		}

		llclen = pb->p80211_payloadlen;
		buflen = wlan_max( llclen + sizeof(wlan_ethhdr_t), WLAN_MIN_ETHFRM_LEN);
		pb->ethhostbuf = dev_alloc_skb(buflen + 2); /* +2 is attempt to align IP header */
		if ( pb->ethhostbuf == NULL ) return 1;
		skb_reserve( (struct sk_buff*)pb->ethhostbuf, 2);		
		skb_put( (struct sk_buff*)pb->ethhostbuf, buflen);	/* make room */
		pb->ethbuf = ((struct sk_buff*)pb->ethhostbuf)->data;
		pb->ethbuflen = buflen;
/*		memset( pb->ethbuf, 0, buflen);	*/		/* zero for possible PAD */

		/* set up the pointers */
		pb->eth_hdr = (wlan_ethhdr_t*)pb->ethbuf;
		pb->eth_llc = (wlan_llc_t*)(pb->ethbuf + sizeof(wlan_ethhdr_t));
		pb->eth_snap = (wlan_snap_t*)
			(pb->ethbuf + sizeof(wlan_ethhdr_t) + sizeof(wlan_llc_t));
		pb->eth_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
		pb->eth_payloadlen = llclen;

		/* set up the 802.3 header */	
		pb->eth_hdr->type = htons(pb->eth_payloadlen);
		memcpy( pb->eth_hdr->daddr, daddr, WLAN_ETHADDR_LEN);
		memcpy( pb->eth_hdr->saddr, saddr, WLAN_ETHADDR_LEN);

		/* now copy the data from the 80211 frame */
		memcpy( pb->eth_payload, pb->p80211_payload, pb->p80211_payloadlen);
	}
	else if (	pb->p80211_llc->dsap == 0xaa &&
				pb->p80211_llc->ssap == 0xaa &&
				pb->p80211_llc->ctl == 0x03 )
	{
		/* it's an 802.1h frame || (an RFC1042 && protocol is not in STT) */
		/* build a DIXII + RFC894 */

		dixlen = pb->p80211_payloadlen - sizeof(wlan_llc_t) - sizeof(wlan_snap_t);

		/* Test for an overlength frame */
		if ( dixlen + WLAN_ETHHDR_LEN > WLAN_MAX_ETHFRM_LEN)
		{
			/* A bogus length ethfrm has been sent. */
			/* Is someone trying an oflow attack? */
			return 1;
		}

		dixlen = pb->p80211_payloadlen - sizeof(wlan_llc_t) - sizeof(wlan_snap_t);
		buflen = wlan_max( dixlen + sizeof(wlan_ethhdr_t), WLAN_MIN_ETHFRM_LEN);
		pb->ethhostbuf = dev_alloc_skb(buflen + 2); /* +2 is attempt to align IP header */
		if ( pb->ethhostbuf == NULL ) return 1;
		skb_reserve( (struct sk_buff*)pb->ethhostbuf, 2);		
		skb_put( (struct sk_buff*)pb->ethhostbuf, buflen);	/* make room */
		pb->ethbuf = ((struct sk_buff*)pb->ethhostbuf)->data;
		pb->ethbuflen = buflen;
/*		memset( pb->ethbuf, 0, buflen);	*/		/* zero for possible PAD */

		/* set up the pointers */
		pb->eth_hdr = (wlan_ethhdr_t*)pb->ethbuf;
		pb->eth_llc = NULL;
		pb->eth_snap = NULL;
		pb->eth_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
		pb->eth_payloadlen = dixlen;

		/* make sure the llc and snap ptrs are set */
		pb->p80211_llc = (wlan_llc_t*)pb->p80211_payload;
		pb->p80211_snap = (wlan_snap_t*)
			(pb->p80211_payload + sizeof(wlan_llc_t));

		/* set up the DIXII header */	
		pb->eth_hdr->type = pb->p80211_snap->type;
		memcpy( pb->eth_hdr->daddr, daddr, WLAN_ETHADDR_LEN);
		memcpy( pb->eth_hdr->saddr, saddr, WLAN_ETHADDR_LEN);

		/* now copy the data from the 80211 frame */
		memcpy( pb->eth_payload, 
				pb->p80211_payload + sizeof(wlan_llc_t) + sizeof(wlan_snap_t), 
				dixlen);
	}
	else
	{
		/* any NON-ENCAP */
		/* it's a generic 80211+LLC or IPX 'Raw 802.3' */
		/*  build an 802.3 frame */
		/* allocate space and setup hostbuf */

		/* Test for an overlength frame */
		if ( pb->p80211_payloadlen + WLAN_ETHHDR_LEN > WLAN_MAX_ETHFRM_LEN)
		{
			/* A bogus length ethfrm has been sent. */
			/* Is someone trying an oflow attack? */
			return 1;
		}

		llclen = pb->p80211_payloadlen;
		buflen = wlan_max( llclen + sizeof(wlan_ethhdr_t), WLAN_MIN_ETHFRM_LEN);
		pb->ethhostbuf = dev_alloc_skb(buflen + 2); /* +2 is attempt to align IP header */
		if ( pb->ethhostbuf == NULL ) return 1;
		skb_reserve( (struct sk_buff*)pb->ethhostbuf, 2);		
		skb_put( (struct sk_buff*)pb->ethhostbuf, buflen);	/* make room */
		pb->ethbuf = ((struct sk_buff*)pb->ethhostbuf)->data;
		pb->ethbuflen = buflen;
/*		memset( pb->ethbuf, 0, buflen);	*/		/* zero for possible PAD */

		/* set up the pointers */
		pb->eth_hdr = (wlan_ethhdr_t*)pb->ethbuf;
		pb->eth_llc = (wlan_llc_t*)(pb->ethbuf + sizeof(wlan_ethhdr_t));
		pb->eth_snap = (wlan_snap_t*)
			(pb->ethbuf + sizeof(wlan_ethhdr_t) + sizeof(wlan_llc_t));
		pb->eth_payload = pb->ethbuf + sizeof(wlan_ethhdr_t);
		pb->eth_payloadlen = llclen;

		/* set up the 802.3 header */	
		pb->eth_hdr->type = htons(pb->eth_payloadlen);
		memcpy( pb->eth_hdr->daddr, daddr, WLAN_ETHADDR_LEN);
		memcpy( pb->eth_hdr->saddr, saddr, WLAN_ETHADDR_LEN);

		/* now copy the data from the 80211 frame */
		memcpy( pb->eth_payload, pb->p80211_payload, pb->p80211_payloadlen);
	}
	return 0;
}


/*----------------------------------------------------------------
*	am930llc_pbfreeskb
*	Free method for wlan_pb's that have skbs in them.  Called
*	via ptr from am930llc_pbfree.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_pbfreeskb( void *buf, int size )
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0))
	dev_kfree_skb( (struct sk_buff*)buf, FREE_WRITE);
#else
	dev_kfree_skb( (struct sk_buff*)buf );
#endif
}


/*----------------------------------------------------------------
*	am930llc_pbkfree_s
*	Free method for wlan_pb's that have linux kmalloc'd 
*	buffer's in them.  Called via ptr from am930llc_pbfree.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_pbkfree_s( void *buf, int size)
{
	kfree_s( buf, size);
}


/*----------------------------------------------------------------
*	am930llc_pballoc
*	Allocates and zeros a wlan_pb structure.  Largely here for
*	symmetry with the pbfree routine.
*
*	returns: A valid wlan_pb ptr on success,
*			NULL otherwise.
----------------------------------------------------------------*/
wlan_pb_t* am930llc_pballoc(void)
{
	wlan_pb_t	*pb;
	pb = kmalloc( sizeof(wlan_pb_t), GFP_ATOMIC);
	if ( pb != NULL )
	{
		memset(pb, 0, sizeof(wlan_pb_t));
	}
	return pb;
}


/*----------------------------------------------------------------
*	am930llc_pballoc_p80211
*	Allocates a buffer for an 80211 frame and sets the ptrs in a 
*	given pb.  Primarily used by the receive path (see hw).  Handled
*	here so that the allocation used and the free method are set
*	in one place (helps with portability?).
*
*	returns: nothing
*			caller should check pb->p80211hostbuf for NULL
----------------------------------------------------------------*/
void am930llc_pballoc_p80211(wlan_pb_t *pb, UINT len)
{
	pb->p80211hostbuf = kmalloc( len, GFP_ATOMIC);
	if ( pb->p80211hostbuf != NULL )
	{
		pb->p80211free = am930llc_pbkfree_s;
		pb->p80211buf = (UINT8*)pb->p80211hostbuf;
		pb->p80211buflen = len;
		pb->p80211frmlen = len;		/* initially assume frm is same as buf */
		pb->p80211_hdr = (p80211_hdr_t*)pb->p80211buf;
		pb->p80211_payload = pb->p80211buf + WLAN_HDR_A3_LEN;
		pb->p80211_payloadlen = pb->p80211buflen - WLAN_HDR_A3_LEN - WLAN_CRC_LEN;
	}
	return;
}


/*----------------------------------------------------------------
*	am930llc_pbfree
*	Frees the ethhostbuf and the p80211hostbuf elements of a wlan_pb
*	if there is a free method for each.  Then frees the wlan_pb itself.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_pbfree(wlan_pb_t* pb)
{
	if ( pb->ethhostbuf != NULL && pb->ethfree != NULL)
	{
		(*(pb->ethfree))(pb->ethhostbuf, pb->ethbuflen);
	}
	if ( pb->p80211hostbuf != NULL && pb->p80211free != NULL)
	{
		(*(pb->p80211free))(pb->p80211hostbuf, pb->p80211buflen);
	}
	kfree_s(pb, sizeof(wlan_pb_t));
}


/*----------------------------------------------------------------
*	am930llc_rxframe
*	Event method called from mac when a frame has been received.
*
*	returns: zero
----------------------------------------------------------------*/
void am930llc_rxframe( am930llc_t *llc, wlan_pb_t *pb)
{
	struct sk_buff	*skb;

	DBFENTER;

	llc->dev->interrupt = 1;

	if ( llc->dev->start )
	{
		if ( am930llc_pb_p80211_to_ether(llc, pb) == 0 )
		{
			llc->dev->last_rx = jiffies;

			/* take ownership of skb from pb */
			skb = (struct sk_buff*)pb->ethhostbuf;
			pb->ethhostbuf = NULL;
			pb->ethfree = NULL;
			skb->dev = llc->dev;

			WLAN_HEX_DUMP(3,"llcethrx:", skb->data, skb->len);

			skb->protocol = eth_type_trans( skb, llc->dev);

			V2P(llc->dev->priv)->stats.rx_packets++;

			netif_rx(skb);
		}
	}

	llc->dev->interrupt = 0;

	am930llc_pbfree(pb);

	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930llc_rxframe_err
*
*	returns: zero
----------------------------------------------------------------*/
void am930llc_rxframe_err( am930llc_t *llc)
{
	DBFENTER;

	V2P(llc->dev->priv)->stats.rx_errors++;

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930llc_sendtosniffer
*	
*	If a sniffer is registered, this function fills in the 
*	sniffer msg header and uses a netlink unicast delivery to send
*	the frame up to the sniffer.
*	
*	Note: the caller is relinquishes ownership of the skb passed to
*			this function.
*	Arguments:
*		skb		ptr to an skb allocated by the caller and filled with
*				a wlan_sniffer_t + an 802.11 frame.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930llc_sendtosniffer(am930llc_t *llc, struct sk_buff *skb)
{
	if ( llc->nlsk == NULL || llc->snifferpid == 0)
	{
		/* This really shouldn't happen */
		WLAN_LOG_WARNING0("sendtosniffer called when no nl sk or registered sniffern");
		kfree_skb(skb);
	}
	else
	{
		netlink_unicast( llc->nlsk, skb, llc->snifferpid, 1);
	}
	return;
}


/*----------------------------------------------------------------
*	am930llc_stt_findproto
*	Searches the 802.1h Selective Translation Table for a given 
*	protocol.
*
*	returns: 1 - if the table is empty or a match is found.
*			 0 - if the table is non-empty and a match is not found.
----------------------------------------------------------------*/
int am930llc_stt_findproto(am930llc_t *llc, UINT16 proto)
{
	/* Always return found for now.  This is the behavior used by the */
	/*  Zoom Win95 driver when 802.1h mode is selected */
	/* TODO: If necessary, add an actual search */

	return 1;
}


/*----------------------------------------------------------------
*	am930llc_stt_addproto
*	Adds a protocol number to the 802.1h Selective Translation Table.
*
*	returns: 0 - on success
*			 non-zero - if the table is full, or a memory allocation fails.
----------------------------------------------------------------*/
int am930llc_stt_addproto(am930llc_t *llc, UINT16 proto)
{
	return 0;
}


/*----------------------------------------------------------------
*	am930llc_devset_multicast_list
*	set_multicast_list method for the linux net device. 
*
*	returns: nothing
----------------------------------------------------------------*/
#define HW_MAX_ADDRS 4
void am930llc_devset_multicast_list(device_t *dev)
{
	DBFENTER;

	/* First handle our promisc state on its own */
	/* We can handle it here by itself since the card supports ALLMULTI */
	if ( dev->flags & IFF_PROMISC )
	{
		/* Enable Promiscuous Mode */
		UINT8 one = 1;
		am930mgr_mibsetitem(V2P(dev->priv)->llc->mgr, 
			MAC_PROMISC_EN, &one, sizeof(one));
		WLAN_LOG_DEBUG0(2, "am930 promisc enabled\n");

	}
	else
	{
		UINT8 zero = 0;
		am930mgr_mibsetitem(V2P(dev->priv)->llc->mgr, 
			MAC_PROMISC_EN, &zero, sizeof(zero));
		WLAN_LOG_DEBUG0(2, "am930 promisc disabled\n");
	}

	/* Now handle the multicast state */
	if ( dev->flags & IFF_ALLMULTI || (dev->mc_count > HW_MAX_ADDRS) )
	{
		/* Enable the hw "rcv all multicast" mode, either because of 
		   the IFF_ALLMULTI switch or because we have more multicast
		   hw addresses than the hw filter can hold (1.0 fw has four).
		*/
		printk(KERN_DEBUG "IFF_ALLMULTI currently unsupported.\n");
	}
	else if ( dev->mc_count )
	{
		/* Load the contents of the mc_list from the dev structure into
		   the filter list in the hardware (effectively reset it from
		   scratch).
		*/
		printk(KERN_DEBUG "Multicast filters currently unsupported.\n");
	}
	else
	{
		/* Disable allmulti, and clear the multicast filter list */
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930llc_ontxcomplete
*
*	returns: zero
----------------------------------------------------------------*/
void am930llc_ontxcomplete( am930llc_t *llc, UINT32 txresult)
{
	DBFENTER;

	if ( txresult != 0 && txresult != 0xffffffffUL) /* tx failure */
	{
		V2P(llc->dev->priv)->stats.tx_errors++;
	}
	else
	{
		V2P(llc->dev->priv)->stats.tx_packets++;
	}

	llc->dev->tbusy = 0;
	mark_bh(NET_BH);

	DBFEXIT;
	return;
}


