/* am930_mac.c: Handles the 802.11 mac functions
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*
*/


#include <linux/config.h>
#include <wlan/wlan_compat.h>

#if (WLAN_OS == WLAN_LINUX_KERNEL)

/* The following prevents "kernel_version" from being set in this file. */
#define __NO_VERSION__

/* PCMCIA headers generated during PCMCIA package installation */
#ifdef WLAN_PCMCIA
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#endif

/* Module related headers, non-module drivers should not include */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

/* Standard driver includes */
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/in.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/kernel_stat.h>
#include <asm/bitops.h>


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,11)
#include <asm/spinlock.h>
#else
#include <linux/spinlock.h>
#endif
#include <linux/errno.h>


/* Card and Driver services includes */
#ifdef WLAN_PCMCIA
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#endif

/* Local Includes */
#include <wlan/version.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_stable.h>
#include <wlan/wlan_ioctl.h>

#ifdef WLAN_PCMCIA
#include "am930di.h"
#endif

#include "am930llc.h"
#include "am930mac.h"
#include "am930hw.h"
#include "am930mgr.h"

#elif ( WLAN_OS == WLAN_LWOS )

#include <wlan/wlan_compat.h>

#include <stddef.h>
#include <stdlibsys.h>
#include <direct.h>
#include <hal.h>
#include <mem.h>
#include <kernel.h>
#include <clock.h>

#include <wlan/version.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_stable.h>
#include <lw-wlan.h>
#include <wlan/am930mac.h>
#include <wlan/am930hw.h>
#include <wlan/am930mgr.h>

#else
	#error "No WLAN_OS match!"
#endif

/*================================================================*/
/* Local Constants  */

#define LLCQ_LEN	20
#define MACQ_LEN	20
#define RXQ_LEN		40

#define WLAN_CRC32_POLY_LE	(0xedb88320UL)
#define WLAN_CRC32_POLY_BE 	(0x04c11db7UL)

/*================================================================*/
/* Local Functions  */

static void am930mac_rxmgmt( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats);
static void am930mac_rxdata( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats);

static wlan_pb_t *am930mac_defrag( am930mac_t *mac, wlan_pb_t *pb);
static wlan_pb_t *am930mac_defragfinish( am930mac_t *mac, UINT8 hash );

static void am930mac_timerfunc(am930mac_t *mac);

static void am930mac_crc32_mktable(am930mac_t *mac);
static UINT32 am930mac_crc32_block(am930mac_t *mac, UINT32 reg, UINT8 *p, UINT len);
static UINT32 am930mac_crc32_blockfinish(am930mac_t *mac, UINT32 reg);
/*  static UINT32 am930mac_crc32_accum(am930mac_t *mac, UINT32 reg, UINT8 d); */

/*================================================================*/
/* Externals  */

#if (WLAN_OS == WLAN_LWOS)
am930mac_t gmac;
#endif

/*----------------------------------------------------------------
*	am930mac_construct
*
*	Allocates and initializes the mac object and it's subobjects.
*
*	returns: NULL on failure, the address of the new object
*            otherwise.
----------------------------------------------------------------*/
#if (WLAN_OS == WLAN_LINUX_KERNEL)
am930mac_t* am930mac_construct(void *di, UINT32 iobase, UINT32 membase, UINT32 irq)
#elif (WLAN_OS == WLAN_LWOS )
am930mac_t* am930mac_construct(wlanInfo_t *lw, UINT32 iobase, UINT32 membase, UINT32 irq)
#else
	#error "No WLAN_OS match!"
#endif

{
	am930mac_t	*mac;
	int			i;

	DBFENTER;

#if (WLAN_OS == WLAN_LINUX_KERNEL)
	/* Allocate the memory */
	mac = kmalloc( sizeof(am930mac_t), GFP_KERNEL);
#elif (WLAN_OS == WLAN_LWOS )
	mac = &gmac;
#else
	#error "No WLAN_OS match!"
#endif
	if ( mac == NULL )
	{
		WLAN_LOG_ERROR0("mac allocation failed!\n");
	}
	else
	{
		/* zero the mac object */
		memset( mac, 0, sizeof(am930mac_t));

		/* init the crc table */
		am930mac_crc32_mktable(mac);

		/* mark all the frag pools as avail */
		for ( i = 0; i < AM930MAC_FRAGPOOLS_MAX; i++)
		{
			mac->defragpool[i].seq = AM930MAC_FRAGPOOL_FREE;
		}

		/* intialize the locks */
		mac->fragpool_lock = SPIN_LOCK_UNLOCKED;

#if  (WLAN_OS == WLAN_LINUX_KERNEL)
#ifdef WLAN_PCMCIA
		/* save a reference to the di */
		mac->di = di;

		/* set the di priv ptr to mac */
		((dev_link_t*)di)->priv = mac;
#endif

		/* Create the hw instance */
		mac->hw = am930hw_construct( irq, iobase, membase, mac) ;

#elif (WLAN_OS == WLAN_LWOS )

		mac->lw = lw;
		mac->llc = lw;

		/* Create the hw instance */
		mac->hw = am930hw_construct( irq, iobase, membase, mac) ;

#else
	#error "No WLAN_OS match!"
#endif

		if ( mac->hw  == NULL)
		{
			WLAN_LOG_ERROR0("hw allocation failed!\n");
			am930mac_destruct(mac);
			mac = NULL;
#if (WLAN_OS == WLAN_LINUX_KERNEL)
#ifdef WLAN_PCMCIA
			((dev_link_t*)di)->priv = NULL;
#endif
#endif
		}
		else
		{
			/* Set up the transmit queues */
			mac->llcq = am930q_init( LLCQ_LEN );
			mac->macq = am930q_init( MACQ_LEN );
			mac->rxq = am930q_init( RXQ_LEN );

			/* Set the default MIB values (where zero is unacceptable) */
			mac->maxrxlifetime = 1024;

			/* Now, call the second part of the hw initialization */
			if ( am930hw_init_rx_tx( mac->hw ) != 0 )
			{
				WLAN_LOG_ERROR0("hw_init_rx_tx failed!\n");
				am930mac_destruct(mac);
				mac = NULL; 
#if (WLAN_OS == WLAN_LINUX_KERNEL)
#ifdef WLAN_PCMCIA
				((dev_link_t*)di)->priv = NULL;
#endif
#endif
			}
			else
			{
				mac->mgr = am930mgr_construct( mac, mac->hw );
				if ( mac->mgr == NULL )
				{
					WLAN_LOG_ERROR0("mgr allocation failed!\n");
					am930mac_destruct(mac);
					mac = NULL; 
#if (WLAN_OS == WLAN_LWOS)
				}
#elif (WLAN_OS == WLAN_LINUX_KERNEL)
#ifdef WLAN_PCMCIA
					((dev_link_t*)di)->priv = NULL;
#endif
				}
				else
				{
					mac->llc = am930llc_construct( mac, mac->mgr);
					if ( mac->llc == NULL )
					{
						WLAN_LOG_ERROR0("llc allocation failed!\n");
						am930mac_destruct(mac);
						mac = NULL; 
#ifdef WLAN_PCMCIA
						((dev_link_t*)di)->priv = NULL;
#endif
					}
				}
#else
	#error "No WLAN_OS match!"
#endif
				mac->timerfunc = am930mac_timerfunc;

				/* Hard coded to match the AM930 phy mib default. */
				/* Lame, I know....the new UI will fix it. */
				mac->oprates[0] = 1;
				mac->oprates[1] = 2;
				mac->oprates[2] = 0x82;
				mac->oprates[3] = 0x84;
			}
		}
	}

	DBFEXIT;
	return mac;
}


/*----------------------------------------------------------------
*	am930mac_destruct
*
*	Destroys the subobjects and the mac object. Caller needs to be
*   aware that the pointer passed in is invalid after this call.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_destruct( am930mac_t* mac)
{
	wlan_pb_t*	pb;

	DBFENTER;

	mac->timerfunc = NULL;

	if ( mac->hw != NULL )
	{
		am930hw_destruct( mac->hw );
		mac->hw = NULL;
	}

#if (WLAN_OS == WLAN_LINUX_KERNEL)
	if ( mac->llc != NULL )
	{
		am930llc_destruct( mac->llc);
		mac->llc = NULL;
	}
#endif

	/* TODO: handle destruction cleanly, call subobject destructors
	         after testing subobject pointers for NULL */

	/* get rid of the queues */
	while ( (mac->llcq != NULL) && (pb = (wlan_pb_t*)am930q_dequeue(mac->llcq)) )
	{
		am930shim_pbfree(pb);
	}
	kfree_s( mac->llcq, sizeof(am930q_t));

	while ( (mac->macq != NULL) && (pb = (wlan_pb_t*)am930q_dequeue(mac->macq)) )
	{
		am930shim_pbfree(pb);
	}
	kfree_s( mac->macq, sizeof(am930q_t));

	/* all subobjects handled, now free the mac object */
	kfree_s( mac, sizeof(am930mac_t));

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mac_crc32_block
*
*	Calculate the crc over a block of bytes.  To get the final crc value,
*	don't forget to call crc32_blockfinish.
*	Arguments: 
*		reg		initial accumulator value, passing it in allows us to 
*				crc non-contiguous blocks.  On the first call to this 
*				function for a given crc calculation, reg should be
*				0xffffffffUL.
*		p		ptr to the block
*		len		size of block
*	returns: 
*		crc32 value
----------------------------------------------------------------*/
UINT32 am930mac_crc32_block(am930mac_t *mac, UINT32 reg, UINT8 *p, UINT len)
{
	while(len)
	{
		reg = mac->crc32_table[(reg ^ *p) & 0x000000ffUL] ^ (reg >> 8);
		len--;
		p++;
	}
	return reg;
}


UINT32 am930mac_crc32_blockfinish(am930mac_t *mac, UINT32 reg)
{
	return reg ^ 0xffffffffUL;
}


/*----------------------------------------------------------------
*	am930mac_crc32_accum
*
*	Accumulates, in a given register, the crc of a series of bytes
*	passed to this function.
*	Arguments: 
*		reg		current accumulated crc value
*		d		next byte in series
*	returns: 
*		new accumulated crc32 value
----------------------------------------------------------------*/
#ifdef NOTUSED	/* we're not using it right now, ifdef'd to stop warning */
UINT32 am930mac_crc32_accum(am930mac_t *mac, UINT32 reg, UINT8 d)
{
	reg = mac->crc32_table[(reg ^ d) & 0x000000ffUL] ^ (reg >> 8);
	return reg;
}
#endif



/*----------------------------------------------------------------
*	am930mac_crc32_mktable
*
*	Constructs the table used for the crc32 calculation.
*	Arguments: none
*	returns:   nothing
----------------------------------------------------------------*/
void am930mac_crc32_mktable(am930mac_t *mac)
{
	UINT	i;
	UINT	k;
	UINT32	c;

	mac->crc32_table[0] = 0;
	for ( i = 1; i < 256; i++)
	{
		c = 0;
		for ( k = i | 256; k != 1; k >>= 1)
		{
			c = (c & 1) ? 
				((c >> 1) ^ WLAN_CRC32_POLY_LE) : 
				(c >> 1);
			if ( k & 1 )
			{
				c ^= WLAN_CRC32_POLY_LE;
			}
		}
		mac->crc32_table[i] = c;
	}
}


/*----------------------------------------------------------------
*	am930mac_ISR
*
*	ISR called from the di ISR. I know this is kindof a waste, but
*	I didn't want the di to know anything about the hw interface.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_ISR( am930mac_t *mac )
{
	DBFENTER;
	if ( mac->hw != NULL )
	{
		am930hw_ISR( mac->hw );
	}
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930mac_mk_capinfo
*
*	Uses the MIB and the current MAC state variables to construct
*	a cap_info halfword.  The value returned is in HOST order!
*
*	Arguments:
*		none
*
*	returns: 
*		The capinfo halfword in HOST order!
----------------------------------------------------------------*/
UINT16  am930mac_mk_capinfo(am930mac_t *mac)
{
	UINT16	capinfo = 0;
	if ( mac->mode == AM930_MACMODE_ESS_STA ||
		 mac->mode == AM930_MACMODE_ESS_AP )
	{
		capinfo |= WLAN_SET_MGMT_CAP_INFO_ESS(1);
	}
	else if ( mac->mode == AM930_MACMODE_IBSS_STA )
	{
		capinfo |= WLAN_SET_MGMT_CAP_INFO_IBSS(1);
	}

	if ( mac->exclude_unencrypted )
	{
		capinfo |= WLAN_SET_MGMT_CAP_INFO_PRIVACY(1);
	}

	return capinfo;
}


/*----------------------------------------------------------------
*	am930mac_ontxcomplete
*
*	Callback from hw when tx is signaled complete.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_ontxcomplete( am930mac_t *mac, UINT32 txresult )
{
	UINT32		txdone = 1;
	wlan_pb_t	*pb = NULL;
	am930q_t	*q;
	int			i = 0;

	DBFENTER;

	do 
	{
		WLAN_LOG_DEBUG3( 3,"nextq=%x llcq.len=%d macq.len=%d\n", 
			mac->nextq, am930q_length(mac->llcq), am930q_length(mac->macq));

		q = NULL;
		if ( am930q_length(mac->llcq) > 0 && am930q_length(mac->macq) > 0 )
		{
			q = mac->nextq & 0x01 ? mac->llcq : mac->macq;
		}
		else if ( am930q_length(mac->llcq) > 0 )
		{
			q = mac->llcq;
		}
		else if ( am930q_length(mac->macq) > 0 )
		{
			q = mac->macq;
		}

		if ( q != NULL )
		{
			pb = (wlan_pb_t*)am930q_dequeue(q);

			if ( pb != NULL )
			{
				txdone = am930hw_txdata( mac->hw, pb, mac->mgr->curr_rate);
				am930shim_pbfree(pb);
				pb = NULL;
			}
			else
			{
				txdone = 1;
			}
		}
		mac->nextq ^= 1;
	} while( q != NULL && txdone == 0 && i++ < 4);

#if (WLAN_OS == WLAN_LINUX_KERNEL)
	if ( txresult != 0xffffffff )
	{
		am930shim_ontxcomplete( mac->llc, txresult);
	}
#elif (WLAN_OS == WLAN_LWOS)
	am930shim_ontxcomplete( mac->llc, txresult);
#else
		#error "No WLAN_OS match!"
#endif


	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930mac_rxdata
*
*	Private helper for the rxframe method.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_rxdata( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats)
{
	wlan_pb_t *pb2;	/* a clone */

	DBFENTER;

	if ( mac->mode != AM930_MACMODE_ESS_AP )
	{
		am930shim_rxframe( mac->llc, pb);
	}
	else
	{
		int 					isme;
		int						isgroup;
		wlan_stable_item_t		*sta = NULL;

		isme = (memcmp(pb->p80211_hdr->a3.a3, mac->hw->addr, WLAN_ADDR_LEN) == 0);
		isgroup = (pb->p80211_hdr->a3.a3[0] & 0x01);

		/* Determine if we need two copies of the frame */
		if ( !isgroup )
		{
			/* It's a directed frame, we only need one copy.  That */
			/*  one copy will be sent either out the the WM, or up */
			/*  to the higher layers                               */
			/* TODO: We're making some assumptions about the hdr format here. */
			/*		 Needs review.	*/
			sta = am930mgr_stable_lookup( mac->mgr, pb->p80211_hdr->a3.a3);
			if ( sta != NULL )
			{
				if (!(sta->sta.info & WLAN_STA_INFO_AUTH) )
				{
					am930mgr_class2err(mac->mgr, pb->p80211_hdr->a3.a2);
					am930shim_pbfree(pb);
					kfree_s( stats, sizeof(am930rxstats_t));
					return;
				}
			}

			if ( sta != NULL )
			{
				if ( !(sta->sta.info & WLAN_STA_INFO_ASSOC) )
				{
					am930mgr_class3err(mac->mgr, pb->p80211_hdr->a3.a2);
					am930shim_pbfree(pb);
					kfree_s( stats, sizeof(am930rxstats_t));
					return;
				}
			}

			pb2 = pb;
		}
		else
		{
			/* It's a multi/broadcast frame, we need two copies. One  */
			/* for retransmission on the WM, one to hand up to higher */
			/* layers                                                 */
			/* clone the pb */
			pb2 = am930shim_pballoc();

			if ( pb2 != NULL )
			{
				am930shim_pballoc_p80211(pb2, pb->p80211buflen);

				if ( pb2->p80211hostbuf != NULL )
				{
					memcpy(pb2->p80211hostbuf, pb->p80211hostbuf, pb->p80211buflen);
					pb2->p80211frmlen = pb->p80211frmlen - WLAN_CRC_LEN;
					pb2->p80211_payloadlen = pb->p80211_payloadlen;
					pb2->wep_iscrypt = pb->wep_iscrypt;
					pb2->wep_iv = pb->wep_iv;
					pb2->wep_icv = pb->wep_icv;
				}
				else /* alloc failure, drop the frame */
				{
					am930shim_pbfree(pb2);
					am930shim_pbfree(pb);
					kfree_s( stats, sizeof(am930rxstats_t));
					return;
				}
			}
			else /* alloc failure, drop the frame */
			{
				am930shim_pbfree(pb);
				kfree_s( stats, sizeof(am930rxstats_t));
				return;
			}
		}

		/* If the frame needs to go back out on the WM, do it */
		if ( isgroup || sta != NULL )
		{
			UINT8	a[WLAN_ADDR_LEN];
			UINT16	fc;

			/* switch the addresses around and send it out */
			memcpy( a, pb2->p80211_hdr->a3.a1, WLAN_BSSID_LEN); 	/* save bssid */
			memcpy( pb2->p80211_hdr->a3.a1, pb2->p80211_hdr->a3.a3, WLAN_ADDR_LEN);	/* DA to a1 */
			memcpy( pb2->p80211_hdr->a3.a3, pb2->p80211_hdr->a3.a2, WLAN_ADDR_LEN); /* SA to a3 */
			memcpy( pb2->p80211_hdr->a3.a2, a, WLAN_BSSID_LEN);	/* bssid to a2 */
	
			fc = ieee2host16(pb2->p80211_hdr->a3.fc);
			fc &= ~(WLAN_SET_FC_TODS(1) |
					WLAN_SET_FC_RETRY(1) |
					WLAN_SET_FC_PWRMGT(1) );
			fc |=  WLAN_SET_FC_FROMDS(1);
			pb2->p80211_hdr->a3.fc = host2ieee16(fc);
		
			pb2->p80211_hdr->a3.dur = host2ieee16(0);


			/* check to see if we need to encrypt */
			if ( 
			((pb2->p80211_hdr->a3.a1[0] & 0x01) && mac->exclude_unencrypted))
			{
					am930mac_wep_encrypt(mac, pb2);
			}
			else if ( sta != NULL )
			{
				if ( ( sta->sta.info & WLAN_STA_INFO_WEPON) || 
					mac->exclude_unencrypted)
				{
						am930mac_wep_encrypt(mac, pb2);
				}
			}

			/* enqueue the frame for delivery */
			if ( am930mac_txmac( mac, pb2) > 1 ) /* if not queued */
			{
				am930shim_pbfree(pb2);
			}
		}

		/* If the frame needs to go up to higher layers, do that.  In the */
		/*  case of multi/broadcast frames, both will be done.            */
		if( isme || isgroup || sta == NULL )
		{
			am930shim_rxframe( mac->llc, pb);
		}
	}
	kfree_s( stats, sizeof(am930rxstats_t));
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mac_rxenqueue
*
*	Enqueue a received frame for later processing.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_rxenqueue( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats)
{
	am930mac_rxitem_t	*item;

	DBFENTER;

	item = kmalloc(sizeof(am930mac_rxitem_t), GFP_ATOMIC);
	if ( item != NULL )
	{
		item->f = pb;
		item->s = stats;
		if ( am930q_enqueue( mac->rxq, item) >= 2 )
		{
			/* drop the frame */
			am930shim_pbfree( pb );
			am930shim_free(stats, sizeof(am930rxstats_t));
			kfree_s(item, sizeof(am930mac_rxitem_t));
		}
		else
		{
			#if (WLAN_OS == WLAN_LWOS )
				am930lw_readerkick();
			#endif
		}
	}
	else
	{
		/* drop the frame */
		am930shim_pbfree( pb );
		am930shim_free(stats, sizeof(am930rxstats_t));
	}

	
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mac_rxframe
*
*	Public method called when a frame has been received at a
*   lower layer and needs to be handled by this layer.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_rxframe( am930mac_t *mac)
{
	UINT32				ftype;
	wlan_pb_t			*pb;
	am930rxstats_t		*stats;
	am930mac_rxitem_t	*item;
	DBFENTER;

	while (((item) = am930q_dequeue(mac->rxq)) != NULL)
	{
		pb = item->f;
		stats = item->s;
		kfree_s( item, sizeof(*item));
		
		ftype = WLAN_GET_FC_FTYPE(ieee2host16(pb->p80211_hdr->a3.fc));
		if ( ftype != WLAN_FTYPE_MGMT && ftype != WLAN_FTYPE_DATA )
		{
			/* we've got a frame we shouldn't...drop it */
			WLAN_LOG_WARNING0("Ctl or unknown frame type received, dropped\n"); 
			kfree_s( stats, sizeof(am930rxstats_t));
			am930shim_pbfree(pb);
		}

		/* first, give the frag filter a chance */
		pb = am930mac_defrag(mac, pb);

		if ( pb == NULL )
		{
			/* the frame was absorbed by a fragment pool, do no more */
			kfree_s( stats, sizeof(am930rxstats_t));
		}
		else
		{
			/* OK, we still have the frame, give the crypto a chance */
			pb = am930mac_wep_decrypt(mac, pb);
			if ( pb == NULL )
			{
				/* the frame failed decryption somehow, do no more */
				kfree_s( stats, sizeof(am930rxstats_t));
			}
			else
			{
				switch( ftype )
				{
					case WLAN_FTYPE_MGMT:
						am930mac_rxmgmt( mac, pb, stats);
						break;
					case WLAN_FTYPE_DATA:
						am930mac_rxdata( mac, pb, stats);
						break;
				}
			}
		}
	}
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mac_defragfinish
*
*	Takes a fragment pool, validates it and assembles the MSDU
*	into a new pb buffer.  If the fragments are encrypted, each
*	fragment is decrypted as it's being added to the assembly.
*
*	When defragmentation is complete, we discard the pb's for
*	the fragments and mark the fragpool as free.
*
*	Arguments:
*		mac		- the receiver
*		hash	- index of the fragpool to assemble
*	
*	returns: 
*	<a new pb>	- if the fragpool was succesfully reassembled
*		NULL	- The fragpool was missing fragments, or the decryption
*					of a fragment failed.
----------------------------------------------------------------*/
wlan_pb_t *am930mac_defragfinish( am930mac_t *mac, UINT8 hash )
{
	wlan_pb_t	*pb = NULL;
	UINT32		nfrag = 0;
	UINT32		iscrypt = 0;
	UINT32  	seenall = 0;
	UINT32  	offset = 0;
	UINT32  	isbad = 0;
	UINT32		i;
	UINT32		len = WLAN_HDR_A3_LEN + WLAN_CRC_LEN;
					/* calcs elsewhere assume CRC is present, so we need it */

	DBFENTER;

	/* add a lot to the lifetime counter to prevent removal */
	mac->defragpool[hash].lifetime += 100000UL;

	/* validate the fragment pool */
	for (i=0; i < WLAN_FRAGS_MAX && !seenall; i++)
	{
		pb = mac->defragpool[hash].pb[i]; 
		if ( pb == NULL )
		{
			/* Missing fragments. It's bad */
			seenall = 1;
			isbad = 1;
			WLAN_LOG_DEBUG0(1, "defragfinish, missing frags, dropped .\n");
		}
		else
		{
			if ( i == 0 && WLAN_GET_FC_ISWEP(ieee2host16(pb->p80211_hdr->a3.fc)) )
			{
				/* If the first frag is crypt, they must all be */
				iscrypt = 1;
			}

			if ( WLAN_GET_FC_ISWEP(ieee2host16(pb->p80211_hdr->a3.fc)) && !iscrypt )
			{
				/* mix of crypt and non-crypt. It's bad */
				seenall = 1;
				isbad = 1;
				WLAN_LOG_DEBUG0(1, "defragfinish, not all crypt, dropped .\n");
			}

			if ( WLAN_GET_FC_MOREFRAG(ieee2host16(pb->p80211_hdr->a3.fc)) == 0 )
			{
				/* Ok, this is the last one. As long as isbad is clear */
				/*  we're on our way */
				seenall = 1;
			}
			nfrag++;
			len += pb->p80211_payloadlen;

		}
	}

	if ( !isbad )
	{
		/* validation was successful, decrypt and defrag */
		#if (WLAN_OS == WLAN_LINUX_KERNEL)
			pb = am930llc_pballoc();
		#elif (WLAN_OS == WLAN_LWOS)
			pb = am930lw_pballoc();
		#else
			#error "No WLAN_OS match!"
		#endif
		if ( pb == NULL )
		{
			WLAN_LOG_DEBUG0(2,"pballoc failed.\n");
		}
		else
		{
			#if (WLAN_OS == WLAN_LINUX_KERNEL)
				am930llc_pballoc_p80211(pb, len);
			#elif (WLAN_OS == WLAN_LWOS)
				am930lw_pballoc_p80211(pb, len);
			#else
				#error "No WLAN_OS match!"
			#endif
			if ( pb->p80211hostbuf == NULL )
			{
				WLAN_LOG_DEBUG0(2,"pballoc_p80211 failed.\n");
				#if (WLAN_OS == WLAN_LINUX_KERNEL)
					am930llc_pbfree(pb);
				#elif (WLAN_OS == WLAN_LWOS)
					am930lw_pbfree(pb);
				#else
					#error "No WLAN_OS match!"
				#endif
			}
			else
			{
				/* copy the header from the first frame and clear MoreFrag */
				memcpy( 
					pb->p80211buf, 
					mac->defragpool[hash].pb[0]->p80211hostbuf,
					WLAN_HDR_A3_LEN);
				pb->p80211_hdr->a3.fc &= host2ieee16(~WLAN_SET_FC_MOREFRAG(1));

				offset = 0;		/* current offset into data portion of frame */

				/* now, copy  the data from each fragment */
				for( i = 0; i < nfrag && !isbad; i++)
				{
					if ( iscrypt )
					{
						mac->defragpool[hash].pb[i] = 
							am930mac_wep_decrypt(mac, mac->defragpool[hash].pb[i]);
					}
					
					if ( mac->defragpool[hash].pb[i] == NULL )
					{
						/* decryption failure */
						isbad = 1;
						WLAN_LOG_DEBUG0(1, "defragfinish, decrypt fail, dropped .\n");
					}
					else
					{
						/* we're ok, move the data */
						memcpy( 
							pb->p80211_payload + offset, 
							mac->defragpool[hash].pb[i]->p80211_payload,
							mac->defragpool[hash].pb[i]->p80211_payloadlen );
		
						offset += 
							mac->defragpool[hash].pb[i]->p80211_payloadlen;
					}
					
				}
			}
		}
	}

	/* deallocate the fragpool */
	for ( i = 0; i < WLAN_FRAGS_MAX; i++)
	{
		if ( mac->defragpool[hash].pb[i] != NULL )
		{
			#if (WLAN_OS == WLAN_LINUX_KERNEL)
				am930llc_pbfree(mac->defragpool[hash].pb[i]);
			#elif (WLAN_OS == WLAN_LWOS)
				am930lw_pbfree(mac->defragpool[hash].pb[i]);
			#else
				#error "No WLAN_OS match!"
			#endif
		}
		mac->defragpool[hash].pb[i] = NULL;
	}
	mac->defragpool[hash].lifetime = 0;
	mac->defragpool[hash].seq = UINT16_MAX;

	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mac_defrag
*
*	Checks a frame to see if it's part of a fragmented MSDU.  If
*	so, then it puts it into a defragpool and returns NULL.  
*	If the MoreFrag field is clear, then we invoke the 
*	defragmentation process build a new pb and return that to
*	the caller.  If the frame doesn't need any kind of defrag
*	behavior, we just return it to caller.
*
*	Arguments:
*		mac		- the receiver
*		pb		- packet buffer containing the frame
*	
*	returns: 
*		pb		- if the frame doesn't need any defrag handling
*	<a new pb>	- if the frame passed in was the last fragment and we
*					successfully defragmented.
*		NULL	- The frame passed in was consumed by the defrag code
*					because:
*					- the frame passed in is a middle fragment and we
*						have no fragment pool
*					- the frame was the last fragment in a series 
*						and we failed to finish the defragmentation 
*						due to missing fragments or decryption failure.
*					- the frame was successfully placed in a fragment
*						pool.
*	NOTE:
*	The 802.11 SDL implies that an 802.11 STA associated with an
*	AP will always and only receive frames from the AP and that
*	the AP is responsible for generating any fragments as part of a
*	_single_ MA-UNITDATA.request .  Therefore
*	it should be impossible for the STA MAC to have to deal w/ multiple
*	fragmented frames simultaneously.  To cover possible implementation
*	differences, we're still going to keep track of (potentially) 
*	multiple fragment pools identified by SRC address and sequence
*	number.  In all other cases, multiple pools are required.
----------------------------------------------------------------*/
wlan_pb_t *am930mac_defrag( am930mac_t *mac, wlan_pb_t *pb)
{
	UINT32			hash = 0;
	UINT8			*srcaddr = NULL;
	UINT32			morefrag;
	UINT32			seq;
	UINT32			fragnum;
	wlan_flags_t	flags;
	UINT32			i;

	DBFENTER;

	/*  collect the fragpool table index */
	switch(mac->mode)
	{
		case AM930_MACMODE_ESS_AP:		/* source addr is associated STA */
		case AM930_MACMODE_IBSS_STA:	/* source addr is peer STA */ 
			hash = WLAN_MACADDR_HASH8(pb->p80211_hdr->a3.a2) & AM930MAC_FRAGPOOL_MASK;
			srcaddr = pb->p80211_hdr->a3.a2;
			break;

		/* TODO: consider wisdom of using SA */
		case AM930_MACMODE_ESS_STA:		/* source addr is !AP, use SA */
			hash = WLAN_MACADDR_HASH8(pb->p80211_hdr->a3.a3) & AM930MAC_FRAGPOOL_MASK;
			srcaddr = pb->p80211_hdr->a3.a3;
			break;
	}

	morefrag = WLAN_GET_FC_MOREFRAG( ieee2host16(pb->p80211_hdr->a3.fc)); 
	seq = WLAN_GET_SEQ_SEQNUM(ieee2host16(pb->p80211_hdr->a3.seq));
	fragnum = WLAN_GET_SEQ_FRGNUM(ieee2host16(pb->p80211_hdr->a3.seq));
	WLAN_LOG_DEBUG3(3, "defrag more=%ld seq=%ld fnum=%ld\n", morefrag, seq, fragnum);
	
	if ( (morefrag == 0) && (fragnum != 0) )
	{
		/* It's the final in a fragment sequence */
		/*  This next comparison relies on the fact that it is _extremely_ */
		/*  unlikely that two MAC addresses that hash the same will be     */
		/*  using the same sequence # at the same time. If it occurs,      */
		/*  we'll nonfatally trash one MSDU from each STA.                 */


		spin_lock_irqsave( &mac->fragpool_lock, flags);
		if ( mac->defragpool[hash].seq == seq )
		{
			/* Ok, it's the last in a sequence that we're keeping */
			mac->defragpool[hash].pb[fragnum] = pb; /* pb owned by pool now */
			mac->defragpool[hash].seq = AM930MAC_FRAGPOOL_COMPLETE;
			spin_unlock_irqrestore( &mac->fragpool_lock, flags);
			pb = am930mac_defragfinish( mac, hash); /* also deallocs pool */

			/* pb is either defragged frame in a new pb, or NULL if defrag */
			/*  failed */
		}
		else
		{
			spin_unlock_irqrestore( &mac->fragpool_lock, flags);
			/* Either we've a hash collision or there's something bogus */
			/*  about our fragpools. Drop the frame. */
			#if (WLAN_OS == WLAN_LINUX_KERNEL)
				am930llc_pbfree(pb);
			#elif (WLAN_OS == WLAN_LWOS)
				am930lw_pbfree(pb);
			#else
				#error "No WLAN_OS match!"
			#endif
			pb = NULL;
			WLAN_LOG_DEBUG0(1, "defrag final mismatch, dropped.\n");
		}
	}
	else if ( morefrag != 0 )
	{
		if ( fragnum == 0 )
		{
			/* is fragpool avail? */
			spin_lock_irqsave( &mac->fragpool_lock, flags);
			if ( 0 == 
				memcmp(srcaddr, mac->defragpool[hash].srcaddr, WLAN_ADDR_LEN) &&
				seq != mac->defragpool[hash].seq )
			{
				/* it's the same STA, it's given up on prior frame */
				/*  we can reclaim this fragpool */
				mac->defragpool[hash].seq = AM930MAC_FRAGPOOL_COMPLETE;
				for ( i = 0; i < WLAN_FRAGS_MAX; i++)
				{
					if ( mac->defragpool[hash].pb[i] != NULL )
					{
						#if (WLAN_OS == WLAN_LINUX_KERNEL)
							am930llc_pbfree(mac->defragpool[hash].pb[i]);
						#elif (WLAN_OS == WLAN_LWOS)
							am930lw_pbfree(mac->defragpool[hash].pb[i]);
						#else
							#error "No WLAN_OS match!"
						#endif
					}
					mac->defragpool[hash].pb[i] = NULL;
				}
				mac->defragpool[hash].lifetime = 0;
				mac->defragpool[hash].seq = AM930MAC_FRAGPOOL_FREE;
			}

			if ( mac->defragpool[hash].seq == AM930MAC_FRAGPOOL_FREE )
			{
				/* empty slot, create new pool */
				mac->defragpool[hash].seq = seq;
				memcpy( mac->defragpool[hash].srcaddr, srcaddr, WLAN_ADDR_LEN);
				mac->defragpool[hash].lifetime = 
					jiffies + wlan_tu2ticks(mac->maxrxlifetime);
				mac->defragpool[hash].pb[fragnum] = pb;	/* fragpool owns pb */
				spin_unlock_irqrestore( &mac->fragpool_lock, flags);
				pb = NULL;		/* tell caller we've consumed it */
			}
			else
			{
				/* collision on hash, frag sequence in progress:      */
				/*   from same STA (impossible?), or                  */
				/*   frame DUP from same STA (f/w should prevent), or */
				/*   from hash colliding STA.  Drop frame.            */
				/* check to see if it's same STA */
				spin_unlock_irqrestore( &mac->fragpool_lock, flags);
				#if (WLAN_OS == WLAN_LINUX_KERNEL)
				am930llc_pbfree(pb);
				#elif (WLAN_OS == WLAN_LWOS)
				am930lw_pbfree(pb);
				#else
				#error "No WLAN_OS match!"
				#endif
				pb = NULL;
				WLAN_LOG_DEBUG0(1, "defrag initial mismatch, dropped.\n");
			}
		}
		else	/* fragnum != 0 */
		{
			spin_lock_irqsave( &mac->fragpool_lock, flags);
			if ( mac->defragpool[hash].seq == seq &&
				 mac->defragpool[hash].pb[fragnum] == NULL )
			{
				mac->defragpool[hash].pb[fragnum] = pb;
				spin_unlock_irqrestore( &mac->fragpool_lock, flags);
				pb = NULL;
			}
			else
			{
				/* collision on hash, frag sequence in progress:      */
				/*   from same STA (diff seq, impossible?), or        */
				/*   from hash colliding STA.  Drop frame.            */
				spin_unlock_irqrestore( &mac->fragpool_lock, flags);
				#if (WLAN_OS == WLAN_LINUX_KERNEL)
					am930llc_pbfree(pb);
				#elif (WLAN_OS == WLAN_LWOS)
					am930lw_pbfree(pb);
				#else
					#error "No WLAN_OS match!"
				#endif
				pb = NULL;
				WLAN_LOG_DEBUG0(1, "defrag intermediate mismatch, dropped.\n");
			}
		}
	}
	/* else morefrag==0 && fragnum==0, just return pb */

	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mac_rxframe_err
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_rxframe_err(am930mac_t *mac)
{
#if (WLAN_OS == WLAN_LINUX_KERNEL)
	am930llc_rxframe_err(mac->llc);
#elif (WLAN_OS == WLAN_LWOS)
	am930lw_rxframe_err( mac->lw);
#else
	#error "No WLAN_OS match!"
#endif
	return;
}


/*----------------------------------------------------------------
*	am930mac_rxmgmt
*
*	Private helper for the rxframe method.  Checks to make sure
*	the received frame is actually directed to us (or is multi/broadcast)
*	before handing the frame off to the mgr component.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_rxmgmt( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats)
{
	int		notme = 0;

	DBFENTER;

	WLAN_LOG_DEBUG0(3, "mgmt frm received\n");

	/* Make sure this frame is really for me in case we're in promiscuous mode */
	if ( (pb->p80211_hdr->a3.a1[0] & 0x01) == 0 ) /* is multicast? */
		notme = memcmp( pb->p80211_hdr->a3.a1, mac->hw->addr, WLAN_ADDR_LEN);

	if ( notme )
	{
		/* mgmt frame with DA that's not me...silently drop it */
		am930shim_pbfree( pb );
		kfree_s(stats, sizeof(am930rxstats_t));
	}
	else
	{
		am930mgr_rxmgmt( mac->mgr, pb, stats);
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mac_timerfunc
*
*	Called 10 times per second by llc.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_timerfunc(am930mac_t *mac)
{
	int 			i;
	int 			j;
	wlan_flags_t	flags;

	for ( i = 0; i < AM930MAC_FRAGPOOLS_MAX; i++)
	{
		/* lock the whole table */
		spin_lock_irqsave( &mac->fragpool_lock, flags);
		if ( mac->defragpool[i].seq != AM930MAC_FRAGPOOL_FREE && 
			mac->defragpool[i].seq != AM930MAC_FRAGPOOL_COMPLETE && 
			time_after_eq( jiffies, mac->defragpool[i].lifetime) )
		{
			/* mark it as complete so noone messes with it */
			mac->defragpool[i].seq = AM930MAC_FRAGPOOL_COMPLETE;
			for ( j=0; j < WLAN_FRAGS_MAX; j++)
			{
				if ( mac->defragpool[i].pb[j] != NULL )
				{
				#if (WLAN_OS == WLAN_LINUX_KERNEL)
					am930llc_pbfree(mac->defragpool[i].pb[j]);
				#elif (WLAN_OS == WLAN_LWOS)
					am930lw_pbfree(mac->defragpool[i].pb[j]);
				#else
					#error "No WLAN_OS match!"
				#endif
				}
				mac->defragpool[i].pb[j] = NULL;
			}
			mac->defragpool[i].seq = AM930MAC_FRAGPOOL_FREE;
		}
		spin_unlock_irqrestore( &mac->fragpool_lock, flags);
	}
}


/*----------------------------------------------------------------
*	am930mac_txllc
*
*	Public method called from the llc object to send a frame. This 
*	function takes a pb and addresses, formats the 802.11 header
*	and then enqueues the frame.  The frame will be transmitted in
*	response to some future tx complete interrupt.
*
*	returns: 
*		0 success 
*		1 alloc failure
*		2 queue full
*		3 queue busy
----------------------------------------------------------------*/
UINT32 am930mac_txllc( am930mac_t *mac, UINT8 *daddr, UINT8 *saddr, 
						wlan_pb_t *pb)
{
	UINT32				result = 0;
	wlan_stable_item_t	*sta;

	DBFENTER;

	if ( mac->mode == AM930_MACMODE_NOTRUNNING ||
		 mac->mode == AM930_MACMODE_NOTJOINED )
	{
		WLAN_LOG_WARNING0("tx attempt, not joined, frame dropped\n");
		return result;
	}

	switch( mac->mode )
	{
		case AM930_MACMODE_IBSS_STA:
			/* copy the addresses to the 802.11 header */
			memcpy( pb->p80211_hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( pb->p80211_hdr->a3.a2, saddr, WLAN_ADDR_LEN);
			memcpy( pb->p80211_hdr->a3.a3, mac->mgr->curr_bssid, WLAN_BSSID_LEN);
	
			/* set the rest of the 802.11 header */
			pb->p80211_hdr->a3.fc = host2ieee16( 
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_DATA ) |
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_DATAONLY) );
			pb->p80211_hdr->a3.dur = 0;
			pb->p80211_hdr->a3.seq = 0;

			/* let the cryptor determine the need */
			am930mac_wep_encrypt( mac, pb );
			break;

		case AM930_MACMODE_ESS_STA:
			/* copy the addresses to the 802.11 header */
			memcpy( pb->p80211_hdr->a3.a1, mac->mgr->curr_bssid, WLAN_BSSID_LEN);
			memcpy( pb->p80211_hdr->a3.a2, saddr, WLAN_ADDR_LEN);
			memcpy( pb->p80211_hdr->a3.a3, daddr, WLAN_ADDR_LEN);
	
			/* set the rest of the 802.11 header */
			pb->p80211_hdr->a3.fc = host2ieee16( 
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_DATA ) |
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_DATAONLY) |
				WLAN_SET_FC_TODS(1) );
			pb->p80211_hdr->a3.dur = 0;
			pb->p80211_hdr->a3.seq = 0;

			/* let the cryptor determine the need */
			am930mac_wep_encrypt( mac, pb );
			break;

		case AM930_MACMODE_ESS_AP:
			/* Check that we want to tx this frame */
			sta = am930mgr_stable_lookup( mac->mgr, daddr);
			if ( (daddr[0] & 0x01) || sta != NULL)
			{
				/* copy the addresses to the 802.11 header */
				memcpy( pb->p80211_hdr->a3.a1, daddr, WLAN_ADDR_LEN);
				memcpy( pb->p80211_hdr->a3.a2, mac->mgr->curr_bssid, WLAN_BSSID_LEN);
				memcpy( pb->p80211_hdr->a3.a3, saddr, WLAN_ADDR_LEN);
	
				/* set the rest of the 802.11 header */
				pb->p80211_hdr->a3.fc = host2ieee16( 
					WLAN_SET_FC_FTYPE(WLAN_FTYPE_DATA ) |
					WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_DATAONLY) |
					WLAN_SET_FC_FROMDS(1) );
				pb->p80211_hdr->a3.dur = 0;
				pb->p80211_hdr->a3.seq = 0;
				
				if ( ((daddr[0] & 0x01) && mac->exclude_unencrypted) || /* ismulti */
					 ( sta != NULL &&  /* is directed (relying on short-circuit) */
					  	( (sta->sta.info & WLAN_STA_INFO_WEPON) || /* sta uses wep */
						 mac->exclude_unencrypted)) )               /* MUST use wep */
				{
					/* let the cryptor determine the need */
					am930mac_wep_encrypt(mac, pb);
				}
			}
			else
			{
				/* silently drop it, (the tx would fail anyway) */
				am930shim_pbfree(pb);
				return 0;
			}
			break;
	}

	/* Enqueue for transmission */
	result = am930q_enqueue( mac->llcq, pb);
	
	am930hw_txkick(mac->hw);

	#if (WLAN_OS == WLAN_LWOS)
		am930lw_readerkick();
	#endif

	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930mac_txmac
*
*	Public method called from mac and mgr to send a frame.  In this
*	case, the 80211 header has already been set up.  All we need
*	to do is queue the frame.
*
*	returns: 
*		0 success 
*		1 alloc failure
*		2 queue full
*		3 queue busy
----------------------------------------------------------------*/
UINT32 am930mac_txmac( am930mac_t *mac, wlan_pb_t *pb)
{
	UINT32			result = 0;

	DBFENTER;

	if ( mac->mode == AM930_MACMODE_NOTRUNNING ||
		 mac->mode == AM930_MACMODE_NOTJOINED )
	{
		WLAN_LOG_WARNING0("tx attempt, not joined, frame dropped\n");
		return result;
	}

	result = am930q_enqueue( mac->macq, pb);

	#if (WLAN_OS == WLAN_LINUX_KERNEL )
		am930hw_txkick(mac->hw);
	#elif (WLAN_OS == WLAN_LWOS )
		am930lw_readerkick();
	#else
		#error  "No WLAN_OS match!"
	#endif


	DBFEXIT;
	return result;
}

/*----------------------------------------------------------------
*	am930q_init
*
*	Initilizes a new queue.  To get rid of it, the caller should 
*	empty it and then use kfree_s(q, sizeof(am930q_t)).
*	
*	returns:
*		NULL - alloc failure
*		else - success
----------------------------------------------------------------*/
am930q_t *am930q_init(UINT maxlen)
{
	am930q_t *q = NULL;
	q = kmalloc(sizeof(am930q_t), GFP_ATOMIC);
	if ( q != NULL )
	{
		memset(q, 0, sizeof(am930q_t));
		q->maxlen = maxlen;
	}
	return q;
}


/*----------------------------------------------------------------
*	am930q_enqueue
*
*	Adds an item to the given queue if it won't exceed the max
*	length.
*
*	returns:
*		0 success, more space avail 
*		1 success, no more space
*		2 alloc failure
*		3 queue full
*		4 queue busy
----------------------------------------------------------------*/
int am930q_enqueue(am930q_t *q, void *data)
{
	int				result = 0;
	wlan_flags_t	flags;
	am930qitem_t	*qitem;

	WLAN_INT_DISABLE(flags);
	
	/* TODO: need safe lock here */
	if ( q->busy == 0 )
	{
		q->busy = 1;

		if ( q->len < q->maxlen )
		{
			qitem = kmalloc(sizeof(am930qitem_t), GFP_ATOMIC);
			if ( qitem != NULL )
			{
				memset(qitem, 0, sizeof(am930qitem_t));
				qitem->data = data;
				qitem->q = q;
				qitem->prev = q->tail;
				if (q->tail != NULL)
				{
					q->tail->next = qitem;
					q->tail = qitem;
				}
				else
				{
					q->head = q->tail = qitem;
				}
				q->len++;
				result = ((q->len < q->maxlen) ? 0 : 1);
			}
			else
			{
				result = 2;
			}
		}
		else
		{
			result = 3;
		}
		q->busy = 0;
		WLAN_INT_ENABLE(flags);
	}
	else
	{
		WLAN_INT_ENABLE(flags);

		WLAN_LOG_DEBUG0(2, "queue is busy\n");
		result = 4;
	}
	return result;
}


/*----------------------------------------------------------------
*	am930q_dequeue
*
*	Retrieves an item from the front of the queue.
*	
*	returns:
*		NULL - queue empty or busy
*		else - success
----------------------------------------------------------------*/
void *am930q_dequeue(am930q_t *q)
{
	void			*data = NULL;
	am930qitem_t	*qitem;
	wlan_flags_t	flags;

	WLAN_INT_DISABLE(flags);

	if ( q->busy == 0 )
	{
		q->busy = 1;


		if ( q->len > 0 && q->head != NULL)
		{
			data = q->head->data;
			qitem = q->head;
			if ( q->head != q->tail )
			{
				q->head = qitem->next;
				q->head->prev = NULL;
			}
			else
			{
				q->head = q->tail = NULL;
			}
			kfree_s(qitem, sizeof(am930qitem_t));
			q->len--;
		}
		q->busy = 0;
		WLAN_INT_ENABLE(flags);
	}
	else
	{
		WLAN_INT_ENABLE(flags);
		WLAN_LOG_DEBUG0(2, "queue is busy\n");
	}
	return data;
}


/*----------------------------------------------------------------
*	am930q_requeue
*
*	TODO: add code to reinsert an item into the front of a queue.
*			Don't test maxlen.
*	returns:
*		0 success 
*		1 alloc failure
----------------------------------------------------------------*/
/*
int am930q_requeue(am930q_t *q, void *data)
{
}
*/


/*----------------------------------------------------------------
*	am930q_peek
*
*	TODO: add a function that will return the data ptr from the
*		front of the queue without removing the item.
*	
*	returns:
*		NULL - queue empty
*		else - success
----------------------------------------------------------------*/
/*
void *am930q_peek(am930q_t *q, void *data)
{
	return
		NULL - queue empty
		else - success
	
}
*/


#ifdef WLAN_INCLUDE_WEP

	#include "wepcode.c.inc"

#else
/*----------------------------------------------------------------
*	am930mac_wep_initprng
*
*	Initializes the WEP Psuedo-Random number generator using a
*	given seed.  The seed is usually the private key.
*	Arguments:
*		prng	- a structure, allocated by the caller, that will 
*				maintain state information during the use of this
*				prng.
*		k		- an array of bytes containing the seed
*		klen	- length of seed
*	
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_wep_initprng( wlan_wepprng_t *prng, UINT8 *k,  UINT klen)
{
}


/*----------------------------------------------------------------
*	am930mac_wep_nextprn
*
*	Retrieves the next value from a prng initialized with
*	wep_prng.
*	Arguments:
*		prng	- a prng structure previosly initialized with
*				wep_initprng.
*	
*	returns: next psuedo-random number
----------------------------------------------------------------*/
UINT8	am930mac_wep_nextprn( wlan_wepprng_t *prng)
{
	return 0;
}


/*----------------------------------------------------------------
*	am930mac_wep_block
*
*	WEP encrypts a block of bytes in place.
*	Arguments:
*		prng	- a prng structure previosly initialized with
*				wep_initprng.
*		d		- ptr to the block of bytes to be encrypted.
*		dlen	- length of the block
*	
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_wep_block( wlan_wepprng_t *prng, UINT8 *d, UINT dlen)
{
	return;
}


/*----------------------------------------------------------------
*	am930mac_wep_decrypt
*
*	Inspects the given frame, if it doesn't require decryption
*	just return it.  If the frame does require decryption, check
*	to see that we set up for decryption.  If we're not set up
*	for decryption, drop the frame and return NULL.  If the frame
*	needs decryption _and_ we support and are configured for it,
*	decrypt the frame inside the given pb and return it to the 
*	caller.
*	Arguments:
*		mac		- the receiver
*		pb		- packet buffer containing the frame
*	
*	returns: 
*		pb		- if the frame doesn't need decryption or the frame
*					is successfully decrypted
*		NULL	- if the frame requires decryption and:
*					- we don't support it 
*					- we haven't been configured for it. 
*						(i.e. mac->privacy_invoked == 0, no keys set, etc.)
*					- the ICV check failed.
----------------------------------------------------------------*/
wlan_pb_t *am930mac_wep_decrypt( am930mac_t *mac, wlan_pb_t *pb)
{
	return pb;
}


/*----------------------------------------------------------------
*	am930mac_wep_encrypt
*
*	Takes the given frame, and if privacy_invoked==true, encrypts
*	it.
*	Arguments:
*		mac		- the receiver
*		pb		- packet buffer containing the frame
*	
*	returns: nothing
*
*	note: after the call, the pb->iscrypt field indicates whether
*			the frame was successfully encrypted.
----------------------------------------------------------------*/
void am930mac_wep_encrypt( am930mac_t *mac, wlan_pb_t *pb)
{
	return;
}

#endif /* WLAN_INCLUDE_WEP */

#ifdef WLAN_INCLUDE_SNIF
/*----------------------------------------------------------------
*	am930mac_snifframe
*
*	Takes the given frame tacks on a sniffer_t to the front and
*	sends it up to the LLC layer.
*
*	Arguments:
*		mac		- the receiver
*		pb		- packet buffer containing the frame
*		stats	- info about the frame
*	
*	returns: nothing
----------------------------------------------------------------*/
void am930mac_snifframe(am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats)
{
	wlan_sniffer_t	snif;
	struct sk_buff	*skb;

	snif.time = jiffies;

	/* determine the pb type and size */
	if ( stats != NULL )
	{
		snif.rssi = stats->rssi;
		snif.ch = stats->ch;
		snif.rate = stats->rate;
		snif.istx = 0;
		/* it's an rx frame, the whole thing is in p80211buf */
		snif.len = pb->p80211frmlen;

	}
	else
	{
		snif.rssi = 0;
		snif.ch = 0;
		snif.rate = 0;
		snif.istx = 1;
		/* note: tx frames, we add 4 for the missing FCS */
		if ( pb->ethhostbuf != NULL )
		{
			/* it's a tx frame, that's been converted from ether */
			snif.len = pb->p80211buflen + pb->p80211_payloadlen + 4;
		}
		else
		{
			/* it's a tx frame, that that has everything in the p80211buf */
			/*  Note the FCS fix again... */
			snif.len = pb->p80211frmlen + WLAN_FCS_LEN;
		}
	}

	if ( pb->wep_iscrypt )
	{
		snif.len += WLAN_WEP_IV_LEN + WLAN_WEP_ICV_LEN;
	}

	/* allocate an skb for sniffer_t + frame */
	skb = alloc_skb(snif.len + sizeof(snif), GFP_ATOMIC);
	skb_put(skb, snif.len + sizeof(snif));

	/* fill skb */
	memcpy( skb->data, &snif, sizeof(snif));
	if ( !snif.istx )
	{
		/* It's an rx frame, all in p80211buf (except iv and icv) */
		UINT8	*src = pb->p80211buf;
		UINT8	*dst = skb->data + sizeof(snif);

		memcpy( dst, src, WLAN_HDR_A3_LEN);
		dst += WLAN_HDR_A3_LEN;
		src += WLAN_HDR_A3_LEN;

		if ( pb->wep_iscrypt )
		{
			memcpy( dst, &(pb->wep_iv), WLAN_WEP_IV_LEN);
			dst += WLAN_WEP_IV_LEN;
		}

		memcpy( dst, src, pb->p80211frmlen - WLAN_HDR_A3_LEN - WLAN_FCS_LEN);
		dst += pb->p80211frmlen - WLAN_HDR_A3_LEN - WLAN_FCS_LEN;
		src += pb->p80211frmlen - WLAN_HDR_A3_LEN - WLAN_FCS_LEN;

		if ( pb->wep_iscrypt )
		{
			memcpy( dst, &(pb->wep_icv), WLAN_WEP_ICV_LEN);
			dst += WLAN_WEP_ICV_LEN;
		}

		memcpy( dst, src, WLAN_FCS_LEN);
	}
	else
	{
		/* We don't support TX sniffing right now */
		#ifdef NOTUSED
		if ( pb->ethhostbuf != NULL )
		{
			memcpy( skb->data + sizeof(snif), pb->p80211buf, pb->p80211buflen);
			memcpy( skb->data + sizeof(snif) + pb->p80211buflen,
						pb->p80211_payload, pb->p80211_payloadlen);
			memset( 
				skb->data + sizeof(snif) + pb->p80211buflen + pb->p80211_payloadlen,
				0,
				WLAN_CRC_LEN);
		}
		else
		{
			memcpy( skb->data + sizeof(snif), pb->p80211buf, snif.len - WLAN_CRC_LEN);
			memset( skb->data + sizeof(snif) + snif.len - WLAN_CRC_LEN, 0, WLAN_CRC_LEN);
		}
		#endif
	}

	/* call llc */
	am930llc_sendtosniffer(mac->llc, skb);
	return;
}
#endif /* WLAN_INCLUDE_SNIF */




