/* am930mac.h: Handles the 802.11 mac functions
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*
*	--------------------------------------------------------------------
*/

#ifndef _LINUX_AM930MAC_H
#define _LINUX_AM930MAC_H

/*=============================================================*/
/*--- Constants -----------------------------------------------*/
/*=============================================================*/

#define WLAN_WEPMODE_OFF		3	/* sta & ap: no wep used at all */
#define WLAN_WEPMODE_WEPONLY	1	/* sta & ap: only communicate with */
									/* wep enabled sta */
#define WLAN_WEPMODE_BOTH		2	/* ap: pass wired frames to all stations, */
									/* forward between stations with */
									/* matching wep settings */

#define WLAN_WEP_FULLKEYLEN		8
#define WLAN_WEP_SBOXLEN		256
#define WLAN_WEP_SWAP(a,b)		{ UINT8 c; c = (a); (a) = (b); (b) = (c); }

#define AM930MAC_FRAGPOOLS_MAX	256
#define AM930MAC_FRAGPOOL_MASK	0xff

#define AM930MAC_FRAGPOOL_FREE		0xffff
#define AM930MAC_FRAGPOOL_COMPLETE	0xf0ff

/* Primary MAC modes */
#define AM930_MACMODE_NOTRUNNING		0
#define AM930_MACMODE_NOTJOINED			1
#define AM930_MACMODE_IBSS_STA			2
#define AM930_MACMODE_ESS_STA			3
#define AM930_MACMODE_ESS_AP			4

/* Active MAC states */
#define AM930_MACSTATE_NOAUTH			0
#define AM930_MACSTATE_AUTHPENDING		1
#define AM930_MACSTATE_AUTH				2
#define AM930_MACSTATE_ASSOCPENDING		3
#define AM930_MACSTATE_ASSOC			4


/*=============================================================*/
/*--- Types ---------------------------------------------------*/
/*=============================================================*/
struct am930hw;
struct am930mgr;

#if (WLAN_OS == WLAN_LINUX_KERNEL)
struct am930llc;
#elif (WLAN_OS == WLAN_LWOS)
struct wlanInfo;
#else
	#error "No WLAN_OS match!"
#endif

/* State maintenance for wep PRNG */
typedef struct wlan_wepprng
{
	UINT8	s[WLAN_WEP_SBOXLEN];
	UINT	i;
	UINT	j;
} wlan_wepprng_t;

/* received frames, passed up from hw */
typedef struct am930rxstats
{
	UINT32	local_time;
	UINT8	rssi;
	UINT8	ch;
	UINT8	rate;
} am930rxstats_t;

/* mac queue types, see functions below */
struct am930qitem;

typedef struct am930queue
{
	struct am930qitem	*head;
	struct am930qitem	*tail;
	UINT				len;
	UINT				maxlen;
	UINT32				busy;
} am930q_t;

typedef struct am930qitem
{
	struct am930qitem	*prev;
	struct am930qitem	*next;
	struct am930queue	*q;
	void				*data;
} am930qitem_t;

typedef struct am930mac_rxitem
{
	wlan_pb_t		*f;
	am930rxstats_t	*s;
} am930mac_rxitem_t;


/* Fragment pool type, used for collecting fragments prior to reassembly */
typedef struct am930fragpool
{
	UINT8					srcaddr[WLAN_ADDR_LEN];
	UINT16					seq;			/* 0xffff means unused */
	INT32					lifetime;
	wlan_pb_t				*pb[WLAN_FRAGS_MAX];
} am930fragpool_t;


/*----------------------------------------------------------*/
/* mac structure                                            */
typedef struct am930mac
{
	/* References to other objects */
	struct am930hw	*hw;
	struct am930mgr	*mgr;
#if (WLAN_OS == WLAN_LINUX_KERNEL)
	void			*di;
	struct am930llc	*llc;
#elif (WLAN_OS == WLAN_LWOS)
	struct wlanInfo	*lw;	/* these two are set to the same value */
	struct wlanInfo	*llc;
#else
	#error "No WLAN_OS match!"
#endif

	/* Operation state variables */
	UINT			mode;			/* see constants above */
	UINT			state;			/* see constants above */

	void			(*timerfunc)(struct am930mac *mac);	/* called by llc */
	UINT			snifflags;		/* constants in wlan_ioctl.h */

	UINT32			last_beacon_time;	/* ESS_STA beacon time */
	UINT32			last_beacon_rssi;	/* ESS_STA beacon strength */

	/* Frame Queues */
	UINT			nextq;		/* selects which queue to tx from next */
	am930q_t		*llcq;
	am930q_t		*macq;
	am930q_t		*rxq;

	/* Host maintained MIB items */
	/* Privacy */
	UINT			privacy_invoked;
	UINT			exclude_unencrypted;
	UINT32			wep_defkeyid;
	UINT8			wep_key[WLAN_WEP_NKEYS][WLAN_WEP_KEYLEN];
	UINT32			crc32_table[256];
	UINT32			undecryptable;	/* in AP implies WEP required */

	/* StationConfig */
	UINT8			oprates[8];		/* stored as in mgmt frm, see 802.11ch7 */

	/* MAC operation */
	UINT32			maxrxlifetime;	/* in ms (supposed to be 1024us) */

	/* TODO: add the key mappings table */

	/* Fragment reassembly pool table */
	am930fragpool_t	defragpool[AM930MAC_FRAGPOOLS_MAX];
	spinlock_t		fragpool_lock;

} am930mac_t;


/*=============================================================*/
/*--- Methods -------------------------------------------------*/
/*=============================================================*/

#if (WLAN_OS == WLAN_LINUX_KERNEL)
am930mac_t *am930mac_construct(void *di, UINT32 iobase, UINT32 membase, UINT32 irq);
#elif (WLAN_OS == WLAN_LWOS)
am930mac_t *am930mac_construct(struct wlanInfo *lw, UINT32 iobase, UINT32 membase, UINT32 irq);
#else
	#error "No WLAN_OS match!"
#endif

void am930mac_destruct( am930mac_t *mac );

/* Instructors/Modifiers */
UINT32 am930mac_txllc( am930mac_t *mac, UINT8 *daddr, UINT8 *saddr, 
						wlan_pb_t *pb);
UINT32 am930mac_txmac( am930mac_t *mac, wlan_pb_t *pb);

/* Event handlers */
void am930mac_ISR( am930mac_t *mac );
void am930mac_ontxcomplete( am930mac_t *mac, UINT32 txresult );
void am930mac_rxframe( am930mac_t *mac);
void am930mac_rxenqueue( am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats);
void am930mac_rxframe_err( am930mac_t *mac);

/* Queue routines */
am930q_t *am930q_init(UINT maxlen);

int am930q_enqueue(am930q_t *q, void *data);
void *am930q_dequeue(am930q_t *q);

#define am930q_length(q) ((q)->len)
#define am930q_maxlen(q) ((q)->maxlen)

/* WEP routines */
void	am930mac_wep_initprng( wlan_wepprng_t *prng, UINT8 *k,  UINT klen);
UINT8	am930mac_wep_nextprn( wlan_wepprng_t *prng );
void	am930mac_wep_block( wlan_wepprng_t *prng, UINT8 *d, UINT dlen);
void	am930mac_wep_encrypt( am930mac_t *mac, wlan_pb_t *pb);
wlan_pb_t *am930mac_wep_decrypt( am930mac_t *mac, wlan_pb_t *pb);

/* Utility routines */
UINT16  am930mac_mk_capinfo(am930mac_t *mac);
void	am930mac_snifframe(am930mac_t *mac, wlan_pb_t *pb, am930rxstats_t *stats);

#endif  /* _LINUX_AM930MAC_H */
