/* am930mgr.c: Handles the 802.11 mac management functions.
*	--------------------------------------------------------------------
*
*   Linux WLAN 
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.0 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   The initial developer of the original code is Mark S. Mathews
*   <mark@absoval.com>.  Portions created by Mark S. Mathews
*   are Copyright (C) 1998 AbsoluteValue Software, Inc.  All Rights Reserved.
*   
*	--------------------------------------------------------------------
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Thanks to David Hinds, Donald Becker, and the rest of the Linux
*	developers worldwide for making all of this possible.
*
*	--------------------------------------------------------------------
*	
*/


#include <linux/config.h>
#include <wlan/wlan_compat.h>

#if (WLAN_OS == WLAN_LINUX_KERNEL)

/* The following prevents "kernel_version" from being set in this file. */
#define __NO_VERSION__

/* PCMCIA headers generated during PCMCIA package installation */
#ifdef WLAN_PCMCIA
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#endif

/* Module related headers, non-module drivers should not include */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>

/* Ethernet and network includes */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

/* Standard driver includes */
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/in.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <asm/bitops.h>
#include <linux/errno.h>

/* Card and Driver services includes */
#ifdef WLAN_PCMCIA
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>
#endif

#include <wlan/version.h>
#include <wlan/am930mib.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/wlan_ioctl.h>
#include <wlan/wlan_stable.h>

#ifdef PCMCIA
#include "am930di.h"
#endif

#include "am930llc.h"
#include "am930mac.h"
#include "am930hw.h"
#include "am930mgr.h"

#elif (WLAN_OS == WLAN_LWOS)

#include <stddef.h>
#include <stdlibsys.h>
#include <direct.h>
#include <hal.h>
#include <mem.h>
#include <kernel.h>
#include <spi.h>
#include <clock.h>

#include <wlan/wlan_compat.h>
#include <wlan/version.h>
#include <wlan/am930mib.h>
#include <wlan/p80211hdr.h>
#include <wlan/p80211mgmt.h>
#include <wlan/am930mib.h>
#include <wlan/wlan_stable.h>
#include <wlan/wlan_ioctl.h>
#include <lw-wlan.h>
#include <wlan/am930mac.h>
#include <wlan/am930hw.h>
#include <wlan/am930mgr.h>

#else
	#error "No WLAN_OS match!"
#endif

/*================================================================*/
/* Local Constants  */

/*================================================================*/
/* Local Functions  */

static void am930mgr_authen1_rx( am930mgr_t *mgr, wlan_fr_authen_t *f );
static void am930mgr_authen2_rx( am930mgr_t *mgr, wlan_fr_authen_t *f );
static void am930mgr_authen3_rx( am930mgr_t *mgr, wlan_fr_authen_t *f );
static void am930mgr_authen4_rx( am930mgr_t *mgr, wlan_fr_authen_t *f );


static void am930mgr_mkchallengetext( am930mgr_t *mgr, 
	UINT8 *chbuf, UINT chbuflen, UINT8 *key, UINT keylen);

static wlan_pb_t *am930mgr_mkfrm_assocreq( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 listen_int,
	wlan_ie_ssid_t *ssid,  wlan_ie_supp_rates_t *rates);

static wlan_pb_t *am930mgr_mkfrm_assocresp( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 status, UINT16 aid, 
	wlan_ie_supp_rates_t *rates);

static wlan_pb_t *am930mgr_mkfrm_auth( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 alg, UINT16 seq, UINT16 status, UINT8 *challenge);

static wlan_pb_t *am930mgr_mkfrm_deauth( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 reason);

static wlan_pb_t *am930mgr_mkfrm_disassoc( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 reason);

static wlan_pb_t *am930mgr_mkfrm_reassocresp( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 status, UINT16 aid, 
	wlan_ie_supp_rates_t *rates);

static int am930mgr_rates_basicissubset( am930mgr_t *mgr, 
	wlan_ie_supp_rates_t *r1, wlan_ie_supp_rates_t *r2);

static UINT8 am930mgr_rates_max(wlan_ie_supp_rates_t *rates);

static void am930mgr_timerfunc(am930mgr_t *mgr);

/*================================================================*/
/* Local Statics  */

/*----------------------------------------------------------------
*	am930mgr_construct
*
*	Allocates and initializes the mgr object and it's subobjects.
*
*	returns: NULL on failure, the address of the new object
*            otherwise.
----------------------------------------------------------------*/
am930mgr_t* am930mgr_construct(am930mac_t *mac, am930hw_t *hw)
{
	am930mgr_t* mgr = NULL;
	DBFENTER;

	if ( mac != NULL )
	{
		mgr = kmalloc( sizeof(am930mgr_t), GFP_KERNEL);
		if ( mgr != NULL )
		{
			memset( mgr, 0, sizeof(am930mgr_t));
			mgr->mac = mac;
			mgr->hw = hw;

			/* TODO: make this parameterized! */
			mgr->challengekey[0] = 0xa4;
			mgr->challengekey[1] = 0xf6;
			mgr->challengekey[2] = 0xd2;
			mgr->challengekey[3] = 0x54;
			mgr->challengekey[4] = 0xc6;
			mgr->challengekey[5] = 0xc6;
			mgr->challengekey[6] = 0x56;
			mgr->challengekey[7] = 0xe6;
			mgr->challengekey[8] = 0x02;
			mgr->challengekey[9] = 0x64;
			mgr->challengekey[10] = 0xe2;
			mgr->challengekey[11] = 0x02;
			mgr->challengekey[12] = 0xd4;
			mgr->challengekey[13] = 0x16;
			mgr->challengekey[14] = 0x47;
			mgr->challengekey[15] = 0x86;
			mgr->challengekey[16] = 0x56;
			mgr->challengekey[17] = 0x77;
			mgr->challengekey[18] = 0x37;

/*			mgr->timerfunc = am930mgr_timerfunc; */

			/* Management transaction timeout, default 10 seconds */
			/*  for auth and assoc */
			mgr->mgmt_trans_timeout = wlan_seconds2ticks(10);
			
			/* Authenticated station timeout (AP), default 60 minutes */
			mgr->ap_auth_timeout = wlan_minutes2ticks(60);

			/* Associated station timeout (AP), default 30 minutes */
			mgr->ap_assoc_timeout = wlan_minutes2ticks(30);

		}
	}
	DBFEXIT;
	return mgr;
}


/*----------------------------------------------------------------
*	am930mgr_timerfunc
*
*	Timer function called from llc.  We currently assume that this
*	function will be called every 1/10th second.  Lets code
*	all the time stuff so it doesn't matter....use absolute time
*	from the system time register.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_timerfunc(am930mgr_t *mgr)
{
	static UINT 		count = 0;
	wlan_stable_item_t	*sta;
	wlan_stable_item_t	*sta_dealloc;
	UINT				i;

	DBFENTER;

	if ( mgr->mac->mode == AM930_MACMODE_ESS_AP && ++count > 9 )
	{
		/* only scan the stable once per second. */
		count = 0;
		for ( i = 0; i < WLAN_STABLE_LEN; i++ )
		{
			sta = mgr->stable[i];
			while (sta != NULL)
			{
				/* Looking for: */
				/* stations that need deallocation */
				if ( sta->sta.state == WLAN_STA_STATE_FREE )
				{
					if (sta->prev == NULL && sta->next == NULL)
					{
						mgr->stable[i] = NULL;
					}
					else if  ( sta->prev == NULL && sta->next != NULL )
					{
						mgr->stable[i] = sta->next;
						sta->next->prev = NULL;
					}
					else if ( sta->prev != NULL && sta->next == NULL )
					{
						sta->prev->next = NULL;
					}
					else
					{
						sta->prev->next = sta->next;
						sta->next->prev = sta->prev;
					}
					mgr->aidtable[sta->sta.aid] = NULL;
					sta_dealloc = sta;
					sta = sta->next;
					am930shim_free(sta_dealloc, sizeof(wlan_stable_item_t));
					continue;
				}

				/* timed out authentication transactions */
				if ( (sta->sta.state == WLAN_STA_STATE_PENDING_ACTIVE) &&
				     (sta->sta.info & WLAN_STA_INFO_AUTH) &&
					 time_after_eq( jiffies, sta->sta.timeout) )
				{
					sta->sta.state = WLAN_STA_STATE_FREE;
					sta->sta.info = 0;
				}

				/* inactive stations (mark them for future dealloc) */
				/*
				if ( (sta->sta.state == WLAN_STA_STATE_ACTIVE) &&
					 time_after_eq( jiffies, sta->
				*/
				sta = sta->next;
			}
		}
	}
	else
	{
		/* if we're a STA in INFRA we need to watch for:             */
		/*    - Timed out authentication transactions                */
		/*    - Timed out association transactions                   */
		/*    - if associated, Inactivity from the AP                */
	}
	
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_destruct
*
*	destroys the mgr object
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_destruct( am930mgr_t *mgr )
{
	knownbss_t	*bss;
	DBFENTER;
	if ( mgr != NULL )
	{
		/* Free the remaining bss's */
		while ( mgr->bsslist != NULL )
		{
			bss = mgr->bsslist;
			mgr->bsslist = bss->next;
			/* TODO: when we add ttl, need to kill any timer funcs */
			kfree_s( bss, sizeof( knownbss_t )); 
		}
		kfree_s(mgr, sizeof(am930mgr_t));
	}
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930mgr_assoc_begin_sta
*
*	Start the station association procedure.  Namely, send an
*	association request frame to the AP.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_assoc_begin_sta(am930mgr_t *mgr)
{
	wlan_pb_t				*pb;
	UINT16					capinfo;

	/* bogus hardcode, needs fixing */
	UINT8					rates[] = {WLAN_EID_SUPP_RATES, 2, 0x82, 0x84};


	DBFENTER;

	capinfo = WLAN_SET_MGMT_CAP_INFO_ESS(1);

	if ( mgr->mac->exclude_unencrypted )
	{
		capinfo |= WLAN_SET_MGMT_CAP_INFO_PRIVACY(1);
	}

	/* build an assocreq frame and send it */
	pb = am930mgr_mkfrm_assocreq( mgr, 
			mgr->curr_bssid,
			capinfo,
			0, /* listen_int, indicates no PS mode? */
			(wlan_ie_ssid_t*)mgr->curr_ssid,
			(wlan_ie_supp_rates_t*)rates);
				
				
	if (pb != NULL )
	{
		/* send the frame */
		am930mac_txmac( mgr->mac, pb);
		mgr->mac->state = AM930_MACSTATE_ASSOCPENDING;
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_assoc_stop
*
*	Stop and clean up a pending association attempt.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_assoc_stop(am930mgr_t *mgr)
{
	if ( mgr->mac->state == AM930_MACSTATE_ASSOCPENDING ) {
		mgr->mac->state = AM930_MACSTATE_AUTH;
	}
}

/*----------------------------------------------------------------
*	am930mgr_assocreq_rx
*
*	Handle incoming association response frames.
*	NOTE: The reassocreq_rx is basically a cut-n-paste of this 
*		function (bad Mark, bad, bad).  If you change anything
*		here it should probably be changed in reassocreq as well.
*
*	Arguments:
*		rxpb	packet buffer containing assocreq frame
*		stats	receive statitistics for assocreq frame
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_assocreq_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_assocreq_t	f;
	wlan_stable_item_t	*sta;
	wlan_pb_t			*pb;
	DBFENTER;

	/* decode the frame */
	f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
	f.buf = rxpb->p80211buf;
	wlan_mgmt_decode_assocreq(&f);

	/* Check for class2 error */
	sta = am930mgr_stable_lookup( mgr, f.hdr->a3.a2);
	if ( sta == NULL || !(sta->sta.info & WLAN_STA_INFO_AUTH) )
	{
    	pb = am930mgr_mkfrm_deauth( mgr, 
				f.hdr->a3.a2, 
				WLAN_MGMT_REASON_CLASS2_NONAUTH);
		if ( pb != NULL ) 
		{
			am930mac_txmac( mgr->mac, pb);
		}
    	return;
	}

	/* Check to make sure we're the SS the station says we are */
	/* assume bad BSSID would be rejected by rx filter */
	if ( memcmp( f.ssid->ssid, mgr->curr_ssid + WLAN_IEHDR_LEN, 
				mgr->curr_ssid[1]) != 0 )
	{
		pb = am930mgr_mkfrm_assocresp( mgr, 
				f.hdr->a3.a2, 
				am930mac_mk_capinfo(mgr->mac),
				WLAN_MGMT_STATUS_ASSOC_DENIED_UNSPEC,
				0, /* AID */
				(wlan_ie_supp_rates_t*)mgr->mac->oprates );
		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Capability Info check */
	/*   condition 1: is it not asking for an ESS? */
	/*   condition 2: is it asking for polling? (we don't support it) */
	/*   condition 3: is it asking for WEP and our wep keys aren't set? */
   	if (
		(WLAN_GET_MGMT_CAP_INFO_ESS(ieee2host16(*f.cap_info)) == 0) ||
       	((WLAN_GET_MGMT_CAP_INFO_CFPOLLABLE(ieee2host16(*f.cap_info)) == 1) && 	
	 		(WLAN_GET_MGMT_CAP_INFO_CFPOLLREQ(ieee2host16(*f.cap_info)) == 0)) ||
       	((WLAN_GET_MGMT_CAP_INFO_PRIVACY(ieee2host16(*f.cap_info)) == 1) && 
	 		(mgr->mac->privacy_invoked == 0)) )
	{
       	pb = am930mgr_mkfrm_assocresp( mgr,
			f.hdr->a3.a2,
			am930mac_mk_capinfo(mgr->mac),
			WLAN_MGMT_STATUS_CAPS_UNSUPPORTED,
			0, /* AID */
			(wlan_ie_supp_rates_t*)mgr->mac->oprates );

		if  ( pb != NULL )
		{
        	am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Basic Rate Check */
	if ( !am930mgr_rates_basicissubset( mgr, 
			(wlan_ie_supp_rates_t*)mgr->mac->oprates, f.supp_rates) )
	{
		/* mismatch in basic rates */
		pb = am930mgr_mkfrm_assocresp( mgr,
				f.hdr->a3.a2,
				am930mac_mk_capinfo(mgr->mac),
				WLAN_MGMT_STATUS_ASSOC_DENIED_RATES,
				0, /* AID */
				(wlan_ie_supp_rates_t*)mgr->mac->oprates );

		if  ( pb != NULL )
		{
        	am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Made it! we're now ready to handle success! */
	if ( (sta->sta.listen_int = ieee2host16(*f.listen_int)) )
	{
		sta->sta.info |= WLAN_STA_INFO_PSMODE;
	} 
	
	if ( WLAN_GET_MGMT_CAP_INFO_PRIVACY(ieee2host16(*f.cap_info)) )
	{
		sta->sta.info |= WLAN_STA_INFO_WEPON;
	}
	
	sta->sta.rate = sta->sta.max_rate = am930mgr_rates_max(f.supp_rates);
	sta->sta.info |= WLAN_STA_INFO_ASSOC;
	pb = am930mgr_mkfrm_assocresp( mgr,
			f.hdr->a3.a2,
			am930mac_mk_capinfo(mgr->mac),
			WLAN_MGMT_STATUS_SUCCESS,
			sta->sta.aid | BIT14 | BIT15,
			(wlan_ie_supp_rates_t*)mgr->mac->oprates );

    if ( pb != NULL )
	{
		am930mac_txmac( mgr->mac, pb);
	}

	DBFEXIT;
    return;
}



/*----------------------------------------------------------------
*	am930mgr_assocresp_rx
*
*	Handle incoming association response frames
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_assocresp_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_assocresp_t	f;
	DBFENTER;

	/* We better be assoc pending, otherwise ignore it */
	if ( mgr->mac->state == AM930_MACSTATE_ASSOCPENDING )
	{
		/* decode the frame */
		f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
		f.buf = rxpb->p80211buf;
		wlan_mgmt_decode_assocresp(&f);
	
		/* save values and set mode */
		if ( ieee2host16(*(f.status)) == WLAN_MGMT_STATUS_SUCCESS )
		{
			mgr->curr_aid = ieee2host16(*(f.aid));
			mgr->mac->state = AM930_MACSTATE_ASSOC;
			if ( (mgr->curr_aid >> 14) != (BIT0 | BIT1) )
			{
				WLAN_LOG_WARNING0("AID from AP, has two msb clear.\n");
			}

			WLAN_LOG_NOTICE1(
				"Association Successful, AID=%d.\n", 
				mgr->curr_aid & ~(BIT14|BIT15));
			am930llc_cmdcomplete(mgr->mac->llc, 0);
		}
		else
		{
			/* jump back to the auth state and indicate the error */
			mgr->mac->state = AM930_MACSTATE_AUTH;
			am930mgr_prstatus(mgr, ieee2host16(*(f.status)) );
			am930llc_cmdcomplete(mgr->mac->llc, ieee2host16(*(f.status)) );
		}
	}
	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930mgr_auth_begin_sta
*
*	Start the station authentication procedure.  Namely, send an
*	auth frame to the AP.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen_begin_sta(am930mgr_t *mgr, UINT16 authalg)
{
	wlan_fr_authen_t	*f;
	wlan_pb_t			*pb;

	DBFENTER;

	/* initial setup of frame */
	pb = am930shim_pballoc();

	if (pb != NULL )
	{
		f = kmalloc( sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN, GFP_ATOMIC );
		if ( f != NULL )
		{
			memset( f, 0, sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN);
			f->buf = ((UINT8*)f) + sizeof(wlan_fr_authen_t);
			f->len = WLAN_AUTHEN_FR_MAXLEN;
			wlan_mgmt_encode_authen(f);	
		
			/* insert values */
			f->hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_AUTHEN));
			memcpy( f->hdr->a3.a1, mgr->curr_bssid, WLAN_BSSID_LEN);
			memcpy( f->hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( f->hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);
	
			*(f->auth_alg) = host2ieee16(authalg);
			*(f->auth_seq) = host2ieee16(1);
		
			/* now copy the constructed frame to a pb and tx */
			am930shim_pballoc_p80211( pb, f->len);

			/* bogus: add 4 because pballoc assumes rx+FCS and subtracts 4 */
			pb->p80211_payloadlen += 4;

			if ( pb->p80211hostbuf != NULL )  /* note silent failure */
			{	
				memcpy( pb->p80211buf, f->buf, f->len);

				/* send the frame */
				am930mac_txmac( mgr->mac, pb);

				mgr->mac->state = AM930_MACSTATE_AUTHPENDING;
			}	
			else
			{
				am930shim_pbfree( pb );
			}
		
			/* the response(s) will be handled in auth_rx */
			kfree_s( f, sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN);
		}
	}
	DBFEXIT;
	return;
}

/*----------------------------------------------------------------
*	am930mgr_authen_stop
*
*	Clear the auth pending state.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen_stop( am930mgr_t *mgr )
{
	mgr->mac->state = AM930_MACSTATE_NOAUTH;
}


/*----------------------------------------------------------------
*	am930mgr_authen_rx
*
*	Handle incoming authentication frames
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_authen_t	f;
	DBFENTER;

	/* we better be an AP or a STA in AUTHPENDING otherwise ignore */
	if ( !(mgr->mac->mode == AM930_MACMODE_ESS_AP ||
			mgr->mac->state == AM930_MACSTATE_AUTHPENDING) )
	{
		return;
	}

	/* OK, we're in an OK mode and state, if it's encrypted, decrypt */
	if ( rxpb->wep_iscrypt )
	{
		if ( (rxpb = am930mac_wep_decrypt( mgr->mac, rxpb)) == NULL )
		{
			WLAN_LOG_NOTICE0("authen_rx: rx'd undecryptable frame\n");
			return;
		}
	}

	/* decode the frame */
	f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
	f.buf = rxpb->p80211buf;
	wlan_mgmt_decode_authen(&f);

	switch ( ieee2host16(*(f.auth_seq )) )
	{
		case 1:
			am930mgr_authen1_rx( mgr, &f);
			break;
		case 2:
			am930mgr_authen2_rx( mgr, &f);
			break;
		case 3:
			am930mgr_authen3_rx( mgr, &f);
			break;
		case 4:
			am930mgr_authen4_rx( mgr, &f);
			break;
		default:
			WLAN_LOG_WARNING1("f.auth_seq not in [1234], seq = %d\n", 
				ieee2host16(*(f.auth_seq)));
			break;
	}

	DBFEXIT;
	return;
}



/*----------------------------------------------------------------
*	am930mgr_authen1_rx
*
*	Handles incoming authen frames with sequence 1.  Currently 
*	assumes we're an AP.  So far, no one appears to use authentication
*	in Ad-Hoc mode.
*
*	returns: nothing.
----------------------------------------------------------------*/
void am930mgr_authen1_rx( am930mgr_t *mgr, wlan_fr_authen_t *f )
{
	wlan_stable_item_t	*sta;
	wlan_pb_t			*pb;

	DBFENTER;

	/* Check to make  sure we support the requested alg */
	if ( ((ieee2host16(*(f->auth_alg)) == WLAN_AUTH_ALG_SHAREDKEY) && 
		 (mgr->mac->privacy_invoked == 0)) ||
		 ((ieee2host16(*(f->auth_alg)) != WLAN_AUTH_ALG_SHAREDKEY) &&
		 (ieee2host16(*(f->auth_alg)) != WLAN_AUTH_ALG_OPENSYSTEM)) )
	{
		/* we haven't any wep keys and can't do SHAREDKEY _or_ */
		/* the STA has asked for an alg we know nothing about  */
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				2,
				WLAN_MGMT_STATUS_UNSUPPORTED_AUTHALG,
				NULL );

		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Check for the sender station in our stable */
	sta = am930mgr_stable_lookup( mgr, f->hdr->a3.a2);
	if ( sta != NULL )
	{
		/* Change it's state to PENDING and mark authenticated */
		sta->sta.state = WLAN_STA_STATE_PENDING_ACTIVE;
		sta->sta.info |= WLAN_STA_INFO_AUTH;
		sta->sta.timeout = jiffies + mgr->mgmt_trans_timeout;
		sta->sta.timer_state = WLAN_STA_TSTATE_AUTH;
	}
	else
	{
		/* It's a new one, allocate and add to stable */
		sta = kmalloc( sizeof(wlan_stable_item_t), GFP_ATOMIC);
		if ( sta != NULL )
		{
			memset( sta, 0, sizeof(wlan_stable_item_t));
			memcpy( sta->sta.addr, f->hdr->a3.a2, WLAN_ADDR_LEN);
			sta->sta.state = WLAN_STA_STATE_PENDING_ACTIVE;
			sta->sta.info |= WLAN_STA_INFO_AUTH;
			sta->sta.timeout = jiffies + mgr->mgmt_trans_timeout;
			if ( (sta->sta.aid = am930mgr_stable_alloc_aid(mgr, sta)) > 0 )
			{
				am930mgr_stable_insert( mgr, sta);
			}
			else
			{
				/* Can't alloc an AID, tell the STA it can't authenticate */
				pb = am930mgr_mkfrm_auth( mgr, 
						f->hdr->a3.a2, 
						ieee2host16(*(f->auth_alg)),
						2,
						WLAN_MGMT_STATUS_UNSPEC_FAILURE,
						NULL );

				if ( pb != NULL )
				{
					am930mac_txmac( mgr->mac, pb);
				}

				am930shim_free(sta, sizeof(wlan_stable_item_t));
				return;
			}
		}
	}

	/* At this point, we've got a sta in the stable and we support the alg */
	/*  time to send a SUCCESS response */
	if (ieee2host16(*(f->auth_alg)) == WLAN_AUTH_ALG_OPENSYSTEM )
	{
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				2,
				WLAN_MGMT_STATUS_SUCCESS,
				NULL );

		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}

		/* No more pending, this guy's authenticated */
		sta->sta.state = WLAN_STA_STATE_ACTIVE;
		sta->sta.timeout = jiffies;
		sta->sta.timer_state = 0;
	}
	else
	{
		UINT8	ctext[WLAN_CHALLENGE_LEN];
		/* Shared key (alg num validated previously) */
		am930mgr_mkchallengetext( mgr, 
			ctext, 
			WLAN_CHALLENGE_LEN, 
			mgr->challengekey,
			sizeof(mgr->challengekey));

		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				2,
				WLAN_MGMT_STATUS_SUCCESS,
				ctext );
		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
	}
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_authen2_rx
*
*	Handles incoming auth frames with sequence number 2.  Currently
*	assumes we're a station.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen2_rx( am930mgr_t *mgr, wlan_fr_authen_t *f )
{
	DBFENTER;

	switch (ieee2host16(*(f->auth_alg)))
	{
		case WLAN_AUTH_ALG_OPENSYSTEM:
			if ( ieee2host16(*(f->status)) == WLAN_MGMT_STATUS_SUCCESS )
			{
				WLAN_LOG_NOTICE0("802.11 Authen (OPEN) Successful.\n");

				mgr->mac->state = AM930_MACSTATE_AUTH;

				am930llc_cmdcomplete(mgr->mac->llc, ieee2host16(*(f->status)));
			}
			else
			{
				WLAN_LOG_NOTICE0("802.11 Authen (OPEN) Failed.\n");
				am930mgr_prstatus(mgr, ieee2host16(*(f->status)) );
				mgr->mac->state = AM930_MACSTATE_NOAUTH;
				am930llc_cmdcomplete(mgr->mac->llc, ieee2host16(*(f->status)));
			}
			break;

#ifdef TODO_WEP
		case WLAN_AUTH_ALG_SHAREDKEY:
		{
			wlan_fr_authen_t	*reply;
			wlan_pb_t			*pb;

			/* Initial setup of response frame */
			pb = am930shim_pballoc();

			if (pb != NULL )
			{
				reply = kmalloc( sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN, GFP_ATOMIC );
				if ( reply != NULL )
				{
					memset( reply, 0, sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN);
	
					/* Set up the reply frame */
					reply->buf = ((UINT8*)reply) + sizeof(wlan_fr_authen_t);
					reply->len = WLAN_AUTHEN_FR_MAXLEN;
					wlan_mgmt_encode_authen(reply);	
		
					/* insert values */
					reply->hdr->a3.fc = host2ieee16(
						WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
						WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_AUTHEN));
					memcpy( reply->hdr->a3.a1, f->hdr->a3.a2, WLAN_BSSID_LEN);
					memcpy( reply->hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
					memcpy( reply->hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);
					*(reply->auth_alg) = *(f->auth_alg);
					*(reply->auth_seq) = host2ieee16(3);

					*(reply->status) = host2ieee16(WLAN_MGMT_STATUS_SUCCESS);
					reply->challenge = (wlan_ie_challenge_t*)reply->buf + reply->len;
					reply->len += WLAN_CHALLENGE_IE_LEN;
					reply->challenge->eid = WLAN_EID_CHALLENGE;
					reply->challenge->len = WLAN_CHALLENGE_LEN;
					memcpy( reply->challenge->challenge, f->challenge->challenge, WLAN_CHALLENGE_LEN);

					/* now copy the constructed frame to a pb and tx */
					am930shim_pballoc_p80211( pb, reply->len);
		
					if ( pb->p80211hostbuf != NULL )  /* note silent failure */
					{	
						memcpy( pb->p80211buf, reply->buf, reply->len);

						am930mac_wep_encrypt( mgr->mac, pb );

						/* send the frame */
						am930mac_txmac( mgr->mac, pb);
					}	
					else
					{
						am930shim_pbfree( pb );
					}
		
					/* discard the buffer */
					kfree_s( reply, sizeof(wlan_fr_authen_t) + WLAN_AUTHEN_FR_MAXLEN);
				}
			}


			break;
		}
#endif /* TODO_WEP */

		default:
			WLAN_LOG_WARNING1("rx'd auth.seq = 2 unknown auth_alg=%d\n", ieee2host16(*(f->auth_alg)) );
			break;
	}
	DBFEXIT;
}


/*----------------------------------------------------------------
*	am930mgr_authen3_rx
*
*	Handles incoming authen frames with sequence 3.  Currently 
*	assumes we're an AP.  This function assumes the frame has
*	already been successfully decrypted.
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen3_rx( am930mgr_t *mgr, wlan_fr_authen_t *f )
{
	UINT8				ctext[WLAN_CHALLENGE_LEN];
	wlan_pb_t			*pb;
	wlan_stable_item_t	*sta;
	
	DBFENTER;

	/* First, find out if we're in a auth exchange w/ sender */
	sta = am930mgr_stable_lookup( mgr, f->hdr->a3.a2);
	if ( sta == NULL )
	{
		/* We're not in an exchange with this STA */
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				4,
				WLAN_MGMT_STATUS_RX_AUTH_NOSEQ,
				NULL );

		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		return;
	}
	else if ( sta->sta.state == WLAN_STA_STATE_PENDING_REMOVAL )
	{
		/* A timeout has occurred */
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				4,
				WLAN_MGMT_STATUS_AUTH_TIMEOUT,
				NULL );
	
		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		sta->sta.state = WLAN_STA_STATE_FREE;	/* next dealloc pass will remove it */
		return;
	}

	/* Check the challenge text */
	am930mgr_mkchallengetext( mgr, 
		ctext, 
		WLAN_CHALLENGE_LEN, 
		mgr->challengekey,
		sizeof(mgr->challengekey));

	if (memcmp( f->challenge->challenge, ctext, WLAN_CHALLENGE_LEN) == 0)
	{
		/* Success! the STA is now authenticated */
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				4,
				WLAN_MGMT_STATUS_SUCCESS,
				NULL );

		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		sta->sta.state = WLAN_STA_STATE_ACTIVE;
	}
	else
	{
		/* challenge failure */
		pb = am930mgr_mkfrm_auth( mgr, 
				f->hdr->a3.a2, 
				ieee2host16(*(f->auth_alg)),
				4,
				WLAN_MGMT_STATUS_CHALLENGE_FAIL,
				NULL );

		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		sta->sta.state = WLAN_STA_STATE_PENDING_REMOVAL;
		sta->sta.info = 0;
	}
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_authen4_rx
*
*	Handles incoming authen frames with sequence 4
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_authen4_rx( am930mgr_t *mgr, wlan_fr_authen_t *f )
{
	DBFENTER;

	if ( ieee2host16(*(f->status)) == WLAN_MGMT_STATUS_SUCCESS )
	{
		WLAN_LOG_NOTICE0("802.11 Authen (SHAREDKEY) Successful.\n");

		mgr->mac->state = AM930_MACSTATE_AUTH;

		am930llc_cmdcomplete(mgr->mac->llc, ieee2host16(*(f->status)));
	}
	else
	{
		WLAN_LOG_NOTICE0("802.11 Authen (SHAREDKEY) Failed.\n");
		am930mgr_prstatus(mgr, ieee2host16(*(f->status)) );
		mgr->mac->state = AM930_MACSTATE_NOAUTH;
		am930llc_cmdcomplete(mgr->mac->llc, ieee2host16(*(f->status)));
	}
	DBFEXIT;
}

/*----------------------------------------------------------------
*	am930mgr_beacon_rx
*
*	Handles beacon frames.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_beacon_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	DBFENTER;

	if ( mgr->mac->mode == AM930_MACMODE_ESS_STA )
	{
		mgr->mac->last_beacon_time = jiffies;
		mgr->mac->last_beacon_rssi = stats->rssi;
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_class2err
*
*	Generates a deauth frame to the given daddr.
*	Arguments:
*		daddr	Destination of the death
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_class2err(am930mgr_t *mgr, UINT8 *daddr)
{
	wlan_pb_t	*pb;
	
	pb = am930mgr_mkfrm_deauth( mgr, daddr, WLAN_MGMT_REASON_CLASS2_NONAUTH);
	if ( pb != NULL )
	{
		am930mac_txmac( mgr->mac, pb);
	}

	return;
}


/*----------------------------------------------------------------
*	am930mgr_class3err
*
*	Generates a disassoc frame to the given daddr.
*	Arguments:
*		daddr	Destination of the disassoc
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_class3err(am930mgr_t *mgr, UINT8 *daddr)
{
	wlan_pb_t	*pb;

	pb = am930mgr_mkfrm_disassoc( mgr, daddr, WLAN_MGMT_REASON_CLASS3_NONASSOC);
	if ( pb != NULL )
	{
		am930mac_txmac( mgr->mac, pb);
	}
	return;
}


/*----------------------------------------------------------------
*	am930mgr_createbss
*
*	Instructs the hw object to create a bss using the supplied
*	attributes. Note that this implementation only supports Ad-Hoc
*	BSS creation.
*	Arguments:
*		ch		The DS channel to use (1-12)
*		ssid	The SSID to use, formatted as a wlan_ie_ssid_t
*		bcnrate	The beacon rate for the BSS.
*
*	returns: zero on success, non-zero on failure
----------------------------------------------------------------*/
UINT32 am930mgr_createbss(	am930mgr_t *mgr, 
							UINT32 ch, 
							wlan_ie_ssid_t *ie_ssid,
							UINT32 bcn, 
							UINT32 atimwindow)
{
	UINT32			result = 0;
	UINT8			bssid[WLAN_BSSID_LEN] = {0,0,0,0,0,0};
	wlan_bss_ts_t	ts = {0,0,0,0,0,0,0,0};
	UINT8			one = 1;
	UINT16			basic_rate = host2amd16(20);
	char			buf[WLAN_IEHDR_LEN + WLAN_SSID_MAXLEN];

	DBFENTER;
	
	/* set the 'sta basic rate' to 20 and be done with it. */
	am930mgr_mibsetitem( mgr, MAC_STA_BASIC_RATE, &basic_rate, sizeof(basic_rate));
				
	if ( am930_isap )
	{
		if ( am930hw_mibsetitem( mgr->hw, LOC_ACTING_AS_AP, &one, 
					MI_SZ(LOC_ACTING_AS_AP)) != AM930HW_SUCCESS )
		{
			result = 1;
		}

		memcpy(bssid, mgr->hw->addr, WLAN_BSSID_LEN);
	}
	if  ( result == 0 )
	{

		if ( !result  && (am930hw_joinbss( mgr->hw, 
								ch,
								1, 		/* newBSS */
								bssid,
								ie_ssid,
								bcn,
								ts, 	/* ts (timestamp) */
								0)		/* sutro local time */
				== AM930HW_SUCCESS ) )
		{
			mgr->curr_ch = ch;
			am930mgr_mibgetitem(
				mgr,
				MAC_STA_BASIC_RATE,
				&mgr->curr_rate,
				MI_SZ(MAC_STA_BASIC_RATE));
			mgr->curr_rate = amd2host16(mgr->curr_rate);
	
			mgr->curr_cap_info = 
				am930_isap ? 
					WLAN_SET_MGMT_CAP_INFO_ESS(1) :
					WLAN_SET_MGMT_CAP_INFO_IBSS(1);
			memcpy( mgr->curr_ssid, ie_ssid, WLAN_IEHDR_LEN + WLAN_SSID_MAXLEN);
			memcpy( mgr->curr_bssid, bssid, WLAN_BSSID_LEN );
			am930mgr_mibgetitem( mgr, MGMT_CURRENT_BSSID, 
								mgr->curr_bssid, WLAN_BSSID_LEN);
			mgr->mac->mode = am930_isap ? 
								AM930_MACMODE_ESS_AP :
								AM930_MACMODE_IBSS_STA;

			mgr->mac->state = AM930_MACSTATE_NOAUTH;

			memcpy( buf, ie_ssid->ssid, ie_ssid->len);
			buf[ie_ssid->len] = '\0';
			WLAN_LOG_NOTICE4("%s BSS created, channel=%ld, rate=%d SSID=%s\n",
								am930_isap ? "Infrastructure" : "AdHoc",
								ch, mgr->curr_rate, buf);
		}
		else
		{
			result = 1;
			WLAN_LOG_ERROR0("Failed to create BSS.\n");
		}
	}


	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930mgr_deauth_rx
*
*	Handles incoming deauthentication frames
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_deauth_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_deauthen_t 	f;
	wlan_stable_item_t	*sta;

	DBFENTER;

	if ( mgr->mac->mode == AM930_MACMODE_ESS_AP )
	{
		f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
		f.buf = rxpb->p80211buf;
		wlan_mgmt_decode_deauthen(&f);

		sta = am930mgr_stable_lookup( mgr, f.hdr->a3.a2);
		if ( sta != NULL )
		{
			WLAN_LOG_DEBUG1(1, "Station deauthed itself, reason=%d\n",
				ieee2host16(*(f.reason)) );
			sta->sta.info &= ~WLAN_STA_INFO_AUTH;
			sta->sta.state = WLAN_STA_STATE_PENDING_REMOVAL;

		}

		/* TODO: set up a notification to llc and ideally up to user-mode 
	             so the user-mode software can react (if necessary). */
	}
	else if ( mgr->mac->mode == AM930_MACMODE_ESS_STA )
	{
		f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
		f.buf = rxpb->p80211buf;
		wlan_mgmt_decode_deauthen(&f);

		mgr->mac->state = AM930_MACSTATE_NOAUTH;
		WLAN_LOG_DEBUG1(1, "AP deauthed me, reason=%d.\n", 
			ieee2host16(*(f.reason)) );
		/* TODO: set up a notification to llc and ideally up to user-mode 
		         so the user-mode software can react. */
	}
	/* else, ignore it.  TODO: IBSS authentication service
	   would be implemented here */

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_disassoc_rx
*
*	Handles incoming disassociation frames
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_disassoc_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_disassoc_t  f;
	wlan_stable_item_t	*sta;

	DBFENTER;

	if ( mgr->mac->mode == AM930_MACMODE_ESS_AP )
	{
		f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
		f.buf = rxpb->p80211buf;
		wlan_mgmt_decode_disassoc(&f);

		sta = am930mgr_stable_lookup( mgr, f.hdr->a3.a2);
		if ( sta != NULL )
		{
			WLAN_LOG_DEBUG1(1, "Station disassociated itself, reason=%d\n",
				ieee2host16(*(f.reason)) );
			sta->sta.info &= ~WLAN_STA_INFO_ASSOC;
		}
		/* TODO: set up a notification to llc and ideally up to user-mode 
		         so the user-mode software can react. */
	}
	else if ( mgr->mac->mode == AM930_MACMODE_ESS_STA )
	{
		f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
		f.buf = rxpb->p80211buf;
		wlan_mgmt_decode_disassoc(&f);

		mgr->mac->state = AM930_MACSTATE_AUTH;
		WLAN_LOG_DEBUG1(1, "AP disassociated me, reason=%d.\n", 
			ieee2host16(*(f.reason)) );
		/* TODO: set up a notification to llc and ideally up to user-mode 
		         so the user-mode software can react. */
	}
	/* else, ignore it */

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_joinbss
*
*	Instructs the hw object to join a bss using the supplied
*	attributes. The only argument is the bssid, the rest of the
*	attributes are obtained from the known bss list.
*
*	Arguments:
*		bssid	the 48 bit id of the bss to join
*
*	returns: zero on success, 
*				1 if the bss isn't found in the list,
*				2 if the hw sync command fails.
----------------------------------------------------------------*/
UINT32 am930mgr_joinbss( am930mgr_t *mgr, UINT8 *bssid )
{
	UINT32		result = 0;
	knownbss_t	*curr;
	UINT8		mibbuf;
	UINT16		basic_rate = host2amd16(20);
	char		buf[20];

	DBFENTER;

	/* set the 'sta basic rate' to 20 and be done with it. */
	am930mgr_mibsetitem( mgr, MAC_STA_BASIC_RATE, &basic_rate, sizeof(basic_rate));

	/* first find the bss in the known list */
	for (curr = mgr->bsslist; curr != NULL; curr = curr->next)
	{
		if ( memcmp( bssid, curr->bssid, WLAN_BSSID_LEN) == 0 )
			break;
	}
	if ( curr == NULL )
	{
		result = 1;
	}
	else
	{
		if ( WLAN_GET_MGMT_CAP_INFO_ESS(curr->cap_info) )
		{
			/* infrastructure network */
			mibbuf = 1;		/* localmib.networkmode = 1  */
			am930mgr_mibsetitem( mgr, LOC_NETWORK_MODE, &mibbuf, sizeof(mibbuf));

			if ( am930hw_joinbss( mgr->hw, 
									curr->channel, 0, bssid, 
									(wlan_ie_ssid_t*)curr->ssid,
									curr->bcn_int, curr->bss_timestamp,
									curr->sutro_reftime) == AM930HW_SUCCESS )
			{
				mgr->curr_ch = curr->channel;
				mgr->curr_rate = curr->rate;
				mgr->curr_cap_info = curr->cap_info;
				memcpy( mgr->curr_ssid, curr->ssid, 
						WLAN_IEHDR_LEN + WLAN_SSID_MAXLEN);
				p802addr_to_str( buf, bssid);
				WLAN_LOG_DEBUG3(1, "ESS joined, ch=%ld, rate=%ld bssid=%s\n",
					curr->channel, curr->rate, buf);
				memcpy( mgr->curr_bssid, bssid, WLAN_BSSID_LEN );

				mgr->mac->mode = AM930_MACMODE_ESS_STA;
				mgr->mac->state = AM930_MACSTATE_NOAUTH;
			}
			else
			{
				result = 2;
			}
		}
		else
		{
			/* adhoc network */
			if ( am930hw_joinbss( mgr->hw, 
									curr->channel, 0, bssid, 
									(wlan_ie_ssid_t*)curr->ssid,
									curr->bcn_int, curr->bss_timestamp,
									curr->sutro_reftime) == AM930HW_SUCCESS )
			{
				mgr->curr_ch = curr->channel;
				mgr->curr_rate = curr->rate;
				p802addr_to_str( buf, bssid);
				WLAN_LOG_DEBUG3(1, "IBSS joined, ch=%ld, rate=%ld bssid=%s\n",
					curr->channel, curr->rate, buf);
				memcpy( mgr->curr_bssid, bssid, WLAN_BSSID_LEN );

				mgr->mac->mode = AM930_MACMODE_IBSS_STA;
				mgr->mac->state = AM930_MACSTATE_NOAUTH;

			}
			else
			{
				result = 2;
			}
		}
	}
	DBFEXIT;
	return result;
}


/*----------------------------------------------------------------
*	am930mgr_[get|set]mib[item]
*
*	These are pass-through calls used to prevent the llc object
*	from knowing about the hw object.
*
*	returns: zero on success, non-zero on failure (see hw)
----------------------------------------------------------------*/
UINT32 am930mgr_mibget( am930mgr_t *mgr, UINT32 mibsec, UINT32 len, void *buf)
{
	return am930hw_mibget( mgr->hw, mibsec, 0, len, buf);
}

UINT32 am930mgr_mibgetitem( am930mgr_t *mgr, UINT32 itemid, 
							void *buf, UINT32 len)
{
	return am930hw_mibgetitem( mgr->hw, itemid, buf, len);
}

UINT32 am930mgr_mibset( am930mgr_t *mgr, UINT32 mibsec, UINT32 len, void *buf)
{
	return am930hw_mibset( mgr->hw, mibsec, 0, len, buf);
}

UINT32 am930mgr_mibsetitem( am930mgr_t *mgr, UINT32 itemid, 
							void *buf, UINT32 len)
{
	return am930hw_mibsetitem( mgr->hw, itemid, buf, len);
}



/*----------------------------------------------------------------
*	am930mgr_mkchallengetext
*
*	Constructs the challengetext used primarily for shared key
*	authentication.  Uses the given key to initialize the wep
*	PRNG, then generates chbuflen PRNs.
*
*	Arguments:
*		chbuf		challenge text buffer
*		chbuflen	challenge text buffer length
*		key			key used to initialize the wep  PRNG
*		keylen		length of the key
*
*	Returns: 
*		nothing
----------------------------------------------------------------*/
void am930mgr_mkchallengetext( am930mgr_t *mgr, 
	UINT8 *chbuf, UINT chbuflen, UINT8 *key, UINT keylen)
{
	wlan_wepprng_t	prng;
	int				i;

	am930mac_wep_initprng(&prng, key, keylen);
	for ( i = 0; i < chbuflen; i++)
	{
		chbuf[i] = am930mac_wep_nextprn(&prng);
	}
	return;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_auth
*
*	Constructs an auth frame using the arguments provided.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		alg			auth algorithm code (see p80211mgmt.h)
*		seq			1-4
*		status		status code (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_auth( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 alg, UINT16 seq, UINT16 status, UINT8 *challenge)
{
	wlan_pb_t 			*pb;
	wlan_fr_authen_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_AUTHEN_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure.  We assume there's no challenge */
			/*   included.  If challenge is present, we fix up the len later */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_AUTHEN_FR_MAXLEN;
			wlan_mgmt_encode_authen(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_AUTHEN));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set up the alg and seq, they're validated below */
			*(reply.auth_alg) = host2ieee16(alg);
			*(reply.auth_seq) = host2ieee16(seq);

			/* Set up the status and validate the alg and seq */
			if ( alg == WLAN_AUTH_ALG_OPENSYSTEM )
			{
				switch (seq)
				{
					case 1:
						/* open(1), status is always zero */
						WLAN_ASSERT(status == 0); 
						*(reply.status) = 0;
						break;
					case 2:
						*(reply.status) = host2ieee16(status);
						break;
					default:
						WLAN_LOG_ERROR1("OPENSYSTEM invalid seq=%d\n", seq);
						am930shim_pbfree(pb);
						pb = NULL;
						break;
				}
			}
			else if ( alg == WLAN_AUTH_ALG_SHAREDKEY )
			{
				switch (seq)
				{
					case 1:
						/* sharedkey(1), status is always zero */
						WLAN_ASSERT(status == 0);
						*(reply.status) = 0;
						break;
					case 2:
						*(reply.status) = host2ieee16(status);
						break;
					case 3:
						/* sharedkey(3), status is always zero */
						WLAN_ASSERT(status == 0);
						*(reply.status) = 0;
						break;
					case 4:
						*(reply.status) = host2ieee16(status);
						break;
					default:
						WLAN_LOG_ERROR1("SHAREDKEY invalid seq=%d\n", seq);
						am930shim_pbfree(pb);
						pb = NULL;
						break;
				}
			}
			else
			{
				WLAN_LOG_ERROR1("Invalid alg=%d\n", alg);
				am930shim_pbfree(pb);
				pb = NULL;
				
			}

			/* Set up the challenge text */
			if ( pb != NULL )
			{
				if ( alg == WLAN_AUTH_ALG_SHAREDKEY )
				{
					if ( (seq == 2 && status == WLAN_MGMT_STATUS_SUCCESS) || 
						 (seq == 3) )
					{
						if ( challenge != NULL )
						{
							reply.challenge = 
								(wlan_ie_challenge_t*)(reply.buf + reply.len);
							reply.len += WLAN_CHALLENGE_IE_LEN;
							reply.challenge->eid = WLAN_EID_CHALLENGE;
							reply.challenge->len = WLAN_CHALLENGE_LEN;
							memcpy(reply.challenge->challenge, challenge, WLAN_CHALLENGE_LEN);
						}
						else
						{
							WLAN_LOG_ERROR0("SHAREDKEY, seq=2|3, null challenge\n");
							am930shim_pbfree(pb);
							pb = NULL;
						}
					}
				}
			}

			/* Adjust the length fields */
			if ( pb != NULL )
			{
				pb->p80211frmlen = reply.len;
				pb->p80211_payloadlen = reply.len - WLAN_HDR_A3_LEN;
			}
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_assocreq
*
*	Constructs an assocreq frame using the arguments provided.  Arguments
*	are in host order.
*
*	Note: The mkfrm_reassocreq will probably be a 
*			cut-n-paste of this function,
*			changes here will probably have to go there as well.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		status		status code (see p80211mgmt.h)
*		aid			association ID (see p80211mgmt.h)
*		oprates		operational rate set (as an ie) (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_assocreq( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 listen_int,
	wlan_ie_ssid_t *ssid,
	wlan_ie_supp_rates_t *rates)
{
	wlan_pb_t 			*pb;
	wlan_fr_assocreq_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_ASSOCREQ_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure. */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_ASSOCREQ_FR_MAXLEN;
			wlan_mgmt_encode_assocreq(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_ASSOCREQ));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set the status and aid */
			*(reply.cap_info) = host2ieee16(capinfo);
			*(reply.listen_int) = host2ieee16(listen_int);

			/* Copy the SSID */
			reply.ssid = (wlan_ie_ssid_t*)(reply.buf + reply.len);
			reply.len += ssid->len + WLAN_IEHDR_LEN;
			memcpy(reply.ssid, ssid, ssid->len + WLAN_IEHDR_LEN);

			/* Copy the rate set */
			reply.supp_rates = (wlan_ie_supp_rates_t*)(reply.buf + reply.len);
			reply.len += rates->len + WLAN_IEHDR_LEN;
			memcpy(reply.supp_rates, rates, rates->len + WLAN_IEHDR_LEN);
			
			/* Adjust the length fields */
			if ( pb != NULL )
			{
				pb->p80211frmlen = reply.len;
				pb->p80211_payloadlen = reply.len - WLAN_HDR_A3_LEN;
			}
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_assocresp
*
*	Constructs an assocresp frame using the arguments provided.  Arguments
*	are in host order.
*
*	Note: The mkfrm_reassocresp is a cut-n-paste of this function,
*			changes here will probably have to go there as well.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		status		status code (see p80211mgmt.h)
*		aid			association ID (see p80211mgmt.h)
*		oprates		operational rate set (as an ie) (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_assocresp( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 status, UINT16 aid, 
	wlan_ie_supp_rates_t *rates)
{
	wlan_pb_t 			*pb;
	wlan_fr_assocresp_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_ASSOCRESP_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure. */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_ASSOCRESP_FR_MAXLEN;
			wlan_mgmt_encode_assocresp(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_ASSOCRESP));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set the status and aid */
			*(reply.cap_info) = host2ieee16(capinfo);
			*(reply.status) = host2ieee16(status);
			*(reply.aid) = host2ieee16(aid);

			/* Copy the rate set */
			reply.supp_rates = (wlan_ie_supp_rates_t*)(reply.buf + reply.len);
			reply.len += rates->len + WLAN_IEHDR_LEN;
			memcpy(reply.supp_rates, rates, rates->len + WLAN_IEHDR_LEN);
			
			/* Adjust the length fields */
			if ( pb != NULL )
			{
				pb->p80211frmlen = reply.len;
				pb->p80211_payloadlen = reply.len - WLAN_HDR_A3_LEN;
			}
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_deauth
*
*	Constructs a deauth frame using the arguments provided. Arguments
*	are in host order.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		reason		reason code (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_deauth( am930mgr_t *mgr, UINT8 *daddr, UINT16 reason)
{
	wlan_pb_t 			*pb;
	wlan_fr_deauthen_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_DEAUTHEN_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure. */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_DEAUTHEN_FR_MAXLEN;
			wlan_mgmt_encode_deauthen(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_DEAUTHEN));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set up the reason */
			*(reply.reason) = host2ieee16(reason);
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_disassoc
*
*	Constructs a disassoc frame using the arguments provided. Arguments
*	are in host order.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		reason		reason code (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error or allocation failure.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_disassoc( am930mgr_t *mgr, UINT8 *daddr, UINT16 reason)
{
	wlan_pb_t 			*pb;
	wlan_fr_disassoc_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_DISASSOC_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure. */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_DISASSOC_FR_MAXLEN;
			wlan_mgmt_encode_disassoc(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_DISASSOC));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set up the reason */
			*(reply.reason) = host2ieee16(reason);
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_mkfrm_reassocresp
*
*	Constructs a reassocresp frame using the arguments provided.  Arguments
*	are in host order.
*
*	NOTE: This is a cut-n-paste of the mkfrm_assocresp function.
*			Changes made here will probably need to be made in 
*			assocresp as well.
*
*	Arguments:
*		daddr		destination address (6 bytes)
*		status		status code (see p80211mgmt.h)
*		aid			association ID (see p80211mgmt.h)
*		oprates		operational rate set (as an ie) (see p80211mgmt.h)
*
*	Returns: 
*		A ptr to a pb containing the frame or NULL on a severe
*		argument validation error.
----------------------------------------------------------------*/
wlan_pb_t *am930mgr_mkfrm_reassocresp( am930mgr_t *mgr, 
	UINT8 *daddr, UINT16 capinfo, UINT16 status, UINT16 aid, 
	wlan_ie_supp_rates_t *rates)
{
	wlan_pb_t 				*pb;
	wlan_fr_reassocresp_t	reply;
	DBFENTER;
	
	pb = am930shim_pballoc();
	if ( pb != NULL )
	{
		am930shim_pballoc_p80211( pb, WLAN_REASSOCRESP_FR_MAXLEN);
		if (pb->p80211hostbuf == NULL )
		{
			am930shim_pbfree(pb);
			pb = NULL;
		}
		else
		{
			/* finish buffer setup */
			pb->p80211_payloadlen += 4;  /* length fix again, see pballoc_p80211 */

			/* Setup the reply structure. */
			reply.buf = pb->p80211buf;
			reply.len = WLAN_REASSOCRESP_FR_MAXLEN;
			wlan_mgmt_encode_reassocresp(&reply);

			/* Setup the header */
			reply.hdr->a3.fc = host2ieee16(
				WLAN_SET_FC_FTYPE(WLAN_FTYPE_MGMT) | 
				WLAN_SET_FC_FSTYPE(WLAN_FSTYPE_REASSOCRESP));
			memcpy( reply.hdr->a3.a1, daddr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a2, mgr->hw->addr, WLAN_ADDR_LEN);
			memcpy( reply.hdr->a3.a3, mgr->curr_bssid, WLAN_BSSID_LEN);

			/* Set the status and aid */
			*(reply.cap_info) = host2ieee16(capinfo);
			*(reply.status) = host2ieee16(status);
			*(reply.aid) = host2ieee16(aid);

			/* Copy the rate set */
			reply.supp_rates = (wlan_ie_supp_rates_t*)(reply.buf + reply.len);
			reply.len += rates->len + WLAN_IEHDR_LEN;
			memcpy(reply.supp_rates, rates, rates->len + WLAN_IEHDR_LEN);
			
			/* Adjust the length fields */
			if ( pb != NULL )
			{
				pb->p80211frmlen = reply.len;
				pb->p80211_payloadlen = reply.len - WLAN_HDR_A3_LEN;
			}
		}
	}
	DBFEXIT;
	return pb;
}


/*----------------------------------------------------------------
*	am930mgr_reassocreq_rx
*
*	Handle incoming reassociation request frames.
*	NOTE: This is basically a cut-n-paste of the assocreq_rx 
*		function (bad Mark, bad, bad).  If you change anything
*		here it should probably be changed in assocreq.
*
*	Arguments:
*		rxpb	packet buffer containing assocreq frame
*		stats	receive statitistics for assocreq frame
*
*	returns: nothing for now
----------------------------------------------------------------*/
void am930mgr_reassocreq_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	wlan_fr_reassocreq_t	f;
	wlan_stable_item_t		*sta;
	wlan_pb_t				*pb;
	DBFENTER;

	/* decode the frame */
	f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
	f.buf = rxpb->p80211buf;
	wlan_mgmt_decode_reassocreq(&f);

	/* Check for class2 error */
	sta = am930mgr_stable_lookup( mgr, f.hdr->a3.a2);
	if ( sta == NULL || !(sta->sta.info & WLAN_STA_INFO_AUTH) )
	{
    	pb = am930mgr_mkfrm_deauth( mgr, 
				f.hdr->a3.a2, 
				WLAN_MGMT_REASON_CLASS2_NONAUTH);
		if ( pb != NULL ) 
		{
			am930mac_txmac( mgr->mac, pb);
		}
    	return;
	}

	/* Check to make sure we're the SS the station says we are */
	/* assume bad BSSID would be rejected by rx filter */
	if ( memcmp( f.ssid->ssid, mgr->curr_ssid + WLAN_IEHDR_LEN, mgr->curr_ssid[1]) != 0 )
	{
		pb = am930mgr_mkfrm_reassocresp( mgr, 
				f.hdr->a3.a2, 
				am930mac_mk_capinfo(mgr->mac),
				WLAN_MGMT_STATUS_ASSOC_DENIED_UNSPEC,
				0, /* AID */
				(wlan_ie_supp_rates_t*)mgr->mac->oprates );
		if ( pb != NULL )
		{
			am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Capability Info check */
	/*   condition 1: is it not asking for an ESS? */
	/*   condition 2: is it asking for polling? (we don't support it) */
	/*   condition 3: is it asking for WEP and our wep keys aren't set? */
   	if (
		(WLAN_GET_MGMT_CAP_INFO_ESS(ieee2host16(*f.cap_info)) == 0) ||
       	((WLAN_GET_MGMT_CAP_INFO_CFPOLLABLE(ieee2host16(*f.cap_info)) == 1) && 	
	 		(WLAN_GET_MGMT_CAP_INFO_CFPOLLREQ(ieee2host16(*f.cap_info)) == 0)) ||
       	((WLAN_GET_MGMT_CAP_INFO_PRIVACY(ieee2host16(*f.cap_info)) == 1) && 
	 		(mgr->mac->privacy_invoked == 0)) )
	{
       	pb = am930mgr_mkfrm_reassocresp( mgr,
			f.hdr->a3.a2,
			am930mac_mk_capinfo(mgr->mac),
			WLAN_MGMT_STATUS_CAPS_UNSUPPORTED,
			0, /* AID */
			(wlan_ie_supp_rates_t*)mgr->mac->oprates );

		if  ( pb != NULL )
		{
        	am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Basic Rate Check */
	if ( !am930mgr_rates_basicissubset( mgr, 
			(wlan_ie_supp_rates_t*)mgr->mac->oprates, f.supp_rates) )
	{
		/* mismatch in basic rates */
		pb = am930mgr_mkfrm_reassocresp( mgr,
				f.hdr->a3.a2,
				am930mac_mk_capinfo(mgr->mac),
				WLAN_MGMT_STATUS_ASSOC_DENIED_RATES,
				0, /* AID */
				(wlan_ie_supp_rates_t*)mgr->mac->oprates );

		if  ( pb != NULL )
		{
        	am930mac_txmac( mgr->mac, pb);
		}
		return;
	}

	/* Made it! we're now ready to handle success! */
	if ( (sta->sta.listen_int = ieee2host16(*f.listen_int)) )
	{
		sta->sta.info |= WLAN_STA_INFO_PSMODE;
	} 
	
	if ( WLAN_GET_MGMT_CAP_INFO_PRIVACY(ieee2host16(*f.cap_info)) )
	{
		sta->sta.info |= WLAN_STA_INFO_WEPON;
	}
	
	sta->sta.rate = sta->sta.max_rate = am930mgr_rates_max(f.supp_rates);
	sta->sta.info |= WLAN_STA_INFO_ASSOC;
	pb = am930mgr_mkfrm_reassocresp( mgr,
			f.hdr->a3.a2,
			am930mac_mk_capinfo(mgr->mac),
			WLAN_MGMT_STATUS_SUCCESS,
			sta->sta.aid | BIT14 | BIT15,
			(wlan_ie_supp_rates_t*)mgr->mac->oprates );

    if ( pb != NULL )
	{
		am930mac_txmac( mgr->mac, pb);
	}

	DBFEXIT;
    return;
}


/*----------------------------------------------------------------
*	am930mgr_proberesp_rx
*
*	Handles probe response management frames. This function shold
*	only be called when we are in an active scanning state.
*	Otherwise the frame is ignored.
*	When the probe response is received, we check to see if the
*	bss owning the responder is a "known bss". If not, we add this
*	bss to the list.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_proberesp_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	knownbss_t			*bssptr;
	wlan_fr_proberesp_t	f;
	DBFENTER;

	/* decode the frame */
	f.len = rxpb->p80211buflen - WLAN_CRC_LEN;
	f.buf = rxpb->p80211buf;
	wlan_mgmt_decode_proberesp(&f);

	/* search for the bss */
	bssptr = mgr->bsslist;
	while ( bssptr != NULL )
	{
		if ( memcmp(bssptr->bssid, f.hdr->a3.a3, WLAN_BSSID_LEN) == 0 )
			break;
		bssptr = bssptr->next;
	}

	if ( bssptr == NULL )
	{
		/* it's not in the list, add it */
		bssptr = kmalloc( sizeof( knownbss_t ), GFP_ATOMIC );
		if ( bssptr != NULL )  /* handle failure by just dropping the Presp. */
		{
			memset( bssptr, 0, sizeof(knownbss_t) );

			/* save the bss info */
			memcpy( bssptr->bssid, f.hdr->a3.a3, WLAN_BSSID_LEN);
			memcpy( bssptr->bss_timestamp, f.ts, sizeof(UINT64)); 
			bssptr->bcn_int = ieee2host16(*f.bcn_int);
			bssptr->cap_info = ieee2host16(*f.cap_info);
			memcpy( bssptr->ssid, f.ssid, f.ssid->len + WLAN_IEHDR_LEN);
			bssptr->ssid[f.ssid->len + WLAN_IEHDR_LEN] = '\0';
WLAN_LOG_DEBUG1(2, "proberesp: ssid=%s\n", bssptr->ssid + WLAN_IEHDR_LEN);
			bssptr->channel = f.ds_parms->curr_ch;
			bssptr->rate = 20; /*TODO: parse the supp rates IE */
			bssptr->sutro_reftime = amd2host32(stats->local_time); // am930 specific 
			bssptr->ttl = 0;

			bssptr->next = mgr->bsslist;
			mgr->bsslist = bssptr;
		}
	}
	else
	{
		/* the bss is in the list, update ttl */
	}

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_probereq_rx
*
*	Handles probe request management frames.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_probereq_rx( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	DBFENTER;

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_prstatus
*
*	Log a warning message based on the contents of the Status
*	Code field of an 802.11 management frame.  Defines are in
*	p80211mgmt.h and are derived from 802.11-1997 pp 54.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_prstatus(am930mgr_t *mgr, UINT16 status )
{
	switch( status )
	{
		case WLAN_MGMT_STATUS_UNSPEC_FAILURE:
			WLAN_LOG_ERROR0("Status code == Unspecified error.\n");
			break;
		case WLAN_MGMT_STATUS_CAPS_UNSUPPORTED:
			WLAN_LOG_ERROR0("Status code == Can't support all requested capabilities.\n");
			break;
		case WLAN_MGMT_STATUS_REASSOC_NO_ASSOC:
			WLAN_LOG_ERROR0("Status code == Reassoc denied, can't confirm original Assoc.\n");
			break;
		case WLAN_MGMT_STATUS_ASSOC_DENIED_UNSPEC:
			WLAN_LOG_ERROR0("Status code == Assoc denied, because we don't feel like it.\n");
			break;
		case WLAN_MGMT_STATUS_UNSUPPORTED_AUTHALG:
			WLAN_LOG_ERROR0("Status code == Peer doesn't support authen algorithm.\n");
			break;
		case WLAN_MGMT_STATUS_RX_AUTH_NOSEQ:
			WLAN_LOG_ERROR0("Status code == Authen frame received out of sequence.\n");
			break;
		case WLAN_MGMT_STATUS_CHALLENGE_FAIL:
			WLAN_LOG_ERROR0("Status code == Authen rejected, challenge failure.\n");
			break;
		case WLAN_MGMT_STATUS_AUTH_TIMEOUT:
			WLAN_LOG_ERROR0("Status code == Authen rejected, timeout waiting for next frame.\n");
			break;
		case WLAN_MGMT_STATUS_ASSOC_DENIED_BUSY:
			WLAN_LOG_ERROR0("Status code == Assoc denied, AP too busy.\n");
			break;
		case WLAN_MGMT_STATUS_ASSOC_DENIED_RATES:
			WLAN_LOG_ERROR0("Status code == Assoc denied, we haven't enough basic rates.\n");
			break;
		default:
			WLAN_LOG_ERROR1("Unknown status code %d.\n", status);
			break;
	}
}


/*----------------------------------------------------------------
*	am930mgr_rates_max
*
*	Returns the maximum rate (basic or not) in information element r.
*
*	Arguments:
*		r		an information element containing rates
*
*	returns: 
*		the value of the maximum rate.
----------------------------------------------------------------*/
UINT8 am930mgr_rates_max(wlan_ie_supp_rates_t *r)
{
	int		i;
	UINT8	maxrate = 0;

	DBFENTER;
	for ( i=0; i < r->len; i++)
	{
		maxrate = maxrate > r->rates[i] ? maxrate : r->rates[i];
	}
	DBFEXIT;
	return maxrate;
}


/*----------------------------------------------------------------
*	am930mgr_rates_basicissubset
*
*	Checks if the basic rates in information element r2 are
*	a subset (or the same as) the rates in r1.
*
*	Arguments:
*		r1		an information element containing the rates we're
*				checking against.
*		r2		an information element containing the rates we're
*				checking.
*
*	returns: 
*		true if the basic rates in r2 are a subset of the basic
*		rates in r1.
----------------------------------------------------------------*/
int am930mgr_rates_basicissubset( am930mgr_t *mgr, 
	wlan_ie_supp_rates_t *r1, wlan_ie_supp_rates_t *r2)
{
	int	i;
	int	j;
	int	badrates = 0;
	int foundrate = 0;
	DBFENTER;

	/* for each rate in r2 */
	for ( i=0; i < r2->len; i++)
	{
		if ( WLAN_MGMT_ISBASICRATE(r2->rates[i]) )
		{
			/* find matching rate in r1 */
			foundrate = 0;
			for  ( j=0; j < r1->len; j++)
			{
				if ( WLAN_MGMT_GET_RATE(r2->rates[i]) == 
					 WLAN_MGMT_GET_RATE(r1->rates[j]) )
				{
					foundrate = 1;
					if (!WLAN_MGMT_ISBASICRATE(r1->rates[j]))
					{
						badrates = 1;
						break;
					}
				}
			}
				
			if ( !foundrate )
			{
				badrates = 1;
				break;
			}
		}
   	} 

	DBFEXIT;
	return !badrates;
}


/*----------------------------------------------------------------
*	am930mgr_rxmgmt
*
*	Entry point for the reception and handling of 802.11 management
*	frames.	Makes a determination of the frame type and then calls
*	the appropriate function.
*
*	returns: zero on success, non-zero on failure
----------------------------------------------------------------*/
void am930mgr_rxmgmt( am930mgr_t *mgr, wlan_pb_t *rxpb, am930rxstats_t *stats)
{
	DBFENTER;

	switch( WLAN_GET_FC_FSTYPE(ieee2host16(rxpb->p80211_hdr->a3.fc)) )
	{
		case WLAN_FSTYPE_ASSOCREQ:
			WLAN_LOG_DEBUG0(3, "rx assocreq\n");
			am930mgr_assocreq_rx( mgr, rxpb, stats);
			break;

		case WLAN_FSTYPE_ASSOCRESP:
			WLAN_LOG_DEBUG0(3, "rx assocresp\n");
			am930mgr_assocresp_rx( mgr, rxpb, stats);
			break;
		case WLAN_FSTYPE_REASSOCREQ:
			WLAN_LOG_DEBUG0(3, "rx reassocreq\n");
			am930mgr_reassocreq_rx( mgr, rxpb, stats);
			break;

		case WLAN_FSTYPE_REASSOCRESP:
			WLAN_LOG_DEBUG0(3, "rx reassocresp\n");
			break;

		case WLAN_FSTYPE_PROBEREQ:
			WLAN_LOG_DEBUG0(3, "rx probereq\n");
			/* doesn't do anything right now
			am930mgr_probereq_rx( mgr, rxpb, stats);
			*/
			break;

		case WLAN_FSTYPE_PROBERESP:
			WLAN_LOG_DEBUG0(3, "rx proberesp\n");
			am930mgr_proberesp_rx( mgr, rxpb, stats);
			break;

		case WLAN_FSTYPE_BEACON:
			WLAN_LOG_DEBUG0(3, "rx beacon\n");
			/* doesn't do anything right now
			am930mgr_beacon_rx( mgr, rxpb, stats);
			*/
			break;

		case WLAN_FSTYPE_ATIM:
			WLAN_LOG_DEBUG0(3, "rx atim\n");
			break;

		case WLAN_FSTYPE_DISASSOC:
			WLAN_LOG_DEBUG0(3, "rx disassoc\n");
			break;

		case WLAN_FSTYPE_AUTHEN:
			WLAN_LOG_DEBUG0(3, "rx authen\n");
			am930mgr_authen_rx( mgr, rxpb, stats);
			break;

		case WLAN_FSTYPE_DEAUTHEN:
			WLAN_LOG_DEBUG0(3, "rx deauthen\n");
			break;

		default:
			WLAN_LOG_DEBUG0(3, "rx unknown mgmt\n");
	}

	am930shim_pbfree( rxpb );
	kfree_s(stats, sizeof(am930rxstats_t));

	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_scanbegin
*
*	Set up scanning managed by the mgr object. This function
*	begins the scanning of a sequence of channels. We call the
*	hw scan method registering the callback so we can setup the
*	scan the of the next channel. We pick up scan results via the
*	rx_mgmt method via probe response and beacon frames.
*	scantype:	0: active	1: passive
*	bsstype:	BIT0: independent	BIT1: infrastructure
*	timech is in Kus
*
*	returns: zero on success, 
*			1 - already scanning
*			2 - scan command failed
----------------------------------------------------------------*/
UINT32 am930mgr_scanbegin( am930mgr_t *mgr, UINT scantype, UINT bsstype,
						UINT8 *bssid, UINT startch, UINT endch, 
						UINT timech, wlan_ie_ssid_t *ie_ssid)
{
	UINT32 result = 0;
	UINT32 cntl = 0;
	knownbss_t	*bss;
	DBFENTER;

	if ( mgr->scanstate == MGR_ISSCANNING )
	{
		WLAN_LOG_WARNING0("mgr_scan called when already scanning!\n");
		result = 1;
	}
	else
	{
		/* clear the bsslist  */
		while( mgr->bsslist != NULL)
		{
			bss = mgr->bsslist;
			mgr->bsslist = mgr->bsslist->next;
			kfree_s( bss, sizeof( knownbss_t )); 
		}

		mgr->scanstate = MGR_ISSCANNING;
		mgr->scantype = scantype;
		mgr->scanch = startch;
		mgr->scanendch = endch;
		mgr->scantimech = timech;
		memcpy( mgr->scanbssid, bssid, sizeof(bssid));
		mgr->scanbsstype = bsstype;
		memcpy( mgr->scanssid, (UINT8 *)ie_ssid, ie_ssid->len + WLAN_IEHDR_LEN);
		if ( scantype == 0 )	cntl |= AM930HWSCAN_ACTIVE;
		else 					cntl |= AM930HWSCAN_PASSIVE;
		if ( bsstype & BIT0 )	cntl |= AM930HWSCAN_IBSS;
		if ( bsstype & BIT1 )	cntl |= AM930HWSCAN_ESS;
		if ( am930hw_scan( mgr->hw, cntl, bssid, startch, timech, 
							ie_ssid, am930mgr_scancomplete, mgr )
		     == AM930HW_FAILURE )
		{
			WLAN_LOG_ERROR0("Initial scan command failed!\n");
			result = 2;
			mgr->scanstate = 0;
		}
		WLAN_LOG_INFO1("scanning channel %d\n", mgr->scanch);

		/* Now the mgr object just waits for mgmt frames and the scan
			complete call to signal the completion of the scan of each
			channel
		*/
	}
	DBFEXIT;
	return result;
}

/*----------------------------------------------------------------
*	am930mgr_scancomplete
*
*	Private method that handles the scan complete interrupt (when
*	scanning is begun by mgr). It checks to see if we are finished 
*	with all of the channels to be scanned. If so, then we set the
*	mgr object to a non-scanning state. If not, then we send a new
*	scan command to the hw.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_scancomplete( void *argp )
{
	am930mgr_t	*mgr = (am930mgr_t *)argp;
	UINT32 cntl = 0;
	DBFENTER;

	if ( mgr->scanstate != MGR_ISSCANNING )
	{
		WLAN_LOG_WARNING0("scancomplet called when not in scanning state!\n");
	}
	else
	{
		if ( mgr->scanch >= mgr->scanendch )
		{
			/* we're done, clear scanning state and call llc */
			mgr->scanstate = 0;
			am930llc_cmdcomplete(mgr->mac->llc, 0);

		}
		else
		{
			/* set up for the next channel */
			if ( mgr->scantype == 0 )	cntl |= AM930HWSCAN_ACTIVE;
			else 						cntl |= AM930HWSCAN_PASSIVE;
			if ( mgr->scanbsstype & BIT0 )	cntl |= AM930HWSCAN_IBSS;
			if ( mgr->scanbsstype & BIT1 )	cntl |= AM930HWSCAN_ESS;

			mgr->scanch++;

			if ( am930hw_scan( mgr->hw, cntl, mgr->scanbssid, mgr->scanch, 
					   mgr->scantimech, 
					   (wlan_ie_ssid_t *)mgr->scanssid,
					   am930mgr_scancomplete, mgr ) 
				== AM930HW_FAILURE )
			{
				WLAN_LOG_WARNING0("2nd or later scan command failed!\n");
				mgr->scanstate = 0;
			}
			WLAN_LOG_DEBUG1(1,"scanning channel %d\n", mgr->scanch);
		}
	}
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_scanstop
*
*	Called from elsewhere to clear our scanning state, usually 
*	due to a timeout.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_scanstop(am930mgr_t *mgr)
{
	DBFENTER;
	/* if scanning */
	if ( mgr->scanstate == MGR_ISSCANNING ) {
		/* Issue a fw.resume command to clear the scan */
		am930hw_resumefromscan(mgr->hw);
		mgr->scanstate = 0;
		mgr->scanch = mgr->scanendch = 0;
		/* empty the scan results? nah, we'll keep em for now */
	}
	/* else do nothing */
	DBFEXIT;
	return;
}


/*----------------------------------------------------------------
*	am930mgr_setwep
*
*	Entry point from higher layer to establish wep keys and 
*	usage settings.
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_setwep( am930mgr_t *mgr, 
	UINT privacy_invoked, 
	UINT exclude_unencrypted,
	UINT32 wep_defkeyid,
	UINT8 wep_key[WLAN_WEP_NKEYS][WLAN_WEP_KEYLEN])
{
	int		i;
	UINT8	mibval8;

	/* copy the values to the mac element */
	mgr->mac->privacy_invoked = privacy_invoked;
	mgr->mac->exclude_unencrypted = exclude_unencrypted;
	mgr->mac->wep_defkeyid = wep_defkeyid;
	WLAN_LOG_DEBUG3(2, 
		"privacy_invoked=%d exclude_unencrypted=%d wep_defkeyid=%ld\n", 
		privacy_invoked, exclude_unencrypted, wep_defkeyid);

	/* lets do this in a loop, just in case someone decides padding is a good idea */
	for ( i = 0; i < WLAN_WEP_NKEYS; i++)
	{
		if ( wep_key != NULL ) {
			memcpy( mgr->mac->wep_key[i], wep_key[i], WLAN_WEP_KEYLEN);
			WLAN_HEX_DUMP(2, "wepkey", wep_key[i], WLAN_WEP_KEYLEN);
		} else {
			memset( mgr->mac->wep_key[i], 0, WLAN_WEP_KEYLEN);
		}
	}

	/* now, set up the firmware */
	mibval8 = mgr->mac->exclude_unencrypted ? 
				SUMIB_WEP_REQUIRED_YES : SUMIB_WEP_REQUIRED_NO;

	am930mgr_mibsetitem(mgr, MGMT_WEP_REQUIRED, &mibval8, MI_SZ(MGMT_WEP_REQUIRED));

	return;
}


/*----------------------------------------------------------------
*	am930mgr_stable_alloc_aid
*
*	Finds the first free aid and allocates it for use by the given
*	sta.
*
*	returns: on success, a non-zero aid
*			on failure, zero
----------------------------------------------------------------*/
UINT16 am930mgr_stable_alloc_aid( am930mgr_t *mgr, wlan_stable_item_t *sta)
{
	UINT16			result;
	wlan_flags_t	flags;

	spin_lock_irqsave( &mgr->stable_lock, flags);

	for ( result = 1; result < WLAN_AIDTABLE_LEN; result++)
	{
		if ( mgr->aidtable[result] == NULL )
			break;
	}

	if  ( result >= WLAN_AIDTABLE_LEN )
	{
		result = 0;
	}
	else
	{
		mgr->aidtable[result] = sta;
		sta->sta.aid = result;
	}
	spin_unlock_irqrestore( &mgr->stable_lock, flags);
	return result;
}


/*----------------------------------------------------------------
*	am930mgr_stable_insert
*
*	Inserts a given entry into the stable
*
*	returns: nothing
----------------------------------------------------------------*/
void am930mgr_stable_insert( am930mgr_t *mgr, wlan_stable_item_t *sta)
{
	int 			index;
	wlan_flags_t	flags;

	spin_lock_irqsave( &mgr->stable_lock, flags);

	index = WLAN_MACADDR_HASH8(sta->sta.addr) & WLAN_STABLE_MASK;

	if ( mgr->stable[index] != NULL )
	{
		sta->next = mgr->stable[index];
		sta->next->prev = sta;

	}
	sta->prev = NULL;
	mgr->stable[index] = sta;

	spin_unlock_irqrestore( &mgr->stable_lock, flags);
}


/*----------------------------------------------------------------
*	am930mgr_stable_lookup
*
*	Finds the sta entry for a given address
*
*	returns: if found, a ptr to the sta,
			 NULL otherwise.
----------------------------------------------------------------*/
wlan_stable_item_t *am930mgr_stable_lookup( am930mgr_t *mgr, UINT8 *addr )
{
	UINT32				index;
	wlan_stable_item_t	*result = NULL;
	wlan_flags_t		flags;

	spin_lock_irqsave( &mgr->stable_lock, flags);
	
	index = WLAN_MACADDR_HASH8(addr) & WLAN_STABLE_MASK;

	for (result = mgr->stable[index]; result != NULL; result = result->next)
	{
		if (memcmp(addr, result->sta.addr, WLAN_ADDR_LEN) == 0 )
			break;
	}

	spin_unlock_irqrestore( &mgr->stable_lock, flags);

	return result;
}


