/* p80211mgmt.h: Defines the macros, types, and functions for dealing
*				with 802.11 managment frames.
*	--------------------------------------------------------------------
*
*	Written 1997-99 by Mark Mathews   mark@absoval.com
*
*	Copyright (c) 1999 AbsoluteValue Software, Inc.
*	http://www.absoval.com
*
*	This software mayy be used and distributed according to the terms
*	of the GNU Public License, incoporated herein by reference.
*
*	The author may be reached as mark@absoval.com, or C/O AbsoluteValue
*	Software Inc., P.O. Box 941149, Maitland, FL, 32794-1149
*
*	Description:
*
*	This file declares the constants and types used in the interface
*	between a wlan driver and the user mode utilities.
*
*	Note: Constant values are always in HOST byte order.  To assign
*			values to multi-byte fields they _must_ be converted to
*			ieee byte order.  To retrieve multi-byte values from incoming
*			frames, they must be converted to host order.
*
*	Note: The len member of the frame structure does NOT!!! include
*			the MAC CRC.  Therefore, the len field on rx'd frames should
*			have 4 subtracted from it.
*
*	All functions declared here are implemented in p80211.c
*
*	The types, macros, and functions defined here are primarily
*	used for encoding and decoding management frames.  They are
*	designed to follow these patterns of use:
*
*	DECODE:
*	1) a frame of length len is received into buffer b
*	2) using the hdr structure and macros, we determine the type
*	3) an appropriate mgmt frame structure, mf, is allocated and zeroed
*	4) mf.hdr = b
*	   mf.buf = b
*	   mf.len = len
*	5) call mgmt_decode( mf )
*	6) the frame field pointers in mf are now set.  Note that any 
*	   multi-byte frame field values accessed using the frame field 
*	   pointers are in ieee byte order and will have to be converted
*	   to host order.
*
*	ENCODE:
*	1) Library client allocates buffer space for maximum length
*	   frame of the desired type
*	2) Library client allocates a mgmt frame structure, called mf, 
*	   of the desired type
*	3) Set the following:
*	   mf.type = <desired type>
*	   mf.buf = <allocated buffer address>
*	4) call mgmt_encode( mf )
*	5) all of the fixed field pointers and fixed length information element
*	   pointers in mf are now set to their respective locations in the 
*	   allocated space (fortunately, all variable length information elements
*	   fall at the end of their respective frames).
*	5a) The length field is set to include the last of the fixed and fixed
*	    length fields.  It may have to be updated for optional or variable 
*		length information elements.
*	6) Optional and variable length information elements are special cases 
*	   and must be handled individually by the client code.
*	--------------------------------------------------------------------
*/

#ifndef _P80211MGMT_H
#define _P80211MGMT_H

#ifndef  _WLAN_COMPAT_H
#include <wlan_compat.h>
#endif

#ifndef  _P80211HDR_H
#include <p80211hdr.h>
#endif


/*=============================================================*/
/*--- Constants & Macros --------------------------------------*/
/*=============================================================*/

/*== Information Element IDs ====================*/
#define WLAN_EID_SSID			0
#define WLAN_EID_SUPP_RATES		1
#define WLAN_EID_FH_PARMS		2
#define WLAN_EID_DS_PARMS		3
#define WLAN_EID_CF_PARMS		4
#define WLAN_EID_TIM			5
#define WLAN_EID_IBSS_PARMS		6
/*-- values 7-15 reserved --*/
#define WLAN_EID_CHALLENGE		16
/*-- values 17-31 reserved for challenge text extension --*/
/*-- values 32-255 reserved --*/

/*== Reason Codes ===============================*/
#define WLAN_MGMT_REASON_RSVD					0	
#define WLAN_MGMT_REASON_UNSPEC					1
#define WLAN_MGMT_REASON_PRIOR_AUTH_INVALID		2
#define WLAN_MGMT_REASON_DEAUTH_LEAVING			3
#define WLAN_MGMT_REASON_DISASSOC_INACTIVE		4
#define WLAN_MGMT_REASON_DISASSOC_AP_BUSY		5
#define WLAN_MGMT_REASON_CLASS2_NONAUTH			6
#define WLAN_MGMT_REASON_CLASS3_NONASSOC		7
#define WLAN_MGMT_REASON_DISASSOC_STA_HASLEFT	8
#define WLAN_MGMT_REASON_CANT_ASSOC_NONAUTH		9

/*== Status Codes ===============================*/
#define WLAN_MGMT_STATUS_SUCCESS				0
#define WLAN_MGMT_STATUS_UNSPEC_FAILURE			1
#define WLAN_MGMT_STATUS_CAPS_UNSUPPORTED		10
#define WLAN_MGMT_STATUS_REASSOC_NO_ASSOC		11
#define WLAN_MGMT_STATUS_ASSOC_DENIED_UNSPEC	12
#define WLAN_MGMT_STATUS_UNSUPPORTED_AUTHALG	13
#define WLAN_MGMT_STATUS_RX_AUTH_NOSEQ			14
#define WLAN_MGMT_STATUS_CHALLENGE_FAIL			15
#define WLAN_MGMT_STATUS_AUTH_TIMEOUT			16
#define WLAN_MGMT_STATUS_ASSOC_DENIED_BUSY		17
#define WLAN_MGMT_STATUS_ASSOC_DENIED_RATES		18

/*== Capability Field ===========================*/
#define WLAN_GET_MGMT_CAP_INFO_ESS(n)			((n) & BIT0)
#define WLAN_GET_MGMT_CAP_INFO_IBSS(n)			(((n) & BIT1) >> 1)
#define WLAN_GET_MGMT_CAP_INFO_CFPOLLABLE(n)	(((n) & BIT2) >> 2)
#define WLAN_GET_MGMT_CAP_INFO_CFPOLLREQ(n)		(((n) & BIT3) >> 3)
#define WLAN_GET_MGMT_CAP_INFO_PRIVACY(n)		(((n) & BIT4) >> 4)

#define WLAN_SET_MGMT_CAP_INFO_ESS(n)			(n)
#define WLAN_SET_MGMT_CAP_INFO_IBSS(n)			((n) << 1)
#define WLAN_SET_MGMT_CAP_INFO_CFPOLLABLE(n)	((n) << 2)
#define WLAN_SET_MGMT_CAP_INFO_CFPOLLREQ(n)		((n) << 3)
#define WLAN_SET_MGMT_CAP_INFO_PRIVACY(n)		((n) << 4)

/*== Rates ======================================*/
#define WLAN_MGMT_ISBASICRATE(b)	((b) & BIT7)
#define WLAN_MGMT_GET_RATE(b)		((b) & ~BIT7)

/*== Auth Algorithm Field ===========================*/
#define WLAN_AUTH_ALG_OPENSYSTEM				0
#define WLAN_AUTH_ALG_SHAREDKEY					1

/*== Management Frame Field Offsets =============*/
/* Note: Not all fields are listed because of variable lengths,   */
/*       see the code in p80211.c to see how we search for fields */
/* Note: These offsets are from the start of the frame data       */

#define WLAN_BEACON_OFF_TS					0
#define WLAN_BEACON_OFF_BCN_INT				8
#define WLAN_BEACON_OFF_CAPINFO				10
#define WLAN_BEACON_OFF_SSID				12

#define WLAN_DISASSOC_OFF_REASON			0

#define WLAN_ASSOCREQ_OFF_CAP_INFO			0
#define WLAN_ASSOCREQ_OFF_LISTEN_INT		2
#define WLAN_ASSOCREQ_OFF_SSID				4

#define WLAN_ASSOCRESP_OFF_CAP_INFO			0
#define WLAN_ASSOCRESP_OFF_STATUS			2
#define WLAN_ASSOCRESP_OFF_AID				4
#define WLAN_ASSOCRESP_OFF_SUPP_RATES		6

#define WLAN_REASSOCREQ_OFF_CAP_INFO		0
#define WLAN_REASSOCREQ_OFF_LISTEN_INT		2
#define WLAN_REASSOCREQ_OFF_CURR_AP			4
#define WLAN_REASSOCREQ_OFF_SSID			10

#define WLAN_REASSOCRESP_OFF_CAP_INFO		0
#define WLAN_REASSOCRESP_OFF_STATUS			2
#define WLAN_REASSOCRESP_OFF_AID			4
#define WLAN_REASSOCRESP_OFF_SUPP_RATES		6

#define WLAN_PROBEREQ_OFF_SSID				0

#define WLAN_PROBERESP_OFF_TS				0
#define WLAN_PROBERESP_OFF_BCN_INT			8
#define WLAN_PROBERESP_OFF_CAP_INFO			10
#define WLAN_PROBERESP_OFF_SSID				12

#define WLAN_AUTHEN_OFF_AUTH_ALG			0
#define WLAN_AUTHEN_OFF_AUTH_SEQ			2
#define WLAN_AUTHEN_OFF_STATUS				4
#define WLAN_AUTHEN_OFF_CHALLENGE			6

#define WLAN_DEAUTHEN_OFF_REASON			0

/*=============================================================*/
/*--- Types and Structures ------------------------------------*/
/*=============================================================*/

/*== Information Element Types ====================*/
/* prototype structure, all IEs start with these members */
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie
{
	UINT8	eid			__WLAN_ATTRIB_PACK__;
	UINT8	len			__WLAN_ATTRIB_PACK__;
} wlan_ie_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- Service Set Identity (SSID)  -----------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_ssid
{
	UINT8	eid						__WLAN_ATTRIB_PACK__;
	UINT8	len						__WLAN_ATTRIB_PACK__;
	UINT8	ssid[WLAN_MIN_ARRAY]	__WLAN_ATTRIB_PACK__;  /* may be zero, ptrs may overlap */
} wlan_ie_ssid_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- Supported Rates  -----------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_supp_rates
{
	UINT8	eid						__WLAN_ATTRIB_PACK__;
	UINT8	len						__WLAN_ATTRIB_PACK__;
	UINT8	rates[WLAN_MIN_ARRAY]	__WLAN_ATTRIB_PACK__;  /* had better be at LEAST one! */
} wlan_ie_supp_rates_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- FH Parameter Set  ----------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_fh_parms
{
	UINT8	eid			__WLAN_ATTRIB_PACK__;
	UINT8	len			__WLAN_ATTRIB_PACK__;
	UINT16	dwell		__WLAN_ATTRIB_PACK__;
	UINT8	hopset		__WLAN_ATTRIB_PACK__;
	UINT8	hoppattern	__WLAN_ATTRIB_PACK__;
	UINT8	hopindex	__WLAN_ATTRIB_PACK__;
} wlan_ie_fh_parms_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- DS Parameter Set  ----------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_ds_parms
{
	UINT8	eid			__WLAN_ATTRIB_PACK__;
	UINT8	len			__WLAN_ATTRIB_PACK__;
	UINT8	curr_ch		__WLAN_ATTRIB_PACK__;
} wlan_ie_ds_parms_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- CF Parameter Set  ----------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_cf_parms
{
	UINT8	eid			__WLAN_ATTRIB_PACK__;
	UINT8	len			__WLAN_ATTRIB_PACK__;
	UINT8	cfp_cnt		__WLAN_ATTRIB_PACK__;
	UINT8	cfp_period	__WLAN_ATTRIB_PACK__;
	UINT16	cfp_maxdur	__WLAN_ATTRIB_PACK__;
	UINT16	cfp_durremaining	__WLAN_ATTRIB_PACK__;
} wlan_ie_cf_parms_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- TIM ------------------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_tim
{
	UINT8	eid						__WLAN_ATTRIB_PACK__;
	UINT8	len						__WLAN_ATTRIB_PACK__;
	UINT8	dtim_cnt				__WLAN_ATTRIB_PACK__;
	UINT8	dtim_period				__WLAN_ATTRIB_PACK__;
	UINT8	bitmap_ctl				__WLAN_ATTRIB_PACK__;
	UINT8	virt_bm[WLAN_MIN_ARRAY]	__WLAN_ATTRIB_PACK__;
} wlan_ie_tim_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- IBSS Parameter Set ---------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_ibss_parms
{
	UINT8	eid			__WLAN_ATTRIB_PACK__;
	UINT8	len			__WLAN_ATTRIB_PACK__;
	UINT16	atim_win	__WLAN_ATTRIB_PACK__;
} wlan_ie_ibss_parms_t;
__WLAN_PRAGMA_PACKDFLT__

/*-- Challenge Text  ------------------------------*/
__WLAN_PRAGMA_PACK1__
typedef struct wlan_ie_challenge
{
	UINT8	eid							__WLAN_ATTRIB_PACK__;
	UINT8	len							__WLAN_ATTRIB_PACK__;
	UINT8	challenge[WLAN_MIN_ARRAY]	__WLAN_ATTRIB_PACK__;
} wlan_ie_challenge_t;
__WLAN_PRAGMA_PACKDFLT__


/*=================================================*/
/*== Frame Types ==================================*/
/* prototype structure, all mgmt frame types will start with these members */
typedef struct wlan_fr_mgmt
{
	UINT16					type;
	UINT16					len;	/* DOES NOT include CRC !!!!*/
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	/*-- info elements ----------*/
} wlan_fr_mgmt_t;

/*-- Beacon ---------------------------------------*/
typedef struct wlan_fr_beacon
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT64					*ts;
	UINT16					*bcn_int;
	UINT16					*cap_info;
	/*-- info elements ----------*/
	wlan_ie_ssid_t			*ssid;
	wlan_ie_supp_rates_t	*supp_rates;
	wlan_ie_fh_parms_t		*fh_parms;
	wlan_ie_ds_parms_t		*ds_parms;
	wlan_ie_cf_parms_t		*cf_parms;
	wlan_ie_ibss_parms_t	*ibss_parms;
	wlan_ie_tim_t			*tim;

} wlan_fr_beacon_t;


/*-- IBSS ATIM ------------------------------------*/
typedef struct wlan_fr_ibssatim
{
	UINT16					type;
	UINT16					len;
	UINT8*					buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */

	/*-- fixed fields -----------*/
	/*-- info elements ----------*/

	/* this frame type has a null body */

} wlan_fr_ibssatim_t;

/*-- Disassociation -------------------------------*/
typedef struct wlan_fr_disassoc
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*reason;

	/*-- info elements ----------*/

} wlan_fr_disassoc_t;

/*-- Association Request --------------------------*/
typedef struct wlan_fr_assocreq
{
	UINT16					type;
	UINT16					len;
	UINT8*					buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*cap_info;
	UINT16					*listen_int;
	/*-- info elements ----------*/
	wlan_ie_ssid_t			*ssid;
	wlan_ie_supp_rates_t	*supp_rates;

} wlan_fr_assocreq_t;

/*-- Association Response -------------------------*/
typedef struct wlan_fr_assocresp
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*cap_info;
	UINT16					*status;
	UINT16					*aid;
	/*-- info elements ----------*/
	wlan_ie_supp_rates_t	*supp_rates;

} wlan_fr_assocresp_t;

/*-- Reassociation Request ------------------------*/
typedef struct wlan_fr_reassocreq
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*cap_info;
	UINT16					*listen_int;
	UINT8					*curr_ap;
	/*-- info elements ----------*/
	wlan_ie_ssid_t			*ssid;
	wlan_ie_supp_rates_t	*supp_rates;

} wlan_fr_reassocreq_t;

/*-- Reassociation Response -----------------------*/
typedef struct wlan_fr_reassocresp
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*cap_info;
	UINT16					*status;
	UINT16					*aid;
	/*-- info elements ----------*/
	wlan_ie_supp_rates_t	*supp_rates;

} wlan_fr_reassocresp_t;

/*-- Probe Request --------------------------------*/
typedef struct wlan_fr_probereq
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	/*-- info elements ----------*/
	wlan_ie_ssid_t			*ssid;
	wlan_ie_supp_rates_t	*supp_rates;

} wlan_fr_probereq_t;

/*-- Probe Response -------------------------------*/
typedef struct wlan_fr_proberesp
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT64					*ts;
	UINT16					*bcn_int;
	UINT16					*cap_info;
	/*-- info elements ----------*/
	wlan_ie_ssid_t			*ssid;
	wlan_ie_supp_rates_t	*supp_rates;
	wlan_ie_fh_parms_t		*fh_parms;
	wlan_ie_ds_parms_t		*ds_parms;
	wlan_ie_cf_parms_t		*cf_parms;
	wlan_ie_ibss_parms_t	*ibss_parms;
} wlan_fr_proberesp_t;

/*-- Authentication -------------------------------*/
typedef struct wlan_fr_authen
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*auth_alg;
	UINT16					*auth_seq;
	UINT16					*status;
	/*-- info elements ----------*/
	wlan_ie_challenge_t		*challenge;

} wlan_fr_authen_t;

/*-- Deauthenication -----------------------------*/
typedef struct wlan_fr_deauthen
{
	UINT16					type;
	UINT16					len;
	UINT8					*buf;
	p80211_hdr_t			*hdr;
	void					*priv;	/* used for target specific data, skb in Linux */
	/*-- fixed fields -----------*/
	UINT16					*reason;

	/*-- info elements ----------*/

} wlan_fr_deauthen_t;


void wlan_mgmt_encode_beacon( wlan_fr_beacon_t  *f );
void wlan_mgmt_decode_beacon( wlan_fr_beacon_t  *f );
void wlan_mgmt_encode_disassoc( wlan_fr_disassoc_t  *f );
void wlan_mgmt_decode_disassoc( wlan_fr_disassoc_t  *f );
void wlan_mgmt_encode_assocreq( wlan_fr_assocreq_t  *f );
void wlan_mgmt_decode_assocreq( wlan_fr_assocreq_t  *f );
void wlan_mgmt_encode_assocresp( wlan_fr_assocresp_t  *f );
void wlan_mgmt_decode_assocresp( wlan_fr_assocresp_t  *f );
void wlan_mgmt_encode_reassocreq( wlan_fr_reassocreq_t  *f );
void wlan_mgmt_decode_reassocreq( wlan_fr_reassocreq_t  *f );
void wlan_mgmt_encode_reassocresp( wlan_fr_reassocresp_t  *f );
void wlan_mgmt_decode_reassocresp( wlan_fr_reassocresp_t  *f );
void wlan_mgmt_encode_probereq( wlan_fr_probereq_t  *f );
void wlan_mgmt_decode_probereq( wlan_fr_probereq_t  *f );
void wlan_mgmt_encode_proberesp( wlan_fr_proberesp_t  *f );
void wlan_mgmt_decode_proberesp( wlan_fr_proberesp_t  *f );
void wlan_mgmt_encode_authen( wlan_fr_authen_t  *f );
void wlan_mgmt_decode_authen( wlan_fr_authen_t  *f );
void wlan_mgmt_encode_deauthen( wlan_fr_deauthen_t  *f );
void wlan_mgmt_decode_deauthen( wlan_fr_deauthen_t  *f );


#endif /* _P80211MGMT_H */
